﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/hio/atk_HioAsyncChannel.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/viewer/detail/hio/atk_HioProtocol.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_COM_DEBUG
#endif

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

//----------------------------------------------------------
void
HioAsyncChannel::RegisterMessageHandler(HioMessageHandler& handler) NN_NOEXCEPT
{
    m_MessageHandlerRefs.push_back(handler.GetReference());
}

//----------------------------------------------------------
void
HioAsyncChannel::UnregisterMessageHandler(HioMessageHandler& handler) NN_NOEXCEPT
{
    // HACK: 多重 erase しないようにするための保護処理
    //if(handler.GetReference().node.GetNext() == NULL &&
    //    handler.GetReference().node.GetPrev() == NULL)
    //{
    //    return;
    //}

    m_MessageHandlerRefs.erase(m_MessageHandlerRefs.iterator_to(handler.GetReference()));
}

//----------------------------------------------------------
HioResult
HioAsyncChannel::Update() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return HioResult(HioResultType_TargetNotInitialized);
    }

    if(!GetStream().IsAvailable())
    {
        return HioResult(HioResultType_TargetRecieveError);
    }

    // メッセージヘッダを読み込みます。
    if(GetStream().CanReadNewPacket())
    {
        HioResult result = GetStream().TryReadHeader();

        if(!result.IsTrue())
        {
            return result;
        }
    }

    HioPacketHeader& header = GetRecvPacketBuffer()->GetHeader();

    if(!header.IsValid())
    {
        NN_DETAIL_ATK_INFO("[nn::atk::viewer::detail::HioAsyncChannel] detail error : receive invalid packet.\n");
        return HioResult(HioResultType_TargetRecieveError);
    }

    HioMessageHandler* handler = GetMessageHandler(header);

    if(handler == NULL)
    {
        NN_ATK_WARNING("unknown HIO message. : ID=0x%08x, Size=%d", uint32_t(header.message), uint32_t(header.bodySize));

        // 処理できないメッセージをスキップします。
        if(!GetStream().SkipBody().IsTrue())
        {
            NN_ATK_WARNING("failed to Skip().");
            return HioResult(HioResultType_TargetRecieveError);
        }

        return HioResult(HioResultType_True);
    }

#if defined(NN_ATK_ENABLE_COM_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[nn::atk::viewer::detail::HioAsyncChannel] receive packet : message=%s(0x%08x), Size=%d, BodySize=%d.\n",
        header.message.ToString(),
        uint32_t(header.message),
        sizeof(HioPacketHeader) + header.bodySize,
        static_cast<uint32_t>(header.bodySize));
#endif

    return handler->Invoke(GetStream());
}

//----------------------------------------------------------
HioResult
HioAsyncChannel::Send(const HioPacket& message) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return HioResult(HioResultType_TargetNotInitialized);
    }

    // メッセージを送信します。
    return GetStream().WritePacket(message);
}

//----------------------------------------------------------
HioMessageHandler*
HioAsyncChannel::GetMessageHandler(const HioPacketHeader& header) const NN_NOEXCEPT
{
    for(HioMessageHandlerRefList::const_iterator it = m_MessageHandlerRefs.begin();
        it != m_MessageHandlerRefs.end();
        ++it)
    {
        NN_SDK_ASSERT_NOT_NULL(it->value);

        if(it->value->CanRespond(header))
        {
            return it->value;
        }
    }

    return NULL;
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
