﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/handler/atk_QueryItemsReplyHandler.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/viewer/atk_SoundArchiveEditor.h>
#include <nn/atk/viewer/detail/atk_SoundEditConnection.h>
#include <nn/atk/viewer/detail/hio/atk_HioProtocol.h>
#include <nn/atk/viewer/detail/hio/atk_HioPacketStream.h>
#include <nn/atk/viewer/detail/protocol/atk_QueryItemsPacket.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_COM_DEBUG
#endif

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

namespace {

//----------------------------------------------------------
static void* GetTemporaryBuffer(QueryItemsReplyPacket& packet) NN_NOEXCEPT
{
    if(packet.GetBody().items.GetItemCount() == 0)
    {
        return NULL;
    }

    // パケットの可変長領域をテンポラリバッファとして使いまわします。
    uint32_t offsetFromBody =
        sizeof(QueryItemsReplyPacket::Body) +
        packet.GetBody().items.GetItemCount() * sizeof(Offset);

    return atk::detail::fnd::AddOffsetToPtr(&packet.GetBody(), offsetFromBody);
}

}

//----------------------------------------------------------
QueryItemsReplyHandler::QueryItemsReplyHandler() NN_NOEXCEPT :
m_Connection(NULL),
m_ItemSize(0),
m_ReadingItemIndex(0)
{
}

//----------------------------------------------------------
void
QueryItemsReplyHandler::Initialize(SoundEditConnection& connection, int maxItemName) NN_NOEXCEPT
{
    NN_SDK_ASSERT(maxItemName > 0);

    m_Connection = &connection;
    m_ItemSize = QueryItemsReplyPacket::ItemInfo::GetRequiredSize(maxItemName);
    m_ReadingItemIndex = 0;
}

//----------------------------------------------------------
void
QueryItemsReplyHandler::Finalize() NN_NOEXCEPT
{
    m_Connection = NULL;
    m_ItemSize = 0;
    m_ReadingItemIndex = 0;
}

//----------------------------------------------------------
HioResult
QueryItemsReplyHandler::OnInvoke(HioPacketStream& stream) NN_NOEXCEPT
{
    if(m_Connection == NULL)
    {
        NN_SDK_ASSERT(false, "QueryItemsReplyHandler is not initialized.\n");
        stream.SkipBody();
        return HioResult(HioResultType_TargetNotInitialized);
    }

    if(!m_Connection->IsInitialized() ||
        m_Connection->GetSoundArchiveEditor() == NULL)
    {
        stream.SkipBody();
        return HioResult(HioResultType_True);
    }

    HioResult result = ReadBodyFixedLengthArea(stream);

    if(result.IsFalse())
    {
        return result;
    }
    else if(!result.IsTrue())
    {
        stream.SkipBody();
        return result;
    }

    QueryItemsReplyPacket* targetPacket =
        reinterpret_cast<QueryItemsReplyPacket*>(stream.GetReadingPacket());
    NN_SDK_ASSERT_NOT_NULL(targetPacket);

    // actionForAllItems が指定されていたらすべてのアイテムについて処理します。
    switch(targetPacket->GetBody().actionForAllItems)
    {
    case QueryItemsReplyPacket::ActionType_UpdateItem:
        m_Connection->GetSoundArchiveEditor()->detail_UpdateAllItemCaches();
        stream.SkipBody();
        return HioResult(HioResultType_True);

    case QueryItemsReplyPacket::ActionType_RemoveItem:
        NN_SDK_ASSERT_NOT_NULL(m_Connection->GetSoundArchiveEditor());
        m_Connection->GetSoundArchiveEditor()->detail_RemoveAllItemCaches();
        stream.SkipBody();
        return HioResult(HioResultType_True);

    default:
        break;
    }

    ResOffsetTableData& itemsData =
        *reinterpret_cast<ResOffsetTableData*>(&targetPacket->GetBody().items);

    // テーブルのオフセットをすべて読み込みます。
    result = ReadItemsOffsets(itemsData, stream);

    if(!result.IsTrue())
    {
        stream.SkipBody();
        return result;
    }

    // アイテム数が多くても使用メモリを抑えるために、
    // ResOffsetTable::itemOffsets[] 以降の可変長メモリ領域を
    // 逐次読み込みのテンポラリ領域として使いまわします。
    QueryItemsReplyPacket::ItemInfo* item = reinterpret_cast<QueryItemsReplyPacket::ItemInfo*>(
        GetTemporaryBuffer(*targetPacket)
        );

    for(uint32_t index = m_ReadingItemIndex; index < itemsData.itemCount; ++index)
    {
        size_t itemSize = ReadItemLength(stream, itemsData, index);

        if(stream.GetReadableBytes() < itemSize)
        {
            return HioResult(HioResultType_False);
        }

        // アイテム名が長すぎる場合は、スキップします。
        if(itemSize > m_ItemSize)
        {
            NN_ATK_WARNING("[nn::atk::viewer::detail::QueryItemsReplyHandler] itemName is too long.");
            stream.SkipBodyPart(itemSize - sizeof(atk::detail::fnd::BinU32));
            continue;
        }

        result = stream.TryReadBodyPart(item, itemSize);

        if(!result.IsTrue())
        {
            stream.SkipBody();
            return result;
        }

        ++m_ReadingItemIndex;

        // 終端文字を設定します（保護処理）
        *atk::detail::fnd::AddOffsetToPtr<uint8_t*>(item, itemSize) = '\0';

#if defined(NN_ATK_ENABLE_COM_DEBUG)
        NN_DETAIL_ATK_INFO(
            "[sndedit] QueryItemsReply : itemName[%d]=%s, action=%s, file=%s\n",
            index,
            item->name.GetName(),
            item->action & QueryItemsReplyPacket::ActionType_Update ? "UPDATE" : "REMOVE");
#endif

        NN_SDK_ASSERT_NOT_NULL(m_Connection->GetSoundArchiveEditor());

        switch(item->action)
        {
        case QueryItemsReplyPacket::ActionType_UpdateItem:
            // アイテムキャッシュを更新します。
            m_Connection->GetSoundArchiveEditor()->detail_UpdateItemCache(item->name.GetName());
            break;

        case QueryItemsReplyPacket::ActionType_RemoveItem:
            // アイテムキャッシュを削除します。
            // 即時にメモリを解放するために、無効化ではなく削除します。
            m_Connection->GetSoundArchiveEditor()->detail_RemoveItemCache(item->name.GetName());
            break;

        default:
            break;
        }
    }

    return HioResult(HioResultType_True);
}

//----------------------------------------------------------
HioResult
QueryItemsReplyHandler::ReadBodyFixedLengthArea(HioPacketStream& stream) NN_NOEXCEPT
{
    // 固定長領域を読み込み済の場合は、処理しません。
    if(stream.GetReadBodySize() > sizeof(QueryItemsReplyPacket::Body))
    {
        return HioResult(HioResultType_True);
    }

    // 固定長領域が中途半端に読み込まれることはありません。
    if(stream.GetReadBodySize() > 0)
    {
        NN_SDK_ASSERT(false, "QueryItemsReplyHandler has invalid status.\n");
        return HioResult(HioResultType_Failed);
    }

    // 固定長領域を一括で読み込めない場合は、処理しません。
    if(stream.GetReadableBytes() < sizeof(QueryItemsReplyPacket::Body))
    {
        return HioResult(HioResultType_False);
    }

    m_ReadingItemIndex = 0;

    HioResult result = stream.ReadBodyPart(
        stream.GetReadingPacket()->GetBody(),
        sizeof(QueryItemsReplyPacket::Body));

#if defined(NN_ATK_ENABLE_COM_DEBUG)
    if(result.IsTrue())
    {
        QueryItemsReplyPacket::Body* body =
            reinterpret_cast<QueryItemsReplyPacket::Body*>(stream.GetReadingPacket()->GetBody());

        NN_DETAIL_ATK_INFO(
            "[sndedit] QueryItemsReply : isAllItems=%d, itemNamesCount=%d\n",
            body->actionForAllItems ? 1 : 0,
            body->items.GetItemCount());
    }
#endif

    return result;
}

//----------------------------------------------------------
HioResult
QueryItemsReplyHandler::ReadItemsOffsets(
    ResOffsetTableData& items,
    HioPacketStream& stream) NN_NOEXCEPT
{
    uint32_t itemsOffsetsSize = items.itemCount * sizeof(Offset);

    // 読み込み済の場合は、処理しません。
    if(stream.GetReadBodySize() >= sizeof(QueryItemsReplyPacket::Body) + itemsOffsetsSize)
    {
        return HioResult(HioResultType_True);
    }

    if(items.itemCount == 0)
    {
        return HioResult(HioResultType_True);
    }

    return stream.ReadBodyPart(items.itemOffsets, itemsOffsetsSize);
}

//----------------------------------------------------------
size_t
QueryItemsReplyHandler::ReadItemLength(
    HioPacketStream& stream,
    ResOffsetTableData& itemsData,
    uint32_t index) NN_NOEXCEPT
{
    if(itemsData.itemCount <= index)
    {
        NN_SDK_ASSERT(false, "ItemNameTable index.\n");
        return 0;
    }

    if(index + 1 < itemsData.itemCount)
    {
        // テーブルアイテムオフセットの差分
        NN_SDK_ASSERT(itemsData.itemOffsets[index + 1] > itemsData.itemOffsets[index]);
        return util::BytePtr(itemsData.itemOffsets[index].to_ptr()).Distance(
            itemsData.itemOffsets[index + 1].to_ptr());
    }
    else
    {
        // 最後のアイテムは、残りパケットボディサイズ
        NN_SDK_ASSERT(stream.GetReadingPacket()->GetHeader().bodySize > stream.GetReadBodySize());
        return stream.GetReadingPacket()->GetHeader().bodySize - stream.GetReadBodySize();
    }
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
