﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/atk_SoundEditConnection.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/fnd/string/atkfnd_String.h>
#include <nn/atk/fnd/string/atkfnd_Path.h>
#include <nn/atk/viewer/atk_SoundArchiveEditor.h>
#include <nn/atk/viewer/detail/hio/atk_HioSyncChannel.h>
#include <nn/atk/viewer/detail/hio/atk_HioAsyncChannel.h>
#include <nn/atk/viewer/detail/protocol/atk_ItemCachePacket.h>
#include <nn/atk/viewer/detail/protocol/atk_QueryItemsPacket.h>
#include <nn/atk/viewer/detail/protocol/atk_QueryItemInfoPacket.h>
#include <nn/atk/viewer/detail/res/atk_ResItemInfo.h>


//--------------------------------------------------------------------------------------------------
// ミドルウェア情報
#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#define NN_DETAIL_XX_MACRO_TO_STRING(x) NN_DETAIL_XX_MACRO_TO_STRING_(x)
#define NN_DETAIL_XX_MACRO_TO_STRING_(x) #x

#define NW_ATK_VIEWER_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_Atk_Viewer_For_Develop-" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MAJOR) "_" \
NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MINOR) "_" NN_DETAIL_XX_MACRO_TO_STRING(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

namespace {
#if defined(NN_SDK_BUILD_DEBUG)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_ATK_VIEWER_MIDDLEWARE_SYMBOL(Debug));
#elif defined(NN_SDK_BUILD_DEVELOP)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_ATK_VIEWER_MIDDLEWARE_SYMBOL(Develop));
#elif defined(NN_SDK_BUILD_RELEASE)
    NN_DEFINE_MIDDLEWARE(g_MiddlewareInfo, "Nintendo", NW_ATK_VIEWER_MIDDLEWARE_SYMBOL(Release));
#endif
}

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

//----------------------------------------------------------
void
SoundEditConnection::Finalize() NN_NOEXCEPT
{
    Close();
    UnsetupSyncChannel();

    m_SendPacketBuffer = NULL;
    m_SendPacketBufferSize = 0;

    m_CacheSyncInterval = 0;
    m_SendTimeout = 0;
    m_HioManager = NULL;
    m_SyncChannel = NULL;
    m_FuncChannel = NULL;
}

//----------------------------------------------------------
Result
SoundEditConnection::Open(SoundArchiveEditor& soundArchiveEditor) NN_NOEXCEPT
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "SoundEditConnection is not initialized.\n");
        return Result(ResultType_NotInitialized);
    }

    if(IsOpened() && m_SoundArchiveEditor->IsStarted())
    {
        return Result(ResultType_True);
    }

    Result result = soundArchiveEditor.detail_Start(*this);

    if(!result.IsTrue())
    {
        Close();
        return result;
    }

    m_CacheSyncStopWatch.Start();

    m_SoundArchiveEditor = &soundArchiveEditor;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundEditConnection::Close() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "SoundEditConnection is not initialized.\n");
        return;
    }

    m_CacheSyncStopWatch.Stop();

    if(m_SoundArchiveEditor != NULL)
    {
        m_SoundArchiveEditor->detail_Stop();
        m_SoundArchiveEditor = NULL;
    }
}

//----------------------------------------------------------
size_t
SoundEditConnection::GetRequiredMemorySize(int maxItemName) const NN_NOEXCEPT
{
    // QueryItemInfoPacket 送信用のバッファサイズ
    size_t result = detail::QueryItemInfoPacket::GetRequiredSize(maxItemName);

    // BeginItemCachePacket 送信用のバッファサイズ
    result = std::max(result, detail::BeginItemCachePacket::GetRequiredSize(maxItemName));

    // EndItemCachePacket 送信用のバッファサイズ
    result = std::max(result, detail::EndItemCachePacket::GetRequiredSize(maxItemName));

    return result;
}

//----------------------------------------------------------
size_t
SoundEditConnection::GetRequiredMemorySizeForOpenFile() NN_NOEXCEPT
{
    return HioManager::GetRequiredMemorySizeForOpenFile();
}

//----------------------------------------------------------
void
SoundEditConnection::Update() NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "SoundEditConnection is not initialized.\n");
        return;
    }

    if(!IsOpened())
    {
        return;
    }

    // 無効アイテムチェック間隔を経過したら...
    // 無効アイテム一覧を要求します。
    if(m_CacheSyncStopWatch.GetElapsedTime().ToMilliSeconds() >= m_CacheSyncInterval)
    {
        BeginCacheSync();
    }
}

//----------------------------------------------------------
Result
SoundEditConnection::BeginUpdateItemCache(const char* itemName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT(*itemName != '\0');

    NN_SDK_ASSERT_NOT_NULL(m_SendPacketBuffer);

    detail::BeginItemCachePacket* packet =
        detail::BeginItemCachePacket::CreateInstance(m_SendPacketBuffer, itemName);

    packet->GetBody().SetAction(detail::CacheAction_Update);

    const detail::HioPacket* replyPacket = NULL;

    detail::HioResult result =
        m_FuncChannel->Send(replyPacket, *packet, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::BeginRemoveItemCache(const char* itemName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT(*itemName != '\0');

    NN_SDK_ASSERT_NOT_NULL(m_SendPacketBuffer);

    detail::BeginItemCachePacket* packet =
        detail::BeginItemCachePacket::CreateInstance(m_SendPacketBuffer, itemName);

    packet->GetBody().SetAction(detail::CacheAction_Remove);

    const detail::HioPacket* replyPacket = NULL;

    detail::HioResult result =
        m_FuncChannel->Send(replyPacket, *m_SendPacketBuffer, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::BeginRemoveAllItemCaches() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_SendPacketBuffer);

    detail::BeginItemCachePacket* packet =
        detail::BeginItemCachePacket::CreateInstance(m_SendPacketBuffer, NULL);

    packet->GetBody().SetIsTargetAll(true);
    packet->GetBody().SetAction(detail::CacheAction_Remove);

    const detail::HioPacket* replyPacket = NULL;

    detail::HioResult result =
        m_FuncChannel->Send(replyPacket, *m_SendPacketBuffer, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::EndUpdateItemCache(Result itemCacheResult, const char* itemName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT(*itemName != '\0');

    NN_SDK_ASSERT_NOT_NULL(m_SendPacketBuffer);

    detail::EndItemCachePacket* packet =
        detail::EndItemCachePacket::CreateInstance(m_SendPacketBuffer, itemName, itemCacheResult);

    packet->GetBody().SetAction(detail::CacheAction_Update);

    const detail::HioPacket* replyPacket = NULL;

    detail::HioResult result =
        m_FuncChannel->Send(replyPacket, *packet, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::EndRemoveItemCache(Result itemCacheResult, const char* itemName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(itemName);
    NN_SDK_ASSERT(*itemName != '\0');

    NN_SDK_ASSERT_NOT_NULL(m_SendPacketBuffer);

    detail::EndItemCachePacket* packet =
        detail::EndItemCachePacket::CreateInstance(m_SendPacketBuffer, itemName, itemCacheResult);

    packet->GetBody().SetAction(detail::CacheAction_Remove);

    const detail::HioPacket* replyPacket = NULL;

    detail::HioResult result =
        m_FuncChannel->Send(replyPacket, *packet, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::EndRemoveAllItemCaches(Result itemCacheResult) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_SendPacketBuffer);

    detail::EndItemCachePacket* packet =
        detail::EndItemCachePacket::CreateInstance(m_SendPacketBuffer, NULL, itemCacheResult);

    packet->GetBody().SetIsTargetAll(true);

    const detail::HioPacket* replyPacket = NULL;

    detail::HioResult result =
        m_FuncChannel->Send(replyPacket, *packet, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
Result
SoundEditConnection::QueryItemInfo(const detail::HioPacket*& replyPacket, const char* itemName) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        NN_SDK_ASSERT(false, "SoundEditConnection is not initialized.\n");
        return Result(ResultType_NotInitialized);
    }

    if(!IsOpened())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(ResultType_NotConnected);
    }

    if(itemName == NULL || *itemName == '\0')
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        replyPacket = &detail::HioPacket::Null();
        return Result(ResultType_Failed);
    }

    NN_SDK_ASSERT_NOT_NULL(m_SendPacketBuffer);
    detail::QueryItemInfoPacket::CreateInstance(m_SendPacketBuffer, itemName);

    detail::HioResult result =
        m_FuncChannel->Send(replyPacket, *m_SendPacketBuffer, m_SendTimeout);

    if(!result.IsTrue())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(static_cast<ResultType>(result));
    }

    HioResult packetResult = replyPacket->GetHeader().GetResult();

    if(!packetResult.IsTrue())
    {
        replyPacket = &detail::HioPacket::Null();
        return Result(static_cast<ResultType>(packetResult));
    }

    return Result(static_cast<ResultType>(result));
}

//----------------------------------------------------------
void
SoundEditConnection::SetupSyncChannel(uint32_t maxItemName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_SyncChannel);

    m_QueryItemsReplyHandler.Initialize(*this, maxItemName);
    m_SyncChannel->RegisterMessageHandler(m_QueryItemsReplyHandler);
}

//----------------------------------------------------------
void
SoundEditConnection::UnsetupSyncChannel() NN_NOEXCEPT
{
    if(m_SyncChannel != NULL)
    {
        m_SyncChannel->UnregisterMessageHandler(m_QueryItemsReplyHandler);
    }

    m_QueryItemsReplyHandler.Finalize();
}

//----------------------------------------------------------
bool
SoundEditConnection::BeginCacheSync() NN_NOEXCEPT
{
    m_CacheSyncStopWatch.Reset();

    if(!IsInitialized() || !IsOpened())
    {
        return false;
    }

    return m_SyncChannel->Send(detail::QueryItemsPacket()).IsSucceeded();
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
