﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WavOutRecorder.h>

#include <nn/atk/atk_HardwareManager.h>

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::MaxChannels );
NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::BytesPerSample );
NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::MaxFrameLength );
NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::FlIndex );
NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::FrIndex );
NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::FcIndex );
NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::LfeIndex );
NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::RlIndex );
NN_DEFINE_STATIC_CONSTANT( const int WavOutRecorder::RrIndex );
const char* WavOutRecorder::DeviceName = "TV";

WavOutRecorder* WavOutRecorder::g_pThis = NULL;

//---------------------------------------------------------------------------
WavOutRecorder::WavOutRecorder() NN_NOEXCEPT :
DeviceOutRecorder(DeviceName)
{
}

//---------------------------------------------------------------------------
WavOutRecorder::~WavOutRecorder() NN_NOEXCEPT
{
}

uint32_t WavOutRecorder::GetValidChannels(uint32_t /* channels */) const NN_NOEXCEPT
{
    return nn::atk::detail::driver::HardwareManager::GetInstance().GetChannelCount();
}


//---------------------------------------------------------------------------
void
WavOutRecorder::OnStart() NN_NOEXCEPT
{
    if(g_pThis != NULL)
    {
        NN_SDK_ASSERT(false, "WavOutRecorder instance is already existed.\n");
        return;
    }

    g_pThis = this;
#if 0 // 参考用 : NW4F 実装
#ifdef NN_ATK_CONFIG_ENABLE_APPFRAMECALLBACK
    AXPB_ERROR_CODE error = AXRegisterAppFrameCallback(AxAppFrameCallbackFunc);
    NN_SDK_ASSERT(error == AXPB_ERROR_NONE);
#endif
#endif
}

//---------------------------------------------------------------------------
void
WavOutRecorder::OnStop() NN_NOEXCEPT
{
#if 0 // 参考用 : NW4F 実装
#ifdef NN_ATK_CONFIG_ENABLE_APPFRAMECALLBACK
    AXPB_ERROR_CODE error = AXDeregisterAppFrameCallback(AxAppFrameCallbackFunc);
    NN_SDK_ASSERT(error == AXPB_ERROR_NONE);
#endif
#endif

    if(g_pThis == this)
    {
        g_pThis = NULL;
    }
}

//---------------------------------------------------------------------------
uint32_t
WavOutRecorder::OnProcessSamples(int16_t* sampleBuffer, uint32_t samples) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(sampleBuffer);

    // 現状では内部的に 6ch 固定で処理をしているので、録音先のチャンネル数(outputChannels)は設定値に依存し、
    // 録音元のチャンネル数(inputChannels)は 6ch 固定。
    uint32_t outputChannels = GetRecordingChannels();
    uint32_t inputChannels = nn::atk::detail::driver::HardwareManager::GetInstance().GetChannelCountMax();

    uint32_t destIndex = 0;

    for(uint32_t srcIndex = 0; srcIndex < samples ; srcIndex += inputChannels, destIndex += outputChannels)
    {
        NN_SDK_ASSERT(srcIndex >= destIndex);

        const int16_t fl = ResolveSampleEndian(sampleBuffer[srcIndex + ChannelIndex_FrontLeft]);
        const int16_t fr = ResolveSampleEndian(sampleBuffer[srcIndex + ChannelIndex_FrontRight]);

        sampleBuffer[destIndex + FlIndex]  = fl;
        sampleBuffer[destIndex + FrIndex]  = fr;

        if(outputChannels == 6)
        {
            const int16_t fc  = ResolveSampleEndian(sampleBuffer[srcIndex + ChannelIndex_FrontCenter]);
            const int16_t lfe = ResolveSampleEndian(sampleBuffer[srcIndex + ChannelIndex_Lfe]);
            const int16_t rl  = ResolveSampleEndian(sampleBuffer[srcIndex + ChannelIndex_RearLeft]);
            const int16_t rr  = ResolveSampleEndian(sampleBuffer[srcIndex + ChannelIndex_RearRight]);

            sampleBuffer[destIndex + FcIndex]  = fc;
            sampleBuffer[destIndex + LfeIndex] = lfe;
            sampleBuffer[destIndex + RlIndex]  = rl;
            sampleBuffer[destIndex + RrIndex]  = rr;
        }
    }

    return destIndex;
}

} // atk
} // nn
