﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_ThreadInfoReader.h>

namespace
{
    //  バッファの最後に書かれる情報です。
    struct ThreadInfoBackToTop
    {
        void* bufferHead;
    };

    //  ThreadInfoRecorder で次の情報がバッファの先頭に書かれていることを示す ID です。
    const uint32_t ThreadInfoBackToTopId = 0xff;
    //  ThreadInfoRecorder のフッターの ID です。
    const uint32_t ThreadInfoFooterId = ThreadInfoBackToTopId - 1;
    NN_STATIC_ASSERT( nn::atk::detail::SoundThreadInfoId_Count <= ThreadInfoFooterId );

    const int InfoHeaderShiftSizeCount = 8; //  情報ヘッダにサイズを記録する際にビットシフトする桁数
    const uint32_t InfoHeaderIdMask = ( 1u << InfoHeaderShiftSizeCount ) - 1u;

    //  id と size を uint32_t の情報ヘッダに変換します。
    inline uint32_t CreateInfoHeader(uint32_t id, size_t size) NN_NOEXCEPT
    {
        return ( static_cast<uint32_t>( size ) << InfoHeaderShiftSizeCount ) | ( id & InfoHeaderIdMask );
    }
    //  情報ヘッダから id を取得します。
    inline uint32_t GetIdFromInfoHeader(const void* pInfoHeader) NN_NOEXCEPT
    {
        return ( *reinterpret_cast<const uint32_t*>( pInfoHeader ) ) & InfoHeaderIdMask;
    }
    //  情報ヘッダから size を取得します。
    inline size_t GetSizeFromInfoHeader(const void* pInfoHeader) NN_NOEXCEPT
    {
        return ( *reinterpret_cast<const uint32_t*>( pInfoHeader ) )>> InfoHeaderShiftSizeCount;
    }

    //  情報ヘッダからデータ部を指すポインタを取得します。
    inline void* GetInfoDataAddress(void* pInfo) NN_NOEXCEPT
    {
        return nn::util::BytePtr( pInfo, sizeof(uint32_t) ).Get();
    }
    //  情報ヘッダからデータ部を指すポインタを取得します。
    inline const void* GetInfoDataAddress(const void* pInfo) NN_NOEXCEPT
    {
        return nn::util::ConstBytePtr( pInfo, sizeof(uint32_t) ).Get();
    }
    //  情報ヘッダ分の大きさを加えたサイズを取得します。
    inline size_t GetTotalInfoSize(size_t infoDataSize) NN_NOEXCEPT
    {
        return sizeof(int32_t) + infoDataSize;
    }

    //  id の情報があるアドレスを取得します。見つからなかった場合は nullptr を返します。
    const void* FindInfo(const void* buffer, uint32_t id, bool isSkipFirst) NN_NOEXCEPT
    {
        nn::util::ConstBytePtr ptr( buffer );

        for(;;)
        {
            const uint32_t current = GetIdFromInfoHeader( ptr.Get() );
            if( id == current )
            {
                if( isSkipFirst )
                {
                    isSkipFirst = false;
                }
                else
                {
                    return ptr.Get();
                }
            }

            if( current == ThreadInfoFooterId )
            {
                return nullptr;
            }

            if( current == ThreadInfoBackToTopId )
            {
                const ThreadInfoBackToTop* pBackToTop = reinterpret_cast<const ThreadInfoBackToTop*>( GetInfoDataAddress( ptr.Get() ) );
                ptr.Reset( pBackToTop->bufferHead );
            }
            else
            {
                ptr.Advance( GetSizeFromInfoHeader( ptr.Get() ) );
            }
        }
    }
}

namespace nn {
namespace atk {
namespace detail {


//  バッファの最小サイズです。現在は 1 KiB に設定しています。
const size_t ThreadInfoRecorder::BufferMinimumSize = 1024;

ThreadInfoRecorder::ThreadInfoRecorder() NN_NOEXCEPT
    : m_Buffer( nullptr )
    , m_BufferSize( 0 )
    , m_WritePosition( 0 )
    , m_ReadPosition( 0 )
    , m_RecordFrameCount( 0 )
    , m_IsAllocationFailed( false )
{
}
void ThreadInfoRecorder::Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( buffer );
    NN_SDK_REQUIRES_GREATER_EQUAL( bufferSize, BufferMinimumSize );

    m_Buffer = reinterpret_cast<uint8_t*>( buffer );

    //  ThreadInfoBackToTop を書き込むために小さくします。
    m_BufferSize = bufferSize - GetTotalInfoSize( sizeof(ThreadInfoBackToTop) );

    m_WritePosition = 0;
    m_ReadPosition = 0;
    m_RecordFrameCount = 0;
    m_IsAllocationFailed = false;
}
int ThreadInfoRecorder::GetRecordedFrameCount() const NN_NOEXCEPT
{
    return m_RecordFrameCount;
}
const void* ThreadInfoRecorder::GetRingBufferHeadAddress() const NN_NOEXCEPT
{
    return nn::util::BytePtr( m_Buffer, m_ReadPosition ).Get();
}
void ThreadInfoRecorder::MoveToNextFrame() NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER( m_RecordFrameCount.load(), 0 );

    //  FrameFooter まで進みます。
    nn::util::ConstBytePtr ptr( FindInfo( GetRingBufferHeadAddress(), ThreadInfoFooterId, false ) );
    NN_SDK_ASSERT_NOT_NULL( ptr.Get() );  //  FrameFooter なので必ずあるはずです。

    //  m_ReadPosition が FrameFooter の次の情報を指すようにします。
    ptr.Advance( GetSizeFromInfoHeader( ptr.Get() ) );
    m_ReadPosition = -ptr.Distance( m_Buffer );
    m_RecordFrameCount.fetch_sub( 1 );
}
void ThreadInfoRecorder::RecordFooter() NN_NOEXCEPT
{
    FrameFooter* pFooter = reinterpret_cast<FrameFooter*>( AllocateBuffer( ThreadInfoFooterId, sizeof(FrameFooter), 0 ) );

    if( pFooter != nullptr )
    {
        pFooter->recordTick = nn::os::GetSystemTick();
        pFooter->isInfoDropped = m_IsAllocationFailed;
        m_RecordFrameCount.fetch_add( 1 );
    }
    m_IsAllocationFailed = false;
}
void* ThreadInfoRecorder::AllocateInfoBuffer(uint32_t id, size_t size) NN_NOEXCEPT
{
    //  FrameFooter を 1 つ確実に確保できるときだけ確保を行います。
    void* buffer = AllocateBuffer( id, size, GetTotalInfoSize( sizeof(FrameFooter) ) );

    if( buffer == nullptr )
    {
        m_IsAllocationFailed = true;
    }

    return buffer;
}
void* ThreadInfoRecorder::AllocateBuffer(uint32_t infoId, size_t infoDataSize, size_t reservedSize) NN_NOEXCEPT
{
    const size_t allocateSize = GetTotalInfoSize( infoDataSize );
    const size_t readPosition = m_ReadPosition;

    //  バッファを確保可能かどうか調べます。
    if( m_WritePosition < readPosition )
    {
        if( readPosition - m_WritePosition < allocateSize + reservedSize )
        {
            return nullptr;
        }
    }
    else
    {
        //  m_ReadPosition と m_WritePosition が同じときに m_RecordFrameCount > 0 であれば、
        //  バッファを使い切った状態であるためバッファの確保はできません。
        if( readPosition == m_WritePosition && m_RecordFrameCount > 0 )
        {
            return nullptr;
        }

        //  まずは allocateSize 分を確保できるかどうかを調べます。
        if( m_BufferSize - m_WritePosition < allocateSize )
        {
            //  バッファの先頭から allocateSize と reservedSize 分を確保できるかどうかを調べます。
            if( readPosition < allocateSize + reservedSize )
            {
                return nullptr;
            }

            //  バッファの先頭から確保できるため、バッファの終端目印をつけて先頭から確保します。
            const size_t size = GetTotalInfoSize( sizeof(ThreadInfoBackToTop) );
            nn::util::BytePtr buffer( m_Buffer, m_WritePosition );
            *buffer.Get<uint32_t>() = CreateInfoHeader( ThreadInfoBackToTopId, size );

            ThreadInfoBackToTop* pInfo = reinterpret_cast<ThreadInfoBackToTop*>( GetInfoDataAddress( buffer.Get() ) );
            pInfo->bufferHead = m_Buffer;

            m_WritePosition = 0;
        }
        else
        {
            //  reservedSize 分を追加で確保できるかどうかを調べます。
            if( m_BufferSize - m_WritePosition < allocateSize + reservedSize &&
                readPosition < reservedSize )
            {
                return nullptr;
            }
        }
    }

    //  バッファを確保
    nn::util::BytePtr buffer( m_Buffer, m_WritePosition );
    m_WritePosition += allocateSize;

    *buffer.Get<uint32_t>() = CreateInfoHeader( infoId, allocateSize );
    return GetInfoDataAddress( buffer.Get() );
}


} // namespace nn::atk::detail


SoundThreadInfoReader::SoundThreadInfoReader(SoundThreadInfoRecorder& recorder) NN_NOEXCEPT
    : m_Recorder( recorder )
{
    ResetInfo();
}
bool SoundThreadInfoReader::HasFrameInfo() const NN_NOEXCEPT
{
    return m_Recorder.GetRecordedFrameCount() > 0;
}
void SoundThreadInfoReader::SetupFrameInfo() NN_NOEXCEPT
{
    if( HasFrameInfo() )
    {
        const void* ringBufferHeadAddress = m_Recorder.GetRingBufferHeadAddress();
        for(uint32_t i = 0; i < detail::SoundThreadInfoId_Count; i++)
        {
            m_pCurrentInfo[i] = FindInfo( ringBufferHeadAddress, i, false );
        }
        m_pFooterInfo = FindInfo( ringBufferHeadAddress, ThreadInfoFooterId, false );
    }
    else
    {
        ResetInfo();
    }
}
void SoundThreadInfoReader::MoveToNextFrame() NN_NOEXCEPT
{
    if( HasFrameInfo() )
    {
        m_Recorder.MoveToNextFrame();
    }
    ResetInfo();
}
bool SoundThreadInfoReader::ReadFrameGlobalInfo(FrameGlobalInfo* pInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pInfo );

    if( m_pFooterInfo == nullptr )
    {
        return false;
    }

    const detail::ThreadInfoRecorder::FrameFooter* pFooterInfo = reinterpret_cast<const detail::ThreadInfoRecorder::FrameFooter*>( GetInfoDataAddress( m_pFooterInfo ) );
    pInfo->recordTick = pFooterInfo->recordTick;
    pInfo->isInfoDropped = pFooterInfo->isInfoDropped;

    return true;
}
bool SoundThreadInfoReader::ReadSoundProfile(SoundProfile* pProfile) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pProfile );
    const uint32_t id = detail::SoundThreadInfoId_SoundProfile;
    const void* ptr = m_pCurrentInfo[id];

    if( ptr == nullptr )
    {
        return false;
    }

    *pProfile = *reinterpret_cast<const SoundProfile*>( GetInfoDataAddress( ptr ) );
    m_pCurrentInfo[id] = FindInfo( ptr, id, true );

    return true;
}
bool SoundThreadInfoReader::ReadSoundThreadUpdateProfile(SoundThreadUpdateProfile* pProfile) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pProfile );
    const uint32_t id = detail::SoundThreadInfoId_SoundThreadUpdateProfile;
    const void* ptr = m_pCurrentInfo[id];

    if( ptr == nullptr )
    {
        return false;
    }

    *pProfile = *reinterpret_cast<const SoundThreadUpdateProfile*>( GetInfoDataAddress( ptr ) );
    m_pCurrentInfo[id] = FindInfo( ptr, id, true );

    return true;
}
void SoundThreadInfoReader::ResetInfo() NN_NOEXCEPT
{
    std::fill( m_pCurrentInfo, m_pCurrentInfo + detail::SoundThreadInfoId_Count, nullptr );
    m_pFooterInfo = nullptr;
}


} // namespace nn::atk
} // namespace nn

