﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_ChannelMixVolume.h>

namespace nn { namespace atk {

NN_DEFINE_STATIC_CONSTANT(const int ChannelMixVolume::ChannelCountMax);

ChannelMixVolume::ChannelMixVolume() NN_NOEXCEPT
: m_ChannelCount(6)
{
    InitializeChannelVolume();
}

ChannelMixVolume::ChannelMixVolume(const MixVolume & mixVolume) NN_NOEXCEPT
: m_ChannelCount(sizeof(mixVolume.channel) / sizeof(mixVolume.channel[0]))
{
    InitializeChannelVolume();
    for(int i = 0; i < sizeof(mixVolume.channel) / sizeof(mixVolume.channel[0]); i++)
    {
        m_ChannelVolume[i] = mixVolume.channel[i];
    }
}

ChannelMixVolume::ChannelMixVolume(const float *pVolume, int volumeCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(volumeCount, 1, ChannelCountMax);
    NN_SDK_REQUIRES_NOT_NULL(pVolume);
    InitializeChannelVolume();

    if(!SetChannelCount(volumeCount))
    {
        m_ChannelCount = 6;
        return;
    }

    for(int i = 0; i < m_ChannelCount; i++)
    {
        m_ChannelVolume[i] = pVolume[i];
    }
}

bool ChannelMixVolume::SetChannelCount(int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(channelCount, 1, ChannelCountMax);

    if(channelCount <= 0 || channelCount > ChannelCountMax)
    {
        return false;
    }

    m_ChannelCount = channelCount;
    return true;
}

bool ChannelMixVolume::SetChannelVolume(int channelIndex, float volume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(channelIndex, 0, ChannelCountMax);
    if(channelIndex < 0 || channelIndex >= m_ChannelCount)
    {
        return false;
    }

    m_ChannelVolume[channelIndex] = volume;
    return true;
}

float ChannelMixVolume::GetChannelVolume(int channelIndex) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(channelIndex, 0, ChannelCountMax);
    NN_SDK_REQUIRES_RANGE(channelIndex, 0, m_ChannelCount);
    if(channelIndex < 0 || channelIndex >= m_ChannelCount)
    {
        return 0.0f;
    }

    return m_ChannelVolume[channelIndex];
}

bool ChannelMixVolume::SetChannelVolume(int channelIndex, const float *pVolume, int volumeCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(channelIndex, 0, ChannelCountMax);
    NN_SDK_REQUIRES_NOT_NULL(pVolume);
    NN_SDK_REQUIRES_GREATER(volumeCount ,0);
    if(channelIndex < 0 || channelIndex + (volumeCount - 1) < ChannelCountMax)
    {
        return false;
    }

    for(int i = 0; i < volumeCount; i++)
    {
        m_ChannelVolume[channelIndex + i] = pVolume[i];
    }
    return true;
}

void ChannelMixVolume::InitializeChannelVolume() NN_NOEXCEPT
{
    for(float volume : m_ChannelVolume)
    {
        volume = 0.0f;
    }
}

}}
