﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/os/atkfnd_Thread.h>
#include <nn/fs/fs_Priority.h>
#include <nn/nn_TimeSpan.h>

namespace
{
nn::fs::Priority GetNnFsPriority(nn::atk::detail::fnd::Thread::FsPriority priority)
{
    switch (priority)
    {
    case nn::atk::detail::fnd::Thread::FsPriority_RealTime :
        return nn::fs::Priority_Realtime;
    case nn::atk::detail::fnd::Thread::FsPriority_Normal :
        return nn::fs::Priority_Normal;
    case nn::atk::detail::fnd::Thread::FsPriority_Low :
        return nn::fs::Priority_Low;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
}

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  スレッドのメイン関数オブジェクトです。
//---------------------------------------------------------------------------
class Thread::ThreadMain
{
public:
    static void Run(void* ptrArg) NN_NOEXCEPT
    {
        Thread* owner = reinterpret_cast<Thread*>(ptrArg);

        NN_SDK_ASSERT_NOT_NULL(owner);
        NN_SDK_ASSERT_NOT_NULL(owner->m_Handler);

        owner->m_IsTerminated = false;
        owner->OnRun();

        owner->SetFsPriority(owner->m_FsPriority);
#if defined(NN_ATK_FND_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
        owner->m_CoreNumber = nn::os::GetCurrentCoreNumber();
#endif
        owner->m_Handler->Run(owner->m_Param);

        owner->OnExit();
        owner->m_IsTerminated = true;
    }
};

//---------------------------------------------------------------------------
Thread::Thread() NN_NOEXCEPT :
m_State(State_NotRun),
m_Id(InvalidId),
m_Priority(DefaultThreadPriority),
#if defined(NN_BUILD_CONFIG_OS_WIN32)
m_Name(""),
#endif
m_Handler(NULL),
#if defined(NN_ATK_FND_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
m_CoreNumber(InvalidCoreNumber),
#endif
m_IsTerminated(false)
{
}

//---------------------------------------------------------------------------
void
Thread::SetPriority(int32_t value) NN_NOEXCEPT
{
    nn::os::ChangeThreadPriority(&m_Handle, value);
}

//---------------------------------------------------------------------------
void
Thread::SetFsPriority(FsPriority value) NN_NOEXCEPT
{
    m_FsPriority = value;
    nn::fs::SetPriorityOnCurrentThread(GetNnFsPriority(m_FsPriority));
}

//---------------------------------------------------------------------------
Thread::FsPriority Thread::GetFsPriority() const NN_NOEXCEPT
{
    return m_FsPriority;
}

//---------------------------------------------------------------------------
void
Thread::Sleep(const TimeSpan& timeSpan) NN_NOEXCEPT
{
    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(timeSpan.ToNanoSeconds()));
}

//---------------------------------------------------------------------------
bool
Thread::Create(Handle& handle, int64_t& id, const RunArgs& args) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::os::GetThreadAvailableCoreMask() & (1ULL << args.idealCoreNumber), "Unsupported core number[%d].", args.idealCoreNumber);
    NN_UNUSED(id);

    if (!nn::os::CreateThread(
        &handle,
        ThreadMain::Run,
        this,
        args.stack,
        args.stackSize,
        args.priority,
        args.idealCoreNumber).IsSuccess()
    )
    {
        m_Id = InvalidId;
        return false;
    }

    m_FsPriority = args.fsPriority;

    nn::os::StartThread(&handle);

    // OSThread へのポインタを ID として利用します。
    m_Id = reinterpret_cast<uint64_t>(&handle);

    return true;
}

//---------------------------------------------------------------------------
void
Thread::Detach() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsTerminated()); // 事前に Join が呼ばれていれば ThreadMain::Run() の処理は完了しているはず

    nn::os::DestroyThread(&m_Handle);
#if defined(NN_ATK_FND_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
    m_CoreNumber = InvalidCoreNumber;
#endif
}

//---------------------------------------------------------------------------
void
Thread::SetName(const char* name) NN_NOEXCEPT
{
    nn::os::SetThreadNamePointer(&m_Handle, name == NULL ? "" : name);
}

//---------------------------------------------------------------------------
void
Thread::SetAffinityMask(int idealCoreNumber, AffinityMask value) NN_NOEXCEPT
{
    nn::os::SetThreadCoreMask(&m_Handle, idealCoreNumber, value);
}

//---------------------------------------------------------------------------
void
Thread::Resume() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
void
Thread::Join() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_Handle);
}

//---------------------------------------------------------------------------
bool
Thread::IsTerminated() const NN_NOEXCEPT
{
    return m_IsTerminated;
}

} // namespace nn::atk::detail::fnd
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn


