﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_MultiVoiceManager.h>
#include <nn/atk/atk_DisposeCallbackManager.h>
#include <nn/atk/atk_MultiVoice.h>
#include <nn/atk/atk_SoundThread.h>

namespace nn {
namespace atk {
namespace detail {
namespace driver {

/*--------------------------------------------------------------------------------*
    Name:           GetInstance

    Description:    シングルトンのインスタンスを取得します

    Arguments:      なし

    Returns:        なし
 *--------------------------------------------------------------------------------*/
MultiVoiceManager& MultiVoiceManager::GetInstance() NN_NOEXCEPT
{
    static MultiVoiceManager instance;
    return instance;
}


MultiVoiceManager::MultiVoiceManager() NN_NOEXCEPT
: m_Initialized( false )
{
}

size_t MultiVoiceManager::GetObjectSize(const detail::SoundInstanceConfig & config) NN_NOEXCEPT
{
    size_t result = sizeof( MultiVoice );
    const size_t AdditionalParamBufferSize = OutputAdditionalParam::GetRequiredMemSize( config );
    if(AdditionalParamBufferSize != 0)
    {
        result += sizeof( OutputAdditionalParam );
        result += AdditionalParamBufferSize;
    }
    return result;
}

size_t MultiVoiceManager::GetRequiredMemSize( int voiceCount, const detail::SoundInstanceConfig& config ) NN_NOEXCEPT
{
    return GetObjectSize(config) * voiceCount;
}

void MultiVoiceManager::Initialize( void* mem, size_t memSize, const detail::SoundInstanceConfig& config ) NN_NOEXCEPT
{
    if ( m_Initialized ) return;

    const size_t TotalObjectSize = GetObjectSize(config);
    const int VoiceCount = static_cast<int>(memSize / TotalObjectSize);

    uint8_t* ptr = reinterpret_cast<uint8_t*>( mem );
    // ボイスをリスト管理
    for ( int i = 0; i < VoiceCount; i++ )
    {
        // Voice 用バッファ
        uint8_t* voicePtr = ptr;
        ptr += sizeof(MultiVoice);

        // AdditionalParam 用バッファ
        OutputAdditionalParam* additionalParamPtr = nullptr;
        const size_t AdditionalParamBufferSize = OutputAdditionalParam::GetRequiredMemSize( config );
        if(AdditionalParamBufferSize != 0)
        {
            additionalParamPtr = reinterpret_cast<OutputAdditionalParam*>(ptr);
            new(additionalParamPtr) OutputAdditionalParam();
            ptr += sizeof( OutputAdditionalParam );

            void* additionalParamBufferPtr = ptr;
            additionalParamPtr->Initialize(additionalParamBufferPtr, AdditionalParamBufferSize, config);
            ptr += AdditionalParamBufferSize;
        }

        m_FreeVoiceList.push_back( *new( voicePtr )MultiVoice(additionalParamPtr));
    }
    NN_SDK_ASSERT( ptr <= reinterpret_cast<uint8_t*>( mem ) + memSize );

    m_Initialized = true;
}

void MultiVoiceManager::Finalize() NN_NOEXCEPT
{
    if ( !m_Initialized ) return;

    StopAllVoices();

    // ボイスをリストから削除
    while ( ! m_FreeVoiceList.empty() )
    {
        MultiVoice& voice = m_FreeVoiceList.front();
        m_FreeVoiceList.pop_front();
        voice.~MultiVoice(); // デストラクタ呼び出し
    }

    m_Initialized = false;
}

void MultiVoiceManager::StopAllVoices() NN_NOEXCEPT
{
    // 再生中のボイスを全てFree
    while ( ! m_PrioVoiceList.empty() )
    {
        MultiVoice& voice = m_PrioVoiceList.front();
        voice.Stop();
        if ( voice.m_Callback != NULL )
        {
            voice.m_Callback(
                &voice,
                MultiVoice::VoiceCallbackStatus_Cancel,
                voice.m_pCallbackData
            );
        }
        voice.Free();
    }
}

MultiVoice* MultiVoiceManager::AllocVoice(
    int voiceChannelCount,
    int priority,
    MultiVoice::VoiceCallback callback,
    void* callbackData
) NN_NOEXCEPT
{
    // ボイスが不足している場合にプライオリティが低いボイスを停止させる
    if ( m_FreeVoiceList.empty() )
    {
        if ( DropLowestPriorityVoice( priority ) == 0 ) return NULL;
    }

    // フリーリストから空きボイスを取得
    MultiVoice& voice = m_FreeVoiceList.front();

    if ( ! voice.Alloc( voiceChannelCount, priority, callback, callbackData ) )
    {
        return NULL;
    }

    // プライオリティを設定してボイスリストに追加
    voice.m_Priority = static_cast<uint8_t>( priority );
    m_FreeVoiceList.erase( m_FreeVoiceList.iterator_to( voice ) );
    AppendVoiceList( &voice );

    return &voice;
}

void MultiVoiceManager::FreeVoice( MultiVoice* voice ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( voice );

    RemoveVoiceList( voice );
}

void MultiVoiceManager::UpdateAllVoiceStatus() NN_NOEXCEPT
{
    // 波形の終端 or DSP 処理落ち防止用のボイスドロップ結果を反映
    for ( VoiceList::iterator itr = m_PrioVoiceList.begin();
          itr != m_PrioVoiceList.end();
        )
    {
        VoiceList::iterator curItr = itr++;
        curItr->UpdateVoiceStatus();
    }
}

void MultiVoiceManager::UpdateAudioFrameVoiceStatus() NN_NOEXCEPT
{
    // 波形の終端 or DSP 処理落ち防止用のボイスドロップ結果を反映
    for ( VoiceList::iterator itr = m_PrioVoiceList.begin();
          itr != m_PrioVoiceList.end();
        )
    {
        VoiceList::iterator curItr = itr++;
        if ( curItr->m_UpdateType == UpdateType_AudioFrame )
        {
            curItr->UpdateVoiceStatus();
        }
    }
}

void MultiVoiceManager::UpdateAllVoices() NN_NOEXCEPT
{
    // すべてのボイスのパラメータ計算
    for ( VoiceList::iterator itr = m_PrioVoiceList.begin();
          itr != m_PrioVoiceList.end();
    )
    {
        VoiceList::iterator curItr = itr++;
        curItr->Calc();
    }

    // すべてのボイスを更新
    for ( VoiceList::iterator itr = m_PrioVoiceList.begin();
          itr != m_PrioVoiceList.end();
    )
    {
        VoiceList::iterator curItr = itr++;
        curItr->Update();
    }
}

void MultiVoiceManager::UpdateAudioFrameVoices() NN_NOEXCEPT
{
    // 更新頻度がオーディオフレームのボイスを更新
    for ( VoiceList::iterator itr = m_PrioVoiceList.begin();
          itr != m_PrioVoiceList.end();
    )
    {
        VoiceList::iterator curItr = itr++;
        if ( curItr->GetUpdateType() == UpdateType_AudioFrame )
        {
            curItr->Calc();
        }
    }

    // 更新頻度がオーディオフレームのボイスを更新
    for ( VoiceList::iterator itr = m_PrioVoiceList.begin();
          itr != m_PrioVoiceList.end();
    )
    {
        VoiceList::iterator curItr = itr++;
        if ( curItr->GetUpdateType() == UpdateType_AudioFrame )
        {
            curItr->Update();
        }
    }
}

void MultiVoiceManager::AppendVoiceList( MultiVoice* voice ) NN_NOEXCEPT
{
    // プライオリティリストに挿入、同プライオリティは後着の音を後ろに繋ぐ
    // 逆順に探索したほうが、同プライオリティの場合に速い。
    VoiceList::reverse_iterator itr = m_PrioVoiceList.rbegin();
    while ( itr != m_PrioVoiceList.rend() )
    {
        if ( itr->m_Priority <= voice->m_Priority )
        {
            break;
        }
        (void)++itr;
    }

    m_PrioVoiceList.insert( itr.base(), *voice );
}
void MultiVoiceManager::RemoveVoiceList( MultiVoice* voice ) NN_NOEXCEPT
{
    m_PrioVoiceList.erase( m_PrioVoiceList.iterator_to( *voice ) );
    m_FreeVoiceList.push_back( *voice );
}

void MultiVoiceManager::ChangeVoicePriority( MultiVoice* voice ) NN_NOEXCEPT
{
    m_PrioVoiceList.erase( m_PrioVoiceList.iterator_to( *voice ) );
    AppendVoiceList( voice );
}

void MultiVoiceManager::UpdateAllVoicesSync( uint32_t syncFlag ) NN_NOEXCEPT
{
    for ( VoiceList::iterator itr = m_PrioVoiceList.begin();
          itr != m_PrioVoiceList.end();
        )
    {
        VoiceList::iterator curItr = itr++;
        if ( curItr->m_IsActive ) curItr->m_SyncFlag |= syncFlag;
    }
}

int MultiVoiceManager::DropLowestPriorityVoice( int priority ) NN_NOEXCEPT
{
    if ( m_PrioVoiceList.empty() ) return 0;

    MultiVoice* dropVoice = &m_PrioVoiceList.front();
    if ( dropVoice->GetPriority() == MultiVoice::PriorityNoDrop )
    {
        return 0;
    }
    if ( dropVoice->GetPriority() > priority ) return 0;

    // もっとも優先度の低いボイスをDrop
    int dropCount = dropVoice->GetSdkVoiceCount();

    MultiVoice::VoiceCallback callbackFunc = dropVoice->m_Callback;
    void* callbackData = dropVoice->m_pCallbackData;

    dropVoice->Stop();
    dropVoice->Free();

    if ( callbackFunc != NULL )
    {
        callbackFunc(
            dropVoice,
            MultiVoice::VoiceCallbackStatus_DropVoice,
            callbackData
        );
    }

    return dropCount;
}

int MultiVoiceManager::GetVoiceCount() const NN_NOEXCEPT
{
    AtkStateAndParameterUpdateLock lock;

    int voiceCount = 0;
    for (
        VoiceList::const_iterator itr = m_PrioVoiceList.begin();
        itr != m_PrioVoiceList.end();
        (void)++itr
    )
    {
        voiceCount += itr->GetSdkVoiceCount();
    }

    return voiceCount;
}

unsigned long MultiVoiceManager::GetActiveCount() const NN_NOEXCEPT
{
    AtkStateAndParameterUpdateLock lock;

    return m_PrioVoiceList.size();
}

unsigned long MultiVoiceManager::GetFreeCount() const NN_NOEXCEPT
{
    AtkStateAndParameterUpdateLock lock;

    return m_FreeVoiceList.size();
}

const MultiVoiceManager::VoiceList& MultiVoiceManager::GetVoiceList() const NN_NOEXCEPT
{
    return m_PrioVoiceList;
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

