﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_LowLevelVoice.h>

#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/atk_Voice.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/atk/atk_Debug.h>
#include <cmath>

namespace
{
    // 以下の値は初期化に使われますが、デフォルト値は外部から与えられます
    const int32_t LowLevelVoiceInitialPriority = nn::audio::VoiceType::PriorityHighest;
    const int LowLevelVoiceInitialSampleRate = 32000;
    const nn::atk::SampleFormat LowLevelVoiceInitialSampleFormat = nn::atk::SampleFormat_PcmS16;
    const nn::atk::detail::VoiceState  LowLevelVoiceInitialVoiceState = nn::atk::detail::VoiceState_Stop;

    //  LowLevelVoice のアライメントを考慮したサイズ
    const size_t LowLevelVoiceAlignedSize = nn::util::align_up( sizeof(nn::atk::detail::LowLevelVoice), nn::audio::BufferAlignSize );

    int ConvertAtkPriorityToAudioPriority(int32_t atkPriority) NN_NOEXCEPT
    {
        if (atkPriority == nn::atk::detail::Voice::PriorityNoDrop)
        {
            return nn::audio::VoiceType::PriorityHighest;
        }
        // atk は優先度が大きいほど高優先だが、nn::audio は優先度が小さいほど高優先のため値をひっくり返す
        return nn::audio::VoiceType::PriorityLowest - atkPriority;
    }

    inline int GetOutputReceiverMixBufferIndex(const nn::atk::OutputReceiver* pOutputReceiver, int channel, int bus)
    {
        return pOutputReceiver->GetChannelCount() * bus + channel;
    }
}

namespace nn { namespace atk { namespace detail {

    LowLevelVoice::LowLevelVoice() NN_NOEXCEPT
    : m_AdpcmParam()
    , m_Voice()
    , m_IsAvailable(false)
    , m_IsSetVoiceSlot(false)
#if defined(ENABLE_NN_ATK_LOWLEVELVOICE_APPEND_PER_BUFFERJUMP)
    , m_IsWaitingForBufferJump(false)
#endif
    , m_VoiceParam()
    , m_Priority(LowLevelVoiceInitialPriority)
    , m_State(LowLevelVoiceInitialVoiceState)
    , m_SampleRate(LowLevelVoiceInitialSampleRate)
    , m_SampleFormat(LowLevelVoiceInitialSampleFormat)
    , m_PlayPosition(0)
    , m_pOutputReceiver(nullptr)
    , m_WaveBufferListBegin(nullptr)
    , m_WaveBufferListEnd(nullptr)
    , m_LastAppendBuffer(nullptr)
    , m_pVoice(nullptr)
    , m_NodeId(0)
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    , m_IsVoiceParamDirty(true)
#endif
    {
    }

    void LowLevelVoice::Initialize() NN_NOEXCEPT
    {
        m_IsAvailable = true;

#if defined(ENABLE_NN_ATK_LOWLEVELVOICE_APPEND_PER_BUFFERJUMP)
        m_IsWaitingForBufferJump = false;
#endif
        m_Priority = LowLevelVoiceInitialPriority;
        m_SampleRate = LowLevelVoiceInitialSampleRate;
        m_SampleFormat = LowLevelVoiceInitialSampleFormat;
        m_PlayPosition = 0;
        m_pOutputReceiver = nullptr;
        m_NodeId = 0;
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        m_IsVoiceParamDirty = true;
#endif
    }

    void LowLevelVoice::Finalize() NN_NOEXCEPT
    {
        FreeAllWaveBuffer();

        if (m_IsSetVoiceSlot)
        {
            nn::audio::ReleaseVoiceSlot(&driver::HardwareManager::GetInstance().GetAudioRendererConfig(), &m_Voice);
            m_IsSetVoiceSlot = false;
        }

        m_pVoice = nullptr;
        m_State = LowLevelVoiceInitialVoiceState;

        m_IsAvailable = false;
    }

    bool LowLevelVoice::IsAvailable() const NN_NOEXCEPT
    {
        return m_IsAvailable;
    }

    void LowLevelVoice::SetAvailable(bool isAvailable) NN_NOEXCEPT
    {
        m_IsAvailable = isAvailable;
    }

    bool LowLevelVoice::IsVoiceDroppedFlagOn() const NN_NOEXCEPT
    {
        if ( !nn::audio::IsVoiceValid(&m_Voice) )
        {
            return false;
        }

        return nn::audio::IsVoiceDroppedFlagOn(&m_Voice);
    }

    void LowLevelVoice::AppendWaveBuffer( WaveBuffer* waveBuffer ) NN_NOEXCEPT
    {
        waveBuffer->next = nullptr;
        waveBuffer->status = WaveBuffer::Status_Wait;

        if ( m_WaveBufferListEnd == nullptr )
        {
            m_WaveBufferListBegin = waveBuffer;
        }
        else
        {
            m_WaveBufferListEnd->next = waveBuffer;
        }
        m_WaveBufferListEnd = waveBuffer;
    }

    void LowLevelVoice::FreeAllWaveBuffer() NN_NOEXCEPT
    {
        WaveBuffer* waveBuffer = m_WaveBufferListBegin;
        while( waveBuffer != nullptr ) {
            waveBuffer->status = WaveBuffer::Status_Done;
            waveBuffer = waveBuffer->next;
        }
        m_WaveBufferListBegin = nullptr;
        m_WaveBufferListEnd = nullptr;
        m_LastAppendBuffer = nullptr;
    }

    void LowLevelVoice::UpdateState(OutputMode outputMode) NN_NOEXCEPT
    {
        // nn::audio::AcquireVoiceSlot() が呼ばれる前に到達した場合はスキップする
        if ( !nn::audio::IsVoiceValid(&m_Voice) )
        {
            return;
        }

        bool isRun = ( nn::audio::GetVoicePlayState(&m_Voice) ==  nn::audio::VoiceType::PlayState_Play );

        switch( m_State ) {
        case VoiceState_Play:
            UpdateStatePlay( isRun, outputMode );
            break;
        case VoiceState_Stop:
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            UpdateStateStop( isRun );
#endif
            break;
        case VoiceState_Pause:
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            UpdateStatePause( isRun, outputMode );
#endif
            break;
        default:
            NN_SDK_ASSERT(false);
            break;
        }
    }

    void LowLevelVoice::SetPriority( int32_t priority ) NN_NOEXCEPT
    {
        m_Priority = priority;
        // nn::audio::AcquireVoiceSlot() が呼ばれた後なら優先度設定関数を呼び出す
        if (nn::audio::IsVoiceValid(&m_Voice))
        {
            nn::audio::SetVoicePriority(&m_Voice, ConvertAtkPriorityToAudioPriority(m_Priority));
        }
    }

    bool LowLevelVoice::AllocVoice() NN_NOEXCEPT
    {
        const nn::audio::AdpcmParameter* pParameter = nullptr;
        size_t paramSize = 0;

        nn::audio::SampleFormat sampleFormat = nn::audio::SampleFormat_PcmInt16;
        switch(m_SampleFormat)
        {
        case SampleFormat_PcmS8:
            sampleFormat = nn::audio::SampleFormat_PcmInt8;
            break;
        case SampleFormat_PcmS16:
            sampleFormat = nn::audio::SampleFormat_PcmInt16;
            break;
        case SampleFormat_DspAdpcm:
            sampleFormat = nn::audio::SampleFormat_Adpcm;

            pParameter = &m_AdpcmParam;
            paramSize = sizeof(nn::audio::AdpcmParameter);

            break;
        default:
            break;
        }

        int audioPriority = ConvertAtkPriorityToAudioPriority(m_Priority);

        bool result = nn::audio::AcquireVoiceSlot(&driver::HardwareManager::GetInstance().GetAudioRendererConfig(), &m_Voice, m_SampleRate, 1, sampleFormat, audioPriority, pParameter, paramSize);

        NN_SDK_ASSERT_NOT_NULL( m_pOutputReceiver );
        switch( m_pOutputReceiver->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
                nn::audio::SetVoiceDestination(&driver::HardwareManager::GetInstance().GetAudioRendererConfig(), &m_Voice, reinterpret_cast<SubMix*>(m_pOutputReceiver)->GetAudioSubMixInstance());
                break;
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
                nn::audio::SetVoiceDestination(&driver::HardwareManager::GetInstance().GetAudioRendererConfig(), &m_Voice, reinterpret_cast<FinalMix*>(m_pOutputReceiver)->GetAudioFinalMixInstance());
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        m_NodeId = nn::audio::GetVoiceNodeId(&m_Voice);

        return result;
    }

    void LowLevelVoice::SetState( VoiceState state ) NN_NOEXCEPT
    {
        m_State = state;

        if (state == VoiceState_Play)
        {
            if (!m_IsSetVoiceSlot)
            {
                if (!AllocVoice())
                {
                    NN_ATK_WARNING("failed to alloc voice.");
                }
                m_IsSetVoiceSlot = true;
            }
        }
        else if (state == VoiceState_Stop)
        {
            if (m_IsSetVoiceSlot)
            {
                nn::audio::ReleaseVoiceSlot(&driver::HardwareManager::GetInstance().GetAudioRendererConfig(), &m_Voice);
                m_IsSetVoiceSlot = false;
            }
        }
    }

    void LowLevelVoice::UpdateVoiceInfo( VoiceInfo* voiceInfo ) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(voiceInfo);

        voiceInfo->userId = m_pVoice;
        voiceInfo->voiceState = m_State;
        volatile WaveBuffer* pListBegin = m_WaveBufferListBegin;
        if ( pListBegin != NULL )
        {
            voiceInfo->waveBufferStatus = pListBegin->status;
            voiceInfo->waveBufferTag = pListBegin->userParam;
        }
        else
        {
            voiceInfo->waveBufferStatus = WaveBuffer::Status_Free;
        }
        voiceInfo->playPosition = static_cast<uint32_t>(m_PlayPosition);
    }

#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    void LowLevelVoice::UpdateVoiceParam( OutputMode outputMode ) NN_NOEXCEPT
    {
        // nn::audio::AcquireVoiceSlot() が呼ばれる前に到達した場合はスキップする
        if(!nn::audio::IsVoiceValid(&m_Voice) || m_State == VoiceState_Stop)
        {
            return;
        }

        if(m_State == VoiceState_Pause && nn::audio::GetVoicePlayState(&m_Voice) ==  nn::audio::VoiceType::PlayState_Play)
        {
            return;
        }

        UpdateVoiceParamImpl(m_VoiceParam, outputMode);
    }

    void LowLevelVoice::ApplyVoiceStatus(OutputMode outputMode) NN_NOEXCEPT
    {
        // nn::audio::AcquireVoiceSlot() が呼ばれる前に到達した場合はスキップする
        if ( !nn::audio::IsVoiceValid(&m_Voice) )
        {
            return;
        }

        bool isRun = ( nn::audio::GetVoicePlayState(&m_Voice) ==  nn::audio::VoiceType::PlayState_Play );

        switch( m_State ) {
        case VoiceState_Play:
            if(!isRun)
            {
                UpdateWaveBufferOnStopState( outputMode );
            }
            break;
        case VoiceState_Stop:
            UpdateStateStop( isRun );
            break;
        case VoiceState_Pause:
            UpdateStatePause( isRun, outputMode );
            break;
        default:
            NN_SDK_ASSERT(false);
            break;
        }
    }
#endif

    void LowLevelVoice::UpdateVoiceParamImpl( const VoiceParam& voiceParam, OutputMode outputMode ) NN_NOEXCEPT
    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (!m_IsVoiceParamDirty)
        {
            // VoiceCommandId_UpdateParam のコマンドが積まれてない場合、更新をスキップ
            return;
        }
        m_IsVoiceParamDirty = false;
#endif
        // Volume
        UpdateVolume( voiceParam );

        // Pitch
        UpdatePitch( voiceParam );

        // Filter
        UpdateBiquadFilter( voiceParam );
        UpdateLowPassFilter( voiceParam );

        // TvMix
        UpdateMixVolume( voiceParam.m_TvMix, outputMode );
    }

    void LowLevelVoice::UpdateStatePlay( bool isRun, OutputMode outputMode ) NN_NOEXCEPT
    {
        UpdateWaveBuffer( isRun, outputMode );
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        UpdateVoiceParamImpl( m_VoiceParam, outputMode );
#endif
    }

    void LowLevelVoice::UpdateStateStop( bool isRun ) NN_NOEXCEPT
    {
        if ( isRun )
        {
            nn::audio::SetVoicePlayState(&m_Voice, nn::audio::VoiceType::PlayState_Stop);
        }
    }

    void LowLevelVoice::UpdateStatePause( bool isRun, OutputMode outputMode ) NN_NOEXCEPT
    {
        NN_UNUSED(outputMode);
        if ( isRun )
        {
            // 即時 Pause 命令が来た場合に対応するため、 Pause 時にもボイスのパラメーター更新を行う
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
            UpdateVoiceParamImpl( m_VoiceParam, outputMode );
#endif
            nn::audio::SetVoicePlayState(&m_Voice, nn::audio::VoiceType::PlayState_Pause);
        }
    }

    void LowLevelVoice::UpdatePlayPosition() NN_NOEXCEPT
    {
        if ( m_IsSetVoiceSlot )
        {
            int64_t playedSampleCount = nn::audio::GetVoicePlayedSampleCount(&m_Voice);
            int64_t startOffset = 0;

            WaveBuffer* currentWaveBuffer = m_WaveBufferListBegin;
            if (currentWaveBuffer != nullptr)
            {
                startOffset = currentWaveBuffer->sampleOffset;
                if (currentWaveBuffer->loopFlag)
                {
                    // nn::audio::GetVoicePlayedSampleCount はループ波形ではサンプル位置が巻き戻らないため補正
                    int64_t loopSampleLength = currentWaveBuffer->sampleLength - currentWaveBuffer->sampleOffset;
                    if (playedSampleCount > loopSampleLength)
                    {
                        playedSampleCount %= loopSampleLength;
                    }
                }
            }

            m_PlayPosition = static_cast<position_t>(startOffset + playedSampleCount);
        }
    }

    void LowLevelVoice::UpdateWaveBuffer( bool isRun, OutputMode outputMode ) NN_NOEXCEPT
    {
        NN_UNUSED(outputMode);
        if ( isRun )
        {
            UpdateWaveBufferOnPlayState();
            UpdatePlayPosition();
        }
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        else
        {
            UpdateWaveBufferOnStopState( outputMode );
        }
#endif
    }

    void LowLevelVoice::UpdateWaveBufferOnPlayState() NN_NOEXCEPT
    {
        WaveBuffer* currentWaveBuffer = m_WaveBufferListBegin;
        if (currentWaveBuffer == nullptr)
        {
            return;
        }

        // 現在の WaveBuffer の再生が終わりバッファジャンプが起きた
        const nn::audio::WaveBuffer* pWaveBuffer = nn::audio::GetReleasedWaveBuffer(&m_Voice);
        if (pWaveBuffer != nullptr)
        {
            // 現在の波形バッファを終了状態に変更
            m_WaveBufferListBegin = currentWaveBuffer->next;
            currentWaveBuffer->status = WaveBuffer::Status_Done;

            // 次の波形バッファがあれば次のバッファに移動
            if (m_WaveBufferListBegin != nullptr)
            {
                currentWaveBuffer = m_WaveBufferListBegin;
                currentWaveBuffer->status = WaveBuffer::Status_Play;
            }
            else
            {
                m_WaveBufferListEnd = nullptr;
                nn::audio::SetVoicePlayState(&m_Voice, nn::audio::VoiceType::PlayState_Stop);
            }

#if defined(ENABLE_NN_ATK_LOWLEVELVOICE_APPEND_PER_BUFFERJUMP)
            m_IsWaitingForBufferJump = false;
#endif
        }

#if defined(ENABLE_NN_ATK_LOWLEVELVOICE_APPEND_PER_BUFFERJUMP)
        if (!m_IsWaitingForBufferJump)
        {
            if (currentWaveBuffer->next != nullptr)
            {
                AppendWaveBufferToVoice(currentWaveBuffer->next);
                m_IsWaitingForBufferJump = true;
            }
        }
#else
        NN_SDK_ASSERT_NOT_NULL( m_LastAppendBuffer );
        if ( m_LastAppendBuffer->next != nullptr )
        {
            if ( AppendWaveBufferToVoice( m_LastAppendBuffer->next ) )
            {
                m_LastAppendBuffer = m_LastAppendBuffer->next;
            }
        }
#endif
    }

    void LowLevelVoice::UpdateWaveBufferOnStopState( OutputMode outputMode ) NN_NOEXCEPT
    {
        WaveBuffer* currentWaveBuffer = m_WaveBufferListBegin;
        if (currentWaveBuffer == nullptr)
        {
            return;
        }

        // 一時停止していた場合
        if ( nn::audio::GetVoicePlayState(&m_Voice) ==  nn::audio::VoiceType::PlayState_Pause )
        {
            nn::audio::SetVoicePlayState(&m_Voice, nn::audio::VoiceType::PlayState_Play);
            return;
        }

        // 再生終了の場合
        if ( currentWaveBuffer->status == WaveBuffer::Status_Play )
        {
            FreeAllWaveBuffer();

            return;
        }

        // 再生待ちの場合
        if ( currentWaveBuffer->status == WaveBuffer::Status_Wait )
        {
            AppendWaveBufferToVoice(currentWaveBuffer);
            m_LastAppendBuffer = currentWaveBuffer;
#if defined(ENABLE_NN_ATK_LOWLEVELVOICE_APPEND_PER_BUFFERJUMP)
            m_IsWaitingForBufferJump = false;
#endif

            nn::audio::SetVoicePlayState(&m_Voice, nn::audio::VoiceType::PlayState_Play);

            // Mix
            UpdateMixVolume(m_VoiceParam.m_TvMix, outputMode);

            currentWaveBuffer->status = WaveBuffer::Status_Play;
        }
    }

    bool LowLevelVoice::AppendWaveBufferToVoice(WaveBuffer* waveBuffer) NN_NOEXCEPT
    {
        const nn::audio::AdpcmContext* pContext = nullptr;
        size_t contextSize = 0;

        if (m_SampleFormat == SampleFormat_DspAdpcm)
        {
            pContext = &waveBuffer->pAdpcmContext->audioAdpcmContext;
            contextSize = sizeof(nn::audio::AdpcmContext);
        }

        nn::audio::WaveBuffer audioWaveBuffer;
        audioWaveBuffer.buffer = waveBuffer->bufferAddress;
        audioWaveBuffer.size = waveBuffer->bufferSize;
        audioWaveBuffer.startSampleOffset = static_cast<int32_t>(waveBuffer->sampleOffset);
        audioWaveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer->sampleLength);
        audioWaveBuffer.loop = waveBuffer->loopFlag ? 1 : 0;
        audioWaveBuffer.pContext = pContext;
        audioWaveBuffer.contextSize = contextSize;
        audioWaveBuffer.isEndOfStream = true;

        NN_ATK_LOG_APICALL_AUDIO_LOW( "AppendWaveBuffer [%p][addr:%p-%p][size:%zu]", &m_Voice, audioWaveBuffer.buffer,
            reinterpret_cast<const int8_t*>( audioWaveBuffer.buffer ) + audioWaveBuffer.size, audioWaveBuffer.size );
        return nn::audio::AppendWaveBuffer(&m_Voice, &audioWaveBuffer);
    }

    void LowLevelVoice::UpdateMixVolume(const OutputMix& outputMix, OutputMode outputMode) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( m_pOutputReceiver );

        const bool isSurroundMode = outputMode == OutputMode_Surround;
        const bool isEffectEnabled = driver::HardwareManager::GetInstance().IsEffectInitialized();
        const int busCount = isEffectEnabled ? m_pOutputReceiver->GetBusCount() : 1;
        const int channelCount = m_pOutputReceiver->GetChannelCount();

        for (auto busIndex = 0; busIndex < busCount; ++ busIndex)
        {
            for (auto channelIndex = 0; channelIndex < channelCount; ++channelIndex)
            {
                const int mixBufferIndex = GetOutputReceiverMixBufferIndex(m_pOutputReceiver, channelIndex, busIndex);

                const float volume = ( isSurroundMode || channelIndex < 2 )
                    ? GetClampedVoiceVolume(outputMix.channelGain[mixBufferIndex]) : 0.0f;
                SetVoiceMixVolume(volume, mixBufferIndex);
            }
        }
    }

    void LowLevelVoice::UpdateVolume(const VoiceParam& voiceParam) NN_NOEXCEPT
    {
        float volume = GetClampedVoiceVolume(voiceParam.m_Volume);
        nn::audio::SetVoiceVolume(&m_Voice, volume);
    }

    void LowLevelVoice::UpdatePitch(const VoiceParam& voiceParam) NN_NOEXCEPT
    {
        float pitch =
            nn::atk::detail::fnd::Clamp(
                voiceParam.m_Pitch,
                nn::audio::VoiceType::GetPitchMin(),
                nn::audio::VoiceType::GetPitchMax()
            );

        nn::audio::SetVoicePitch(&m_Voice, pitch);
    }

    void LowLevelVoice::UpdateBiquadFilter(const VoiceParam& voiceParam) NN_NOEXCEPT
    {
        nn::audio::BiquadFilterParameter parameter;

        parameter.denominator[0] = voiceParam.m_BiquadFilterCoefficients.a1;
        parameter.denominator[1] = voiceParam.m_BiquadFilterCoefficients.a2;
        parameter.numerator[0] = voiceParam.m_BiquadFilterCoefficients.b0;
        parameter.numerator[1] = voiceParam.m_BiquadFilterCoefficients.b1;
        parameter.numerator[2] = voiceParam.m_BiquadFilterCoefficients.b2;

        parameter.enable = voiceParam.m_BiquadFilterFlag;

        nn::audio::SetVoiceBiquadFilterParameter(&m_Voice, 0, parameter);
    }

    void LowLevelVoice::UpdateLowPassFilter(const VoiceParam& voiceParam) NN_NOEXCEPT
    {
        // TODO: 正規化周波数への対応について検討する
        if (voiceParam.m_MonoFilterCutoff > 0 && voiceParam.m_MonoFilterCutoff < 16000 )
        {
            nn::atk::BiquadFilterCoefficients coefficients;
            if (driver::HardwareManager::GetInstance().IsPreviousSdkVersionLowPassFilterCompatible())
            {
                coefficients = nn::atk::detail::Util::CalcLowPassFilterCoefficients(voiceParam.m_MonoFilterCutoff, m_SampleRate, false);
            }
            else
            {
                coefficients = nn::atk::detail::Util::CalcLowPassFilterCoefficients(voiceParam.m_MonoFilterCutoff, driver::HardwareManager::GetInstance().GetRendererSampleRate(), true);
            }
            nn::audio::BiquadFilterParameter parameter;
            parameter.denominator[0] = coefficients.a1;
            parameter.denominator[1] = coefficients.a2;
            parameter.numerator[0] = coefficients.b0;
            parameter.numerator[1] = coefficients.b1;
            parameter.numerator[2] = coefficients.b2;
            parameter.enable = voiceParam.m_MonoFilterFlag;
            nn::audio::SetVoiceBiquadFilterParameter(&m_Voice, 1, parameter);
        }
        else
        {
            // MultiVoice::CalcLpf で計算された値が 16000 以上の場合など、係数計算が必要のない場合の処理
            nn::audio::BiquadFilterParameter parameter;
            parameter.denominator[0] = 0;
            parameter.denominator[1] = 0;
            parameter.numerator[0] = 0;
            parameter.numerator[1] = 0;
            parameter.numerator[2] = 0;
            parameter.enable = voiceParam.m_MonoFilterFlag;
            nn::audio::SetVoiceBiquadFilterParameter(&m_Voice, 1, parameter);
        }
    }

    void LowLevelVoice::SetVoiceMixVolume(float mixVolume, int destinationIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( m_pOutputReceiver );
        switch( m_pOutputReceiver->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
                nn::audio::SetVoiceMixVolume(&m_Voice, reinterpret_cast<SubMix*>(m_pOutputReceiver)->GetAudioSubMixInstance(), mixVolume, 0, destinationIndex);
                break;
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
                nn::audio::SetVoiceMixVolume(&m_Voice, reinterpret_cast<FinalMix*>(m_pOutputReceiver)->GetAudioFinalMixInstance(), mixVolume, 0, destinationIndex);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    float LowLevelVoice::GetClampedVoiceVolume(float volume) NN_NOEXCEPT
    {
        return nn::atk::detail::fnd::FloatClamp(
            volume,
            nn::audio::VoiceType::GetVolumeMin(),
            nn::audio::VoiceType::GetVolumeMax()
        );
    }

    NN_DEFINE_STATIC_CONSTANT( const int LowLevelVoiceAllocator::Unassigned );

    LowLevelVoiceAllocator::LowLevelVoiceAllocator() NN_NOEXCEPT
    : m_pVoiceArray(nullptr)
    , m_ppVoiceTable(nullptr)
    , m_UsingCount(0)
    , m_pAssignedTableIndex(nullptr)
    , m_VoiceCount(0)
    , m_DroppedVoiceCount(0)
    {
    }

    size_t LowLevelVoiceAllocator::GetRequiredMemSize(int voiceCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL( voiceCount, 0 );

        size_t result = 0;
        result += sizeof(LowLevelVoice*) * voiceCount;
        result += sizeof(int) * voiceCount;
        result += nn::audio::BufferAlignSize + LowLevelVoiceAlignedSize * voiceCount;

        return result;
    }
    void LowLevelVoiceAllocator::Initialize(int voiceCount, void* mem, size_t memSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_GREATER_EQUAL( voiceCount, 0 );
        NN_SDK_ASSERT_GREATER_EQUAL( memSize, GetRequiredMemSize( voiceCount ) );
        NN_UNUSED( memSize );
        m_VoiceCount = voiceCount;

        nn::util::BytePtr ptr( mem );
        m_ppVoiceTable = ptr.Get<LowLevelVoice*>();
        ptr.Advance( sizeof(LowLevelVoice*) * m_VoiceCount );
        m_pAssignedTableIndex = ptr.Get<int>();
        ptr.Advance( sizeof(int) * m_VoiceCount );

        ptr.AlignUp( nn::audio::BufferAlignSize );
        m_pVoiceArray = ptr.Get();
        for (int i = 0; i < m_VoiceCount; ++i)
        {
            m_ppVoiceTable[i] = new( ptr.Get<LowLevelVoice>() ) LowLevelVoice();
            ptr.Advance( LowLevelVoiceAlignedSize );
        }

        for (int i = 0; i < m_VoiceCount; ++i)
        {
            m_pAssignedTableIndex[i] = Unassigned;
        }

        m_UsingCount = 0;
    }
    void LowLevelVoiceAllocator::Finalize() NN_NOEXCEPT
    {
        m_pVoiceArray = nullptr;
        m_ppVoiceTable = nullptr;
        m_UsingCount = 0;
        m_pAssignedTableIndex = nullptr;
        m_VoiceCount = 0;
        m_DroppedVoiceCount = 0;
    }

    void LowLevelVoiceAllocator::UpdateAllVoiceState(OutputMode outputMode) NN_NOEXCEPT
    {
        int droppedVoiceCount = 0;
        for (int i = 0; i < m_UsingCount; ++i)
        {
            if ( m_ppVoiceTable[i]->IsVoiceDroppedFlagOn() && m_ppVoiceTable[i]->IsAvailable() )
            {
                m_ppVoiceTable[i]->SetAvailable(false);
                droppedVoiceCount++;
                continue;
            }

            m_ppVoiceTable[i]->UpdateState(outputMode);
        }
        m_DroppedVoiceCount = droppedVoiceCount;
    }

#ifndef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    void LowLevelVoiceAllocator::UpdateAllVoiceParam(OutputMode outputMode) NN_NOEXCEPT
    {
        for (int i = 0; i < m_UsingCount; ++i)
        {
            m_ppVoiceTable[i]->UpdateVoiceParam(outputMode);
            m_ppVoiceTable[i]->ApplyVoiceStatus(outputMode);
        }
    }
#endif

    LowLevelVoice* LowLevelVoiceAllocator::AllocVoice() NN_NOEXCEPT
    {
        if (m_UsingCount < m_VoiceCount)
        {
            const int assignTableIndex = m_UsingCount;
            ++m_UsingCount;

            //  voiceArrayIndex は pVoice が m_VoiceArray の何番目にあるかを表します。
            //  つまり、&m_VoiceArray[voiceArrayIndex] == pVoice を満たします。
            LowLevelVoice* pVoice = m_ppVoiceTable[assignTableIndex];
            const ptrdiff_t voiceArrayIndex = GetVoiceArrayIndex( pVoice );
            NN_SDK_ASSERT_RANGE(voiceArrayIndex, 0, m_VoiceCount);
            m_pAssignedTableIndex[voiceArrayIndex] = assignTableIndex;

            pVoice->Initialize();
            return pVoice;
        }

        return nullptr;
    }

    void LowLevelVoiceAllocator::FreeVoice(LowLevelVoice* pVoice) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pVoice);
        pVoice->Finalize();

        //  voiceArrayIndex は pVoice が m_VoiceArray の何番目にあるかを表します。
        const ptrdiff_t voiceArrayIndex = GetVoiceArrayIndex( pVoice );
        NN_SDK_ASSERT_RANGE(voiceArrayIndex, 0, m_VoiceCount);

        //  voiceTableIndex は pVoice が m_pVoiceTable の何番目に割り当てられているかを表します。
        const int voiceTableIndex = m_pAssignedTableIndex[voiceArrayIndex];
        NN_SDK_ASSERT_NOT_EQUAL(voiceTableIndex, Unassigned);  //  未割り当ての確認

        //  使用中のボイスのうち、m_pVoiceTable の最後のボイスと pVoice を交換します。
        const int lastTableIndex = m_UsingCount - 1;
        --m_UsingCount;
        NN_SDK_ASSERT_GREATER_EQUAL(m_UsingCount, 0);

        const ptrdiff_t lastArrayIndex = GetVoiceArrayIndex( m_ppVoiceTable[lastTableIndex] );
        NN_SDK_ASSERT_RANGE(lastArrayIndex , 0, m_VoiceCount);
        m_ppVoiceTable[voiceTableIndex] = m_ppVoiceTable[lastTableIndex];
        m_ppVoiceTable[lastTableIndex] = pVoice;
        m_pAssignedTableIndex[lastArrayIndex] = voiceTableIndex;
        m_pAssignedTableIndex[voiceArrayIndex] = Unassigned;
    }

    int LowLevelVoiceAllocator::GetDroppedVoiceCount() const NN_NOEXCEPT
    {
        return m_DroppedVoiceCount;
    }

    ptrdiff_t LowLevelVoiceAllocator::GetVoiceArrayIndex(LowLevelVoice* pVoice) NN_NOEXCEPT
    {
        return ( reinterpret_cast<char*>( pVoice ) - reinterpret_cast<char*>( m_pVoiceArray ) ) / LowLevelVoiceAlignedSize;
    }
}}}
