﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_ChannelManager.h>

namespace nn {
namespace atk {
namespace detail {
namespace driver {

/* ========================================================================
        ChannelManager class
   ======================================================================== */

ChannelManager& ChannelManager::GetInstance() NN_NOEXCEPT
{
    static ChannelManager instance;
    return instance;
}

ChannelManager::ChannelManager() NN_NOEXCEPT
: m_IsInitialized( false )
, m_ChannelCount( 0 )
, m_pAdditionalParamPool( nullptr )
, m_pAdditionalParamBufferPool( nullptr )
, m_AdditionalParamBufferSizePerChannel( 0 )
{
}

size_t ChannelManager::GetObjectSize( const detail::SoundInstanceConfig& config ) NN_NOEXCEPT
{
    // Channel (inclide alignment)
    size_t result = nn::util::align_up( sizeof( Channel ), nn::audio::BufferAlignSize );

    const size_t AdditionalParamBufferSize = OutputAdditionalParam::GetRequiredMemSize( config );
    if(AdditionalParamBufferSize != 0)
    {
        result += sizeof( OutputAdditionalParam );
        result += nn::util::align_up( AdditionalParamBufferSize, BufferPool::DefaultAlignment );
    }
    return result;
}

size_t ChannelManager::GetRequiredMemSize( int channelCount, const detail::SoundInstanceConfig& config ) NN_NOEXCEPT
{
    return GetObjectSize(config) * channelCount
        + sizeof( InstancePool<OutputAdditionalParam> ) // インスタンスプール用
        + sizeof( BufferPool )
        + nn::audio::BufferAlignSize;   //  アラインマージン
}

void ChannelManager::Initialize( void* mem, size_t memSize, int channelCount, const detail::SoundInstanceConfig& config ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(mem);
    NN_SDK_ASSERT_GREATER_EQUAL(memSize, GetRequiredMemSize(channelCount, config));
    if ( m_IsInitialized ) return;

    // 各種インスタンスのバッファ計算
    m_AdditionalParamBufferSizePerChannel = nn::util::align_up( OutputAdditionalParam::GetRequiredMemSize( config ), BufferPool::DefaultAlignment );
    const bool IsAdditionalParamEnabled = m_AdditionalParamBufferSizePerChannel != 0;
    const size_t AdditionalParamPoolSize = IsAdditionalParamEnabled ? sizeof( InstancePool<OutputAdditionalParam> ) : 0;
    const size_t AdditionalParamSize = IsAdditionalParamEnabled ? sizeof( OutputAdditionalParam ) * channelCount: 0;
    const size_t AdditionalParamBufferPoolSize = IsAdditionalParamEnabled ? sizeof( BufferPool ) : 0;
    const size_t AdditionalParamBufferSize = IsAdditionalParamEnabled ? m_AdditionalParamBufferSizePerChannel * channelCount : 0;
    const size_t ChannelPoolSize = memSize - AdditionalParamPoolSize - AdditionalParamSize - AdditionalParamBufferPoolSize - AdditionalParamBufferSize;

    // Channel
    uint8_t* ptr = static_cast<uint8_t*>(mem);
    m_ChannelCount = m_Pool.Create( ptr, ChannelPoolSize, nn::audio::BufferAlignSize );
    ptr += ChannelPoolSize;

    int count = 0;
    NN_UNUSED(count);

    // AdditionalParam
    if(IsAdditionalParamEnabled)
    {
        m_pAdditionalParamPool = reinterpret_cast<InstancePool<OutputAdditionalParam>*>(ptr);
        new(m_pAdditionalParamPool) InstancePool<OutputAdditionalParam>();
        ptr += AdditionalParamPoolSize;
        count = m_pAdditionalParamPool->Create( ptr, AdditionalParamSize );
        NN_SDK_ASSERT_EQUAL(count, m_ChannelCount);
        ptr += AdditionalParamSize;

        m_pAdditionalParamBufferPool = reinterpret_cast<BufferPool*>(ptr);
        new(m_pAdditionalParamBufferPool) BufferPool();
        ptr += AdditionalParamBufferPoolSize;
        count = m_pAdditionalParamBufferPool->Create( ptr, AdditionalParamBufferSize, OutputAdditionalParam::GetRequiredMemSize( config ) );
    }

    NN_SDK_ASSERT( ptr <= reinterpret_cast<uint8_t*>( mem ) + memSize );
    m_IsInitialized = true;
    m_SoundInstanceConfig = config;
}

void ChannelManager::Finalize() NN_NOEXCEPT
{
    if ( !m_IsInitialized ) return;

    // リスト中の全ボイスを停止
    for ( ChannelList::iterator itr = m_ChannelList.begin();
          itr != m_ChannelList.end();
        )
    {
        Channel* channel = &*itr;
        itr++;

        channel->Stop();
        channel->CallChannelCallback( Channel::ChannelCallbackStatus_Cancel );
        Free(channel);
    }

    NN_SDK_ASSERT( m_ChannelList.empty() );

    if(m_pAdditionalParamPool != nullptr)
    {
        m_pAdditionalParamPool->Destroy();
        m_pAdditionalParamPool = nullptr;
    }
    if(m_pAdditionalParamBufferPool != nullptr)
    {
        m_pAdditionalParamBufferPool->Destroy();
        m_pAdditionalParamBufferPool = nullptr;
    }
    m_Pool.Destroy();

    m_IsInitialized = false;
}

Channel* ChannelManager::Alloc() NN_NOEXCEPT
{
    // Channel
    Channel* channel = m_Pool.Alloc();
    if ( channel == NULL ) return NULL;

    // Additional Param
    OutputAdditionalParam* pAdditionalParam = nullptr;
    if ( m_pAdditionalParamPool != nullptr )
    {
        pAdditionalParam = m_pAdditionalParamPool->Alloc();
        if ( pAdditionalParam == nullptr )
        {
            m_Pool.Free(channel);
            return nullptr;
        }

        void* additionalParamBuffer = m_pAdditionalParamBufferPool->Alloc();
        if (additionalParamBuffer == nullptr)
        {
            m_Pool.Free(channel);
            m_pAdditionalParamPool->Free(pAdditionalParam);
            return nullptr;
        }

        pAdditionalParam->Initialize(additionalParamBuffer, m_AdditionalParamBufferSizePerChannel, m_SoundInstanceConfig);
        channel->SetTvAdditionalParamAddr(pAdditionalParam);
    }

    m_ChannelList.push_back( *channel );
    return channel;
}
void ChannelManager::Free( Channel* channel ) NN_NOEXCEPT
{
    if(m_pAdditionalParamBufferPool != nullptr && m_pAdditionalParamPool != nullptr)
    {
        OutputAdditionalParam* pAdditionalParam = channel->GetTvAdditionalParamAddr();
        m_pAdditionalParamBufferPool->Free(pAdditionalParam->GetBufferAddr());
        m_pAdditionalParamPool->Free(pAdditionalParam);
    }

    m_ChannelList.erase( m_ChannelList.iterator_to( *channel ) );
    m_Pool.Free( channel );
}

void ChannelManager::UpdateAllChannel() NN_NOEXCEPT
{
    for ( ChannelList::iterator itr = m_ChannelList.begin();
          itr != m_ChannelList.end();
        )
    {
        Channel* channel = &*itr;
        itr++;

        channel->Update( true );
        if ( ! channel->IsActive() ) {
            channel->CallChannelCallback( Channel::ChannelCallbackStatus_Stopped );
            Free(channel);
        }
    }
}

void ChannelManager::UpdateAudioFrameChannel() NN_NOEXCEPT
{
    for ( ChannelList::iterator itr = m_ChannelList.begin();
        itr != m_ChannelList.end();
        )
    {
        Channel* channel = &*itr;
        itr++;

        if ( channel->IsActive() && channel->GetUpdateType() == UpdateType_AudioFrame )
        {
            channel->Update( true );
        }
        if ( ! channel->IsActive() ) {
            channel->CallChannelCallback( Channel::ChannelCallbackStatus_Stopped );
            Free(channel);
        }
    }
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

