﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Channel.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_MultiVoiceManager.h>
#include <nn/atk/atk_ChannelManager.h>
#include <nn/atk/atk_Global.h>
#include <nn/atk/atk_DisposeCallbackManager.h>

namespace nn {
namespace atk {
namespace detail {
namespace driver {

NN_DEFINE_STATIC_CONSTANT( const int Channel::ChannelCount );
NN_DEFINE_STATIC_CONSTANT( const int Channel::ChannelMin );
NN_DEFINE_STATIC_CONSTANT( const int Channel::ChannelMax );
NN_DEFINE_STATIC_CONSTANT( const int Channel::PriorityRelease );
NN_DEFINE_STATIC_CONSTANT( const int Channel::KeyInit );
NN_DEFINE_STATIC_CONSTANT( const int Channel::OriginalKeyInit );
NN_DEFINE_STATIC_CONSTANT( const uint8_t Channel::SilenceVolumeMax );
NN_DEFINE_STATIC_CONSTANT( const uint8_t Channel::SilenceVolumeMin );
NN_DEFINE_STATIC_CONSTANT( const int Channel::WaveBufferMax );
NN_DEFINE_STATIC_CONSTANT( const int Channel::ModCount );

// SILENCE_VOLUME_MAX の逆数です。割り算の掛け算化による高速化が狙いです。
const float Channel::SilenceVolumeMaxR = 1.0f / static_cast<float>(SilenceVolumeMax);

namespace
{

uint8_t GetNwInterpolationTypeFromHardwareManager() NN_NOEXCEPT
{
    uint8_t result = 0;  // 0: ４点補間, 1: 線形補間, 2: 補間なし
                    // (BankFile::VelocityRegion に格納されている値と同じ)
    switch ( HardwareManager::GetInstance().GetSrcType() )
    {
    case SampleRateConverterType_None:
        result = 2;
        break;
    case SampleRateConverterType_Linear:
        result = 1;
        break;
    case SampleRateConverterType_4Tap:
        // 0 のまま
        break;
    default:
        break;
    }

    return result;
}
} // anonymous namespace

/* ========================================================================
        member function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         Channel

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Channel::Channel() NN_NOEXCEPT
: m_PauseFlag( 0 )
, m_ActiveFlag( false )
, m_AllocFlag( false )
, m_pTvAdditionalParam( nullptr )
, m_pVoice( NULL )
{
    m_Disposer.Initialize(this);
    DisposeCallbackManager::GetInstance().RegisterDisposeCallback( &m_Disposer );
}

/*--------------------------------------------------------------------------------*
  Name:         Channel

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Channel::~Channel() NN_NOEXCEPT
{
    DisposeCallbackManager::GetInstance().UnregisterDisposeCallback( &m_Disposer );
}

/*--------------------------------------------------------------------------------*
  Name:         InitParam

  Description:  パラメータの初期化

  Arguments:

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Channel::InitParam( ChannelCallback callback, void* callbackData ) NN_NOEXCEPT
{
    m_pNextLink              = NULL; // ユーザー領域

    m_Callback               = callback;
    m_CallbackData           = callbackData;

    //  msync_flag           = 0; // NOTE: クリアしてはならない！
    m_PauseFlag              = false;
    m_AutoSweep              = true;
    m_ReleasePriorityFixFlag = false;
    m_IsIgnoreNoteOff        = false;

    m_LoopFlag               = false;
    m_LoopStartFrame         = 0;
    m_OriginalLoopStartFrame = 0;

    m_Length                 = 0;
    m_Key                    = KeyInit;
    m_OriginalKey            = OriginalKeyInit;
    m_InitPan                = 0.0f;
    m_InitSurroundPan        = 0.0f;
    m_Tune                   = 1.0f;

    m_Cent                   = 0.0f;
    m_CentPitch              = 1.0f;

    m_UserVolume             = 1.0f;
    m_UserPitch              = 0.0f;
    m_UserPitchRatio         = 1.0f;
    m_UserLpfFreq            = 0.0f;
    m_BiquadType             = BiquadFilterType_None;
    m_BiquadValue            = 0.0f;

    m_OutputLineFlag         = OutputLine_Main;

    m_TvParam.Initialize();
    if( m_pTvAdditionalParam != nullptr )
    {
        m_pTvAdditionalParam->Reset();
    }

    m_SilenceVolume.InitValue( SilenceVolumeMax );

    m_SweepPitch   = 0.0f;
    m_SweepLength  = 0;
    m_SweepCounter = 0;

    m_CurveAdshr.Initialize();
    for (int i = 0; i < ModCount; i++)
    {
        m_Lfo[i].GetParam().Initialize();
        m_LfoTarget[i] = LfoTarget_Invalid;
    }

    m_PanMode = PanMode_Dual;
    m_PanCurve = PanCurve_Sqrt;

    m_KeyGroupId = 0;
    m_InterpolationType = GetNwInterpolationTypeFromHardwareManager();
        // WSD はこの値が反映される。
        // SEQ は Channel::SetInterpolationType によって設定された値
        // (インストのベロシティリージョンに書かれた値) が反映される。
    m_InstrumentVolume = 1.0f;
    m_Velocity         = 1.0f;
}

// #define NN_ATK_CHANNEL_PROFILE

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  チャンネルのフレームワーク

  Arguments:    doPeriodicProc - 周期処理を行うかどうかのフラグ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Channel::Update( bool doPeriodicProc ) NN_NOEXCEPT
{
    if ( ! m_ActiveFlag ) return;

#ifdef NN_ATK_CHANNEL_PROFILE
    nn::os::Tick tick = nn::os::Tick::GetSystemCurrent();
#endif

    if ( m_PauseFlag ) doPeriodicProc = false;

    // volume
    m_SilenceVolume.Update();
    float volume = m_Velocity * m_Velocity * m_InstrumentVolume * m_UserVolume * m_SilenceVolume.GetValue() * SilenceVolumeMaxR;

    // 減衰完了チェック
    if ( m_CurveAdshr.GetStatus() == CurveAdshr::Status_Release &&
         m_CurveAdshr.GetValue() < -90.4f )
    {
        Stop();
        return;
    }

    // pitch
    float cent = m_Key - m_OriginalKey + m_UserPitch + GetSweepValue();
    for (int i = 0; i < ModCount; i++)
    {
        if ( m_LfoTarget[i] == LfoTarget_Pitch )
        {
            cent += m_Lfo[i].GetValue();
        }
    }

    float pitch;
    if ( cent == m_Cent )
    {
        pitch = m_CentPitch;
    }
    else
    {
        pitch = Util::CalcPitchRatio( static_cast<int32_t>( cent * ( 1 << Util::PitchDivisionBit ) ) );
        m_Cent = cent;
        m_CentPitch = pitch;
    }

    pitch = pitch * m_Tune * m_UserPitchRatio;


    OutputParam tvParam = m_TvParam;
    {
        tvParam.pan += m_InitPan;
        for (int i = 0; i < ModCount; i++)
        {
            if ( m_LfoTarget[i] == LfoTarget_Pan )
            {
                tvParam.pan += m_Lfo[i].GetValue();
            }
        }
        tvParam.span += m_InitSurroundPan;
    }

    // カウンタ更新
    if ( doPeriodicProc )
    {
        if ( m_AutoSweep )
        {
            UpdateSweep( HardwareManager::SoundFrameIntervalMsec );
        }
        for (int i = 0; i < ModCount; i++)
        {
            if (m_LfoTarget[i] != LfoTarget_Invalid)
            {
                m_Lfo[i].Update(HardwareManager::SoundFrameIntervalMsec);
            }
        }
        m_CurveAdshr.Update( HardwareManager::SoundFrameIntervalMsec );
    }

    // lpf
    float lpfFreq = 1.0f + m_UserLpfFreq;

    // ボリューム値更新
    volume *= Util::CalcVolumeRatio( m_CurveAdshr.GetValue() );
    for (int i = 0; i < ModCount; i++)
    {
        if ( m_LfoTarget[i] == LfoTarget_Volume )
        {
            volume *= Util::CalcVolumeRatio( m_Lfo[i].GetValue() * 6.0f );
        }
    }

    if ( m_pVoice != NULL )
    {
        m_pVoice->SetVolume( volume );
        m_pVoice->SetPitch( pitch );

        m_pVoice->SetPanMode( m_PanMode );
        m_pVoice->SetPanCurve( m_PanCurve );

        m_pVoice->SetLpfFreq( lpfFreq );
        m_pVoice->SetBiquadFilter( m_BiquadType, m_BiquadValue );

        m_pVoice->SetOutputLine( m_OutputLineFlag );
        m_pVoice->SetTvParam( tvParam );

        if( m_pTvAdditionalParam != nullptr )
        {
            m_pVoice->SetTvAdditionalParam( *m_pTvAdditionalParam );
        }
    }

#ifdef NN_ATK_CHANNEL_PROFILE
    g_Tick += nn::os::Tick::GetSystemCurrent() - tick;
    g_Count += 1;
#endif
}

/*--------------------------------------------------------------------------------*
  Name:         Start

  Description:  チャンネルでPCMを再生します

  Arguments:    wave - 波形情報構造体のポインタ
                data - 波形データアドレス
                length - 発音長

  Returns:      成功したかどうか
 *--------------------------------------------------------------------------------*/
void Channel::Start( const WaveInfo& waveInfo, int length, position_t startOffsetSamples, bool isContextCalculationSkipMode ) NN_NOEXCEPT
{
    // Channel Param
    m_Length = length;
    for (int i = 0; i < ModCount; i++)
    {
        m_Lfo[i].Reset();
    }
    m_CurveAdshr.Reset();
    m_SweepCounter = 0;

    m_pVoice->SetSampleFormat( waveInfo.sampleFormat );
    m_pVoice->SetSampleRate( waveInfo.sampleRate );
    m_pVoice->SetInterpolationType( m_InterpolationType );

    AppendWaveBuffer( waveInfo, startOffsetSamples, isContextCalculationSkipMode );

    m_pVoice->Start();
    m_ActiveFlag = true;
}

void Channel::AppendWaveBuffer( const WaveInfo& waveInfo, position_t startOffsetSamples, bool isContextCalculationSkipMode ) NN_NOEXCEPT
{
    m_LoopFlag = waveInfo.loopFlag;
    m_LoopStartFrame = waveInfo.loopStartFrame;
    m_OriginalLoopStartFrame = waveInfo.originalLoopStartFrame;

    // 8 バイト境界に切り下げる (DSP ADPCM は 8 バイト 14 サンプル単位のエンコードのため)
    if ( startOffsetSamples > 0 &&
         waveInfo.sampleFormat == nn::atk::SampleFormat_DspAdpcm )
    {
        startOffsetSamples = ( startOffsetSamples / 14 ) * 14;
    }

    NN_SDK_ASSERT( waveInfo.loopEndFrame > startOffsetSamples );

    const int sdkVoiceCount = m_pVoice->GetSdkVoiceCount();
    for ( int ch = 0; ch < sdkVoiceCount; ch++ )
    {
        const void* originalDataAddress = waveInfo.channelParam[ch].dataAddress;

        AdpcmContext& adpcmContext = m_AdpcmContext[ch];
        AdpcmContext& adpcmLoopContext = m_AdpcmLoopContext[ch];

        // ADPCM の場合、コンテキストを生成
        if ( waveInfo.sampleFormat == SampleFormat_DspAdpcm )
        {
            const DspAdpcmParam* pParam = &waveInfo.channelParam[ch].adpcmParam;

            // 係数設定 (uint16_t coef[16])
            nn::atk::AdpcmParam param;
            for ( int i = 0; i < 8; i++ )
            {
                for ( int j = 0; j < 2; ++j )
                {
                    param.coefficients[i * 2 + j] = pParam->coef[i][j];
                }
            }

            // コンテキスト設定 (nn::atk::DspVoice が使用するまで SDK ユーザー側で保持)
            if ( startOffsetSamples == 0 )
            {
                adpcmContext.audioAdpcmContext.predScale  = pParam->predScale;
                adpcmContext.audioAdpcmContext.history[0] = pParam->yn1;
                adpcmContext.audioAdpcmContext.history[1] = pParam->yn2;
            }
            else    // startOffsetSamples > 0
            {
                if( isContextCalculationSkipMode )
                {
                    adpcmContext.audioAdpcmContext.predScale  = 0;
                    adpcmContext.audioAdpcmContext.history[0] = 0;
                    adpcmContext.audioAdpcmContext.history[1] = 0;
                }
                else
                {
                    // 8 バイト境界の先頭 1 バイトを pred_scale に入れる。
                    // pred_scale は uint16_t 型だが、上位 8 bit は参照されないので、
                    // *reinterpret_cast<const uint16_t*>( adjustDataAddress ) を pred_scale に代入でよいはず。
                    adpcmContext.audioAdpcmContext.predScale  = pParam->predScale;
                    adpcmContext.audioAdpcmContext.history[0] = pParam->yn1;
                    adpcmContext.audioAdpcmContext.history[1] = pParam->yn2;

                    // 途中まで自前でデコード (通常は DSP でデコードするため、その分 CPU 負荷がかかる)
                    MultiVoice::CalcOffsetAdpcmParam(
                        &adpcmContext,
                        param,
                        startOffsetSamples,
                        originalDataAddress
                    );
                }
            }

            if ( waveInfo.loopFlag )
            {
                const DspAdpcmLoopParam* pLoopParam = &waveInfo.channelParam[ch].adpcmLoopParam;
                adpcmLoopContext.audioAdpcmContext.predScale  = pLoopParam->loopPredScale;
                adpcmLoopContext.audioAdpcmContext.history[0] = pLoopParam->loopYn1;
                adpcmLoopContext.audioAdpcmContext.history[1] = pLoopParam->loopYn2;
            }

            m_pVoice->SetAdpcmParam( ch, param );
        }

        {
            // |------------------|<---------- loop --------->| のような波形を、
            //
            // |<-------------- pBuffer0 -------------------->|
            // |<-------------- pBuffer1 -------------------->|
            // |-loopStartOffset->|                             として利用する。

            // WaveBuffer の登録
            WaveBuffer* pBuffer0 = &m_WaveBuffer[ch][0];
            WaveBuffer* pBuffer1 = &m_WaveBuffer[ch][1];

            pBuffer0->bufferAddress = originalDataAddress;
            pBuffer0->bufferSize = waveInfo.channelParam[ch].dataSize;
            pBuffer0->sampleOffset = startOffsetSamples;
            pBuffer0->sampleLength = waveInfo.loopEndFrame;
            pBuffer0->loopFlag = false;
            if ( waveInfo.sampleFormat == SampleFormat_DspAdpcm )
            {
                pBuffer0->pAdpcmContext = &adpcmContext;
            }
            else
            {
                pBuffer0->pAdpcmContext = NULL;
            }

            if ( waveInfo.loopFlag )
            {
                pBuffer1->bufferAddress = originalDataAddress;
                pBuffer1->bufferSize = waveInfo.channelParam[ch].dataSize;
                pBuffer1->sampleOffset = m_LoopStartFrame;
                pBuffer1->sampleLength = waveInfo.loopEndFrame;
                pBuffer1->loopFlag = true;
                if ( waveInfo.sampleFormat == SampleFormat_DspAdpcm )
                {
                    pBuffer1->pAdpcmContext = &adpcmLoopContext;
                }
                else
                {
                    pBuffer1->pAdpcmContext = NULL;
                }
                m_pVoice->AppendWaveBuffer( ch, pBuffer0, false );
                m_pVoice->AppendWaveBuffer( ch, pBuffer1, true );
            }
            else
            {
                m_pVoice->AppendWaveBuffer( ch, pBuffer0, true );
            }

            // NN_DETAIL_ATK_INFO("fmt(%d) [0] len(%d) lp(%d) a(%d,%d,%d) [1] len(%d) lp(%d) a(%d,%d,%d)\n",
            //         waveInfo.sampleFormat,
            //         pBuffer0->sampleLength, pBuffer0->loopFlag,
            //         pBuffer0->pAdpcmContext.pred_scale,
            //         pBuffer0->pAdpcmContext.yn1,
            //         pBuffer0->pAdpcmContext.yn2,
            //         pBuffer1->sampleLength, pBuffer1->loopFlag,
            //         pBuffer1->pAdpcmContext.pred_scale,
            //         pBuffer1->pAdpcmContext.yn1,
            //         pBuffer1->pAdpcmContext.yn2 );
        }
    }
} // NOLINT(impl/function_size)

/*--------------------------------------------------------------------------------*
  Name:         Release

  Description:  チャンネルをリリース状態にします

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Channel::Release() NN_NOEXCEPT
{
    if ( m_CurveAdshr.GetStatus() != CurveAdshr::Status_Release )
    {
        if ( m_pVoice != NULL )
        {
            if ( ! m_ReleasePriorityFixFlag )
            {
                m_pVoice->SetPriority( Channel::PriorityRelease );
            }
        }
        m_CurveAdshr.SetStatus( CurveAdshr::Status_Release );
    }
    m_PauseFlag = false;
}

/*--------------------------------------------------------------------------------*
  Name:         NoteOff

  Description:  チャンネルをノートオフします
                通常はリリース状態になります

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void Channel::NoteOff() NN_NOEXCEPT
{
    if ( m_IsIgnoreNoteOff ) return;

    Release();
}

/*--------------------------------------------------------------------------------*
    Name:           Stop

    Description:    チャンネルを停止します

    Arguments:      無し

    Returns:        無し
 *--------------------------------------------------------------------------------*/
void Channel::Stop() NN_NOEXCEPT
{
    if ( m_pVoice == NULL ) return;

    m_pVoice->Stop();
    m_pVoice->Free();
    m_pVoice = NULL;

    m_PauseFlag = false;
    m_ActiveFlag = false;

#if 0
    CallChannelCallback( ChannelCallbackStatus_Stopped );
    if ( m_AllocFlag )
    {
        m_AllocFlag = false;
        ChannelManager::GetInstance().Free( this );
    }
#endif
}

void Channel::UpdateSweep( int count ) NN_NOEXCEPT
{
    m_SweepCounter += count;
    if ( m_SweepCounter > m_SweepLength )
    {
        m_SweepCounter = m_SweepLength;
    }
}

void Channel::SetSweepParam( float sweepPitch, int sweepTime, bool autoUpdate ) NN_NOEXCEPT
{
    m_SweepPitch = sweepPitch;
    m_SweepLength = sweepTime;
    m_AutoSweep = autoUpdate;
    m_SweepCounter = 0;
}

float Channel::GetSweepValue() const NN_NOEXCEPT
{
    if ( m_SweepPitch == 0.0f ) return 0.0f;
    if ( m_SweepCounter >= m_SweepLength ) return 0.0f;

    float sweep = m_SweepPitch;
    sweep *= m_SweepLength - m_SweepCounter;

    NN_SDK_ASSERT( m_SweepLength != 0 );
    sweep /= m_SweepLength;

    return sweep;
}

void Channel::SetBiquadFilter( int type, float value ) NN_NOEXCEPT
{
    // ここに来るまでに、INHERITの処理は済んでいるはずなので、BiquadFilterType_DataMinが最小になるはずです。
    NN_SDK_ASSERT( type >= BiquadFilterType_DataMin && type <= BiquadFilterType_Max );
    m_BiquadType = static_cast<uint8_t>( type );
    m_BiquadValue = value;
}

void Channel::SetTvAdditionalParam(const OutputAdditionalParam& param) NN_NOEXCEPT
{
    if ( m_pTvAdditionalParam == nullptr )
    {
        return;
    }

    *m_pTvAdditionalParam = param;
}


position_t Channel::GetCurrentPlayingSample(bool isOriginalSamplePosition) const NN_NOEXCEPT
{
    if ( ! m_ActiveFlag ) return 0;

    NN_SDK_ASSERT_NOT_NULL( m_pVoice );

    position_t playSamplePosition = m_pVoice->GetCurrentPlayingSample();

    // ループ補正前のサンプル位置を計算
    if ( isOriginalSamplePosition && m_LoopFlag )
    {
        position_t loopEnd = m_WaveBuffer[0][0].sampleLength;
        position_t originalLoopEnd = loopEnd - (m_LoopStartFrame - m_OriginalLoopStartFrame);

        if ( playSamplePosition > originalLoopEnd )
        {
            playSamplePosition = m_OriginalLoopStartFrame + ( playSamplePosition - originalLoopEnd );
        }
    }

    return playSamplePosition;
}

/*--------------------------------------------------------------------------------*
    Name:           VoiceCallback

    Description:    ボイスがドロップされたときに呼ばれる

    Arguments:      param: ボイスへのポインタ

    Returns:        none
 *--------------------------------------------------------------------------------*/
void Channel::VoiceCallbackFunc(
    MultiVoice* voice,
    MultiVoice::VoiceCallbackStatus status,
    void* arg
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( arg );

    ChannelCallbackStatus chStatus = ChannelCallbackStatus_Finish;    // 適当な初期化
    switch ( status )
    {
    case MultiVoice::VoiceCallbackStatus_FinishWave:
        chStatus = ChannelCallbackStatus_Finish;
        voice->Free();
        break;
    case MultiVoice::VoiceCallbackStatus_Cancel:
        chStatus = ChannelCallbackStatus_Cancel;
        voice->Free();
        break;
    case MultiVoice::VoiceCallbackStatus_DropVoice:
        chStatus = ChannelCallbackStatus_Drop;
        break;
    case MultiVoice::VoiceCallbackStatus_DropDsp:
        chStatus = ChannelCallbackStatus_Drop;
        break;
    default:
        break;
    }

    Channel* channel = static_cast<Channel*>( arg );

#if 0
    int sdkVoiceCount = channel->m_pVoice->GetSdkVoiceCount();
    for ( int channelIndex = 0; channelIndex < sdkVoiceCount; channelIndex++ )
    {
        for ( int waveBufferIndex = 0; waveBufferIndex < WAVE_BUFFER_MAX; waveBufferIndex++ )
        {
            const WaveBuffer& waveBuffer =
                channel->m_WaveBuffer[ channelIndex ][ waveBufferIndex ];
            NN_SDK_ASSERT( waveBuffer.status == WaveBuffer::Status_Free || waveBuffer.status == WaveBuffer::Status_Done );
        }
    }
#endif

    channel->CallChannelCallback( chStatus );

    channel->m_pVoice = NULL;
    channel->m_PauseFlag = false;
    channel->m_ActiveFlag = false;

    channel->m_AllocFlag = false;
}

/*--------------------------------------------------------------------------------*
  Name:         AllocChannel

  Description:  チャンネルを確保します

  Arguments:    priority - プライオリティ
                callback - チャンネルコールバック関数
                callbackData - チャンネルコールバック関数のユーザー引数

  Returns:      確保したチャンネルのポインタ
                確保できなかった場合はNULL
 *--------------------------------------------------------------------------------*/
Channel* Channel::AllocChannel(
    int voiceChannelCount,
    int priority,
    Channel::ChannelCallback callback,
    void* callbackData
) NN_NOEXCEPT
{
    NN_SDK_ASSERT( priority >= Voice::PriorityMin && priority <= Voice::PriorityMax );

    Channel* channel = ChannelManager::GetInstance().Alloc();
    if ( channel == NULL )
    {
        Util::WarningLogger::GetInstance().Log(
            Util::WarningLogger::LogId_ChannelAllocationFailed );
        return NULL;
    }
    channel->m_AllocFlag = true;

    // ボイス取得
    MultiVoice* voice = MultiVoiceManager::GetInstance().AllocVoice(
        voiceChannelCount,
        priority,
        VoiceCallbackFunc,
        channel
    );
    if ( voice == NULL )
    {
        ChannelManager::GetInstance().Free( channel );
        return NULL;
    }

    channel->m_pVoice = voice;
    channel->InitParam( callback, callbackData );

    return channel;
}

void Channel::FreeChannel( Channel* channel ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( channel );

    ChannelManager::GetInstance().Free( channel );
}

/*--------------------------------------------------------------------------------*
  Name:         DetachChannel

  Description:  チャンネルを解放します

  Arguments:    channel - チャンネルポインタ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void Channel::DetachChannel( Channel* channel ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( channel );

    channel->m_Callback     = NULL;
    channel->m_CallbackData = NULL;
}

void Channel::CallChannelCallback( ChannelCallbackStatus status ) NN_NOEXCEPT
{
    if ( m_Callback != NULL )
    {
        m_Callback( this, status, m_CallbackData );

        m_Callback = NULL;
        m_CallbackData = NULL;
    }
}

void Channel::Disposer::InvalidateData( const void* start, const void* end ) NN_NOEXCEPT
{
    if ( m_pChannel->m_pVoice == NULL ) return;

    // 一つでも start, end 内に波形データを含んでいたらボイス停止
    bool disposeFlag = false;
    int sdkVoiceCount = m_pChannel->m_pVoice->GetSdkVoiceCount();

    for ( int channelIndex = 0; channelIndex < sdkVoiceCount; channelIndex++ )
    {
        for ( int waveBufferIndex = 0; waveBufferIndex < WaveBufferMax; waveBufferIndex++ )
        {
            const WaveBuffer& waveBuffer =
                m_pChannel->m_WaveBuffer[ channelIndex ][ waveBufferIndex ];
            if ( waveBuffer.status == WaveBuffer::Status_Done )
            {
                continue;
            }

            const void* bufferEnd = util::ConstBytePtr(
                    waveBuffer.bufferAddress,
                    Util::GetByteBySample(
                        waveBuffer.sampleLength,
                        m_pChannel->m_pVoice->GetFormat() ) ).Get();

            if ( start <= bufferEnd && end >= waveBuffer.bufferAddress )
            {
                disposeFlag = true;
                break;
            }
        }
    }

    if ( disposeFlag )
    {
        m_pChannel->CallChannelCallback( ChannelCallbackStatus_Cancel );

        m_pChannel->Stop();

        FreeChannel( m_pChannel  );
    }
}


} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

