﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_CurveLfo.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/atk/atk_Util.h>

namespace nn {
namespace atk {
namespace detail {

namespace {

// sine
float CurveSine(float arg) NN_NOEXCEPT
{
    return nn::util::SinTable(static_cast<nn::util::AngleIndex>(arg * 0x100000000));
}

//
// square
//
float CurveSquare(float arg) NN_NOEXCEPT
{
    if (arg < 0.5f)
    {
        return 0.0f;
    }
    return 1.0f;
}

//
// saw
//
float CurveSaw(float arg) NN_NOEXCEPT
{
    // 定義域 (arg) 0.0f → 1.0f の間を、
    // 値域 0.0f → 1.0f まで変化していく
    return arg;
}

//
// triangle
//
float CurveTriangle(float arg) NN_NOEXCEPT
{
    // [0] 0.00 -> 0.25  :  4 * arg;
    // [1] 0.25 -> 0.75  : -4 * arg + 2;
    // [2] 0.75 -> 1.00  :  4 * arg - 3;
    const float gradients[5] = { 4.0f, -4.0f, -4.0f, 4.0f, 4.0f };
    const float intercepts[5] = { 0.0f, 2.0f, 2.0f, -4.0f, 0.0f };
    uint32_t tmp = static_cast<uint32_t>(arg * 4);
    return gradients[tmp] * arg + intercepts[tmp];
}

//
// random
//
float CurveRandom(float /*arg*/) NN_NOEXCEPT
{
    uint16_t rand = Util::CalcRandom();
    float ret = static_cast<float>(rand) / 0xffff;
    return ret;
}

CurveLfo::CurveFunc g_CurveFuncTable[CurveLfoParam::CurveType_Count] = {NULL};

} // anonymous namespace


// カーブテーブルの定義
void CurveLfo::InitializeCurveTable() NN_NOEXCEPT
{
    std::memset(g_CurveFuncTable, 0, sizeof(CurveFunc) * CurveLfoParam::CurveType_Count);

    g_CurveFuncTable[CurveLfoParam::CurveType_Sine] = CurveSine;
    g_CurveFuncTable[CurveLfoParam::CurveType_Triangle] = CurveTriangle;
    g_CurveFuncTable[CurveLfoParam::CurveType_Saw] = CurveSaw;
    g_CurveFuncTable[CurveLfoParam::CurveType_Square] = CurveSquare;
    g_CurveFuncTable[CurveLfoParam::CurveType_Random] = CurveRandom;
}

CurveLfo::CurveFunc CurveLfo::RegisterUserCurve(CurveFunc func, uint32_t index) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
            static_cast<int32_t>(index) >= 0 && static_cast<int32_t>(index) <= CurveLfoParam::CurveType_UserMax - CurveLfoParam::CurveType_UserMin);

    int32_t tableIndex = index + CurveLfoParam::CurveType_UserMin;
    CurveFunc tmp = g_CurveFuncTable[tableIndex];
    g_CurveFuncTable[tableIndex] = func;
    return tmp;
}

CurveLfo::CurveFunc CurveLfo::UnregisterUserCurve(uint32_t index) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
            static_cast<int32_t>(index) >= 0 && static_cast<int32_t>(index) <= CurveLfoParam::CurveType_UserMax - CurveLfoParam::CurveType_UserMin);

    int32_t tableIndex = index + CurveLfoParam::CurveType_UserMin;
    CurveFunc tmp = g_CurveFuncTable[tableIndex];
    g_CurveFuncTable[tableIndex] = NULL;
    return tmp;
}

/* ========================================================================
        public function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         LfoParam::Init

  Description:  LFOパラメータ構造体を初期化します

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void CurveLfoParam::Initialize() NN_NOEXCEPT
{
    depth = 0.0f;
    speed = 6.25f;
    delay = 0;
    range = 1;
    curve = 0;
    phase = 0;
}

/*--------------------------------------------------------------------------------*
  Name:         Reset

  Description:  LFOをリセットします

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void CurveLfo::Reset() NN_NOEXCEPT
{
    m_Counter = 0.0f;
    m_RandomValue = 1.0f;
    m_DelayCounter = 0;
    m_IsStart = false;
    m_IsNext = false;
}

/*--------------------------------------------------------------------------------*
  Name:         Update

  Description:  LFOを更新します

  Arguments:    msec - 更新時間[msec]

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void CurveLfo::Update( int msec ) NN_NOEXCEPT
{
    if ( m_DelayCounter < m_Param.delay )
    {
        if ( m_DelayCounter + msec <= m_Param.delay )
        {
            m_DelayCounter += msec;
            return;
        }
        else
        {
            msec -= m_Param.delay - m_DelayCounter;
            m_DelayCounter = m_Param.delay;
        }
    }

    if (m_Param.speed > 0.0f)
    {
        if (m_IsStart == false)
        {
            m_Counter = m_Param.phase / 127.0f;
            m_IsStart = true;
        }
        m_Counter += m_Param.speed * msec / 1000.0f;
        if (m_Counter >= 1.0f)
        {
            m_IsNext = true;
        }
        else
        {
            m_IsNext = false;
        }
        m_Counter -= static_cast<int>( m_Counter );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         GetValue

  Description:  現在のLFO値を取得します

  Arguments:    なし

  Returns:      現在のLFO値
 *--------------------------------------------------------------------------------*/
float CurveLfo::GetValue() const NN_NOEXCEPT
{
    if ( m_Param.depth == 0.0f ) return 0.0f;
    if ( m_DelayCounter < m_Param.delay ) return 0.0f;

    NN_SDK_ASSERT(
            static_cast<CurveLfoParam::CurveType>(m_Param.curve) >= CurveLfoParam::CurveType_Min &&
            static_cast<CurveLfoParam::CurveType>(m_Param.curve) <= CurveLfoParam::CurveType_Max);
    CurveFunc func = g_CurveFuncTable[m_Param.curve];

    float value = 0.f;
    if (func != NULL)
    {
        NN_SDK_ASSERT(m_Counter >= 0.0f && m_Counter <= 1.0f);

        // ランダムは、１周期中ずっと同じ値を取るため、特別対応とした
        if (m_Param.curve == CurveLfoParam::CurveType_Random)
        {
            if (m_IsNext)
            {
                value = func(m_Counter);
                m_RandomValue = value;
            }
            else
            {
                value = m_RandomValue;
            }
        }
        else
        {
            value = func(m_Counter);
        }
    }
    else
    {
        value = 1.0f;
    }
    value *= m_Param.depth;
    value *= m_Param.range;
    return value;
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

