﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_StreamSoundFileReader.h>
#include <nn/atk/atk_StreamSoundPrefetchFileReader.h>
#include <nn/atk/atk_WaveFile.h>                // EncodeMethod
#include <nn/atk/fnd/binary/atkfnd_Binary.h>
#include <nn/util/util_BinTypes.h>

namespace nn {
namespace atk {
namespace detail {

namespace {

const uint32_t SignatureFileStp              = NN_UTIL_CREATE_SIGNATURE_4( 'F', 'S', 'T', 'P' );
const uint32_t SignatureInfoBlockStp         = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'O' );
const uint32_t SignaturePrefetchdataBlockStp = NN_UTIL_CREATE_SIGNATURE_4( 'P', 'D', 'A', 'T' );

const int SupportedFileVersionStp            = NN_ATK_MAKE_VERSION( 0, 1, 0, 0 );   // ライブラリがサポートする最低バージョン
const int CurrentFileVersionStp              = NN_ATK_MAKE_VERSION( 0, 5, 2, 0 );   // ライブラリがサポートする最新バージョン
const int IncludeRegionInfoVersionStp        = NN_ATK_MAKE_VERSION( 0, 3, 0, 0 );   // リージョン情報が含まれるようになったバージョン
const int IncludeCrc32CheckVersionStp        = NN_ATK_MAKE_VERSION( 0, 4, 0, 0 );   // CRC32 によるハッシュチェックが含まれるようになったバージョン
const int IncludeRegionIndexCheckVersionStp  = NN_ATK_MAKE_VERSION( 0, 5, 1, 0 );   // リージョン番号のアサートチェックが行われるようになったバージョン

} // anonymous namespace

StreamSoundPrefetchFileReader::StreamSoundPrefetchFileReader() NN_NOEXCEPT
    : m_pHeader( NULL )
    , m_pInfoBlockBody( NULL )
    , m_pPrefetchDataBlockBody( NULL )
    , m_RegionDataOffset(0)
    , m_RegionInfoBytes(0)
{
}

StreamSoundPrefetchFileReader::~StreamSoundPrefetchFileReader() NN_NOEXCEPT
{
    Finalize();
}

void StreamSoundPrefetchFileReader::Initialize( const void* streamSoundPrefetchFile ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( streamSoundPrefetchFile );

    if ( ! IsValidFileHeader( streamSoundPrefetchFile ) ) return;

    m_pHeader =
        reinterpret_cast<const StreamSoundPrefetchFile::FileHeader*>( streamSoundPrefetchFile );
    const StreamSoundFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();
    NN_SDK_ASSERT( infoBlock->header.kind == SignatureInfoBlockStp );
    if ( infoBlock->header.kind != SignatureInfoBlockStp )
    {
        return;
    }
    const StreamSoundPrefetchFile::PrefetchDataBlock* dataBlock = m_pHeader->GetPrefetchDataBlock();
    NN_SDK_ASSERT( dataBlock->header.kind == SignaturePrefetchdataBlockStp );
    if ( dataBlock->header.kind != SignaturePrefetchdataBlockStp )
    {
        return;
    }

    m_pInfoBlockBody         = &infoBlock->body;
    m_pPrefetchDataBlockBody = &dataBlock->body;
    NN_SDK_ASSERT( m_pInfoBlockBody->GetStreamSoundInfo()->oneBlockBytes % 32 == 0 );

    m_RegionDataOffset = GetRegionDataOffset();
    m_RegionInfoBytes = GetRegionInfoBytes();
}

void StreamSoundPrefetchFileReader::Finalize() NN_NOEXCEPT
{
    m_pHeader = NULL;
    m_pInfoBlockBody = NULL;
    m_pPrefetchDataBlockBody = NULL;
    m_RegionDataOffset = 0;
    m_RegionInfoBytes = 0;
}



bool StreamSoundPrefetchFileReader::IsIncludeRegionInfo() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>(m_pHeader);
    if ( header.version >= IncludeRegionInfoVersionStp )
    {
        return true;
    }
    return false;
}

bool StreamSoundPrefetchFileReader::IsCrc32CheckAvailable() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>(m_pHeader);
    if ( header.version >= IncludeCrc32CheckVersionStp )
    {
        return true;
    }
    return false;
}

bool StreamSoundPrefetchFileReader::IsRegionIndexCheckAvailable() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>(m_pHeader);
    if ( header.version >= IncludeRegionIndexCheckVersionStp )
    {
        return true;
    }
    return false;
}

bool StreamSoundPrefetchFileReader::ReadStreamSoundInfo(
        StreamSoundFile::StreamSoundInfo* strmInfo ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    const StreamSoundFile::StreamSoundInfo* info =
        m_pInfoBlockBody->GetStreamSoundInfo();

    NN_SDK_ASSERT( util::is_aligned(info->oneBlockBytes, 32) );
    NN_SDK_ASSERT( util::is_aligned(info->lastBlockPaddedBytes, 32) );

    // struct copy
    *strmInfo = *info;

    return true;
}

bool StreamSoundPrefetchFileReader::ReadDspAdpcmChannelInfo(
        DspAdpcmParam* pParam,
        DspAdpcmLoopParam* pLoopParam,
        int channelIndex ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( pParam );
    NN_SDK_ASSERT_NOT_NULL( pLoopParam );

    const StreamSoundFile::DspAdpcmChannelInfo* src =
        m_pInfoBlockBody->
        GetChannelInfoTable()->
        GetChannelInfo( channelIndex )->
        GetDspAdpcmChannelInfo();
    if ( src == NULL ) return false;

    *pParam = src->param;
    *pLoopParam = src->loopParam;
    return true;
}

bool StreamSoundPrefetchFileReader::ReadPrefetchDataInfo(
PrefetchDataInfo* pDataInfo, int prefetchIndex ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pPrefetchDataBlockBody );
    NN_SDK_ASSERT_NOT_NULL( pDataInfo );

    const StreamSoundPrefetchFile::PrefetchData* data = m_pPrefetchDataBlockBody->GetPrefetchData( prefetchIndex );
    NN_SDK_ASSERT_NOT_NULL( data );

    pDataInfo->startFrame   = data->startFrame;
    pDataInfo->prefetchSize = data->prefetchSize;

    const StreamSoundPrefetchFile::PrefetchSample* sample = data->GetPrefetchSample();
    pDataInfo->dataAddress  = sample->GetSampleAddress();

    return true;
}

bool StreamSoundPrefetchFileReader::ReadRegionInfo(StreamSoundFile::RegionInfo* pInfo, uint32_t regionIndex) const NN_NOEXCEPT
{
    if ( m_RegionDataOffset == 0 || m_RegionInfoBytes == 0 )
    {
        return false;
    }

    position_t offset = m_RegionDataOffset + (static_cast<position_t>(m_RegionInfoBytes) * regionIndex);
    nn::util::ConstBytePtr bytePtr( m_pHeader );
    bytePtr.Advance(offset);
    *pInfo = *reinterpret_cast<const StreamSoundFile::RegionInfo*>(bytePtr.Get());

    return true;
}

bool StreamSoundPrefetchFileReader::IsValidFileHeader( const void* streamSoundPrefetchFile ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( streamSoundPrefetchFile );

    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( streamSoundPrefetchFile );

    // シグニチャ確認
    NN_SDK_ASSERT(
            header.signature == SignatureFileStp,
            "invalid file signature. stream data is not available." );
    if ( header.signature != SignatureFileStp ) return false;

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header.byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header.byteOrder );
    if ( header.byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionStp <= header.version) &&
                                   (header.version <= CurrentFileVersionStp) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfstp file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionStp, CurrentFileVersionStp, header.version
    );
    return isSupportedVersion;
}

uint32_t StreamSoundPrefetchFileReader::GetRegionDataOffset() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    if ( IsAvailable() && m_pHeader->HasRegionBlock() )
    {
        uint32_t result = m_pHeader->GetRegionBlockOffset() +
            sizeof( BinaryBlockHeader ) +
            m_pInfoBlockBody->GetStreamSoundInfo()->regionDataOffset.offset;
        return result;
    }
    return 0;
}

uint16_t StreamSoundPrefetchFileReader::GetRegionInfoBytes() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    uint16_t result = m_pInfoBlockBody->GetStreamSoundInfo()->regionInfoBytes;
    return result;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

