﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SequenceSound.h>

#include <nn/atk/atk_SoundInstanceManager.h>
#include <nn/atk/atk_SequenceTrack.h>
#include <nn/atk/atk_SequenceSoundHandle.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_PlayerHeap.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/atk_SoundDataManager.h>
#include <nn/atk/atk_SoundPlayer.h>

namespace nn {
namespace atk {
namespace detail {

NN_DEFINE_STATIC_CONSTANT( const int SequenceSound::BankIndexMin );
NN_DEFINE_STATIC_CONSTANT( const int SequenceSound::BankIndexMax );
NN_DEFINE_STATIC_CONSTANT( const int8_t SequenceSound::TransposeMin );
NN_DEFINE_STATIC_CONSTANT( const int8_t SequenceSound::TransposeMax );
NN_DEFINE_STATIC_CONSTANT( const uint8_t SequenceSound::VelocityRangeMin );
NN_DEFINE_STATIC_CONSTANT( const uint8_t SequenceSound::VelocityRangeMax );

/* ========================================================================
        member function
   ======================================================================== */

/*--------------------------------------------------------------------------------*
  Name:         SequenceSound

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
SequenceSound::SequenceSound( SequenceSoundInstanceManager& manager ) NN_NOEXCEPT
: m_Manager( manager )
, m_InitializeFlag(false)
, m_IsCalledPrepare(false)
{
}

bool SequenceSound::Initialize(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    bool result = BasicSound::Initialize( pOutputReceiver );
    if( !result )
    {
        return false;
    }
    m_pTempSpecialHandle = NULL;
    m_IsCalledPrepare = false;
    m_InitializeFlag = true;

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  サウンドの終了

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::Finalize() NN_NOEXCEPT
{
    if ( ! m_InitializeFlag )
    {
        return;
    }

    m_InitializeFlag = false;
    m_IsCalledPrepare = false;
    BasicSound::Finalize();
    m_Manager.Free( this );
}

void SequenceSound::Setup(
    driver::SequenceTrackAllocator* trackAllocator,
    uint32_t allocTracks,
    driver::NoteOnCallback* noteOnCallback,
    int channelPriority,
    bool isReleasePriorityFix,
    SequenceUserProcCallback userproc,
    void* userprocArg
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( noteOnCallback );
    NN_SDK_ASSERT( channelPriority >= nn::atk::ChannelPriorityMin && channelPriority <= nn::atk::ChannelPriorityMax );

    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandSequenceSoundSetup* command =
            cmdmgr.AllocCommand<DriverCommandSequenceSoundSetup>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_SeqSetup;
        command->player = &m_PlayerInstance;

        driver::SequenceSoundPlayer::SetupArg arg;
        {
            arg.trackAllocator = trackAllocator;
            arg.allocTracks = allocTracks;
            arg.callback = noteOnCallback;
        }
        command->arg = arg;  // 構造体コピー
        command->channelPriority = static_cast<uint8_t>(channelPriority);
        command->isReleasePriorityFix = isReleasePriorityFix;
        command->userproc = reinterpret_cast<uintptr_t>(userproc);
        command->userprocArg = userprocArg;

        cmdmgr.PushCommand(command);
    }
}

void SequenceSound::Prepare(
        const Resource& res,
        const driver::SequenceSoundPlayer::StartInfo& startInfo ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( res.seq );
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandSequenceSoundPrepare* command =
            cmdmgr.AllocCommand<DriverCommandSequenceSoundPrepare>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_SeqPrepare;
        command->player = &m_PlayerInstance;

        driver::SequenceSoundPlayer::PrepareArg arg;
        {
            arg.seqFile = res.seq;
            for (int i = 0; i < SeqBankMax; i++)
            {
                arg.bankFiles[i] = res.banks[i];
                arg.warcFiles[i] = res.warcs[i];
                arg.warcIsIndividuals[i] = res.warcIsIndividuals[i];
            }
            arg.seqOffset = startInfo.seqOffset;
            arg.delayTime = startInfo.delayTime;
            arg.delayCount = startInfo.delayCount;
            arg.updateType = startInfo.updateType;
        }
        command->arg = arg;

        cmdmgr.PushCommand(command);
    }

    if ( startInfo.startOffset > 0 )
    {
        Skip( startInfo.startOffsetType, startInfo.startOffset );
    }

    m_IsCalledPrepare = true;
}



/*--------------------------------------------------------------------------------*
  Name:         Skip

  Description:  スキップ

  Arguments:    tick - スキップするTick数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::Skip( driver::SequenceSoundPlayer::StartOffsetType offsetType, int offset ) NN_NOEXCEPT
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandSequenceSoundSkip* command =
            cmdmgr.AllocCommand<DriverCommandSequenceSoundSkip>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_SeqSkip;
        command->player = &m_PlayerInstance;
        command->offsetType = offsetType;
        command->offset = offset;
        cmdmgr.PushCommand(command);
    }
}

/*--------------------------------------------------------------------------------*
  Name:         SetTempoRatio

  Description:  テンポ比率の変更

  Arguments:    tempoRatio  - テンポの比率

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTempoRatio( float tempoRatio ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( tempoRatio >= 0.0f );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTempoRatio* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTempoRatio>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTempoRatio;
    command->player = &m_PlayerInstance;
    command->tempoRatio = tempoRatio;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetChannelPriority

  Description:  発音プライオリティを変更

  Arguments:    priority - 発音プライオリティ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetChannelPriority( int priority ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( priority >= nn::atk::ChannelPriorityMin && priority <= nn::atk::ChannelPriorityMax );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundChannelPrio* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundChannelPrio>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqChannelPrio;
    command->player = &m_PlayerInstance;
    command->priority = static_cast<uint8_t>(priority);
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         OnUpdatePlayerPriority

  Description:  プレイヤープライオリティを変更

  Arguments:

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::OnUpdatePlayerPriority() NN_NOEXCEPT
{
    int priority = CalcCurrentPlayerPriority();
    m_Manager.UpdatePriority( this, priority );
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackMute

  Description:  トラックミュート

  Arguments:    trackBitFlag - トラックビットフラグ
                flag         - ミュート

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackMute( uint32_t trackBitFlag, SequenceMute mute ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackMute* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackMute>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackMute;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->mute = mute;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackMute

  Description:  トラックミュート

  Arguments:    trackBitFlag - トラックビットフラグ
                muteFlag     - ミュートフラグ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackMute( uint32_t trackBitFlag, bool muteFlag ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackMute* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackMute>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackMute;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->mute = muteFlag ? SequenceMute_Stop : SequenceMute_Off;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackSilence

  Description:  トラックサイレンス

  Arguments:    trackBitFlag - トラックビットフラグ
                silenceFlag  - ミュートフラグ
                fadeFrames   - フェードフレーム数

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackSilence( uint32_t trackBitFlag, bool silenceFlag, int fadeFrames ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackSilence* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackSilence>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackSilence;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->silenceFlag = silenceFlag;
    command->fadeFrames = fadeFrames;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackVolume

  Description:  トラックボリュームの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                volume       - ボリューム

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackVolume( uint32_t trackBitFlag, float volume ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( volume >= 0.0f );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackVolume;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = volume;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackPitch

  Description:  トラックピッチの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                pitch        - 音程

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackPitch( uint32_t trackBitFlag, float pitch ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pitch >= 0.0f );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackPitch;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = pitch;
    cmdmgr.PushCommand(command);
}





void SequenceSound::SetTrackMainOutVolume( uint32_t trackBitFlag, float volume ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackTvVolume;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = volume;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackChannelMixParameter( uint32_t trackBitFlag, uint32_t srcChNo, const MixParameter& mixParam ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( srcChNo < WaveChannelMax );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackMixParameter* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackMixParameter>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackTvMixParameter;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->srcChNo = srcChNo;
    for ( int channel = 0; channel < ChannelIndex_Count; channel++ )
    {
        command->param.ch[channel] = mixParam.ch[channel];
    }
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackPan

  Description:  トラックパンの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                pan          - パン

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackPan( uint32_t trackBitFlag, float pan ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackTvPan;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = pan;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackSurroundPan

  Description:  トラックサラウンドパンの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                surroundPan  - サラウンドパン

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackSurroundPan( uint32_t trackBitFlag, float surroundPan ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackTvSpan;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = surroundPan;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackMainSend( uint32_t trackBitFlag, float send ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackTvMainSend;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackFxSend( uint32_t trackBitFlag, AuxBus bus, float send ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( bus, 0, AuxBus_Count );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackTvFxSend;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    command->uint32Value = bus;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackLpfFreq

  Description:  ローパスフィルタのカットオフの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                lpfFreq  - フィルタのカットオフ値

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackLpfFreq( uint32_t trackBitFlag, float lpfFreq ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackLpf;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = lpfFreq;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         SetTrackBiquadFilter

  Description:  Biquadフィルタの設定

  Arguments:    trackBitFlag - トラックビットフラグ
                type  - フィルタの種類 (0 - 127)
                value - フィルタの値 (0.0f - 1.0f)

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::SetTrackBiquadFilter( uint32_t trackBitFlag, int type, float value ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackBiquad* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackBiquad>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackBiquad;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->type = type;
    command->value = value;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackBankIndex( uint32_t trackBitFlag, int bankIndex ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackBankIndex* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackBankIndex>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackBankIndex;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->bankIndex = bankIndex;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackTranspose( uint32_t trackBitFlag, int8_t transpose ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackTranspose* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackTranspose>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackTranspose;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->transpose = transpose;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackVelocityRange( uint32_t trackBitFlag, uint8_t velocityRange ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackVelocityRange* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackVelocityRange>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackVelocityRange;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->range = velocityRange;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackOutputLine( uint32_t trackBitFlag, uint32_t outputLine ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackOutputLine* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackOutputLine>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackOutputline;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->outputLine = outputLine;
    cmdmgr.PushCommand(command);
}

void SequenceSound::ResetTrackOutputLine( uint32_t trackBitFlag ) NN_NOEXCEPT
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackOutputLine* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackOutputLine>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqTrackOutputlineReset;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    cmdmgr.PushCommand(command);
}

#if 0   // トラック操作コマンドのひな形
void SequenceSound::SetTrackXXX( uint32_t trackBitFlag, yyy zzz ) NN_NOEXCEPT
{
    DriverCommandManager& cmdmgr = DriverCommandManager::GetInstance();
    DriverCommandSequenceSoundTrackXXX* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackXXX>();
    command->id = DriverCommandId_SeqTrackAaa;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->zzz = zzz;
    cmdmgr.PushCommand(command);
}
#endif

/*--------------------------------------------------------------------------------*
  Name:         ReadVariable

  Description:  シーケンスローカル変数の読みとり

  Arguments:    varNo  - 変数番号
                var    - 読み込み先アドレス

  Returns:      成功したかどうか
 *--------------------------------------------------------------------------------*/
bool SequenceSound::ReadVariable( int varNo, int16_t* var ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( var );
    NN_SDK_ASSERT_RANGE( varNo, 0, driver::SequenceSoundPlayer::PlayerVariableCount );

    if ( ! IsPlayerAvailable() )
    {
        *var = driver::SequenceSoundPlayer::VariableDefaultValue;
    }
    else
    {
        *var = m_PlayerInstance.GetLocalVariable( varNo );
    }
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         ReadGlobalVariable

  Description:  シーケンスグローバル変数の読みとり

  Arguments:    varNo - 変数番号
                var   - 読み込み先アドレス

  Returns:      成功したかどうか
 *--------------------------------------------------------------------------------*/
bool SequenceSound::ReadGlobalVariable( int varNo, int16_t* var ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( var );
    NN_SDK_ASSERT_RANGE( varNo, 0, driver::SequenceSoundPlayer::GlobalVariableCount );

    *var = driver::SequenceSoundPlayer::GetGlobalVariable( varNo );
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         ReadTrackVariable

  Description:  シーケンストラック変数の読みとり

  Arguments:    trackNo - トラック番号
                varNo  - 変数番号
                var    - 読み込み先アドレス

  Returns:      成功したかどうか
 *--------------------------------------------------------------------------------*/
bool SequenceSound::ReadTrackVariable( int trackNo, int varNo, int16_t* var ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( var );
    NN_SDK_ASSERT_RANGE( trackNo, 0, driver::SequenceSoundPlayer::TrackCountPerPlayer );
    NN_SDK_ASSERT_RANGE( varNo, 0, driver::SequenceTrack::TrackVariableCount );

    if ( ! IsPlayerAvailable() )
    {
        *var = driver::SequenceSoundPlayer::VariableDefaultValue;
    }
    else
    {
        const driver::SequenceTrack* track = m_PlayerInstance.GetPlayerTrack( trackNo );
        if ( track == NULL )
        {
            *var = driver::SequenceSoundPlayer::VariableDefaultValue;
        }
        else
        {
            *var = track->GetTrackVariable( varNo );
        }
    }
    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         WriteVariable

  Description:  シーケンスローカル変数の書き込み

  Arguments:    varNo  - 変数番号
                var    - 書き込む値

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::WriteVariable( int varNo, int16_t var ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( varNo, 0, driver::SequenceSoundPlayer::PlayerVariableCount );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundSetVar* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundSetVar>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqSetVar;
    command->player = &m_PlayerInstance;
    command->varNo = varNo;
    command->var = var;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         WriteGlobalVariable

  Description:  シーケンスグローバル変数の書き込み

  Arguments:    varNo - 変数番号
                var   - 書き込む値

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::WriteGlobalVariable( int varNo, int16_t var ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( varNo, 0, driver::SequenceSoundPlayer::GlobalVariableCount );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundSetVar* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundSetVar>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqSetGvar;
    command->varNo = varNo;
    command->var = var;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         WriteTrackVariable

  Description:  シーケンストラック変数の書き込み

  Arguments:    trackNo - トラック番号
                varNo  - 変数番号
                var    - 書き込む値

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void SequenceSound::WriteTrackVariable( int trackNo, int varNo, int16_t var ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE( trackNo, 0, driver::SequenceSoundPlayer::TrackCountPerPlayer );
    NN_SDK_ASSERT_RANGE( varNo, 0, driver::SequenceTrack::TrackVariableCount );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundSetVar* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundSetVar>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = DriverCommandId_SeqSetTvar;
    command->player = &m_PlayerInstance;
    command->trackNo = trackNo;
    command->varNo = varNo;
    command->var = var;
    cmdmgr.PushCommand(command);
}

/*--------------------------------------------------------------------------------*
  Name:         GetTick

  Description:  再生開始からのティック数の取得

  Arguments:    None.

  Returns:      ティック数
 *--------------------------------------------------------------------------------*/
uint32_t SequenceSound::GetTick() const NN_NOEXCEPT
{
    if ( ! IsPlayerAvailable() )
    {
        // pre start
        return 0;
    }

    return m_PlayerInstance.GetTickCounter();
}

bool SequenceSound::IsAttachedTempSpecialHandle() NN_NOEXCEPT
{
    return m_pTempSpecialHandle != NULL;
}

void SequenceSound::DetachTempSpecialHandle() NN_NOEXCEPT
{
    m_pTempSpecialHandle->DetachSound();
}

void SequenceSound::RegisterDataLoadTask(
    const driver::SequenceSoundLoader::LoadInfo& loadInfo,
    const driver::SequenceSoundPlayer::StartInfo& startInfo ) NN_NOEXCEPT
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandSequenceSoundLoad* command =
            cmdmgr.AllocCommand<DriverCommandSequenceSoundLoad>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = DriverCommandId_SeqLoad;
        command->player = &m_PlayerInstance;
        command->startInfo = startInfo; // 構造体コピー
        command->arg.soundArchive = loadInfo.soundArchive;
        command->arg.soundDataManager = loadInfo.soundDataManager;
        command->arg.soundPlayer = loadInfo.soundPlayer;
        command->arg.loadInfoSeq = *(loadInfo.loadInfoSeq); // 構造体コピー
        for (int i = 0; i < SoundArchive::SequenceBankMax; i++)
        {
            command->arg.loadInfoBanks[i] = *(loadInfo.loadInfoBanks[i]);
            // command->arg.loadInfoWarcs[i] = *(loadInfo.loadInfoWarcs[i]);
        }

        cmdmgr.PushCommand(command);
    }
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

