﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_MmlParser.h>
#include <nn/atk/atk_MmlCommand.h>
#include <nn/atk/atk_MmlSequenceTrack.h>
#include <nn/atk/atk_SequenceSoundPlayer.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>

namespace nn {
namespace atk {
namespace detail {
namespace driver {

namespace {
const float ModSpeedBase = 0.390625f;                   // x 16 で 6.25Hz
}

NN_DEFINE_STATIC_CONSTANT( const int MmlParser::PanCenter );
NN_DEFINE_STATIC_CONSTANT( const int MmlParser::SurroundPanCenter );
NN_DEFINE_STATIC_CONSTANT( const int MmlParser::TempoMin );
NN_DEFINE_STATIC_CONSTANT( const int MmlParser::TempoMax );
bool MmlParser::mPrintVarEnabledFlag;

/*--------------------------------------------------------------------------------*
  Name:         Parse

  Description:  シーケンス処理を進めます

  Arguments:    player - プレイヤーポインタ
                trackNo - トラックナンバ
                doNoteOn - ノートオンするかどうか

  Returns:      シーケンス継続時には０を、完了時にはー１を返します
 *--------------------------------------------------------------------------------*/
SequenceTrack::ParseResult MmlParser::Parse(
    MmlSequenceTrack* track,
    bool doNoteOn
) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( track );
    SequenceSoundPlayer* player = track->GetSequenceSoundPlayer();
    NN_SDK_ASSERT_NOT_NULL( player );
    SequenceTrack::ParserTrackParam& trackParam = track->GetParserTrackParam();

    SeqArgType argType = SeqArgType_None;
    SeqArgType argType2 = SeqArgType_None;
    bool useArgType = false;
    bool doExecCommand = true;

    uint32_t cmd = ReadByte( &trackParam.currentAddr );

    // 接頭バイナリの処理
    if ( cmd == MmlCommand::Mml_If )
    {
        cmd = ReadByte( &trackParam.currentAddr );
        doExecCommand = trackParam.cmpFlag != 0;
    }

    // 第２パラメータ
    if ( cmd == MmlCommand::Mml_Time )
    {
        cmd = ReadByte( &trackParam.currentAddr );
        argType2 = SeqArgType_S16;
    }
    else if ( cmd == MmlCommand::Mml_TimeRandom )
    {
        cmd = ReadByte( &trackParam.currentAddr );
        argType2 = SeqArgType_Random;
    }
    else if ( cmd == MmlCommand::Mml_TimeVariable )
    {
        cmd = ReadByte( &trackParam.currentAddr );
        argType2 = SeqArgType_Variable;
    }

    // パラメータ
    if ( cmd == MmlCommand::Mml_Random )
    {
        cmd = ReadByte( &trackParam.currentAddr );
        argType = SeqArgType_Random;
        useArgType = true;
    }
    else if ( cmd == MmlCommand::Mml_Variable )
    {
        cmd = ReadByte( &trackParam.currentAddr );
        argType = SeqArgType_Variable;
        useArgType = true;
    }

    if ( ( cmd & 0x80 ) == 0 )
    {
        // ノートコマンド
        const uint8_t velocity = ReadByte( &trackParam.currentAddr );
        const int32_t length = ReadArg(
            &trackParam.currentAddr,
            player,
            track,
            useArgType ? argType : SeqArgType_Vmidi
        );
        int key = static_cast<int>( cmd ) + trackParam.transpose;
        if ( ! doExecCommand ) return SequenceTrack::ParseResult_Continue;

        key = nn::atk::detail::fnd::Clamp( key, 0, 127 );

        if ( ! trackParam.muteFlag && doNoteOn )
        {
            // NN_DETAIL_ATK_INFO("    key(%d) velocity(%d) len(%d)\n", key, velocity, length);
            NoteOnCommandProc(
                track,
                key,
                velocity,
                length > 0 ? length : -1,
                trackParam.tieFlag
            );
        }

        if ( trackParam.noteWaitFlag ) {
            trackParam.wait = length;
            if ( length == 0 ) {
                trackParam.noteFinishWait = true;
            }
        }
    }
    else
    {
        int32_t commandArg1 = 0;
        int32_t commandArg2 = 0;

        switch ( cmd & 0xf0 )
        {
        case 0x80:
        {
            switch ( cmd )
            {
            case MmlCommand::Mml_Wait:
            {
                // ここで処理する
                int32_t arg = ReadArg(
                    &trackParam.currentAddr,
                    player,
                    track,
                    useArgType ? argType : SeqArgType_Vmidi
                );
                // NN_DETAIL_ATK_INFO("    wait(%d)\n", arg);
                if ( doExecCommand )
                {
                    trackParam.wait = arg;
                }
                break;
            }
            case MmlCommand::Mml_Prg:
            {
                commandArg1 = ReadArg(
                    &trackParam.currentAddr,
                    player,
                    track,
                    useArgType ? argType : SeqArgType_Vmidi
                );
                // NN_DETAIL_ATK_INFO("    prg(%d)\n", commandArg1);
                if ( doExecCommand )
                {
                    CommandProc(
                        track,
                        cmd,
                        commandArg1,
                        commandArg2
                    );
                }
                break;
            }
            case MmlCommand::Mml_OpenTrack:
            {
                uint8_t trackNo = ReadByte( &trackParam.currentAddr );
                uint32_t offset = Read24( &trackParam.currentAddr );
                if ( doExecCommand )
                {
                    commandArg1 = trackNo;
                    commandArg2 = static_cast<signed long>( offset );
                    CommandProc(
                        track,
                        cmd,
                        commandArg1,
                        commandArg2
                    );
                }
                // NN_DETAIL_ATK_INFO("    openTrack trkNo(%d) offset(%d/%x)\n", commandArg1, commandArg2, commandArg2);
                break;
            }
            case MmlCommand::Mml_Jump:
            {
                uint32_t offset = Read24( &trackParam.currentAddr );
                if ( doExecCommand )
                {
                    commandArg1 = static_cast<signed long>( offset );
                    CommandProc(
                        track,
                        cmd,
                        commandArg1,
                        commandArg2
                    );
                }
                // NN_DETAIL_ATK_INFO("    jump offset(%d)\n", offset);
                break;
            }
            case MmlCommand::Mml_Call:
            {
                uint32_t offset = Read24( &trackParam.currentAddr );
                if ( doExecCommand )
                {
                    commandArg1 = static_cast<signed long>( offset );
                    CommandProc(
                        track,
                        cmd,
                        commandArg1,
                        commandArg2
                    );
                }
                break;
            }
            default:
                break;

            }
            break; // case 0x88 - 0x8f
        }

        case 0xb0: case 0xc0: case 0xd0: // uint8_t パラメータ ( 一部 int8_t パラメータが混ざってる )
        {
            uint8_t arg = static_cast<uint8_t>( ReadArg(
                &trackParam.currentAddr,
                player,
                track,
                useArgType ? argType : SeqArgType_U8
            ) );

            if ( argType2 != SeqArgType_None )
            {
                commandArg2 = ReadArg(
                    &trackParam.currentAddr,
                    player,
                    track,
                    argType2
                );
            }

            if ( doExecCommand )
            {
                switch ( cmd )
                {
                case MmlCommand::Mml_Transpose:
                case MmlCommand::Mml_PitchBend:
                    commandArg1 = *reinterpret_cast<int8_t*>( &arg );
                    break;
                default:
                    commandArg1 = arg;
                }

                CommandProc(
                    track,
                    cmd,
                    commandArg1,
                    commandArg2
                );
            }
            break;
        }

        case 0x90: // 予備
        {
            if ( doExecCommand )
            {
                CommandProc(
                    track,
                    cmd,
                    commandArg1,
                    commandArg2
                );
            }
            break;
        }

        case 0xe0: // int16_t パラメータ
        {
            commandArg1 = static_cast<int16_t>( ReadArg(
                &trackParam.currentAddr,
                player,
                track,
                useArgType ? argType : SeqArgType_S16
            ) );

            if ( doExecCommand )
            {
                CommandProc(
                    track,
                    cmd,
                    commandArg1,
                    commandArg2
                );
            }
            break;
        }

        case 0xf0: // パラメータ無し or 拡張コマンド
        {

            switch ( cmd )
            {
            case MmlCommand::Mml_AllocTrack:
                (void)Read16( &trackParam.currentAddr );
                NN_SDK_ASSERT( false, "seq: must use alloctrack in startup code");
                break;

            case MmlCommand::Mml_Fin:
                if ( doExecCommand )
                {
                    return SequenceTrack::ParseResult_Finish;
                }
                break;

            case MmlCommand::Mml_ExCommand:
            {
                uint32_t cmdex = ReadByte( &trackParam.currentAddr );

                switch ( cmdex & 0xf0 )
                {
                case 0xa0: case 0xb0: // uint8_t パラメータ
                {
                    commandArg1 = ReadByte( &trackParam.currentAddr );
                    if (doExecCommand)
                    {
                        CommandProc(
                            track,
                            (cmd << 8) + cmdex,
                            commandArg1,
                            commandArg2
                        );
                    }
                    break;
                }

                case 0xe0: // uint16_t パラメータ
                {
                    commandArg1 = static_cast<uint16_t>( ReadArg(
                        &trackParam.currentAddr,
                        player,
                        track,
                        useArgType ? argType : SeqArgType_S16
                    ) );
                    if ( doExecCommand )
                    {
                        CommandProc(
                            track,
                            (cmd << 8) + cmdex,
                            commandArg1,
                            commandArg2
                        );
                    }
                    break;
                }

                case 0x80: case 0x90: // 2パラメータ
                {
                    commandArg1 = ReadByte( &trackParam.currentAddr );
                    commandArg2 = static_cast<int16_t>( ReadArg(
                        &trackParam.currentAddr,
                        player,
                        track,
                        useArgType ? argType : SeqArgType_S16
                    ) );
                    if ( doExecCommand )
                    {
                        CommandProc(
                            track,
                            (cmd << 8) + cmdex,
                            commandArg1,
                            commandArg2
                        );
                    }
                    break;
                }

                default:
                    break;

                }
                break;
            } // case MmlCommand::Mml_ExCommand

            default:
            {
                if ( doExecCommand )
                {
                    CommandProc(
                        track,
                        cmd,
                        commandArg1,
                        commandArg2
                    );
                }
                break;
            }
            } // case 0xf0 - 0xff
            break;
        } // case 0xf0

        case 0xa0: // 接頭コマンド
        {
            // ここに接頭コマンドがあるのは不正
            NN_SDK_ASSERT( false, "Invalid seqdata command: %d", cmd );
        }

        default:
            break;

        }
    }

    return SequenceTrack::ParseResult_Continue;
} // NOLINT(impl/function_size)

/*--------------------------------------------------------------------------------*
  Name:         CommandProc

  Description:  一つのＭＭＬコマンドに対する処理を行います。

  Arguments:    track - トラックポインタ
                trackParam - パーサパラメータ
                cmd     - コマンド
                cmdex   - 拡張コマンド
                commandArg1  - コマンドパラメータ１
                commandArg2  - コマンドパラメータ２

  Returns:      なし
 *--------------------------------------------------------------------------------*/
void MmlParser::CommandProc(
    MmlSequenceTrack* track,
    uint32_t command,
    int32_t commandArg1,
    int32_t commandArg2
) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( track );
    SequenceSoundPlayer* player = track->GetSequenceSoundPlayer();
    NN_SDK_ASSERT_NOT_NULL( player );
    SequenceTrack::ParserTrackParam& trackParam = track->GetParserTrackParam();
    SequenceSoundPlayer::ParserPlayerParam& playerParam = player->GetParserPlayerParam();

    if ( command <= 0xff ) // 1バイトコマンド
    {
        switch ( command )
        {

        case MmlCommand::Mml_Tempo:
            playerParam.tempo = static_cast<uint16_t>(
                nn::atk::detail::fnd::Clamp( static_cast<int>( commandArg1 ), TempoMin, TempoMax )
            );
            // NN_DETAIL_ATK_INFO("tempo(%d) -> (%d)\n", commandArg1, playerParam.tempo);
            break;

        case MmlCommand::Mml_Timebase:
            playerParam.timebase = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_Prg:
            if ( commandArg1 < 0x10000 )
            {
                trackParam.prgNo = static_cast<uint16_t>( commandArg1 );
            }
            else
            {
                NN_ATK_WARNING("nn::atk::MmlParser: too large prg No. %d", commandArg1);
            }
            break;

        case MmlCommand::Mml_Mute:
            track->SetMute( static_cast<SequenceMute>( commandArg1 ) );
            break;

        case MmlCommand::Mml_Volume:
            trackParam.volume.SetTarget(
                static_cast<uint8_t>( commandArg1 ),
                static_cast<int16_t>( commandArg2 )
            );
            break;

        case MmlCommand::Mml_Volume2:
            trackParam.volume2.SetTarget(
                static_cast<uint8_t>( commandArg1 ),
                static_cast<int16_t>( commandArg2 )
            );
            break;

        case MmlCommand::Mml_VelocityRange:
            trackParam.velocityRange = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_MainVolume:
            playerParam.volume.SetTarget(
                static_cast<uint8_t>( commandArg1 ),
                static_cast<int16_t>( commandArg2 )
            );
            break;

        case MmlCommand::Mml_Transpose:
            trackParam.transpose = static_cast<int8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_PitchBend:
            trackParam.pitchBend.SetTarget(
                static_cast<int8_t>( commandArg1 ),
                static_cast<int16_t>( commandArg2 )
            );
            break;

        case MmlCommand::Mml_BendRange:
            trackParam.bendRange = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_Pan:
            trackParam.pan.SetTarget(
                static_cast<int8_t>( commandArg1 - PanCenter ),
                static_cast<int16_t>( commandArg2 )
            );
            break;

        case MmlCommand::Mml_InitPan:
            trackParam.initPan = static_cast<int8_t>( commandArg1 - PanCenter );
            break;

        case MmlCommand::Mml_SurroundPan:
            trackParam.surroundPan.SetTarget(
                static_cast<int8_t>( commandArg1 ),
                static_cast<int16_t>( commandArg2 )
            );
            break;

        case MmlCommand::Mml_Prio:
            trackParam.priority = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_NoteWait:
            trackParam.noteWaitFlag = ( commandArg1 != 0 );
            break;

        case MmlCommand::Mml_FrontBypass:
            trackParam.frontBypassFlag = ( commandArg1 != 0 );
            break;

        case MmlCommand::Mml_PortaTime:
            trackParam.portaTime = static_cast<uint8_t>( commandArg1 );
            break;

        // モジュレーション 1 {{
        case MmlCommand::Mml_ModDepth:
            trackParam.lfoParam[0].depth = static_cast<uint8_t>( commandArg1 ) / 128.0f;
            break;

        case MmlCommand::Mml_ModSpeed:
            trackParam.lfoParam[0].speed = static_cast<uint8_t>( commandArg1 ) * ModSpeedBase;
            break;

        case MmlCommand::Mml_ModPeriod:
            {
                int16_t arg = static_cast<int16_t>(commandArg1);
                if (arg == 0)
                {
                    trackParam.lfoParam[0].speed = 0.0f;
                }
                else
                {
                    trackParam.lfoParam[0].speed = 100.f / arg;
                }
            }
            break;

        case MmlCommand::Mml_ModPhase:
            trackParam.lfoParam[0].phase = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_ModType:
            trackParam.lfoTarget[0] = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_ModRange:
            trackParam.lfoParam[0].range = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_ModDelay:
            trackParam.lfoParam[0].delay = static_cast<uint32_t>( commandArg1 * 5 ); // 単位5ms
            break;

        case MmlCommand::Mml_ModCurve:
            trackParam.lfoParam[0].curve = static_cast<uint8_t>( commandArg1 );
            break;
        // }}


        case MmlCommand::Mml_SweepPitch:
            trackParam.sweepPitch = static_cast<float>( commandArg1 ) / 64.0f;
            break;

        case MmlCommand::Mml_Attack:
            trackParam.attack = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_Decay:
            trackParam.decay = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_Sustain:
            trackParam.sustain = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_Release:
            trackParam.release = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_EnvHold:
            trackParam.envHold = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_EnvReset:
            trackParam.attack = SequenceTrack::InvalidEnvelope;
            trackParam.decay = SequenceTrack::InvalidEnvelope;
            trackParam.sustain = SequenceTrack::InvalidEnvelope;
            trackParam.release = SequenceTrack::InvalidEnvelope;
            trackParam.envHold = SequenceTrack::InvalidEnvelope;
            break;

        case MmlCommand::Mml_Damper:
            trackParam.damperFlag = ( static_cast<uint8_t>( commandArg1 ) >= 64 );
            break;

        case MmlCommand::Mml_Tie:
            trackParam.tieFlag = ( commandArg1 != 0 );
            track->ReleaseAllChannel( -1 );
            track->FreeAllChannel();
            break;

        case MmlCommand::Mml_Monophonic:
            trackParam.monophonicFlag = ( commandArg1 != 0 );
            if ( trackParam.monophonicFlag )
            {
                track->ReleaseAllChannel( -1 );
                track->FreeAllChannel();
            }
            break;

        case MmlCommand::Mml_Porta:
            trackParam.portaKey = static_cast<uint8_t>( commandArg1 + trackParam.transpose );
            trackParam.portaFlag = true;
            break;

        case MmlCommand::Mml_PortaSw:
            trackParam.portaFlag = ( commandArg1 != 0 );
            break;

        case MmlCommand::Mml_LpfCutoff:
            trackParam.lpfFreq = static_cast<float>( commandArg1 - 64 ) / 64.0f;
            break;

        case MmlCommand::Mml_BiquadType:
            trackParam.biquadType = static_cast<int8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_BiquadValue:
            trackParam.biquadValue = static_cast<float>( commandArg1 ) / 127.0f;
            break;

        case MmlCommand::Mml_BankSelect:
            trackParam.bankIndex = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_FxsendA:
            trackParam.fxSend[ AuxBus_A ] = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_FxsendB:
            trackParam.fxSend[ AuxBus_B ] = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_FxsendC:
            trackParam.fxSend[ AuxBus_C ] = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_Mainsend:
            trackParam.mainSend = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::Mml_Printvar:
            if ( mPrintVarEnabledFlag ) {
                const volatile int16_t* const varPtr = GetVariablePtr( player, track, commandArg1 );
#if defined(NN_SDK_BUILD_RELEASE)
                NN_UNUSED(varPtr);
#endif
                NN_SDK_ASSERT_NOT_NULL(varPtr);
                NN_DETAIL_ATK_INFO( "#%08x[%d]: printvar %sVAR_%d(%d) = %d\n",
                    player,
                    track->GetPlayerTrackNo(),
                    ( commandArg1 >= 32 )? "T": ( commandArg1 >= 16 )? "G": "",
                    ( commandArg1 >= 32 )? commandArg1 - 32: ( commandArg1 >= 16 )? commandArg1 - 16: commandArg1,
                    commandArg1,
                    *varPtr );
            }
            break;

        case MmlCommand::Mml_OpenTrack:
        {
            SequenceTrack* newTrack = player->GetPlayerTrack( commandArg1 );
            if ( newTrack == NULL )
            {
                NN_ATK_WARNING("nn::atk::MmlParser: opentrack for not allocated track");
                break;
            }
            if ( newTrack == track )
            {
                NN_ATK_WARNING("nn::atk::MmlParser: opentrack for self track");
                break;
            }
            newTrack->Close();
            newTrack->SetSeqData( trackParam.baseAddr, commandArg2 );
            newTrack->Open();
            break;
        }

        case MmlCommand::Mml_Jump:
            trackParam.currentAddr = trackParam.baseAddr + commandArg1;
            break;

        case MmlCommand::Mml_Call:
        {
            if ( trackParam.callStackDepth >= SequenceTrack::CallStackDepth ) {
                NN_ATK_WARNING("nn::atk::MmlParser: cannot 'call' because already too deep");
                break;
            }

            SequenceTrack::ParserTrackParam::CallStack* callStack = &trackParam.callStack[ trackParam.callStackDepth ];
            callStack->address = trackParam.currentAddr;
            callStack->loopFlag = false;
            trackParam.callStackDepth++;
            trackParam.currentAddr = trackParam.baseAddr + commandArg1;
            break;
        }

        case MmlCommand::Mml_Ret:
        {
            SequenceTrack::ParserTrackParam::CallStack* callStack = NULL;
            while( trackParam.callStackDepth > 0 ) {
                trackParam.callStackDepth--;
                if ( ! trackParam.callStack[ trackParam.callStackDepth ].loopFlag ) {
                    callStack = &trackParam.callStack[ trackParam.callStackDepth ];
                    break;
                }
            }
            if ( callStack == NULL ) {
                NN_ATK_WARNING("nn::atk::MmlParser: unmatched sequence command 'ret'");
                break;
            }
            trackParam.currentAddr = callStack->address;
            break;
        }

        case MmlCommand::Mml_LoopStart:
        {
            if ( trackParam.callStackDepth >= SequenceTrack::CallStackDepth ) {
                NN_ATK_WARNING("nn::atk::MmlParser: cannot 'loop_start' because already too deep");
                break;
            }

            SequenceTrack::ParserTrackParam::CallStack* callStack = &trackParam.callStack[ trackParam.callStackDepth ];
            callStack->address = trackParam.currentAddr;
            callStack->loopCount = static_cast<uint8_t>( commandArg1 );
            callStack->loopFlag = true;
            trackParam.callStackDepth++;
            break;
        }

        case MmlCommand::Mml_LoopEnd:
        {
            if ( trackParam.callStackDepth == 0 ) {
                NN_ATK_WARNING("nn::atk::MmlParser: unmatched sequence command 'loop_end'");
                break;
            }

            SequenceTrack::ParserTrackParam::CallStack* callStack = & trackParam.callStack[ trackParam.callStackDepth - 1 ];
            if ( ! callStack->loopFlag ) {
                NN_ATK_WARNING("nn::atk::MmlParser: unmatched sequence command 'loop_end'");
                break;
            }

            uint8_t loop_count = callStack->loopCount;
            if ( loop_count > 0 ) {
                loop_count--;
                if ( loop_count == 0 ) {
                    trackParam.callStackDepth--;
                    break;
                }
            }

            callStack->loopCount = loop_count;
            trackParam.currentAddr = callStack->address;
            break;
        }

        default:
            break;

        }
    }
    else if ( command <= 0xffff ) // 2バイトコマンド
    {
    #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        uint32_t cmd = command >> 8;
        NN_SDK_ASSERT( cmd == MmlCommand::Mml_ExCommand );
    #endif // NN_SDK_BUILD_DEBUG || NN_SDK_BUILD_DEVELOP
        uint32_t cmdex = command & 0xff;

        // -----------------------------------------------------------------
        // 拡張コマンド

        volatile int16_t* varPtr = NULL;
        if ( (( cmdex & 0xf0 ) == 0x80 ) ||
             (( cmdex & 0xf0 ) == 0x90 )
           )
        {
            // シーケンス変数コマンドのときはシーケンス変数へのポインタを取得
            varPtr = GetVariablePtr( player, track, commandArg1 );
            if ( varPtr == NULL ) return;
        }


        switch ( cmdex )
        {
        case MmlCommand::MmlEx_Setvar:
            *varPtr = static_cast<int16_t>( commandArg2 );
            break;

        case MmlCommand::MmlEx_Addvar:
            *varPtr += static_cast<int16_t>( commandArg2 );
            break;

        case MmlCommand::MmlEx_Subvar:
            *varPtr -= static_cast<int16_t>( commandArg2 );
            break;

        case MmlCommand::MmlEx_Mulvar:
            *varPtr *= static_cast<int16_t>( commandArg2 );
            break;

        case MmlCommand::MmlEx_Divvar:
            if ( commandArg2 != 0 )
            {
                *varPtr /= static_cast<int16_t>( commandArg2 );
            }
            break;

        case MmlCommand::MmlEx_Shiftvar:
            if ( commandArg2 >= 0 )
            {
                *varPtr <<= commandArg2;
            }
            else
            {
                *varPtr >>= -commandArg2;
            }
            break;

        case MmlCommand::MmlEx_Randvar:
        {
            bool minus_flag = false;
            int32_t rand;

            if ( commandArg2 < 0 ) {
                minus_flag = true;
                commandArg2 = static_cast<int16_t>( -commandArg2 );
            }

            rand = Util::CalcRandom();
            rand *= commandArg2 + 1;
            rand >>= 16;
            if ( minus_flag ) rand = -rand;
            *varPtr = static_cast<int16_t>( rand );
            break;

        }

        case MmlCommand::MmlEx_Andvar:
            *varPtr &= commandArg2;
            break;

        case MmlCommand::MmlEx_Orvar:
            *varPtr |= commandArg2;
            break;

        case MmlCommand::MmlEx_Xorvar:
            *varPtr ^= commandArg2;
            break;

        case MmlCommand::MmlEx_Notvar:
            *varPtr = static_cast<int16_t>( ~static_cast<uint16_t>( commandArg2 ) );
            break;

        case MmlCommand::MmlEx_Modvar:
            if ( commandArg2 != 0 )
            {
                *varPtr %= commandArg2;
            }
            break;

        case MmlCommand::MmlEx_CmpEq:
            trackParam.cmpFlag = ( *varPtr == commandArg2 ) ;
            break;

        case MmlCommand::MmlEx_CmpGe:
            trackParam.cmpFlag = ( *varPtr >= commandArg2 ) ;
            break;

        case MmlCommand::MmlEx_CmpGt:
            trackParam.cmpFlag = ( *varPtr > commandArg2 ) ;
            break;

        case MmlCommand::MmlEx_CmpLe:
            trackParam.cmpFlag = ( *varPtr <= commandArg2 ) ;
            break;

        case MmlCommand::MmlEx_CmpLt:
            trackParam.cmpFlag = ( *varPtr < commandArg2 ) ;
            break;

        case MmlCommand::MmlEx_CmpNe:
            trackParam.cmpFlag = ( *varPtr != commandArg2 ) ;
            break;

        case MmlCommand::MmlEx_Userproc:
            player->CallSequenceUserprocCallback(
                static_cast<uint16_t>( commandArg1 ), // procId
                track
            );
            break;

        // モジュレーション 2 {{
        case MmlCommand::MmlEx_Mod2Depth:
            trackParam.lfoParam[1].depth = static_cast<uint8_t>( commandArg1 ) / 128.0f;
            break;

        case MmlCommand::MmlEx_Mod2Speed:
            trackParam.lfoParam[1].speed = static_cast<uint8_t>( commandArg1 ) * ModSpeedBase;
            break;

        case MmlCommand::MmlEx_Mod2Period:
            {
                int16_t arg = static_cast<int16_t>(commandArg1);
                if (arg == 0)
                {
                    trackParam.lfoParam[1].speed = 0.0f;
                }
                else
                {
                    trackParam.lfoParam[1].speed = 100.f / arg;
                }
            }
            break;

        case MmlCommand::MmlEx_Mod2Phase:
            trackParam.lfoParam[1].phase = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod2Type:
            trackParam.lfoTarget[1] = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod2Range:
            trackParam.lfoParam[1].range = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod2Delay:
            trackParam.lfoParam[1].delay = static_cast<uint32_t>( commandArg1 * 5 ); // 単位5ms
            break;

        case MmlCommand::MmlEx_Mod2Curve:
            trackParam.lfoParam[1].curve = static_cast<uint8_t>( commandArg1 );
            break;
        // }}
        // モジュレーション 3 {{
        case MmlCommand::MmlEx_Mod3Depth:
            trackParam.lfoParam[2].depth = static_cast<uint8_t>( commandArg1 ) / 128.0f;
            break;

        case MmlCommand::MmlEx_Mod3Speed:
            trackParam.lfoParam[2].speed = static_cast<uint8_t>( commandArg1 ) * ModSpeedBase;
            break;

        case MmlCommand::MmlEx_Mod3Period:
            {
                int16_t arg = static_cast<int16_t>(commandArg1);
                if (arg == 0)
                {
                    trackParam.lfoParam[2].speed = 0.0f;
                }
                else
                {
                    trackParam.lfoParam[2].speed = 100.f / arg;
                }
            }
            break;

        case MmlCommand::MmlEx_Mod3Phase:
            trackParam.lfoParam[2].phase = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod3Type:
            trackParam.lfoTarget[2] = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod3Range:
            trackParam.lfoParam[2].range = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod3Delay:
            trackParam.lfoParam[2].delay = static_cast<uint32_t>( commandArg1 * 5 ); // 単位5ms
            break;

        case MmlCommand::MmlEx_Mod3Curve:
            trackParam.lfoParam[2].curve = static_cast<uint8_t>( commandArg1 );
            break;
        // }}
        // モジュレーション 4 {{
        case MmlCommand::MmlEx_Mod4Depth:
            trackParam.lfoParam[3].depth = static_cast<uint8_t>( commandArg1 ) / 128.0f;
            break;

        case MmlCommand::MmlEx_Mod4Speed:
            trackParam.lfoParam[3].speed = static_cast<uint8_t>( commandArg1 ) * ModSpeedBase;
            break;

        case MmlCommand::MmlEx_Mod4Period:
            {
                int16_t arg = static_cast<int16_t>(commandArg1);
                if (arg == 0)
                {
                    trackParam.lfoParam[3].speed = 0.0f;
                }
                else
                {
                    trackParam.lfoParam[3].speed = 100.f / arg;
                }
            }
            break;

        case MmlCommand::MmlEx_Mod4Phase:
            trackParam.lfoParam[3].phase = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod4Type:
            trackParam.lfoTarget[3] = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod4Range:
            trackParam.lfoParam[3].range = static_cast<uint8_t>( commandArg1 );
            break;

        case MmlCommand::MmlEx_Mod4Delay:
            trackParam.lfoParam[3].delay = static_cast<uint32_t>( commandArg1 * 5 ); // 単位5ms
            break;

        case MmlCommand::MmlEx_Mod4Curve:
            trackParam.lfoParam[3].curve = static_cast<uint8_t>( commandArg1 );
            break;

        default:
            break;
        // }}

        } // switch ( cmdex )
    }
} // NOLINT(impl/function_size)

/*--------------------------------------------------------------------------------*
  Name:         NoteOnCommandProc

  Description:  ノートオンコマンドを処理します

  Arguments:    track - トラックポインタ
                trackParam - パーサパラメータ
                key      - キー番号
                velocity - ベロシティ
                length   - ノート長

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void MmlParser::NoteOnCommandProc(
    MmlSequenceTrack* track,
    int key,
    int velocity,
    int32_t length,
    bool tieFlag
) const NN_NOEXCEPT
{
    track->NoteOn( key, velocity, length, tieFlag );
}

/*--------------------------------------------------------------------------------*
  Name:         Read16

  Description:  シーケンスデータを２バイト読み込みます

  Arguments:

  Returns:      読み込んだデータ
 *--------------------------------------------------------------------------------*/
uint16_t MmlParser::Read16( const uint8_t** ptr ) const NN_NOEXCEPT
{
    uint16_t ret = ReadByte( ptr );
    ret |= (ReadByte( ptr ) << 8);
    return ret;
}

/*--------------------------------------------------------------------------------*
  Name:         Read24

  Description:  シーケンスデータを３バイト読み込みます

  Arguments:

  Returns:      読み込んだデータ
 *--------------------------------------------------------------------------------*/
uint32_t MmlParser::Read24( const uint8_t** ptr ) const NN_NOEXCEPT
{
    uint32_t ret = ReadByte( ptr );
    ret |= (ReadByte( ptr ) << 8);
    ret |= (ReadByte( ptr ) << 16);
    return ret;
}

/*--------------------------------------------------------------------------------*
  Name:         ReadVar

  Description:  可変長シーケンスデータを読み込みます

  Arguments:

  Returns:      読み込んだデータ
 *--------------------------------------------------------------------------------*/
int32_t MmlParser::ReadVar( const uint8_t** ptr ) const NN_NOEXCEPT
{
    int32_t ret = 0;
    uint8_t b;
    int i;

    for( i = 0 ; ; ++i ) {
        NN_SDK_ASSERT( i < 4 );
        b = ReadByte( ptr );
        ret <<= 7;
        ret |= b & 0x7f;
        if ( ! ( b & 0x80 ) ) break;
    }

    return ret;
}

/*--------------------------------------------------------------------------------*
  Name:         ReadArg

  Description:  シーケンスコマンド引数を読み込みます

  Arguments:    player - プレイヤーポインタ
                argType - 引数タイプ

  Returns:      読み込んだ値
 *--------------------------------------------------------------------------------*/
int32_t MmlParser::ReadArg( const uint8_t** ptr, SequenceSoundPlayer* player, SequenceTrack* track, SeqArgType argType ) const NN_NOEXCEPT
{
    int32_t var = 0;

    switch ( argType ) {

    case SeqArgType_U8:
        var = ReadByte( ptr );
        break;

    case SeqArgType_S16:
        var = Read16( ptr );
        break;

    case SeqArgType_Vmidi:
        var = ReadVar( ptr );
        break;

    case SeqArgType_Variable: {
        uint8_t varNo = ReadByte( ptr );
        const volatile int16_t* varPtr = GetVariablePtr( player, track, varNo );
        if ( varPtr != NULL ) {
            var = *varPtr;
        }
        break;
    }

    case SeqArgType_Random: {
        int32_t rand;
        int16_t min;
        int16_t max;

        min = static_cast<int16_t>( Read16( ptr ) );
        max = static_cast<int16_t>( Read16( ptr ) );

        rand = Util::CalcRandom();  /* 0x0000 - 0xffff */
        rand *= ( max - min ) + 1;
        rand >>= 16;
        rand += min;
        var = rand;
        break;
    }

    default:
        break;
    }

    return var;
}

volatile int16_t* MmlParser::GetVariablePtr( SequenceSoundPlayer* player, SequenceTrack* track, int varNo ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        varNo >= 0 && varNo <= SequenceSoundPlayer::PlayerVariableCount + SequenceSoundPlayer::GlobalVariableCount + SequenceTrack::TrackVariableCount
    );

    if ( varNo < SequenceSoundPlayer::PlayerVariableCount
               + SequenceSoundPlayer::GlobalVariableCount )
    {
        return player->GetVariablePtr( varNo );
    }
    else if ( varNo < SequenceSoundPlayer::PlayerVariableCount
                    + SequenceSoundPlayer::GlobalVariableCount
                    + SequenceTrack::TrackVariableCount )
    {
        return track->GetVariablePtr( varNo - SequenceSoundPlayer::PlayerVariableCount - SequenceSoundPlayer::GlobalVariableCount );
    }
    else
    {
        return NULL;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         ParseAllocTrack [static]

  Description:  シーケンスデータを解析し、alloctrackコマンドを読みとります。
                指定したオフセット位置に alloctrackコマンドがある場合は、
                コマンド引数を読み取り、allocTrack引数へ格納し、
                読み取り後のオフセット位置を返します。
                alloctrackコマンドが無い場合は、
                allocTrack引数へ"1"を代入し、オフセット位置をそのまま返します。

  Arguments:    baseAddress - シーケンスデータの先頭アドレス
                seqOffset - シーケンスデータの開始オフセット
                allocTrack - 確保すべきトラックのビットフラグを格納するアドレス

  Returns:      解析後のシーケンスデータの開始オフセット
  *--------------------------------------------------------------------------------*/
uint32_t MmlParser::ParseAllocTrack( const void* baseAddress, uint32_t seqOffset, uint32_t* allocTrack ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( baseAddress );
    NN_SDK_ASSERT_NOT_NULL( allocTrack );

    const uint8_t* ptr = static_cast<const uint8_t*>( util::ConstBytePtr( baseAddress, seqOffset ).Get() );
    if ( *ptr != MmlCommand::Mml_AllocTrack ) {
        *allocTrack = (1 << 0);
        return seqOffset;
    }
    else {
        ++ptr;
        uint32_t tracks = *ptr;
        tracks <<= 8;
        ++ptr;
        tracks |= *ptr;
        tracks |= (1 << 0);
        *allocTrack = tracks;
        return seqOffset + 3;
    }
}

} // namespace nn::atk::detail::driver
} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

