﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WaveFileReader.h>

#include <nn/atk/atk_Util.h>
#include <nn/atk/fnd/binary/atkfnd_Binary.h>

namespace nn {
namespace atk {
namespace detail {

namespace {

const uint32_t SignatureInfoBlockWav = NN_UTIL_CREATE_SIGNATURE_4( 'I', 'N', 'F', 'O' );
const uint32_t SignatureDataBlockWav = NN_UTIL_CREATE_SIGNATURE_4( 'D', 'A', 'T', 'A' );

const uint32_t SupportedFileVersionWav       = NN_ATK_MAKE_VERSION( 0, 1, 0, 0 );  // ライブラリがサポートする最低バージョン
const uint32_t CurrentFileVersionWav         = NN_ATK_MAKE_VERSION( 0, 1, 2, 0 );  // ライブラリがサポートする最新バージョン
const uint32_t IncludeOriginalLoopVersionWav = NN_ATK_MAKE_VERSION( 0, 1, 2, 0 );

bool IsValidFileHeaderWav( const void* waveFile ) NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( waveFile );

    // シグニチャ確認
    NN_SDK_ASSERT( header.signature == WaveFileReader::SignatureFile,
            "invalid file signature." );
    if ( header.signature != WaveFileReader::SignatureFile )
    {
        return false;
    }

    // バイトオーダーマーク確認
    NN_SDK_ASSERT( header.byteOrder == BinaryFileHeader::ValidByteOrderMark,
            "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", BinaryFileHeader::ValidByteOrderMark, header.byteOrder );
    if ( header.byteOrder != BinaryFileHeader::ValidByteOrderMark )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SupportedFileVersionWav <= header.version) &&
                                   (header.version <= CurrentFileVersionWav) )
    {
        isSupportedVersion = true;

    }
    NN_SDK_ASSERT( isSupportedVersion,
            "bfwav file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SupportedFileVersionWav, CurrentFileVersionWav, header.version
    );
    return isSupportedVersion;
}

} // anonymous namespace

NN_DEFINE_STATIC_CONSTANT( const uint32_t WaveFileReader::SignatureFile );

SampleFormat WaveFileReader::GetSampleFormat( uint8_t format ) NN_NOEXCEPT
{
    switch ( format )
    {
    case WaveFile::EncodeMethod_Pcm8:       return SampleFormat_PcmS8;
    case WaveFile::EncodeMethod_Pcm16:      return SampleFormat_PcmS16;
    case WaveFile::EncodeMethod_DspAdpcm:   return SampleFormat_DspAdpcm;
    default:
        NN_SDK_ASSERT( false, "Unknown wave data format(%d)", format );
        return SampleFormat_DspAdpcm;
    }
}

WaveFileReader::WaveFileReader( const void* waveFile, int8_t waveType ) NN_NOEXCEPT
: m_pHeader(NULL), m_pInfoBlockBody( NULL ), m_pDataBlockBody( NULL ), m_WaveType(waveType)
{
    switch (m_WaveType)
    {
    case WaveType_Nwwav:
        {
            if ( ! IsValidFileHeaderWav( waveFile ) ) return;

            m_pHeader =
                reinterpret_cast<const WaveFile::FileHeader*>(waveFile);

            const WaveFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();
            const WaveFile::DataBlock* dataBlock = m_pHeader->GetDataBlock();

            if ( infoBlock == NULL ) return;
            if ( dataBlock == NULL ) return;

            NN_SDK_ASSERT( infoBlock->header.kind == SignatureInfoBlockWav );
            NN_SDK_ASSERT( dataBlock->header.kind == SignatureDataBlockWav );

            if ( infoBlock->header.kind != SignatureInfoBlockWav )
            {
                return;
            }
            if ( dataBlock->header.kind != SignatureDataBlockWav )
            {
                return;
            }

            m_pInfoBlockBody = &infoBlock->body;
            m_pDataBlockBody = &dataBlock->byte;
        }
        break;
    case WaveType_Dspadpcm:
        m_DspadpcmReader.Initialize(waveFile);
        break;
    default:
        break;
    }
}

// MEMO: 本当に必要？！
// WaveFileReader::WaveFileReader( const WaveFile::InfoBlockBody* infoBlockBody )
// : m_pInfoBlockBody( infoBlockBody )
// {
// }

bool WaveFileReader::IsOriginalLoopAvailable() const NN_NOEXCEPT
{
    const nn::atk::detail::BinaryFileHeader& header =
        *reinterpret_cast<const nn::atk::detail::BinaryFileHeader*>( m_pHeader );
    if (header.version >= IncludeOriginalLoopVersionWav)
    {
        return true;
    }
    return false;
}

bool WaveFileReader::ReadWaveInfo(
    WaveInfo* info,
    const void* waveDataOffsetOrigin ) const NN_NOEXCEPT
{
    switch (m_WaveType)
    {
    case WaveType_Nwwav:
        {
            NN_SDK_ASSERT( m_pInfoBlockBody );

            const SampleFormat format = GetSampleFormat( m_pInfoBlockBody->encoding );
            const int channelCount = m_pInfoBlockBody->GetChannelCount();
            info->sampleFormat   = format;
            info->channelCount   = channelCount;
            info->sampleRate     = m_pInfoBlockBody->sampleRate;
            info->loopFlag       = ( m_pInfoBlockBody->isLoop == 1 );
            info->loopStartFrame = m_pInfoBlockBody->loopStartFrame;
            info->loopEndFrame   = m_pInfoBlockBody->loopEndFrame;
            info->dataSize       = m_pHeader->header.fileSize - sizeof(nn::atk::detail::Util::SoundFileHeader);

            if ( IsOriginalLoopAvailable() )
            {
                info->originalLoopStartFrame = m_pInfoBlockBody->originalLoopStartFrame;
            }
            else
            {
                info->originalLoopStartFrame = m_pInfoBlockBody->loopStartFrame;
            }

            for ( int32_t i = 0; i < channelCount; i++ )
            {
                if ( i >= WaveChannelMax ) continue;

                WaveInfo::ChannelParam& channelParam = info->channelParam[ i ];

                const WaveFile::ChannelInfo& channelInfo = m_pInfoBlockBody->GetChannelInfo( i );
                // if ( channelInfo.offsetToAdpcmInfo != 0 )
                if ( channelInfo.referToAdpcmInfo.offset != 0 )
                {
                    const WaveFile::DspAdpcmInfo& adpcmInfo = channelInfo.GetDspAdpcmInfo();
                    channelParam.adpcmParam = adpcmInfo.adpcmParam;
                    channelParam.adpcmLoopParam = adpcmInfo.adpcmLoopParam;
                }

                channelParam.dataAddress = GetWaveDataAddress( &channelInfo, waveDataOffsetOrigin );
                // 1ch 分のデータサイズを計算
                // (データサイズ) = (データブロックの大きさ) - (最後のチャンネルのデータの、データブロック先頭からのオフセット)
                const WaveFile::ChannelInfo& lastChannelInfo = m_pInfoBlockBody->GetChannelInfo( channelCount - 1 );
                channelParam.dataSize = m_pHeader->GetDataBlock()->header.size - lastChannelInfo.referToSamples.offset - sizeof(m_pHeader->GetDataBlock()->header);
            }
        }
        break;
    case WaveType_Dspadpcm:
        {
            m_DspadpcmReader.ReadWaveInfo(info);
        }
        break;
    default:
        break;
    }

    return true;
}

const void* WaveFileReader::GetWaveDataAddress(
        const WaveFile::ChannelInfo* info,
        const void* waveDataOffsetOrigin ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pInfoBlockBody );
    NN_SDK_ASSERT_NOT_NULL( info );

    NN_UNUSED( waveDataOffsetOrigin );

#if 0
    const void* waveDataAddress = NULL;

    // dataLocationがデータブロックの先頭を指しているか（WaveFileの場合）
    // 波形データの先頭を指しているか（BankFile中のWaveInfoの場合）
    bool offsetIsDataBlock = ( waveDataOffsetOrigin == NULL );

    if ( waveDataOffsetOrigin == NULL ) {
        waveDataOffsetOrigin = m_pInfoBlockBody;
    }

    // MEMO: バンクの波形差し替えで WAVE_DATA_LOCATION_ADDRESS を使っていた？模様。
    //       今回、バンクの波形差し替えは、インストの warcID と waveIndex を
    //       差し替えられるようにして、対応する。
    switch( m_pInfoBlockBody->dataLocationType ) {
    case WaveFile::WAVE_DATA_LOCATION_OFFSET:
        waveDataAddress = util::BytePtr( waveDataOffsetOrigin, m_pInfoBlockBody->dataLocation ).Get();
        if ( offsetIsDataBlock ) {
            waveDataAddress = util::BytePtr( waveDataAddress, 8 ).Get();
        }
        break;
    case WaveFile::WAVE_DATA_LOCATION_ADDRESS:
        waveDataAddress = reinterpret_cast<const void*>( m_pInfoBlockBody->dataLocation );
        break;
    default:
        return NULL;
    }

    waveDataAddress = util::BytePtr( waveDataAddress, waveChannelInfo->channelDataOffset ).Get();

    return waveDataAddress;
#endif

    return info->GetSamplesAddress( m_pDataBlockBody );
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

