﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_OutputAdditionalParam.h>
#include <nn/atk/atk_Util.h>
#include <nn/nn_SdkAssert.h>

namespace
{

inline int GetAdditionalBusCount(int busCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(busCount, nn::atk::detail::DefaultBusCount, nn::atk::OutputReceiver::BusCountMax + 1);
    return busCount - nn::atk::detail::DefaultBusCount;
}

}

namespace nn { namespace atk { namespace detail {

size_t OutputAdditionalParam::GetRequiredMemSize(const SoundInstanceConfig& config) NN_NOEXCEPT
{
    size_t result = 0;
    if (config.busCount > DefaultBusCount)
    {
        result += sizeof(SendArray);
        result += SendArray::GetRequiredMemSize(GetAdditionalBusCount(config.busCount));
    }

    if (config.isBusMixVolumeEnabled)
    {
        result += sizeof(BusMixVolumePacket);
        result += BusMixVolumePacket::GetRequiredMemSize(config.busCount);
    }

    if (config.isVolumeThroughModeEnabled)
    {
        result += sizeof(VolumeThroughModePacket);
        result += VolumeThroughModePacket::GetRequiredMemSize(config.busCount);
    }
    return result;
}

void OutputAdditionalParam::Initialize(void *buffer, size_t bufferSize, const SoundInstanceConfig& config) NN_NOEXCEPT
{
    NN_UNUSED(bufferSize);
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, GetRequiredMemSize(config));

    uint8_t* ptr = static_cast<uint8_t*>(buffer);
    if (config.busCount > DefaultBusCount)
    {
        m_pAdditionalSend = reinterpret_cast<SendArray*>(ptr);
        new(m_pAdditionalSend) SendArray();
        ptr += sizeof(SendArray);

        const int AdditionalSendCount = GetAdditionalBusCount(config.busCount);
        const size_t BufferSize = SendArray::GetRequiredMemSize(AdditionalSendCount);
        m_pAdditionalSend->Initialize(ptr, BufferSize, AdditionalSendCount);
        ptr += BufferSize;
    }

    if (config.isBusMixVolumeEnabled)
    {
        m_pBusMixVolumePacket = reinterpret_cast<BusMixVolumePacket*>(ptr);
        new(m_pBusMixVolumePacket) BusMixVolumePacket();
        ptr += sizeof(BusMixVolumePacket);

        const size_t BufferSize = BusMixVolumePacket::GetRequiredMemSize(config.busCount);
        m_pBusMixVolumePacket->Initialize(ptr, BufferSize, config.busCount);
        ptr += BufferSize;
    }

    if (config.isVolumeThroughModeEnabled)
    {
        m_pVolumeThroughModePacket = reinterpret_cast<VolumeThroughModePacket*>(ptr);
        new(m_pVolumeThroughModePacket) VolumeThroughModePacket();
        ptr += sizeof(VolumeThroughModePacket);

        const size_t BufferSize = VolumeThroughModePacket::GetRequiredMemSize(config.busCount);
        m_pVolumeThroughModePacket->Initialize(ptr, BufferSize, config.busCount);
        ptr += BufferSize;
    }

    NN_SDK_ASSERT(ptr <= static_cast<uint8_t*>(buffer) + bufferSize);
}

void OutputAdditionalParam::Finalize() NN_NOEXCEPT
{
    if (m_pAdditionalSend != nullptr)
    {
        m_pAdditionalSend->Finalize();
        m_pAdditionalSend = nullptr;
    }

    if (m_pBusMixVolumePacket != nullptr)
    {
        m_pBusMixVolumePacket = nullptr;
    }

    if (m_pVolumeThroughModePacket != nullptr)
    {
        m_pVolumeThroughModePacket->Finalize();
        m_pVolumeThroughModePacket = nullptr;
    }
}

void OutputAdditionalParam::Reset() NN_NOEXCEPT
{
    if (m_pAdditionalSend != nullptr)
    {
        m_pAdditionalSend->Reset();
    }
    if (m_pBusMixVolumePacket != nullptr)
    {
        m_pBusMixVolumePacket->Reset();
    }
    if (m_pVolumeThroughModePacket != nullptr)
    {
        m_pVolumeThroughModePacket->Reset();
    }
}

void* OutputAdditionalParam::GetBufferAddr() NN_NOEXCEPT
{
    if (m_pAdditionalSend != nullptr)
    {
        return m_pAdditionalSend;
    }
    if (m_pBusMixVolumePacket != nullptr)
    {
        return m_pBusMixVolumePacket;
    }
    return m_pVolumeThroughModePacket;
}

SendArray* OutputAdditionalParam::GetAdditionalSendAddr() NN_NOEXCEPT
{
    return m_pAdditionalSend;
}

const SendArray* const OutputAdditionalParam::GetAdditionalSendAddr() const NN_NOEXCEPT
{
    return m_pAdditionalSend;
}

float OutputAdditionalParam::TryGetAdditionalSend(int bus) const NN_NOEXCEPT
{
    return m_pAdditionalSend->TryGetValue(Util::GetAdditionalSendIndex(bus));
}

bool OutputAdditionalParam::IsAdditionalSendEnabled() const NN_NOEXCEPT
{
    return m_pAdditionalSend != nullptr;
}

void OutputAdditionalParam::TrySetAdditionalSend(int bus, float send) NN_NOEXCEPT
{
    m_pAdditionalSend->TrySetValue(Util::GetAdditionalSendIndex(bus), send);
}

BusMixVolumePacket* OutputAdditionalParam::GetBusMixVolumePacketAddr() NN_NOEXCEPT
{
    return m_pBusMixVolumePacket;
}

const BusMixVolumePacket * const OutputAdditionalParam::GetBusMixVolumePacketAddr() const NN_NOEXCEPT
{
    return m_pBusMixVolumePacket;
}

float OutputAdditionalParam::GetBusMixVolume(int waveChannel, int mixChannel) const NN_NOEXCEPT
{
    return m_pBusMixVolumePacket->GetBusMixVolume(waveChannel, mixChannel);
}

const OutputBusMixVolume& OutputAdditionalParam::GetBusMixVolume() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBusMixVolumePacket);
    return m_pBusMixVolumePacket->GetBusMixVolume();
}

void OutputAdditionalParam::SetBusMixVolume(int waveChannel, int mixChannel, float volume) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBusMixVolumePacket);
    m_pBusMixVolumePacket->SetBusMixVolume(waveChannel, mixChannel, volume);
}

void OutputAdditionalParam::SetBusMixVolume(const OutputBusMixVolume & param) NN_NOEXCEPT
{
    m_pBusMixVolumePacket->SetBusMixVolume(param);
}

bool OutputAdditionalParam::IsBusMixVolumeUsed() const NN_NOEXCEPT
{
    return m_pBusMixVolumePacket->IsUsed();
}

void OutputAdditionalParam::SetBusMixVolumeUsed(bool isUsed) NN_NOEXCEPT
{
    m_pBusMixVolumePacket->SetUsed(isUsed);
}

bool OutputAdditionalParam::IsBusMixVolumeEnabledForBus(int bus) const NN_NOEXCEPT
{
    return m_pBusMixVolumePacket->IsEnabled(bus);
}

void OutputAdditionalParam::SetBusMixVolumeEnabledForBus(int bus, bool isEnabled) NN_NOEXCEPT
{
    m_pBusMixVolumePacket->SetEnabled(bus, isEnabled);
}

bool OutputAdditionalParam::IsBusMixVolumeEnabled() const NN_NOEXCEPT
{
    return m_pBusMixVolumePacket != nullptr;
}

VolumeThroughModePacket* OutputAdditionalParam::GetVolumeThroughModePacketAddr() NN_NOEXCEPT
{
    return m_pVolumeThroughModePacket;
}

const VolumeThroughModePacket* const OutputAdditionalParam::GetVolumeThroughModePacketAddr() const NN_NOEXCEPT
{
    return m_pVolumeThroughModePacket;
}

float OutputAdditionalParam::GetBinaryVolume() const NN_NOEXCEPT
{
    return m_pVolumeThroughModePacket->GetBinaryVolume();
}

void OutputAdditionalParam::SetBinaryVolume(float volume) NN_NOEXCEPT
{
    if(m_pVolumeThroughModePacket == nullptr)
    {
        return;
    }
    m_pVolumeThroughModePacket->SetBinaryVolume(volume);
}

uint8_t OutputAdditionalParam::TryGetVolumeThroughMode(int bus) const NN_NOEXCEPT
{
    return m_pVolumeThroughModePacket->TryGetVolumeThroughMode(bus);
}

void OutputAdditionalParam::TrySetVolumeThroughMode(int bus, uint8_t mode) NN_NOEXCEPT
{
    if(m_pVolumeThroughModePacket == nullptr)
    {
        return;
    }
    m_pVolumeThroughModePacket->TrySetVolumeThroughMode(bus, mode);
}

bool OutputAdditionalParam::IsVolumeThroughModeEnabled() const NN_NOEXCEPT
{
    return m_pVolumeThroughModePacket != nullptr;
}

bool OutputAdditionalParam::IsVolumeThroughModeUsed() const NN_NOEXCEPT
{
    return m_pVolumeThroughModePacket != nullptr && m_pVolumeThroughModePacket->IsVolumeThroughModeUsed();
}

void OutputAdditionalParam::SetVolumeThroughModeUsed(bool isUsed) NN_NOEXCEPT
{
    if(m_pVolumeThroughModePacket == nullptr)
    {
        return;
    }
    m_pVolumeThroughModePacket->SetVolumeThroughModeUsed(isUsed);
}

OutputAdditionalParam& OutputAdditionalParam::operator=(const OutputAdditionalParam &rhs) NN_NOEXCEPT
{
    if (m_pAdditionalSend != nullptr)
    {
        if (rhs.m_pAdditionalSend != nullptr)
        {
            *m_pAdditionalSend = *rhs.m_pAdditionalSend;
        }
        else
        {
            m_pAdditionalSend->Reset();
        }
    }

    if (m_pBusMixVolumePacket != nullptr && rhs.m_pBusMixVolumePacket != nullptr)
    {
        m_pBusMixVolumePacket->SetUsed( rhs.m_pBusMixVolumePacket->IsUsed() );
        if(m_pBusMixVolumePacket->IsUsed())
        {
            m_pBusMixVolumePacket->SetBusMixVolume(rhs.m_pBusMixVolumePacket->GetBusMixVolume());
            const int CopyCount = std::min(m_pBusMixVolumePacket->GetBusCount(), rhs.m_pBusMixVolumePacket->GetBusCount());

            for (int i = 0; i < CopyCount; ++i)
            {
                const bool isEnabled = rhs.m_pBusMixVolumePacket->IsEnabled(i);
                m_pBusMixVolumePacket->SetEnabled(i, isEnabled);
            }
            if( CopyCount < m_pBusMixVolumePacket->GetBusCount())
            {
                for (int i = CopyCount; m_pBusMixVolumePacket->GetBusCount(); ++i)
                {
                    m_pBusMixVolumePacket->SetEnabled(i, false);
                }
            }
        }
    }

    if (m_pVolumeThroughModePacket != nullptr)
    {
        if (rhs.m_pVolumeThroughModePacket != nullptr)
        {
            *m_pVolumeThroughModePacket = *rhs.m_pVolumeThroughModePacket;
        }
        else
        {
            m_pVolumeThroughModePacket->Reset();
        }
    }

    return *this;
}

}}}
