﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/detail/atk_AdvancedWaveSoundRuntime.h>

#include <nn/atk/atk_SoundDataManager.h>
#include <nn/atk/atk_SoundRuntimeUtility.h>
#include <nn/atk/atk_StartInfoReader.h>

namespace
{
    const int MemAlignSize = nn::audio::BufferAlignSize;
}

namespace nn { namespace atk { namespace detail {

    AdvancedWaveSoundRuntime::AdvancedWaveSoundRuntime() NN_NOEXCEPT
    {
    }

    AdvancedWaveSoundRuntime::~AdvancedWaveSoundRuntime() NN_NOEXCEPT
    {
    }

    bool AdvancedWaveSoundRuntime::Initialize(
        int soundCount,
        void** pOutAllocatedAddr,
        const void* endAddr
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(*pOutAllocatedAddr), 32) );
        if ( reinterpret_cast<uint64_t>(*pOutAllocatedAddr) % MemAlignSize != 0 )
        {
            return false;
        }

        bool isSuccess = false;

        // AdvancedWaveSoundInstanceManager
        {
            const detail::SoundInstanceConfig config = nn::atk::SoundSystem::GetSoundInstanceConfig();

            size_t requireSize = nn::util::align_up<size_t>(
                    m_InstanceManager.GetRequiredMemSize(soundCount, config), MemAlignSize );

            void* estimateEndAddr = nn::util::BytePtr( *pOutAllocatedAddr, requireSize ).Get();
            if ( nn::util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
            {
                return false;
            }

            unsigned long createdCount = m_InstanceManager.Create( *pOutAllocatedAddr, requireSize, config );
            isSuccess = createdCount == static_cast<unsigned long>(soundCount);
            NN_SDK_ASSERT( isSuccess );
            *pOutAllocatedAddr = estimateEndAddr;
        }

        /*
        // AdvancedWaveSoundLoaderManager
        {
            size_t requireSize = nn::util::align_up<size_t>(
                    driver::AdvancedWaveSoundLoaderManager::GetRequiredMemSize(soundCount),
                    MemAlignSize );

            void* estimateEndAddr = nn::util::BytePtr( *allocatedAddr, requireSize ).Get();
            if ( nn::util::ConstBytePtr(endAddr).Distance(estimateEndAddr) > 0 )
            {
                return false;
            }

            uint32_t createdCount = m_LoaderManager.Create( *allocatedAddr, requireSize );
            isSuccess = static_cast<int>(createdCount) == soundCount;
            NN_SDK_ASSERT( isSuccess,
                    "AdvancedWaveSoundLoaderManager::Create createdCount(%d) but soundCount(%d)",
                    createdCount, soundCount);
            *allocatedAddr = estimateEndAddr;
        }

        // 各 AdvancedWaveSound に WaveSoundLoaderManager をセット
        AdvancedWaveSoundInstanceManager::PriorityList& list =
            m_WaveSoundInstanceManager.GetFreeList();
        for (AdvancedWaveSoundInstanceManager::PriorityList::iterator itr = list.begin();
                itr != list.end();
                ++itr)
        {
            itr->SetLoaderManager(m_LoaderManager);
        }
        driver::SoundThread::GetInstance().RegisterSoundFrameCallback(&m_LoaderManager);
        */

        return isSuccess;
    }

    void AdvancedWaveSoundRuntime::Finalize() NN_NOEXCEPT
    {
        //driver::SoundThread::GetInstance().UnregisterSoundFrameCallback(&m_LoaderManager);

        m_InstanceManager.Destroy();
        //m_LoaderManager.Destroy();
    }

    size_t AdvancedWaveSoundRuntime::GetRequiredMemorySize(
        const SoundArchive::SoundArchivePlayerInfo& soundArchivePlayerInfo,
        size_t alignmentSize
    ) NN_NOEXCEPT
    {
        size_t size = 0;

        size += nn::util::align_up<size_t>(
            AdvancedWaveSoundInstanceManager::GetRequiredMemSize(
                soundArchivePlayerInfo.waveSoundCount, nn::atk::SoundSystem::GetSoundInstanceConfig()), alignmentSize );
        // NN_ATK_INIT_LOG("[AWSD] instance: max(%2d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.waveSoundCount, alignmentSize, size);

        /*
        size += nn::util::align_up<size_t>(
            driver::AdvancedWaveSoundLoaderManager::GetRequiredMemSize(
                soundArchivePlayerInfo.waveSoundCount), alignmentSize );
        // NN_ATK_INIT_LOG("[AWSD] loader:   max(%2d) ALIGN(%d) => %8d\n",
        //         soundArchivePlayerInfo.waveSoundCount, alignmentSize, size);
        */

        return size;
    }

    int AdvancedWaveSoundRuntime::GetFreeAdvancedWaveSoundCount() const NN_NOEXCEPT
    {
        return m_InstanceManager.GetFreeCount();
    }

    void AdvancedWaveSoundRuntime::SetupUserParam(void** startAddr, size_t adjustSize) NN_NOEXCEPT
    {
        void* curAddr = *startAddr;

        AdvancedWaveSoundInstanceManager::PriorityList& list = m_InstanceManager.GetFreeList();
        for (AdvancedWaveSoundInstanceManager::PriorityList::iterator itr = list.begin();
                itr != list.end();
                ++itr)
        {
            itr->SetUserParamBuffer(curAddr, adjustSize);

            curAddr = nn::util::BytePtr(curAddr, adjustSize).Get();
        }

        *startAddr = curAddr;
    }

    void AdvancedWaveSoundRuntime::Update() NN_NOEXCEPT
    {
        m_InstanceManager.SortPriorityList();
    }

    AdvancedWaveSound* AdvancedWaveSoundRuntime::AllocSound(
        SoundArchive::ItemId soundId,
        int priority,
        int ambientPriority,
        detail::BasicSound::AmbientInfo* ambientArgInfo,
        OutputReceiver* pOutputReceiver
    ) NN_NOEXCEPT
    {
        return SoundRuntimeUtility::AllocSound<AdvancedWaveSound>(
            &m_InstanceManager,
            soundId,
            priority,
            ambientPriority,
            ambientArgInfo,
            pOutputReceiver
        );
    }

    SoundStartable::StartResult AdvancedWaveSoundRuntime::PrepareImpl(
        const SoundArchive* pSoundArchive,
        const SoundDataManager* pSoundDataManager,
        SoundArchive::ItemId soundId,
        AdvancedWaveSound* sound,
        const SoundArchive::SoundInfo* commonInfo,
        const StartInfoReader& startInfoReader
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( sound );
        NN_SDK_ASSERT_NOT_NULL( pSoundArchive );
        NN_SDK_ASSERT_NOT_NULL( pSoundDataManager );

        // 高機能版ウェーブサウンド情報の取得
        SoundArchive::AdvancedWaveSoundInfo info;
        if ( ! pSoundArchive->detail_ReadAdvancedWaveSoundInfo( soundId, &info ) )
        {
            return SoundStartable::StartResult(
                SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId);
        }

        // 事前にロードが済んでいるウェーブサウンドファイルのアドレスを取得
        const void* pAwsdFile = NULL;
        pAwsdFile = pSoundDataManager->detail_GetFileAddress( commonInfo->fileId );
        if ( pAwsdFile == NULL )
        {
            return SoundStartable::StartResult(
                SoundStartable::StartResult::ResultCode_ErrorNotWsdLoaded);
        }

        // 事前にロードが済んでいる波形アーカイブファイルのアドレスを取得
        const void* pWarcFile = NULL;
        pWarcFile = pSoundDataManager->detail_GetFileAddressByItemId( info.waveArchiveId );
        if ( pWarcFile == NULL )
        {
            return SoundStartable::StartResult(
                SoundStartable::StartResult::ResultCode_ErrorNotWarcLoaded);
        }

        driver::AdvancedWaveSoundPlayer::PrepareParameter parameter;
        parameter.advancedWaveSoundInfo = info;
        parameter.updateType = startInfoReader.GetUpdateType();
        parameter.pAwsdFile = pAwsdFile;
        parameter.pWarcFile = pWarcFile;

        sound->Prepare( parameter );

        return SoundStartable::StartResult(SoundStartable::StartResult::ResultCode_Success);
    }

    void AdvancedWaveSoundRuntime::DumpMemory(const SoundArchive* pSoundArchive) const NN_NOEXCEPT
    {
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
        NN_SDK_ASSERT_NOT_NULL(pSoundArchive);
        SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( pSoundArchive->ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            size_t instanceManagerSize = m_InstanceManager.GetRequiredMemSize( soundArchivePlayerInfo.waveSoundCount, nn::atk::SoundSystem::GetSoundInstanceConfig() );
            //size_t loaderManagerSize   = driver::WaveSoundLoaderManager::GetRequiredMemSize( soundArchivePlayerInfo.waveSoundMax );
            NN_DETAIL_ATK_INFO("    waveSound\n"
                       "      instanceManager        %8zu bytes\n"
                       //"      loaderManager          %8zu bytes\n"
                       , instanceManagerSize/*, loaderManagerSize*/ );
        }
#else
        NN_UNUSED(pSoundArchive);
#endif
    }
}}} // namespace nn::atk::detail
