﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_DisposeCallbackManager.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_SoundDataManager.h>
#include <nn/atk/atk_WaveArchiveFileReader.h>
#include <nn/atk/atk_WaveFileReader.h>

#include <nn/atk/detail/atk_AdvancedWaveSoundPlayer.h>
#include <nn/atk/detail/atk_AdvancedWaveSoundFileReader.h>

namespace
{
    const uint32_t SoundFrameIntervalMicroSeconds =
        nn::atk::detail::driver::HardwareManager::SoundFrameIntervalUsec;
}


namespace nn { namespace atk { namespace detail { namespace driver {

    NN_DEFINE_STATIC_CONSTANT( const int AdvancedWaveSoundPlayer::TrackParam::ClipParamCountMax );
    NN_DEFINE_STATIC_CONSTANT( const int AdvancedWaveSoundPlayer::TrackParamSet::TrackParamCountMax );

    AdvancedWaveSoundPlayer::AdvancedWaveSoundPlayer() NN_NOEXCEPT
    : m_pAwsdFile(nullptr)
    , m_pWarcFile(nullptr)
    , m_CurrentTime(0)
    , m_IsPrepared(false)
    , m_IsInitialized(false)
    , m_IsRegisterPlayerCallback(false)
    {
    }

    AdvancedWaveSoundPlayer::~AdvancedWaveSoundPlayer() NN_NOEXCEPT
    {
        Finalize();
    }

    void AdvancedWaveSoundPlayer::Initialize(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
    {
        if ( !m_IsInitialized )
        {
            BasicSoundPlayer::Initialize( pOutputReceiver );
            m_IsInitialized = true;
        }

        m_IsPrepared = false;
    }

    void AdvancedWaveSoundPlayer::Finalize() NN_NOEXCEPT
    {
        TearDownPlayer();

        if ( IsActive() )
        {
            SetActiveFlag(false);
            ReleaseTracks();
        }

        if ( m_IsInitialized )
        {
            BasicSoundPlayer::Finalize();
            m_IsInitialized = false;
        }
    }

    void AdvancedWaveSoundPlayer::Start() NN_NOEXCEPT
    {
        SetStartedFlag(true);
    }

    void AdvancedWaveSoundPlayer::Stop() NN_NOEXCEPT
    {
        TearDownPlayer();
    }

    void AdvancedWaveSoundPlayer::Pause( bool isPauseEnabled ) NN_NOEXCEPT
    {
        SetPauseFlag(isPauseEnabled);
    }

    void AdvancedWaveSoundPlayer::Prepare(const PrepareParameter& parameter) NN_NOEXCEPT
    {
        if ( IsActive() )
        {
            TearDownPlayer();
        }

        m_AdvancedWaveSoundInfo = parameter.advancedWaveSoundInfo;
        m_UpdateType = parameter.updateType;
        m_pAwsdFile = parameter.pAwsdFile;
        m_pWarcFile = parameter.pWarcFile;

        SetActiveFlag(true);
        SetupPlayer();

        if  ( ! SetupTracks() )
        {
            TearDownPlayer();
            return;
        }

        m_IsPrepared = true;
    }

    void AdvancedWaveSoundPlayer::Update() NN_NOEXCEPT
    {
        // フラグチェック
        {
            if ( !IsActive() )
            {
                return;
            }
            if ( !IsStarted() )
            {
                return;
            }
        }

        if ( !IsPause() )
        {
            if ( UpdateTracks() )
            {
                TearDownPlayer();
                return;
            }
        }
    }

    void AdvancedWaveSoundPlayer::SetupPlayer() NN_NOEXCEPT
    {
        if ( !m_IsRegisterPlayerCallback )
        {
            SoundThread::GetInstance().RegisterPlayerCallback( this );
            m_IsRegisterPlayerCallback = true;
        }

        m_CurrentTime = 0;
    }

    void AdvancedWaveSoundPlayer::TearDownPlayer() NN_NOEXCEPT
    {
        SetFinishFlag(true);

        if ( m_IsRegisterPlayerCallback )
        {
            SoundThread::GetInstance().UnregisterPlayerCallback( this );
            m_IsRegisterPlayerCallback = false;
        }

        if ( IsStarted() )
        {
            SetStartedFlag(false);
        }
    }

    bool AdvancedWaveSoundPlayer::SetupTracks() NN_NOEXCEPT
    {
        AdvancedWaveSoundFileReader fileReader(m_pAwsdFile);

        if ( !fileReader.ReadWaveSoundTrackInfoSet( &m_AdvancedWaveSoundTrackInfoSet ) )
        {
            return false;
        }

        InitializeTrackParams();

        return true;
    }

    bool AdvancedWaveSoundPlayer::UpdateTracks() NN_NOEXCEPT
    {
        bool isAllTrackPlayed = true;

        int trackCount = m_AdvancedWaveSoundTrackInfoSet.waveSoundTrackCount;
        for (auto trackIndex = 0; trackIndex < trackCount; ++trackIndex)
        {
            AdvancedWaveSoundTrackInfo& trackInfo = m_AdvancedWaveSoundTrackInfoSet.waveSoundTrackInfo[trackIndex];
            TrackParam& trackParam = m_TrackParamSet.trackParam[trackIndex];

            bool isAllClipPlayed = true;
            for (auto clipIndex = 0; clipIndex < trackInfo.waveSoundClipCount; ++clipIndex)
            {
                AdvancedWaveSoundClipInfo& waveSoundClipInfo = trackInfo.waveSoundClipInfo[clipIndex];
                ClipParam& clipParam = trackParam.clipParam[clipIndex];

                // 再生開始前
                if (!clipParam.isPlayed)
                {
                    isAllClipPlayed = false;
                    if (waveSoundClipInfo.position * 1000 < m_CurrentTime)
                    {
                        StartClip(&clipParam, &waveSoundClipInfo);
                    }
                }

                // 再生開始後
                // 再生開始と同時にパラメータを反映させるため else ではない
                if (clipParam.isPlayed)
                {
                    if (clipParam.pChannel != nullptr)
                    {
                        // 再生中
                        if (clipParam.pChannel->IsActive())
                        {
                            isAllClipPlayed = false;
                            UpdateClip(&clipParam, &waveSoundClipInfo);
                        }
                        else
                        {
                            // アクティブでなくなったので終了処理
                            clipParam.pChannel = nullptr;
                        }
                    }
                }
            }

            if (!isAllClipPlayed)
            {
                isAllTrackPlayed = false;
            }
        }

        m_CurrentTime += SoundFrameIntervalMicroSeconds;

        return isAllTrackPlayed;
    }

    void AdvancedWaveSoundPlayer::ReleaseTracks() NN_NOEXCEPT
    {
        int trackCount = m_AdvancedWaveSoundTrackInfoSet.waveSoundTrackCount;
        for (auto trackIndex = 0; trackIndex < trackCount; ++trackIndex)
        {
            AdvancedWaveSoundTrackInfo& trackInfo = m_AdvancedWaveSoundTrackInfoSet.waveSoundTrackInfo[trackIndex];
            TrackParam& trackParam = m_TrackParamSet.trackParam[trackIndex];

            for (auto clipIndex = 0; clipIndex < trackInfo.waveSoundClipCount; ++clipIndex)
            {
                ClipParam& clipParam = trackParam.clipParam[clipIndex];
                ReleaseClip(&clipParam);
            }
        }
    }

    bool AdvancedWaveSoundPlayer::StartClip(ClipParam* pClipParam, AdvancedWaveSoundClipInfo* pWaveSoundClipInfo) NN_NOEXCEPT
    {
        // プライオリティの仕様は検討したほうが良いので、ひとまず変更不可かつデフォルト値固定とする
        const int priority = 64 + 64;  // プレイヤーのプライオリティ + トラックのプライオリティ

        // 波形ファイルの取得
        WaveArchiveFileReader warcReader( m_pWarcFile, false );
        const void* pWaveFile = warcReader.GetWaveFile( pWaveSoundClipInfo->waveIndex );
        if ( pWaveFile == NULL )
        {
            return false;
        }

        // 波形情報の取得
        WaveInfo waveInfo;
        {
            detail::WaveFileReader waveReader( pWaveFile, WaveType_Nwwav );
            if ( ! waveReader.ReadWaveInfo( &waveInfo ) )
            {
                return false;
            }
        }

        Channel* pChannel = Channel::AllocChannel(
            std::min( static_cast<int>( waveInfo.channelCount ), 2 ),
            priority,
            nullptr,
            this
        );

        // 鳴らせなかったらエラーでサウンド終了
        if ( pChannel == nullptr )
        {
            return false;
        }

        // エンベロープ設定
        // pChannel->SetAttack( m_WaveSoundInfo.adshr.GetAttack() );
        // pChannel->SetHold( m_WaveSoundInfo.adshr.GetHold() );
        // pChannel->SetDecay( m_WaveSoundInfo.adshr.GetDecay() );
        // pChannel->SetSustain( m_WaveSoundInfo.adshr.GetSustain() );
        // pChannel->SetRelease( m_WaveSoundInfo.adshr.GetRelease() );

        //pChannel->SetReleasePriorityFix( m_ReleasePriorityFixFlag );
        pChannel->SetUpdateType( m_UpdateType );
        pChannel->SetOutputReceiver( GetOutputReceiver() );

        // 開始オフセットを msec → サンプル数に変換
        uint32_t startOffsetSamples = 0;
        startOffsetSamples = static_cast<uint32_t>(
            static_cast<int64_t>( pWaveSoundClipInfo->startOffset ) * waveInfo.sampleRate / 1000
        );

        pChannel->Start( waveInfo, pWaveSoundClipInfo->duration, startOffsetSamples, false );

        pClipParam->pChannel = pChannel;
        pClipParam->isPlayed = true;

        return true;

    }

    void AdvancedWaveSoundPlayer::UpdateClip(ClipParam* pClipParam, AdvancedWaveSoundClipInfo* pWaveSoundClipInfo) NN_NOEXCEPT
    {
        Channel* pChannel = pClipParam->pChannel;

        if (pChannel == nullptr)
        {
            return;
        }

        // 再生残り時間 (msec) の更新
        // pChannel->GetLength() で得られる値を、 msec の単位として解釈して再生残り時間の処理を行う
        int remainingTimeMilliSeconds = pChannel->GetLength() - detail::driver::HardwareManager::SoundFrameIntervalMsec;
        if(remainingTimeMilliSeconds <= 0)
        {
            StopClip(pClipParam);
            return;
        }
        pChannel->SetLength(remainingTimeMilliSeconds);

        // volume
        float volume = 1.0f;
        volume *= GetVolume();
        volume *= static_cast<float>(pWaveSoundClipInfo->volume) / 255.0f;
        pChannel->SetUserVolume(volume);

        // pitch
        float pitchRatio = 1.0f;
        pitchRatio *= GetPitch();
        pitchRatio *= pWaveSoundClipInfo->pitch;
        pChannel->SetUserPitchRatio(pitchRatio);

        // pan
        float panBase = 0.0f;
        panBase = static_cast<float>(pWaveSoundClipInfo->pan - 64) / 64.0f;

        // TV 出力向けパラメータ (データの値と重ね合わせる)
        OutputParam tvParam = GetTvParam();
        {
            tvParam.pan += panBase;
        }
        pChannel->SetTvParam( tvParam );

        pChannel->SetPanMode( GetPanMode() );
        pChannel->SetPanCurve( GetPanCurve() );
    }

    void AdvancedWaveSoundPlayer::ReleaseClip(ClipParam* pClipParam) NN_NOEXCEPT
    {
        Channel* pChannel = pClipParam->pChannel;

        if (pChannel == nullptr)
        {
            return;
        }

        if (pChannel->IsActive())
        {
            pChannel->Release();
        }

        pClipParam->pChannel = nullptr;
    }

    void AdvancedWaveSoundPlayer::StopClip(ClipParam* pClipParam) NN_NOEXCEPT
    {
        // TODO: この関数を利用して、Clip の再生区間を超えた場合の停止を行う
        Channel* pChannel = pClipParam->pChannel;

        if (pChannel == nullptr)
        {
            return;
        }

        if (pChannel->IsActive())
        {
            pChannel->Stop();
        }

        pClipParam->pChannel = nullptr;
    }

    void AdvancedWaveSoundPlayer::InitializeTrackParams() NN_NOEXCEPT
    {
        m_TrackParamSet.isPlayed = false;

        int trackCount = m_AdvancedWaveSoundTrackInfoSet.waveSoundTrackCount;
        for ( auto trackIndex = 0; trackIndex < trackCount; ++trackIndex )
        {
            NN_SDK_ASSERT(trackIndex < AdvancedWaveSoundTrackInfoSet::AdvancedWaveSoundTrackInfoCountMax);

            TrackParam& trackParam = m_TrackParamSet.trackParam[trackIndex];
            trackParam.isPlayed = false;

            int clipCount = m_AdvancedWaveSoundTrackInfoSet.waveSoundTrackInfo[trackIndex].waveSoundClipCount;
            for ( auto clipIndex = 0; clipIndex < clipCount; ++clipIndex )
            {
                NN_SDK_ASSERT(clipIndex < AdvancedWaveSoundTrackInfo::AdvancedWaveSoundClipInfoCountMax);

                ClipParam& clipParam = trackParam.clipParam[clipIndex];
                clipParam.isPlayed = false;
                clipParam.pChannel = nullptr;
            }
        }
    }

}}}} // namespace nn::atk::detail::driver
