﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/detail/atk_AdvancedWaveSoundFileReader.h>

namespace nn { namespace atk { namespace detail {

    namespace
    {

        uint32_t FileVersionToUint32(uint16_t major, uint8_t minor, uint8_t micro) NN_NOEXCEPT
        {
            uint32_t version = static_cast<uint32_t>(
                  (major << 16)
                | (minor <<  8)
                | (micro <<  0) );

            return version;
        }

        bool IsValidBinaryFileHeader(
            const void* file,
            const char signatureString[],
            uint32_t minimumSupportedFileVersion,
            uint32_t currentSupportedFileVersion
        ) NN_NOEXCEPT
        {
            const nn::util::BinaryFileHeader& fileHeader =
                *reinterpret_cast<const nn::util::BinaryFileHeader*>( file );

            // シグネチャのチェック
            bool isValidSignature = fileHeader.signature.IsValid(signatureString);
            NN_SDK_ASSERT(isValidSignature, "invalid file signature.");

            // バイトオーダーマーク確認
            bool isValidByteOrderMark = fileHeader._byteOrderMark == util::ByteOrderMark_Normal;
            NN_SDK_ASSERT( isValidByteOrderMark,
                    "invalid file byte order mark. [expected:0x%04x][this:0x%04x]", util::ByteOrderMark_Normal, fileHeader._byteOrderMark );

            // ファイルバージョンのチェック
            bool isSupportedVersion =
                ( minimumSupportedFileVersion <= fileHeader.version.GetPacked() ) &&
                ( fileHeader.version.GetPacked() <= currentSupportedFileVersion );
            NN_SDK_ASSERT(isSupportedVersion,
                "binary file is not supported version.\n"
                "please reconvert file using new version tools.\n"
                "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
                minimumSupportedFileVersion, currentSupportedFileVersion, fileHeader.version
            );

            return isValidSignature && isValidByteOrderMark && isSupportedVersion;
        }

        bool IsValidBinaryBlock( const void* binaryblock, const char signatureString[] ) NN_NOEXCEPT
        {
            const nn::util::BinaryBlockHeader& blockHeader =
                *reinterpret_cast<const nn::util::BinaryBlockHeader*>( binaryblock );

            // シグネチャのチェック
            bool isValidSignature = blockHeader.signature.IsValid(signatureString);
            NN_SDK_ASSERT(isValidSignature, "invalid block signature.");

            return isValidSignature;
        }

        const char SignatureForFileAwsd[] = "BAWSD   ";
        const char SignatureForInfoBlockAwsd[] = "INFO";

        const uint32_t MinimumSupportedFileVersion = FileVersionToUint32( 1, 0, 0 );   // ライブラリがサポートする最低バージョン
        const uint32_t CurrentSupportedFileVersion = FileVersionToUint32( 1, 0, 0 );   // ライブラリがサポートする最新バージョン

    } // anonymous namespace

    NN_DEFINE_STATIC_CONSTANT( const int AdvancedWaveSoundTrackInfo::AdvancedWaveSoundClipInfoCountMax );
    NN_DEFINE_STATIC_CONSTANT( const int AdvancedWaveSoundTrackInfoSet::AdvancedWaveSoundTrackInfoCountMax );

    AdvancedWaveSoundFileReader::AdvancedWaveSoundFileReader( const void* pFile ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( pFile );

        if ( !IsValidBinaryFileHeader(
            pFile,
            SignatureForFileAwsd,
            MinimumSupportedFileVersion,
            CurrentSupportedFileVersion ) )
        {
            return;
        }

        const AdvancedWaveSoundFile& pAdvancedWaveSoundFile =
            *reinterpret_cast<const AdvancedWaveSoundFile*>( pFile );

        // INFOBLOCKを取得
        const AdvancedWaveSoundFile::InfoBlock* pInfoBlock = pAdvancedWaveSoundFile.GetBlock();
        NN_SDK_ASSERT( pInfoBlock );
        if ( pInfoBlock == nullptr )
        {
            return;
        }

        if ( !IsValidBinaryBlock( pInfoBlock, SignatureForInfoBlockAwsd ) )
        {
            return;
        }

        m_pInfoBlockBody = &pInfoBlock->body;
    }

    int AdvancedWaveSoundFileReader::GetWaveSoundTrackCount() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT( m_pInfoBlockBody );
        return m_pInfoBlockBody->GetTrackCount();
    }

    int AdvancedWaveSoundFileReader::GetWaveSoundClipCount( int trackIndex ) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT( m_pInfoBlockBody );
        NN_SDK_ASSERT( trackIndex < GetWaveSoundTrackCount() );

        const AdvancedWaveSoundFile::WaveSoundTrack& waveSoundTrack =
            m_pInfoBlockBody->GetWaveSoundTrack( trackIndex );
        return waveSoundTrack.GetClipCount();
    }

    bool AdvancedWaveSoundFileReader::ReadWaveSoundTrackInfoSet(
        AdvancedWaveSoundTrackInfoSet* pTrackInfoSet
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( pTrackInfoSet );

        pTrackInfoSet->waveSoundTrackCount = GetWaveSoundTrackCount();
        for (auto trackIndex = 0; trackIndex < pTrackInfoSet->waveSoundTrackCount; ++trackIndex)
        {
            AdvancedWaveSoundTrackInfo& trackInfo = pTrackInfoSet->waveSoundTrackInfo[trackIndex];

            const AdvancedWaveSoundFile::WaveSoundTrack& waveSoundTrack =
                m_pInfoBlockBody->GetWaveSoundTrack( trackIndex );

            trackInfo.waveSoundClipCount = waveSoundTrack.GetClipCount();

            for ( auto index = 0; index < trackInfo.waveSoundClipCount; ++index )
            {
                const AdvancedWaveSoundFile::WaveSoundClip& waveSoundClip =
                    waveSoundTrack.GetWaveSoundClip(index);

                AdvancedWaveSoundClipInfo& waveSoundClipInfo = trackInfo.waveSoundClipInfo[index];
                waveSoundClipInfo.waveIndex = waveSoundClip.waveIndex;
                waveSoundClipInfo.position = waveSoundClip.position;
                waveSoundClipInfo.duration = waveSoundClip.duration;
                waveSoundClipInfo.startOffset = waveSoundClip.startOffset;
                waveSoundClipInfo.pitch = waveSoundClip.pitch;
                waveSoundClipInfo.volume = waveSoundClip.volume;
                waveSoundClipInfo.pan = waveSoundClip.pan;
            }
        }

        return true;
    }

}}} // namespace nn::atk::detail
