﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_Windows.h>
#endif
#include <nn/os/os_Thread.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/atk/atk_AudioRendererPerformanceReader.h>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_MultiVoice.h>
#include <nn/atk/atk_Channel.h>
#include <nn/atk/atk_ChannelManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_SequenceSoundPlayer.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_VoiceCommand.h>
#include <nn/atk/atk_CurveLfo.h>
#include <nn/atk/detail/atk_Macro.h>

#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

// atk のミドルウェア情報です。
#define NW_MIDDLEWARE_SYMBOL(buildOption) "NintendoWare_Atk-" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MAJOR) "_" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MINOR) "_" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

#if defined(NN_SDK_BUILD_DEBUG)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Debug) );
#elif defined(NN_SDK_BUILD_DEVELOP)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Develop) );
#elif defined(NN_SDK_BUILD_RELEASE)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL(Release) );
#endif

namespace
{
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    const int VoiceCommandQueueCount = 32;
#endif

    //  CommandManager が利用するバッファサイズを取得します
    size_t GetRequiredDriverCommandManagerMemSize(const nn::atk::SoundSystem::SoundSystemParam& param ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( nn::util::is_aligned( param.soundThreadCommandBufferSize, 4 ) );
        NN_SDK_REQUIRES( nn::util::is_aligned( param.taskThreadCommandBufferSize, 4 ) );

        size_t result = 0;
        result += nn::atk::detail::DriverCommand::GetInstance().GetRequiredMemSize(param.soundThreadCommandBufferSize, param.soundThreadCommandQueueCount);
        result += nn::atk::detail::DriverCommand::GetInstanceForTaskThread().GetRequiredMemSize(param.taskThreadCommandBufferSize, param.taskThreadCommandQueueCount);
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        result += nn::atk::detail::LowLevelVoiceCommand::GetInstance().GetRequiredMemSize(param.voiceCommandBufferSize, VoiceCommandQueueCount);
        result += nn::atk::detail::VoiceReplyCommand::GetInstance().GetRequiredMemSize(param.voiceCommandBufferSize, VoiceCommandQueueCount);
#endif

        return result;
    }
    //  CommandManager が利用するバッファのうち、メモリプール管理されている必要があるバッファサイズを取得します
    size_t GetRequiredDriverCommandManagerMemSizeForMemoryPool(const nn::atk::SoundSystem::SoundSystemParam& param ) NN_NOEXCEPT
    {
        NN_UNUSED( param );

        size_t result = 0;
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        result += nn::atk::detail::LowLevelVoiceCommand::GetRequiredWaveBufferMemSize( param.voiceCommandWaveBufferPacketCount );
#endif

        return result;
    }
    // SoundSystemParam のメンバのうち、HardwareManager の初期化に必要な値を HardwareManagerParameter へ設定します
    static void SetupHardwareManagerParameter(nn::atk::detail::driver::HardwareManager::HardwareManagerParameter* pOutValue, const nn::atk::SoundSystem::SoundSystemParam& parameter) NN_NOEXCEPT
    {
        pOutValue->SetRendererSampleRate( parameter.rendererSampleRate );
        pOutValue->SetUserEffectCount( parameter.effectCount );
        pOutValue->SetProfilerEnabled( parameter.enableProfiler );
        pOutValue->SetVoiceCount( parameter.voiceCountMax );
        pOutValue->SetEffectEnabled( parameter.enableEffect );
        pOutValue->SetStereoModeEnabled( parameter.enableStereoMode );
        pOutValue->SetRecordingEnabled( parameter.enableRecordingFinalOutputs );
        pOutValue->SetRecordingAudioFrameCount( parameter.recordingAudioFrameCount );
        pOutValue->SetUserCircularBufferSinkEnabled( parameter.enableCircularBufferSink );
        pOutValue->SetSoundThreadEnabled( parameter.enableSoundThread );
        pOutValue->SetVoiceDropEnabled( parameter.enableVoiceDrop );
        pOutValue->SetCompatibleDownMixSettingEnabled( parameter.enableCompatibleDownMixSetting );
        pOutValue->SetPreviousSdkVersionLowPassFilterCompatible( parameter.enableCompatibleLowPassFilter );
        pOutValue->SetUnusedEffectChannelMutingEnabled( parameter.enableUnusedEffectChannelMuting );
        pOutValue->SetCompatibleBusVolumeEnabled( parameter.enableCompatibleBusVolume );
        pOutValue->SetUserThreadRenderingEnabled( parameter.enableUserThreadRendering );
        pOutValue->SetMemoryPoolAttachCheckEnabled( parameter.enableMemoryPoolAttachCheck );

        const int subMixTotalChannelCount = parameter.mixBufferCount == nn::atk::SoundSystem::SoundSystemParam::InvalidMixBufferCount ? parameter.subMixTotalChannelCount : parameter.mixBufferCount;
        pOutValue->SetSubMixParameter(
            parameter.enableStereoMode,
            parameter.enableEffect,
            parameter.enableSubMix,
            parameter.enableAdditionalEffectBus,
            parameter.enableAdditionalSubMix,
            parameter.enableCustomSubMix,
            parameter.subMixCount,
            subMixTotalChannelCount
        );
    }
    //  サウンドシステムが利用するバッファサイズを取得します
    size_t GetRequiredSoundSystemMemSize(const nn::atk::SoundSystem::SoundSystemParam& param, const nn::atk::detail::SoundInstanceConfig& config) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( nn::util::is_aligned(param.soundThreadStackSize, nn::os::ThreadStackAlignment) );
        NN_SDK_REQUIRES( param.soundThreadCommandQueueCount > 0 );
        NN_SDK_REQUIRES( nn::util::is_aligned(param.taskThreadStackSize, nn::os::ThreadStackAlignment) );
        NN_SDK_REQUIRES( param.taskThreadCommandQueueCount > 0 );
        NN_SDK_REQUIRES( param.rendererSampleRate == 32000 || param.rendererSampleRate == 48000 );
        NN_SDK_REQUIRES_MINMAX( param.voiceCountMax, 1, nn::atk::SoundSystem::VoiceCountMax );

        size_t result = 0;
        result += param.taskThreadStackSize;
        result += nn::atk::detail::driver::MultiVoiceManager::GetInstance().GetRequiredMemSize( param.voiceCountMax, config );

        result += param.soundThreadStackSize;
        result += nn::atk::detail::fnd::Thread::StackAlignment;

        nn::atk::detail::driver::HardwareManager::HardwareManagerParameter hardwareManagerParameter;
        SetupHardwareManagerParameter(&hardwareManagerParameter, param);

        if ( param.enableProfiler )
        {
            result += nn::audio::BufferAlignSize;
            result += nn::atk::detail::driver::HardwareManager::GetInstance().GetRequiredPerformanceFramesBufferSize( hardwareManagerParameter ) * nn::atk::detail::MaxPerformanceBufferCount;
        }

        if ( param.enableRecordingFinalOutputs )
        {
            result += nn::atk::detail::driver::HardwareManager::GetInstance().GetRequiredRecorderWorkBufferSize( hardwareManagerParameter );
        }

        result += nn::atk::detail::driver::HardwareManager::GetInstance().GetRequiredMemSize( hardwareManagerParameter );
        result += nn::os::MemoryPageSize;

        return result;
    }

    // サウンドシステムが利用するチャンネル数を取得します
    inline int GetChannelCount(int voiceCount)
    {
        return voiceCount * 2;
    }

    //  サウンドシステムが利用するバッファのうち、メモリプール管理されている必要があるバッファサイズを取得します
    size_t GetRequiredSoundSystemMemSizeForMemoryPool(const nn::atk::SoundSystem::SoundSystemParam& param, const nn::atk::detail::SoundInstanceConfig& config) NN_NOEXCEPT
    {
        size_t result = 0;
        result += nn::atk::detail::driver::ChannelManager::GetInstance().GetRequiredMemSize( GetChannelCount(param.voiceCountMax), config );
        result += nn::atk::detail::driver::HardwareManager::GetInstance().GetRequiredMemSizeForMemoryPool( param.voiceCountMax );

        return result;
    }

    //  AdditionalSubMix のバスインデックス
    const int AdditionalSubMixBusIndex = 0;

    //  AuxBus の Index を SubMix の実際の Bus の Index に変換します
    inline int ConvertAuxBusIndexToSubMixBusIndex(nn::atk::AuxBus bus)
    {
        return bus + 1;
    }

    inline int ClampBusCountMax(int busCountMax) NN_NOEXCEPT
    {
        return nn::atk::detail::fnd::Clamp<int>(busCountMax, 1, nn::atk::OutputReceiver::BusCountMax);
    }
}

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::VoiceCountMax );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundSystem::WorkMemoryAlignSize );
NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::SoundThreadIntervalUsec );

NN_DEFINE_STATIC_CONSTANT( const int32_t SoundSystem::SoundSystemParam::DefaultSoundThreadPriority );
NN_DEFINE_STATIC_CONSTANT( const int32_t SoundSystem::SoundSystemParam::DefaultTaskThreadPriority );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundSystem::SoundSystemParam::DefaultSoundThreadStackSize );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundSystem::SoundSystemParam::DefaultTaskThreadStackSize );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundSystem::SoundSystemParam::DefaultSoundThreadCommandBufferSize );
NN_DEFINE_STATIC_CONSTANT( const uint32_t SoundSystem::SoundSystemParam::DefaultTaskThreadCommandBufferSize );
NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::SoundSystemParam::DefaultSoundThreadCommandQueueCount );
NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::SoundSystemParam::DefaultTaskThreadCommandQueueCount );
NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::SoundSystemParam::DefaultUserEffectCount );
NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::SoundSystemParam::DefaultVoiceCountMax );
NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::SoundSystemParam::DefaultSoundThreadCoreNumber );
NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::SoundSystemParam::DefaultTaskThreadCoreNumber );
NN_DEFINE_STATIC_CONSTANT( const int SoundSystem::SoundSystemParam::DefaultRecordingAudioFrameCount );

bool SoundSystem::g_IsInitialized = false;
bool SoundSystem::g_IsStreamLoadWait = false;
bool SoundSystem::g_IsEnterSleep = false;
bool SoundSystem::g_IsInitializedDriverCommandManager = false;
void* SoundSystem::g_LoadThreadStackPtr = NULL;
size_t SoundSystem::g_LoadThreadStackSize = 0;
FsPriority SoundSystem::g_TaskThreadFsPriority = FsPriority_Normal;
void* SoundSystem::g_SoundThreadStackPtr = NULL;
size_t SoundSystem::g_SoundThreadStackSize = 0;
void* SoundSystem::g_PerformanceFrameBuffer = NULL;
size_t SoundSystem::g_PerformanceFrameBufferSize = 0;
bool SoundSystem::g_IsStreamOpenFailureHalt = true;
bool SoundSystem::g_IsTaskThreadEnabled = true;
bool SoundSystem::g_IsManagingMemoryPool = true;

size_t SoundSystem::g_SoundThreadCommandBufferSize = 0;
int SoundSystem::g_SoundThreadCommandQueueCount = 32;
size_t SoundSystem::g_TaskThreadCommandBufferSize = 0;
int SoundSystem::g_TaskThreadCommandQueueCount = 32;
size_t SoundSystem::g_VoiceCommandBufferSize = 0;
nn::audio::MemoryPoolType SoundSystem::g_MemoryPoolForSoundSystem;

// HardwareManager で使用するパラメータの初期値
int SoundSystem::g_RendererSampleRate = 0;
int SoundSystem::g_VoiceCountMax = SoundSystem::SoundSystemParam::DefaultVoiceCountMax;
int SoundSystem::g_UserEffectCount = SoundSystem::SoundSystemParam::DefaultUserEffectCount;
int SoundSystem::g_CustomSubMixSubMixCount = 0;
int SoundSystem::g_CustomSubMixMixBufferCount = 0;
bool SoundSystem::g_IsProfilerEnabled = false;
bool SoundSystem::g_IsDetailSoundThreadProfilerEnabled = false;
bool SoundSystem::g_IsAdditionalEffectBusEnabled = false;
bool SoundSystem::g_IsAdditionalSubMixEnabled = false;
bool SoundSystem::g_IsEffectEnabled = false;
bool SoundSystem::g_IsRecordingEnabled = false;
bool SoundSystem::g_IsCircularBufferSinkEnabled = false;
bool SoundSystem::g_IsCircularBufferSinkWarningDisplayed = false;
bool SoundSystem::g_IsSubMixEnabled = true;
bool SoundSystem::g_IsPresetSubMixEnabled = true;
bool SoundSystem::g_IsStereoModeEnabled = true;
bool SoundSystem::g_IsSoundThreadEnabled = true;
bool SoundSystem::g_IsVoiceDropEnabled = false;
bool SoundSystem::g_IsPreviousSdkVersionLowPassFilterCompatible = false;
bool SoundSystem::g_IsUnusedEffectChannelMutingEnabled = false;
bool SoundSystem::g_IsUserThreadRenderingEnabled = false;
bool SoundSystem::g_IsCustomSubMixEnabled = false;
bool SoundSystem::g_IsMemoryPoolAttachCheckEnabled = false;
bool SoundSystem::g_IsBusMixVolumeEnabled = false;
bool SoundSystem::g_IsVolumeThroughModeEnabled = false;
int SoundSystem::g_BusCountMax = detail::DefaultBusCount;

WarningCallback SoundSystem::g_WarningCallback = nullptr;

SoundSystem::SoundSystemParam::SoundSystemParam() NN_NOEXCEPT
: soundThreadPriority( DefaultSoundThreadPriority )
, soundThreadStackSize( DefaultSoundThreadStackSize )
, soundThreadCommandBufferSize( DefaultSoundThreadCommandBufferSize )
, soundThreadCommandQueueCount( DefaultSoundThreadCommandQueueCount )
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
, voiceCommandBufferSize( 512 * 1024 )
#endif
, taskThreadPriority( DefaultTaskThreadPriority )
, taskThreadStackSize( DefaultTaskThreadStackSize )
, taskThreadCommandBufferSize( DefaultTaskThreadCommandBufferSize )
, taskThreadCommandQueueCount( DefaultTaskThreadCommandQueueCount )
, taskThreadFsPriority( DefaultTaskThreadFsPriority )
, rendererSampleRate( DefaultRendererSampleRate )
, effectCount( DefaultUserEffectCount )
, voiceCountMax( DefaultVoiceCountMax )
, voiceCommandWaveBufferPacketCount( 512 )
, enableProfiler( false )
, enableDetailSoundThreadProfile( false )
, enableRecordingFinalOutputs( false )
, enableCircularBufferSink(false)
, recordingAudioFrameCount( DefaultRecordingAudioFrameCount )
, soundThreadCoreNumber( DefaultSoundThreadCoreNumber )
, taskThreadCoreNumber( DefaultTaskThreadCoreNumber )
, enableAdditionalEffectBus( false )
, enableAdditionalSubMix( false )
, enableTaskThread( true )
, enableSoundThread( true )
, enableMemoryPoolManagement( true )
, enableCircularBufferSinkBufferManagement( true )
, enableEffect( true )
, enableSubMix( true )
, enableStereoMode( false )
, enableVoiceDrop( true )
, enableCompatibleDownMixSetting( false )
, enableCompatibleLowPassFilter( false )
, enableUnusedEffectChannelMuting( false )
, enableCompatibleBusVolume( false )
, enableUserThreadRendering( false )
, enableCustomSubMix( false )
, subMixCount( 0 )
, subMixTotalChannelCount( 0 )
, mixBufferCount( InvalidMixBufferCount )
, busCountMax( DefaultBusCountMax )
, enableMemoryPoolAttachCheck( false )
, enableBusMixVolume( false )
, enableVolumeThroughMode( false )
{
}

size_t SoundSystem::GetRequiredMemSize( const SoundSystemParam& param ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX( param.busCountMax, 1, OutputReceiver::BusCountMax );
    size_t result = 0;
    result += GetRequiredSoundSystemMemSize( param, GetSoundInstanceConfig(param) );
    result += GetRequiredDriverCommandManagerMemSize( param );

    if( param.enableCircularBufferSink && param.enableCircularBufferSinkBufferManagement )
    {
        result += GetRequiredMemSizeForCircularBufferSink(param);
    }

    if( param.enableMemoryPoolManagement )
    {
        const size_t memoryPoolMemSize = GetRequiredMemSizeForMemoryPool( param );

        //  メモリプール管理するためにサイズを整え, アラインマージンをとります
        result += nn::util::align_up( memoryPoolMemSize, nn::audio::MemoryPoolType::SizeGranularity );
        result += nn::audio::MemoryPoolType::AddressAlignment;
    }

    return result;
}

size_t SoundSystem::GetRequiredMemSizeForMemoryPool( const SoundSystemParam& param ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX( param.busCountMax, 1, OutputReceiver::BusCountMax );
    size_t result = 0;

    result += GetRequiredSoundSystemMemSizeForMemoryPool( param, GetSoundInstanceConfig(param) );
    result += GetRequiredDriverCommandManagerMemSizeForMemoryPool( param );

    return result;
}

size_t SoundSystem::GetRequiredMemSizeForCircularBufferSink( const SoundSystemParam& param ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX( param.busCountMax, 1, OutputReceiver::BusCountMax );
    // 内部でメモリプールアタッチ用のバッファサイズ調整はされているため、ここでは不要
    nn::atk::detail::driver::HardwareManager::HardwareManagerParameter hardwareManagerParameter;
    SetupHardwareManagerParameter(&hardwareManagerParameter, param);
    return nn::atk::detail::driver::HardwareManager::GetInstance().GetRequiredCircularBufferSinkWithMemoryPoolBufferSize( hardwareManagerParameter );

}

bool SoundSystem::detail_InitializeSoundSystem(
    Result* pOutResult,
    const SoundSystemParam& param,
    InitializeBufferSet& bufferSet ) NN_NOEXCEPT
{
    NN_USING_MIDDLEWARE(g_MiddlewareInfo);

    NN_SDK_REQUIRES( nn::util::is_aligned( param.soundThreadStackSize, nn::os::ThreadStackAlignment ) );
    NN_SDK_REQUIRES( param.soundThreadCommandQueueCount > 0 );
    NN_SDK_REQUIRES( nn::util::is_aligned( param.taskThreadStackSize, nn::os::ThreadStackAlignment ) );
    NN_SDK_REQUIRES( param.taskThreadCommandQueueCount > 0 );
    NN_SDK_REQUIRES( param.rendererSampleRate == 32000 || param.rendererSampleRate == 48000 );
    NN_SDK_REQUIRES_MINMAX( param.voiceCountMax, 1, SoundSystem::VoiceCountMax );
    NN_SDK_REQUIRES_MINMAX( param.busCountMax, 1, OutputReceiver::BusCountMax );
    NN_SDK_REQUIRES_ALIGNED( bufferSet.workMem, WorkMemoryAlignSize );
    NN_SDK_REQUIRES_ALIGNED( bufferSet.memoryPoolMem, nn::audio::BufferAlignSize );
    nn::atk::detail::SoundInstanceConfig config = nn::atk::SoundSystem::GetSoundInstanceConfig(param);
    NN_SDK_ASSERT( bufferSet.workMemSize >= GetRequiredSoundSystemMemSize( param, config ) + GetRequiredDriverCommandManagerMemSize( param ),
            "workMemSize(%X) >= RequiredMemSize(%X)",
            bufferSet.workMemSize, GetRequiredSoundSystemMemSize( param, config ) + GetRequiredDriverCommandManagerMemSize( param ) );
    NN_SDK_ASSERT( bufferSet.memoryPoolMemSize >= GetRequiredSoundSystemMemSizeForMemoryPool( param, config ) + GetRequiredDriverCommandManagerMemSizeForMemoryPool( param ),
            "memoryPoolMemSize(%X) >= RequiredMemSize(%X)",
            bufferSet.memoryPoolMemSize, GetRequiredSoundSystemMemSizeForMemoryPool( param, config ) + GetRequiredDriverCommandManagerMemSizeForMemoryPool( param ) );

    // 多重初期化チェック
    if ( g_IsInitialized )
    {
        if ( pOutResult != nullptr )
        {
            *pOutResult = nn::ResultSuccess();
        }
        NN_DETAIL_ATK_INFO("SoundSystem is already initialized\n");
        return true;
    }

    bool result;
    NN_UNUSED(result);

    uintptr_t ptr = bufferSet.workMem;
    uintptr_t ptrMemoryPool = bufferSet.memoryPoolMem;
    uintptr_t ptrCircularBufferSink = bufferSet.circularBufferSinkMem;

    // コマンドバッファの初期化
    if ( g_IsInitializedDriverCommandManager == false )
    {
        size_t workMemSizeForDriverCommandManager =
            GetRequiredDriverCommandManagerMemSize( param );
        size_t workMemSizeForDriverCommandManagerForMemoryPool =
            GetRequiredDriverCommandManagerMemSizeForMemoryPool( param );

        detail_InitializeDriverCommandManager(
                param,
                ptr,
                workMemSizeForDriverCommandManager,
                ptrMemoryPool,
                workMemSizeForDriverCommandManagerForMemoryPool
                );
        ptr += workMemSizeForDriverCommandManager;
        ptrMemoryPool += workMemSizeForDriverCommandManagerForMemoryPool;
    }

    ptr = nn::util::align_up( ptr, nn::os::MemoryPageSize );
    // SDK 関連初期化
    {
        nn::atk::detail::driver::HardwareManager::HardwareManagerParameter hardwareManagerParameter;
        SetupHardwareManagerParameter(&hardwareManagerParameter, param);

        g_VoiceCountMax = param.voiceCountMax; // TODO: Set/GetVoiceCountMax の廃止が完了したら、初期化は param.voiceCountMax で行うように修正する

        {
            void* workBuffer = reinterpret_cast<void*>(ptr);
            void* memoryPoolBuffer = reinterpret_cast<void*>(ptrMemoryPool);
            void* circularBufferSinkBuffer = nullptr;
            size_t workBufferSize = detail::driver::HardwareManager::GetInstance().GetRequiredMemSize(hardwareManagerParameter);
            size_t memoryPoolBufferSize = detail::driver::HardwareManager::GetInstance().GetRequiredMemSizeForMemoryPool(g_VoiceCountMax);
            size_t circularBufferSinkBufferSize = 0u;

            if( param.enableRecordingFinalOutputs )
            {
                workBufferSize += detail::driver::HardwareManager::GetInstance().GetRequiredRecorderWorkBufferSize( hardwareManagerParameter );
            }

            if ( param.enableCircularBufferSink )
            {
                NN_SDK_ASSERT(ptrCircularBufferSink != NULL);
                circularBufferSinkBuffer = reinterpret_cast<void*>(ptrCircularBufferSink);
                circularBufferSinkBufferSize = detail::driver::HardwareManager::GetInstance().GetRequiredCircularBufferSinkWithMemoryPoolBufferSize( hardwareManagerParameter );
            }

            ptr += workBufferSize;
            ptrMemoryPool += memoryPoolBufferSize;
            ptrCircularBufferSink += circularBufferSinkBufferSize;
            nn::Result hardwareManagerResult = detail::driver::HardwareManager::GetInstance().Initialize(
                workBuffer,
                workBufferSize,
                memoryPoolBuffer,
                memoryPoolBufferSize,
                circularBufferSinkBuffer,
                circularBufferSinkBufferSize,
                hardwareManagerParameter );
            result = hardwareManagerResult.IsSuccess();
            if (pOutResult != nullptr)
            {
                *pOutResult = hardwareManagerResult;
            }
        }

        if(!result)
        {
            NN_DETAIL_ATK_INFO("Failed to initialize SoundSystem\n");

            if ( g_IsInitializedDriverCommandManager )
            {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
                detail::LowLevelVoiceCommand::GetInstance().Finalize();
                detail::VoiceReplyCommand::GetInstance().Finalize();
                g_VoiceCommandBufferSize = 0;
#endif
                detail::DriverCommand::GetInstance().Finalize();
                detail::DriverCommand::GetInstanceForTaskThread().Finalize();
                g_SoundThreadCommandBufferSize = 0;
                g_TaskThreadCommandBufferSize = 0;
            }

            return false;
        }

        if ( param.enableProfiler )
        {
            ptr = nn::util::align_up(ptr, nn::audio::BufferAlignSize);
            uintptr_t performanceFrameBuffer = ptr;
            size_t performanceFrameBufferSize = detail::driver::HardwareManager::GetInstance().GetRequiredPerformanceFramesBufferSize( hardwareManagerParameter ) * detail::MaxPerformanceBufferCount;
            ptr += performanceFrameBufferSize;
            g_PerformanceFrameBuffer = reinterpret_cast<void*>(performanceFrameBuffer);
            g_PerformanceFrameBufferSize = performanceFrameBufferSize;
        }
        else
        {
            g_PerformanceFrameBuffer = nullptr;
            g_PerformanceFrameBufferSize = 0;
        }

        const int subMixTotalChannelCount = param.mixBufferCount == nn::atk::SoundSystem::SoundSystemParam::InvalidMixBufferCount ? param.subMixTotalChannelCount : param.mixBufferCount;
        g_RendererSampleRate                          = param.rendererSampleRate;
        g_UserEffectCount                             = param.effectCount;
        g_CustomSubMixSubMixCount                     = param.subMixCount;
        g_CustomSubMixMixBufferCount                  = subMixTotalChannelCount;
        g_IsProfilerEnabled                           = param.enableProfiler;
        g_IsDetailSoundThreadProfilerEnabled          = param.enableDetailSoundThreadProfile;
        g_IsAdditionalEffectBusEnabled                = param.enableAdditionalEffectBus;
        g_IsAdditionalSubMixEnabled                   = param.enableAdditionalSubMix;
        g_IsEffectEnabled                             = param.enableEffect;
        g_IsRecordingEnabled                          = param.enableRecordingFinalOutputs;
        g_IsCircularBufferSinkEnabled                 = param.enableCircularBufferSink;
        g_IsSubMixEnabled                             = param.enableSubMix;
        g_IsPresetSubMixEnabled                       = hardwareManagerParameter.IsPresetSubMixEnabled();
        g_IsStereoModeEnabled                         = param.enableStereoMode;
        g_IsSoundThreadEnabled                        = param.enableSoundThread;
        g_IsVoiceDropEnabled                          = param.enableVoiceDrop;
        g_IsPreviousSdkVersionLowPassFilterCompatible = param.enableCompatibleLowPassFilter;
        g_IsUserThreadRenderingEnabled                = param.enableUserThreadRendering;
        g_IsUnusedEffectChannelMutingEnabled          = param.enableUnusedEffectChannelMuting;
        g_IsCustomSubMixEnabled                       = param.enableCustomSubMix;
        g_IsMemoryPoolAttachCheckEnabled              = param.enableMemoryPoolAttachCheck;
        g_IsBusMixVolumeEnabled                       = param.enableBusMixVolume;
        g_IsVolumeThroughModeEnabled                  = param.enableVolumeThroughMode;
        g_BusCountMax                                 = ClampBusCountMax(param.busCountMax);
    }

    ptr = nn::util::align_up( ptr, detail::fnd::Thread::StackAlignment );
    // nn::atk サウンドスレッドの準備
    {
        uintptr_t soundThreadStackPtr = static_cast<uintptr_t>(NULL);
        soundThreadStackPtr = ptr;
        ptr += param.soundThreadStackSize;
        g_SoundThreadStackPtr = reinterpret_cast<void*>(soundThreadStackPtr);
        g_SoundThreadStackSize = param.soundThreadStackSize;
    }
    detail::driver::SoundThread::GetInstance().Initialize( g_PerformanceFrameBuffer, g_PerformanceFrameBufferSize, g_IsProfilerEnabled, g_IsDetailSoundThreadProfilerEnabled, g_IsUserThreadRenderingEnabled );

    // nn::atk サウンドデータロードスレッドの準備
    {
        uintptr_t loadThreadStackPtr = ptr;
        ptr += param.taskThreadStackSize;
        detail::TaskManager::GetInstance().Initialize( param.enableProfiler );
        // サウンドスレッドのスタックのサイズを nn::atk::detail::fnd::Thread::StackAlignment の倍数で指定する必要アリ
        g_LoadThreadStackPtr = reinterpret_cast<void*>(loadThreadStackPtr);
        g_LoadThreadStackSize = param.taskThreadStackSize;
        g_TaskThreadFsPriority = param.taskThreadFsPriority;
    }

    // MultiVoiceManager初期化
    {
        uintptr_t voiceWork = ptr;
        ptr += detail::driver::MultiVoiceManager::GetInstance().GetRequiredMemSize( g_VoiceCountMax, config );
        detail::driver::MultiVoiceManager::GetInstance().Initialize(
                reinterpret_cast<void*>( voiceWork ),
                detail::driver::MultiVoiceManager::GetInstance().GetRequiredMemSize( g_VoiceCountMax, config ),
                config);
    }

    // ChannelManager初期化
    {
        void* mem = reinterpret_cast<void*>( ptrMemoryPool );
        const size_t memSize = detail::driver::ChannelManager::GetInstance().GetRequiredMemSize( GetChannelCount(g_VoiceCountMax), config );
        ptrMemoryPool += memSize;

        detail::driver::ChannelManager::GetInstance().Initialize( mem, memSize, GetChannelCount(g_VoiceCountMax), config );
    }

    NN_SDK_ASSERT( ptr <= bufferSet.workMem + bufferSet.workMemSize );
    NN_SDK_ASSERT( ptrMemoryPool <= bufferSet.memoryPoolMem + bufferSet.memoryPoolMemSize );

    if ( param.enableCircularBufferSink )
    {
        NN_SDK_ASSERT( ptrCircularBufferSink <= bufferSet.circularBufferSinkMem + bufferSet.circularBufferSinkMemSize );
    }

    // SequenceSoundPlayer初期化
    detail::driver::SequenceSoundPlayer::InitSequenceSoundPlayer();

    uint16_t attribute = 0;

    if ( param.enableTaskThread )
    {
        // データロードスレッド起動
        result = detail::TaskThread::GetInstance().Create(
            param.taskThreadPriority,
            g_LoadThreadStackPtr,
            g_LoadThreadStackSize,
            param.taskThreadCoreNumber,
            attribute,
            g_TaskThreadFsPriority
            );
        NN_SDK_ASSERT( result ); // TODO: 初期化処理を巻き戻して false を返す
        g_IsTaskThreadEnabled = true;
    }
    else
    {
        g_IsTaskThreadEnabled = false;
    }

    // 初期化で積まれた DriverCommand をフラッシュ
    detail::DriverCommand::GetInstance().FlushCommand( false );

    // サウンドスレッド起動
    if ( g_IsSoundThreadEnabled )
    {
        result = detail::driver::SoundThread::GetInstance().CreateSoundThread(
            param.soundThreadPriority,
            g_SoundThreadStackPtr,
            g_SoundThreadStackSize,
            param.soundThreadCoreNumber,
            attribute );
        NN_SDK_ASSERT( result ); // TODO: 初期化処理を巻き戻して false を返す
    }

    // SEQ モジュレーションカーブテーブルの初期化
    detail::CurveLfo::InitializeCurveTable();

    g_IsInitialized = true;
    return true;
} // NOLINT(impl/function_size)

void SoundSystem::detail_InitializeDriverCommandManager(
    const SoundSystemParam& param,
    uintptr_t workMem,
    size_t workMemSize,
    uintptr_t memoryPoolMem,
    size_t memoryPoolMemSize ) NN_NOEXCEPT
{
    if ( g_IsInitializedDriverCommandManager )
    {
        return;
    }

    NN_UNUSED( workMemSize );
    NN_UNUSED( memoryPoolMem );
    NN_UNUSED( memoryPoolMemSize );
    NN_SDK_REQUIRES_ALIGNED( memoryPoolMem, nn::audio::BufferAlignSize );
    NN_SDK_ASSERT( workMemSize >= GetRequiredDriverCommandManagerMemSize( param ),
            "workMemSize(%X) >= RequiredMemSize(%X)",
            workMemSize, GetRequiredDriverCommandManagerMemSize( param ) );
    NN_SDK_ASSERT( memoryPoolMemSize >= GetRequiredDriverCommandManagerMemSizeForMemoryPool( param ),
            "memoryPoolMemSize(%X) >= RequiredMemSize(%X)",
            memoryPoolMemSize, GetRequiredDriverCommandManagerMemSizeForMemoryPool( param ) );

    uintptr_t ptr = workMem;

    // コマンドバッファ初期化
    uintptr_t driverCommandBufferPtr = ptr;
    const auto driverCommandBufferSize = detail::DriverCommand::GetInstance().GetRequiredMemSize(param.soundThreadCommandBufferSize, param.soundThreadCommandQueueCount);
    ptr += driverCommandBufferSize;

    uintptr_t driverCommandForTaskThreadBufferPtr = ptr;
    const auto driverCommandForTaskThreadBufferSize = detail::DriverCommand::GetInstanceForTaskThread().GetRequiredMemSize(param.taskThreadCommandBufferSize, param.taskThreadCommandQueueCount);
    ptr += driverCommandForTaskThreadBufferSize;

    detail::DriverCommand::GetInstance().Initialize(
        reinterpret_cast<void*>( driverCommandBufferPtr ),
        driverCommandBufferSize,
        param.soundThreadCommandBufferSize,
        param.soundThreadCommandQueueCount
    );
    detail::DriverCommand::GetInstanceForTaskThread().Initialize(
        reinterpret_cast<void*>( driverCommandForTaskThreadBufferPtr ),
        driverCommandForTaskThreadBufferSize,
        param.taskThreadCommandBufferSize,
        param.taskThreadCommandQueueCount
    );
    g_SoundThreadCommandBufferSize = param.soundThreadCommandBufferSize;
    g_SoundThreadCommandQueueCount = param.soundThreadCommandQueueCount;
    g_TaskThreadCommandBufferSize = param.taskThreadCommandBufferSize;
    g_TaskThreadCommandQueueCount = param.taskThreadCommandQueueCount;

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    uintptr_t nnVoiceCommandBufferPtr = ptr;
    const size_t nnVoiceCommandBufferSize = detail::LowLevelVoiceCommand::GetInstance().GetRequiredMemSize(param.voiceCommandBufferSize, VoiceCommandQueueCount);
    ptr += nnVoiceCommandBufferSize;

    uintptr_t voiceReplyCommandBufferPtr = ptr;
    const size_t voiceReplyCommandBufferSize = detail::VoiceReplyCommand::GetInstance().GetRequiredMemSize(param.voiceCommandBufferSize, VoiceCommandQueueCount);
    ptr += voiceReplyCommandBufferSize;

    // VoiceCommandManager（LowLevelVoiceCommandなど） の数を変更した時には
    // VoiceCommandManagerCountMax の値も変更する必要があります。
    detail::LowLevelVoiceCommand::GetInstance().Initialize(
        reinterpret_cast<void*>( nnVoiceCommandBufferPtr ),
        nnVoiceCommandBufferSize,
        param.voiceCommandBufferSize,
        VoiceCommandQueueCount,
        reinterpret_cast<void*>( memoryPoolMem ),
        memoryPoolMemSize,
        param.voiceCommandWaveBufferPacketCount
    );
    detail::VoiceReplyCommand::GetInstance().Initialize(
        reinterpret_cast<void*>( voiceReplyCommandBufferPtr ),
        voiceReplyCommandBufferSize,
        param.voiceCommandBufferSize,
        VoiceCommandQueueCount
    );
    g_VoiceCommandBufferSize = param.voiceCommandBufferSize;
#endif

    g_IsInitializedDriverCommandManager = true;
}

bool SoundSystem::Initialize(
    const SoundSystemParam& param,
    uintptr_t workMem,
    size_t workMemSize ) NN_NOEXCEPT
{
    return Initialize( nullptr, param, workMem, workMemSize );
}


bool SoundSystem::Initialize(
    Result* pOutResult,
    const SoundSystemParam& param,
    uintptr_t workMem,
    size_t workMemSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( workMemSize >= GetRequiredMemSize( param ),
            "workMemSize(%X) >= RequiredMemSize(%X)",
            workMemSize, GetRequiredMemSize( param ) );
    NN_SDK_ASSERT( param.enableMemoryPoolManagement, "SoundSystem doesn't manage memory as MemoryPool." );


    InitializeBufferSet bufferSet;
    bufferSet.workMem = workMem;
    bufferSet.workMemSize = workMemSize;
    bufferSet.memoryPoolMem = NULL;
    bufferSet.memoryPoolMemSize = 0u;
    bufferSet.circularBufferSinkMem = NULL;
    bufferSet.circularBufferSinkMemSize = 0u;

    InitializeBufferSet setupBufferSet;
    SetupInitializeBufferSet(&setupBufferSet, param, bufferSet );

    const bool result = detail_InitializeSoundSystem(
        pOutResult,
        param,
        setupBufferSet );

    if( result )
    {
        nn::atk::detail::driver::HardwareManager::GetInstance().AttachMemoryPool( &g_MemoryPoolForSoundSystem, reinterpret_cast<void*>(setupBufferSet.memoryPoolMem ), setupBufferSet.memoryPoolMemSize, true );
    }

    return result;
}

bool SoundSystem::Initialize(
    const SoundSystemParam& param,
    uintptr_t workMem,
    size_t workMemSize,
    uintptr_t memoryPoolMem,
    size_t memoryPoolMemSize ) NN_NOEXCEPT
{
    return Initialize( nullptr, param, workMem, workMemSize, memoryPoolMem, memoryPoolMemSize );
}

bool SoundSystem::Initialize(
    Result* pOutResult,
    const SoundSystemParam& param,
    uintptr_t workMem,
    size_t workMemSize,
    uintptr_t memoryPoolMem,
    size_t memoryPoolMemSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( !param.enableMemoryPoolManagement, "SoundSystem manages work memory as MemoryPool." );

    InitializeBufferSet bufferSet;
    bufferSet.workMem = workMem;
    bufferSet.workMemSize = workMemSize;
    bufferSet.memoryPoolMem = memoryPoolMem;
    bufferSet.memoryPoolMemSize = memoryPoolMemSize;
    bufferSet.circularBufferSinkMem = NULL;
    bufferSet.circularBufferSinkMemSize = 0u;

    InitializeBufferSet setupBufferSet;
    SetupInitializeBufferSet(&setupBufferSet, param, bufferSet );

    const bool result = detail_InitializeSoundSystem(
        pOutResult,
        param,
        setupBufferSet );

    return result;
}

bool SoundSystem::Initialize(
    const SoundSystemParam& param,
    InitializeBufferSet& bufferSet
    ) NN_NOEXCEPT
{
    return Initialize( nullptr, param, bufferSet );
}

bool SoundSystem::Initialize(
    Result* pOutResult,
    const SoundSystemParam& param,
    InitializeBufferSet& bufferSet
    ) NN_NOEXCEPT
{
    InitializeBufferSet setupBufferSet;
    SetupInitializeBufferSet(&setupBufferSet, param, bufferSet );

    const bool result = detail_InitializeSoundSystem(
        pOutResult,
        param,
        setupBufferSet );

    if( result && param.enableMemoryPoolManagement )
    {
        nn::atk::detail::driver::HardwareManager::GetInstance().AttachMemoryPool( &g_MemoryPoolForSoundSystem, reinterpret_cast<void*>(setupBufferSet.memoryPoolMem ), setupBufferSet.memoryPoolMemSize, true );
    }

    return result;
}

void SoundSystem::Finalize() NN_NOEXCEPT
{
    if ( ! g_IsInitialized )
    {
        return;
    }

    if( g_IsPresetSubMixEnabled )
    {
        // 積まれたコマンドを消化しきってから終了処理します
        const auto tag = detail::DriverCommand::GetInstance().FlushCommand( true );
        if( g_IsUserThreadRenderingEnabled )
        {
            // ユーザスレッドの場合、SoundThread 更新がされない場合があるため手動で呼び出します
            // FrameProcess() は m_CriticalSection で排他制御されるため、ユーザスレッドと同時呼び出しになっても大丈夫です。
            detail::driver::SoundThread::GetInstance().FrameProcess();
        }
        detail::DriverCommand::GetInstance().WaitCommandReply( tag );
    }

    // データロードスレッド終了
    if ( g_IsTaskThreadEnabled )
    {
        detail::TaskManager::GetInstance().CancelAllTask();
        detail::TaskThread::GetInstance().Destroy();
        detail::TaskManager::GetInstance().Finalize();
    }
    else
    {
        g_IsTaskThreadEnabled = true;
    }

    if ( g_IsSoundThreadEnabled )
    {
        detail::driver::SoundThread::GetInstance().Destroy();
    }
    else
    {
        g_IsSoundThreadEnabled = true;
    }

    // チャンネルマネージャー破棄
    detail::driver::ChannelManager::GetInstance().Finalize();

    //  メモリプールのデタッチ
    if( g_IsManagingMemoryPool )
    {
        detail::driver::HardwareManager::GetInstance().DetachMemoryPool( &g_MemoryPoolForSoundSystem, false );
    }
    else
    {
        g_IsManagingMemoryPool = true;
    }

    detail::driver::MultiVoiceManager::GetInstance().Finalize();
    detail::driver::HardwareManager::GetInstance().Finalize();

    // サウンドスレッド終了
    detail::driver::SoundThread::GetInstance().Finalize();

    if ( g_IsInitializedDriverCommandManager )
    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        detail::LowLevelVoiceCommand::GetInstance().Finalize();
        detail::VoiceReplyCommand::GetInstance().Finalize();
        g_VoiceCommandBufferSize = 0;
#endif

        detail::DriverCommand::GetInstance().Finalize();
        detail::DriverCommand::GetInstanceForTaskThread().Finalize();
        g_SoundThreadCommandBufferSize = 0;
        g_TaskThreadCommandBufferSize = 0;

        g_IsInitializedDriverCommandManager = false;
    }

    g_IsCircularBufferSinkEnabled = false;
    g_IsCircularBufferSinkWarningDisplayed = false;
    g_IsInitialized = false;
}

void SoundSystem::SetSoundThreadBeginUserCallback(SoundThreadUserCallback callback, uintptr_t arg) NN_NOEXCEPT
{
    detail::driver::SoundThread::GetInstance().RegisterThreadBeginUserCallback(callback, arg);
}

void SoundSystem::ClearSoundThreadBeginUserCallback() NN_NOEXCEPT
{
    detail::driver::SoundThread::GetInstance().ClearThreadBeginUserCallback();
}

void SoundSystem::SetSoundThreadEndUserCallback(SoundThreadUserCallback callback, uintptr_t arg) NN_NOEXCEPT
{
    detail::driver::SoundThread::GetInstance().RegisterThreadEndUserCallback(callback, arg);
}

void SoundSystem::ClearSoundThreadEndUserCallback() NN_NOEXCEPT
{
    detail::driver::SoundThread::GetInstance().ClearThreadEndUserCallback();
}

bool SoundSystem::IsInitialized() NN_NOEXCEPT
{
    return g_IsInitialized;
}

void SoundSystem::SuspendAudioRenderer(nn::TimeSpan fadeTimes) NN_NOEXCEPT
{
    NN_UNUSED(fadeTimes);
    detail::driver::HardwareManager::GetInstance().SuspendAudioRenderer();
}

void SoundSystem::ResumeAudioRenderer(nn::TimeSpan fadeTimes) NN_NOEXCEPT
{
    NN_UNUSED(fadeTimes);
    detail::driver::HardwareManager::GetInstance().ResumeAudioRenderer();
}

void SoundSystem::ExecuteRendering() NN_NOEXCEPT
{
    detail::driver::HardwareManager::GetInstance().ExecuteAudioRendererRendering();
}

int SoundSystem::GetAudioRendererRenderingTimeLimit() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::atk::SoundSystem::IsInitialized());

    return detail::driver::HardwareManager::GetInstance().GetAudioRendererRenderingTimeLimit();
}

void SoundSystem::SetAudioRendererRenderingTimeLimit(int limitPercent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::atk::SoundSystem::IsInitialized());
    NN_SDK_REQUIRES_GREATER(limitPercent, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(limitPercent, 100);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::driver::HardwareManager::GetInstance().SetAudioRendererRenderingTimeLimit(limitPercent));
}

void SoundSystem::AttachMemoryPool(nn::audio::MemoryPoolType* pMemoryPool, void* address, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(nn::atk::SoundSystem::IsInitialized());

    detail::driver::HardwareManager::GetInstance().AttachMemoryPool(pMemoryPool, address, size, true);
}

void SoundSystem::DetachMemoryPool(nn::audio::MemoryPoolType* pMemoryPool) NN_NOEXCEPT
{
    NN_SDK_ASSERT(nn::atk::SoundSystem::IsInitialized());

    detail::driver::HardwareManager::GetInstance().DetachMemoryPool(pMemoryPool, true);
}

void SoundSystem::DumpMemory() NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP) // Dump 用途
    if ( !IsInitialized() )
    {
        NN_DETAIL_ATK_INFO("SoundSystem is not Initialized\n");
        return;
    }

    NN_DETAIL_ATK_INFO("  soundSystem\n"
               "    hardwareManager(Renderer)      %8zu bytes\n"
               "    hardwareManager(LowLevelVoice) %8zu bytes\n"
               "    recorderBuffer                 %8zu bytes\n"
               "    userCircularBufferSinkBuffer   %8zu bytes\n"
               "    multiVoiceManager              %8zu bytes\n"
               "    channelManager                 %8zu bytes\n"
               "    soundThreadStack               %8u bytes\n"
               "    taskThreadStack                %8u bytes\n"
               "    performanceBuffer              %8u bytes\n"
               , GetAudioRendererBufferSize()
               , GetLowLevelVoiceAllocatorBufferSize()
               , GetRecorderBufferSize()
               , GetUserCircularBufferSinkBufferSize()
               , GetMultiVoiceManagerBufferSize()
               , GetChannelManagerBufferSize()
               , g_SoundThreadStackSize
               , g_LoadThreadStackSize
               , g_PerformanceFrameBufferSize );
    size_t totalSize = 0;
    totalSize += GetAudioRendererBufferSize();
    totalSize += GetLowLevelVoiceAllocatorBufferSize();
    totalSize += GetRecorderBufferSize();
    totalSize += GetUserCircularBufferSinkBufferSize();
    totalSize += GetMultiVoiceManagerBufferSize();
    totalSize += GetChannelManagerBufferSize();
    totalSize += g_SoundThreadStackSize;
    totalSize += g_LoadThreadStackSize;
    totalSize += g_PerformanceFrameBufferSize;

    if ( g_IsInitializedDriverCommandManager )
    {
        NN_DETAIL_ATK_INFO("  commandBuffer\n"
                   "    soundThreadCommandTotalBuffer       %8zu bytes\n"
                   "    - soundThreadCommandBuffer       %6zu bytes\n"
                   "    taskThreadCommandTotalBuffer        %8zu bytes\n"
                   "    - taskThreadCommandBuffer        %8zu bytes\n"
                   , GetSoundThreadCommandTotalBufferSize()
                   , g_SoundThreadCommandBufferSize
                   , GetTaskThreadCommandTotalBufferSize()
                   , g_TaskThreadCommandBufferSize );
        totalSize += GetSoundThreadCommandTotalBufferSize();
        totalSize += GetTaskThreadCommandTotalBufferSize();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        NN_DETAIL_ATK_INFO("    lowLevelVoiceCommandTotalBuffer             %8zu bytes\n", GetLowLevelVoiceCommandTotalBufferSize());
        NN_DETAIL_ATK_INFO("    - lowLevelVoiceCommandBuffer             %8zu bytes\n", g_VoiceCommandBufferSize);
        NN_DETAIL_ATK_INFO("    voiceReplyCommandTotalBuffer             %8zu bytes\n", GetVoiceReplyCommandTotalBufferSize());
        NN_DETAIL_ATK_INFO("    - voiceReplyCommandBuffer             %8zu bytes\n", g_VoiceCommandBufferSize);
        totalSize += GetLowLevelVoiceCommandTotalBufferSize();
        totalSize += GetVoiceReplyCommandTotalBufferSize();
#endif
    }
    NN_DETAIL_ATK_INFO( "  total:                     %8zu bytes\n", totalSize );
#endif
}

size_t SoundSystem::GetAudioRendererBufferSize() NN_NOEXCEPT
{
    detail::driver::HardwareManager::HardwareManagerParameter parameter;
    SetupHardwareManagerParameterFromCurrentSetting(&parameter);
    return IsInitialized() ? detail::driver::HardwareManager::GetInstance().GetRequiredMemSize( parameter ) : 0;
}

//! @briefprivate
size_t SoundSystem::GetRecorderBufferSize() NN_NOEXCEPT
{
    detail::driver::HardwareManager::HardwareManagerParameter parameter;
    SetupHardwareManagerParameterFromCurrentSetting(&parameter);

    size_t bufferSize = 0u;
    if (IsInitialized() && g_IsRecordingEnabled)
    {
        bufferSize += detail::driver::HardwareManager::GetInstance().GetRequiredRecorderWorkBufferSize( parameter );
    }

    return bufferSize;
}

size_t SoundSystem::GetUserCircularBufferSinkBufferSize() NN_NOEXCEPT
{
    detail::driver::HardwareManager::HardwareManagerParameter parameter;
    SetupHardwareManagerParameterFromCurrentSetting(&parameter);

    size_t bufferSize = 0u;
    if (IsInitialized() && g_IsCircularBufferSinkEnabled)
    {
        bufferSize += detail::driver::HardwareManager::GetInstance().GetRequiredCircularBufferSinkWithMemoryPoolBufferSize( parameter );
    }

    return bufferSize;
}

size_t SoundSystem::GetLowLevelVoiceAllocatorBufferSize() NN_NOEXCEPT
{
    return IsInitialized() ? detail::driver::HardwareManager::GetInstance().GetRequiredMemSizeForMemoryPool( g_VoiceCountMax ) : 0;
}

size_t SoundSystem::GetMultiVoiceManagerBufferSize() NN_NOEXCEPT
{
    return IsInitialized() ? detail::driver::MultiVoiceManager::GetInstance().GetRequiredMemSize( g_VoiceCountMax, GetSoundInstanceConfig() ) : 0;
}

size_t SoundSystem::GetChannelManagerBufferSize() NN_NOEXCEPT
{
    return IsInitialized() ? detail::driver::ChannelManager::GetInstance().GetRequiredMemSize( GetChannelCount( g_VoiceCountMax ), GetSoundInstanceConfig() ) : 0;
}

size_t SoundSystem::GetSoundThreadCommandTotalBufferSize() NN_NOEXCEPT
{
    return detail::DriverCommand::GetInstance().GetRequiredMemSize(g_SoundThreadCommandBufferSize, g_SoundThreadCommandQueueCount);
}

size_t SoundSystem::GetTaskThreadCommandTotalBufferSize() NN_NOEXCEPT
{
    return detail::DriverCommand::GetInstance().GetRequiredMemSize(g_TaskThreadCommandBufferSize, g_TaskThreadCommandQueueCount);
}

#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
size_t SoundSystem::GetLowLevelVoiceCommandTotalBufferSize() NN_NOEXCEPT
{
    return detail::LowLevelVoiceCommand::GetInstance().GetRequiredMemSize(g_VoiceCommandBufferSize, VoiceCommandQueueCount);
}

size_t SoundSystem::GetVoiceReplyCommandTotalBufferSize() NN_NOEXCEPT
{
    return detail::VoiceReplyCommand::GetInstance().GetRequiredMemSize(g_VoiceCommandBufferSize, VoiceCommandQueueCount);
}
#endif

size_t SoundSystem::GetDriverCommandBufferSize() NN_NOEXCEPT
{
    return detail::DriverCommand::GetInstance().GetCommandBufferSize();
}

size_t SoundSystem::GetAllocatableDriverCommandSize() NN_NOEXCEPT
{
    return detail::DriverCommand::GetInstance().GetAllocatableCommandSize();
}

size_t SoundSystem::GetAllocatedDriverCommandBufferSize() NN_NOEXCEPT
{
    return detail::DriverCommand::GetInstance().GetAllocatedCommandBufferSize();
}


int SoundSystem::GetAllocatedDriverCommandCount() NN_NOEXCEPT
{
    return detail::DriverCommand::GetInstance().GetAllocatedCommandCount();
}

detail::SoundInstanceConfig SoundSystem::GetSoundInstanceConfig() NN_NOEXCEPT
{
    detail::SoundInstanceConfig config;
    config.isBusMixVolumeEnabled = g_IsBusMixVolumeEnabled;
    config.isVolumeThroughModeEnabled = g_IsVolumeThroughModeEnabled;
    config.busCount = g_BusCountMax;
    return config;
}

detail::SoundInstanceConfig SoundSystem::GetSoundInstanceConfig(const SoundSystemParam & param) NN_NOEXCEPT
{
    detail::SoundInstanceConfig config;
    config.isBusMixVolumeEnabled = param.enableBusMixVolume;
    config.isVolumeThroughModeEnabled = param.enableVolumeThroughMode;
    config.busCount = ClampBusCountMax(param.busCountMax);
    return config;
}

void SoundSystem::RegisterAudioRendererPerformanceReader( AudioRendererPerformanceReader& performanceReader ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( performanceReader.IsInitialized() );

    if( !g_IsProfilerEnabled )
    {
        //  performanceReader が初期化されているならば SoundSystem も初期化されています。
        //  SoundSystem が初期化されているとき、プロファイラが有効か無効かは判明しています。
        //  プロファイラが無効であるとき、パフォーマンス情報が取得できないことを警告します。
        NN_ATK_WARNING("Profiler is not enabled. So you can't get audio performance information.");
    }

    detail::driver::SoundThread::GetInstance().RegisterAudioRendererPerformanceReader( performanceReader );
}


#if 0
/*--------------------------------------------------------------------------------*
  Name:         WaitForResetReady

  Description:  リセット準備の完了を待つ

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
void SoundSystem::WaitForResetReady()
{
    if ( ! g_IsInitialized ) return;

    nn::os::Tick tick = nn::os::Tick::GetSystemCurrent();

    while( ! detail::driver::HardwareManager::GetInstance().IsResetReady() )
    {
        int64_t seconds = static_cast<nn::fnd::TimeSpan>(
                nn::os::Tick::GetSystemCurrent() - tick ).GetSeconds();

        if ( seconds > 0 )
        {
            NN_ATK_WARNING("SoundSystem::WaitForResetReady is TIME OUT.");
            break;
        }
    }
}
#endif

bool SoundSystem::AppendEffect( AuxBus bus, EffectBase* pEffect, void* effectBuffer, size_t effectBufferSize ) NN_NOEXCEPT
{
    return AppendEffect(bus, pEffect, effectBuffer, effectBufferSize, OutputDevice_Main);
}

bool SoundSystem::AppendEffect( AuxBus bus, EffectBase* pEffect, void* effectBuffer, size_t effectBufferSize, OutputDevice device ) NN_NOEXCEPT
{
    return AppendEffect(bus, pEffect, effectBuffer, effectBufferSize, device, 0);
}

bool SoundSystem::AppendEffect( AuxBus bus, EffectBase* pEffect, void* effectBuffer, size_t effectBufferSize, OutputDevice device, int subMixIndex ) NN_NOEXCEPT
{
    // バッファの必要ないユーザーエフェクトが存在することを考慮し、 effectBuffer の nullptr チェックは行わない
    NN_SDK_ASSERT_NOT_NULL( pEffect );
    NN_SDK_ASSERT_EQUAL( device, OutputDevice_Main );
    NN_UNUSED( device );

    return detail::driver::HardwareManager::GetInstance().GetSubMix( subMixIndex ).AppendEffect(
        pEffect, ConvertAuxBusIndexToSubMixBusIndex( bus ), effectBuffer, effectBufferSize );
}

bool SoundSystem::AppendEffect( AuxBus bus, EffectAux* pEffect, void* effectBuffer, size_t effectBufferSize ) NN_NOEXCEPT
{
    return AppendEffect(bus, pEffect, effectBuffer, effectBufferSize, OutputDevice_Main);
}

bool SoundSystem::AppendEffect( AuxBus bus, EffectAux* pEffect, void* effectBuffer, size_t effectBufferSize, OutputDevice device) NN_NOEXCEPT
{
    return AppendEffect(bus, pEffect, effectBuffer, effectBufferSize, device, 0);
}

bool SoundSystem::AppendEffect( AuxBus bus, EffectAux* pEffect, void* effectBuffer, size_t effectBufferSize, OutputDevice device, int subMixIndex ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL( device, OutputDevice_Main );
    NN_UNUSED( device );

    return detail::driver::HardwareManager::GetInstance().GetSubMix( subMixIndex ).AppendEffect(
        pEffect, ConvertAuxBusIndexToSubMixBusIndex( bus ), effectBuffer, effectBufferSize );
}

bool SoundSystem::AppendEffectToFinalMix( EffectAux* pEffect, void* effectBuffer, size_t effectBufferSize ) NN_NOEXCEPT
{
    return nn::atk::detail::driver::HardwareManager::GetInstance().GetFinalMix().AppendEffect(
        pEffect, effectBuffer, effectBufferSize );
}

bool SoundSystem::AppendEffectToAdditionalSubMix( EffectAux* pEffect, void* effectBuffer, size_t effectBufferSize ) NN_NOEXCEPT
{
    return nn::atk::detail::driver::HardwareManager::GetInstance().GetAdditionalSubMix().AppendEffect(
        pEffect, AdditionalSubMixBusIndex, effectBuffer, effectBufferSize );
}

size_t SoundSystem::GetRequiredEffectAuxBufferSize( const EffectAux* pEffect ) NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetRequiredEffectAuxBufferSize(pEffect);
}


void SoundSystem::RemoveEffect( AuxBus bus, EffectBase* pEffect ) NN_NOEXCEPT
{
    RemoveEffect(bus, pEffect, OutputDevice_Main);
}

void SoundSystem::RemoveEffect( AuxBus bus, EffectBase* pEffect, OutputDevice device ) NN_NOEXCEPT
{
    RemoveEffect(bus, pEffect, device, 0);
}

void SoundSystem::RemoveEffect( AuxBus bus, EffectBase* pEffect, OutputDevice device, int subMixIndex ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pEffect );
    NN_SDK_ASSERT_EQUAL( device, OutputDevice_Main );
    NN_UNUSED( device );

    detail::driver::HardwareManager::GetInstance().GetSubMix( subMixIndex ).RemoveEffect(
        pEffect, ConvertAuxBusIndexToSubMixBusIndex( bus ) );
}

void SoundSystem::RemoveEffect( AuxBus bus, EffectAux* pEffect ) NN_NOEXCEPT
{
    RemoveEffect(bus, pEffect, OutputDevice_Main);
}

void SoundSystem::RemoveEffect( AuxBus bus, EffectAux* pEffect, OutputDevice device ) NN_NOEXCEPT
{
    RemoveEffect(bus, pEffect, device, 0);
}

void SoundSystem::RemoveEffect( AuxBus bus, EffectAux* pEffect, OutputDevice device, int subMixIndex ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pEffect );
    NN_SDK_ASSERT_EQUAL( device, OutputDevice_Main );
    NN_UNUSED( device );

    detail::driver::HardwareManager::GetInstance().GetSubMix( subMixIndex ).RemoveEffect(
        pEffect, ConvertAuxBusIndexToSubMixBusIndex( bus ) );
}

void SoundSystem::RemoveEffectFromFinalMix( EffectAux* pEffect ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pEffect );

    detail::driver::HardwareManager::GetInstance().GetFinalMix().RemoveEffect( pEffect );
}

void SoundSystem::RemoveEffectFromAdditionalSubMix( EffectAux* pEffect ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pEffect );

    detail::driver::HardwareManager::GetInstance().GetAdditionalSubMix().RemoveEffect( pEffect, AdditionalSubMixBusIndex );
}

void SoundSystem::ClearEffect( AuxBus bus ) NN_NOEXCEPT
{
    ClearEffect(bus, OutputDevice_Main);
}

void SoundSystem::ClearEffect( AuxBus bus, OutputDevice device ) NN_NOEXCEPT
{
    ClearEffect(bus, device, 0);
}

void SoundSystem::ClearEffect( AuxBus bus, OutputDevice device, int subMixIndex ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL( device, OutputDevice_Main );
    NN_UNUSED( device );

    detail::driver::HardwareManager::GetInstance().GetSubMix( subMixIndex ).ClearEffect( ConvertAuxBusIndexToSubMixBusIndex( bus ) );
}

void SoundSystem::ClearEffectFromFinalMix() NN_NOEXCEPT
{
    detail::driver::HardwareManager::GetInstance().GetFinalMix().ClearEffect();
}

void SoundSystem::ClearEffectFromAdditionalSubMix() NN_NOEXCEPT
{
    detail::driver::HardwareManager::GetInstance().GetAdditionalSubMix().ClearEffect( AdditionalSubMixBusIndex );
}

bool SoundSystem::IsClearEffectFinished( AuxBus bus ) NN_NOEXCEPT
{
    return IsClearEffectFinished( bus, OutputDevice_Main );
}

bool SoundSystem::IsClearEffectFinished( AuxBus bus, OutputDevice device ) NN_NOEXCEPT
{
    return IsClearEffectFinished( bus, device, 0 );
}

bool SoundSystem::IsClearEffectFinished( AuxBus bus, OutputDevice device, int subMixIndex ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL( device, OutputDevice_Main );   //  現状、Main のみ
    NN_UNUSED( device );
    return !detail::driver::HardwareManager::GetInstance().GetSubMix( subMixIndex ).HasEffect( ConvertAuxBusIndexToSubMixBusIndex( bus ) );
}

bool SoundSystem::IsClearEffectFromFinalMixFinished() NN_NOEXCEPT
{
    return !detail::driver::HardwareManager::GetInstance().GetFinalMix().HasEffect( 0 );    //  FinalMix は 1 つしかバスを持ちません
}

bool SoundSystem::IsClearEffectFromAdditionalSubMixFinished() NN_NOEXCEPT
{
    return !detail::driver::HardwareManager::GetInstance().GetAdditionalSubMix().HasEffect( 0 ); //  AdditinalSubMix は 1 つしかバスを持ちません
}

void SoundSystem::SetAuxBusVolume(AuxBus bus, float volume, nn::TimeSpan fadeTimes) NN_NOEXCEPT
{
    SetAuxBusVolume(bus, volume, fadeTimes, 0);
}

void SoundSystem::SetAuxBusVolume(AuxBus bus, float volume, nn::TimeSpan fadeTimes, int subMixIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(bus, AuxBus_A, AuxBus_Count);

    const int FadeTimesMsec = static_cast<int>(fadeTimes.GetMilliSeconds());
    const int SoundFrameIntervalMsec = nn::atk::detail::driver::HardwareManager::SoundFrameIntervalMsec;
    const int FadeFrames = (FadeTimesMsec + SoundFrameIntervalMsec - 1) / SoundFrameIntervalMsec;

    // コマンドを投げる
    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    detail::DriverCommandAuxBusVolume* command =
        cmdmgr.AllocCommand<detail::DriverCommandAuxBusVolume>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = detail::DriverCommandId_AuxBusVolume;
    command->bus = bus;
    command->subMixIndex = subMixIndex;
    command->volume = volume;
    command->fadeFrames = FadeFrames;
    cmdmgr.PushCommand(command);

}

float SoundSystem::GetAuxBusVolume(AuxBus bus) NN_NOEXCEPT
{
    return GetAuxBusVolume(bus, 0);
}

float SoundSystem::GetAuxBusVolume(AuxBus bus, int subMixIndex) NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetAuxBusVolume(bus, subMixIndex);
}

void SoundSystem::SetMainBusChannelVolumeForAdditionalEffect(float volume, int srcChannelIndex, int destChannelIndex) NN_NOEXCEPT
{
    detail::driver::HardwareManager::GetInstance().SetMainBusChannelVolumeForAdditionalEffect(
        volume,
        srcChannelIndex,
        destChannelIndex
    );
}

float SoundSystem::GetMainBusChannelVolumeForAdditionalEffect(int srcChannelIndex, int destChannelIndex) NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetMainBusChannelVolumeForAdditionalEffect(srcChannelIndex, destChannelIndex);
}

void SoundSystem::SetAuxBusChannelVolumeForAdditionalEffect(AuxBus bus, float volume, int srcChannelIndex, int destChannelIndex) NN_NOEXCEPT
{
    detail::driver::HardwareManager::GetInstance().SetAuxBusChannelVolumeForAdditionalEffect(
        bus,
        volume,
        srcChannelIndex,
        destChannelIndex
    );
}

float SoundSystem::GetAuxBusChannelVolumeForAdditionalEffect(AuxBus bus, int srcChannelIndex, int destChannelIndex) NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetAuxBusChannelVolumeForAdditionalEffect(bus, srcChannelIndex, destChannelIndex);
}

void SoundSystem::SetAllAuxBusChannelVolumeForAdditionalEffect(float volume, int srcChannelIndex, int destChannelIndex) NN_NOEXCEPT
{
    const int auxBusCountForAdditionalEffect = detail::driver::HardwareManager::GetInstance().GetAuxBusTypeCountForAdditionalEffect();
    for ( int i = 0; i < auxBusCountForAdditionalEffect; i++ )
    {
        SetAuxBusChannelVolumeForAdditionalEffect(static_cast<AuxBus>(i), volume, srcChannelIndex, destChannelIndex);
    }
}

void SoundSystem::SetAllBusChannelVolumeForAdditionalEffect(float volume, int srcChannelIndex, int destChannelIndex) NN_NOEXCEPT
{
    SetMainBusChannelVolumeForAdditionalEffect(volume, srcChannelIndex, destChannelIndex);
    SetAllAuxBusChannelVolumeForAdditionalEffect(volume, srcChannelIndex, destChannelIndex);
}

void SoundSystem::VoiceCommandProcess(UpdateType updateType, uint32_t frameCount) NN_NOEXCEPT
{
    if (g_IsInitialized == false)
    {
        return;
    }

    detail::driver::SoundThread& soundThread = detail::driver::SoundThread::GetInstance();
    detail::LowLevelVoiceCommand& lowLevelVoiceCommand = detail::LowLevelVoiceCommand::GetInstance();

    while (lowLevelVoiceCommand.GetCommandListCount() < frameCount)
    {
        soundThread.FrameProcess( updateType );

        lowLevelVoiceCommand.RecvCommandReply();
        lowLevelVoiceCommand.FlushCommand( true, false );
    }
}

void SoundSystem::VoiceCommandProcess(uint32_t frameCount) NN_NOEXCEPT
{
    VoiceCommandProcess(UpdateType_AudioFrame, frameCount);
}

void SoundSystem::VoiceCommandUpdate() NN_NOEXCEPT
{
    if ( !detail_IsSoundThreadEnabled() )
    {
        nn::atk::detail::driver::SoundThread::GetInstance().UpdateLowLevelVoices();
        nn::atk::detail::driver::SoundThread::GetInstance().EffectFrameProcess();
        nn::atk::detail::driver::HardwareManager::GetInstance().RequestUpdateAudioRenderer();
    }
}

size_t SoundSystem::GetPerformanceFrameBufferSize() NN_NOEXCEPT
{
    return g_PerformanceFrameBufferSize / detail::MaxPerformanceBufferCount;
}

int SoundSystem::GetDroppedLowLevelVoiceCount() NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetDroppedLowLevelVoiceCount();
}

void SoundSystem::SetWarningCallback( WarningCallback callback ) NN_NOEXCEPT
{
    g_WarningCallback = callback;
}

void SoundSystem::ClearWarningCallback() NN_NOEXCEPT
{
    g_WarningCallback = nullptr;
}

void SoundSystem::CallWarningCallback(WarningId id, IWarningCallbackInfo* pInfo) NN_NOEXCEPT
{
    if (g_WarningCallback != nullptr)
    {
        WarningCallbackArg arg;
        arg.warningId = id;
        arg.pInfo = pInfo;
        g_WarningCallback(arg);
    }
}

void SoundSystem::RegisterSoundThreadUpdateProfileReader(SoundThreadUpdateProfileReader& profileReader) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( profileReader.IsInitialized() );
    detail::driver::SoundThread::GetInstance().RegisterSoundThreadUpdateProfileReader( profileReader );
}

void SoundSystem::UnregisterSoundThreadUpdateProfileReader(SoundThreadUpdateProfileReader& profileReader) NN_NOEXCEPT
{
    detail::driver::SoundThread::GetInstance().UnregisterSoundThreadUpdateProfileReader( profileReader );
}

void SoundSystem::RegisterSoundThreadInfoRecorder(SoundThreadInfoRecorder& recorder) NN_NOEXCEPT
{
    detail::driver::SoundThread::GetInstance().RegisterSoundThreadInfoRecorder( recorder );
}

void SoundSystem::UnregisterSoundThreadInfoRecorder(SoundThreadInfoRecorder& recorder) NN_NOEXCEPT
{
    detail::driver::SoundThread::GetInstance().UnregisterSoundThreadInfoRecorder( recorder );
}

size_t SoundSystem::ReadCircularBufferSink(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if( !g_IsCircularBufferSinkEnabled )
    {
        //  CircularBufferSink が無効ならば、1 度だけ警告を出すようにします
        if( !g_IsCircularBufferSinkWarningDisplayed )
        {
            NN_ATK_WARNING( "CircularBufferSink is disabled. See the document of this function." );
            g_IsCircularBufferSinkWarningDisplayed = true;
        }
        return 0;
    }

    return detail::driver::HardwareManager::GetInstance().ReadUserCircularBufferSink(pOutBuffer, bufferSize);
}

size_t SoundSystem::GetCircularBufferSinkBufferSize() NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetUserCircularBufferSinkBufferSize();
}

int SoundSystem::GetRendererSampleCount() NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetRendererSampleCount();
}

int SoundSystem::GetRendererChannelCountMax() NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetChannelCountMax();
}

void SoundSystem::StopCircularBufferSink() NN_NOEXCEPT
{
    detail::driver::HardwareManager& manager = detail::driver::HardwareManager::GetInstance();
    if( manager.GetUserCircularBufferSinkState() == CircularBufferSinkState_Invalid )
    {
        NN_ATK_WARNING("CircularBufferSink is disabled. See the document of this function.");
        return;
    }

    manager.StopUserCircularBufferSink();
}

void SoundSystem::StartCircularBufferSink() NN_NOEXCEPT
{
    detail::driver::HardwareManager& manager = detail::driver::HardwareManager::GetInstance();
    if( manager.GetUserCircularBufferSinkState() == CircularBufferSinkState_Invalid )
    {
        NN_ATK_WARNING("CircularBufferSink is disabled. See the document of this function.");
        return;
    }

    manager.StartUserCircularBufferSink(false);
}

CircularBufferSinkState SoundSystem::GetCircularBufferSinkState() NN_NOEXCEPT
{
    return detail::driver::HardwareManager::GetInstance().GetUserCircularBufferSinkState();
}

void SoundSystem::SetupInitializeBufferSet(
    InitializeBufferSet* pOutValue,
    const SoundSystemParam& param,
    const InitializeBufferSet& bufferSet) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);
    NN_SDK_ASSERT(bufferSet.workMem != NULL);

    g_IsManagingMemoryPool = param.enableMemoryPoolManagement;

    if ( param.enableMemoryPoolManagement )
    {
        const uintptr_t workMemEnd = bufferSet.workMem + bufferSet.workMemSize;

        //  メモリプール管理するバッファの用意
        const size_t memoryPoolMemSize = nn::util::align_up( GetRequiredMemSizeForMemoryPool( param ), nn::audio::MemoryPoolType::SizeGranularity );
        const uintptr_t memoryPoolMem = nn::util::align_up( bufferSet.workMem, nn::audio::MemoryPoolType::AddressAlignment );

        pOutValue->workMem = memoryPoolMem + memoryPoolMemSize;
        pOutValue->workMemSize = workMemEnd - ( memoryPoolMem + memoryPoolMemSize );
        pOutValue->memoryPoolMem = memoryPoolMem;
        pOutValue->memoryPoolMemSize = memoryPoolMemSize;
    }
    else
    {
        NN_SDK_ASSERT(bufferSet.memoryPoolMem != NULL);

        pOutValue->workMem = bufferSet.workMem;
        pOutValue->workMemSize = bufferSet.workMemSize;
        pOutValue->memoryPoolMem = bufferSet.memoryPoolMem;
        pOutValue->memoryPoolMemSize = bufferSet.memoryPoolMemSize;
    }

    if ( param.enableCircularBufferSinkBufferManagement )
    {
        if ( param.enableCircularBufferSink )
        {
            const size_t circularBufferSinkMemSize = GetRequiredMemSizeForCircularBufferSink( param );
            const uintptr_t circularBufferSinkMem = pOutValue->workMem;

            pOutValue->workMem = circularBufferSinkMem + circularBufferSinkMemSize;
            pOutValue->workMemSize = pOutValue->workMemSize - circularBufferSinkMemSize;

            pOutValue->circularBufferSinkMem = circularBufferSinkMem;
            pOutValue->circularBufferSinkMemSize = circularBufferSinkMemSize;
        }
        else
        {
            pOutValue->circularBufferSinkMem = NULL;
            pOutValue->circularBufferSinkMemSize = 0u;
        }
    }
    else
    {
        pOutValue->circularBufferSinkMem = bufferSet.circularBufferSinkMem;
        pOutValue->circularBufferSinkMemSize = bufferSet.circularBufferSinkMemSize;
    }
}

void SoundSystem::SetupHardwareManagerParameterFromCurrentSetting(
    nn::atk::detail::driver::HardwareManager::HardwareManagerParameter* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    pOutValue->SetRendererSampleRate( g_RendererSampleRate );
    pOutValue->SetUserEffectCount( g_UserEffectCount );
    pOutValue->SetProfilerEnabled( g_IsProfilerEnabled );
    pOutValue->SetVoiceCount( g_VoiceCountMax );
    pOutValue->SetEffectEnabled( g_IsEffectEnabled );
    pOutValue->SetRecordingEnabled( g_IsRecordingEnabled );
    pOutValue->SetUserCircularBufferSinkEnabled( g_IsCircularBufferSinkEnabled );
    pOutValue->SetStereoModeEnabled( g_IsStereoModeEnabled );
    pOutValue->SetSoundThreadEnabled( g_IsSoundThreadEnabled );
    pOutValue->SetVoiceDropEnabled( g_IsVoiceDropEnabled );
    pOutValue->SetPreviousSdkVersionLowPassFilterCompatible( g_IsPreviousSdkVersionLowPassFilterCompatible );
    pOutValue->SetUnusedEffectChannelMutingEnabled( g_IsUnusedEffectChannelMutingEnabled );
    pOutValue->SetUserThreadRenderingEnabled( g_IsUserThreadRenderingEnabled );
    pOutValue->SetSubMixParameter(
        g_IsStereoModeEnabled,
        g_IsEffectEnabled,
        g_IsSubMixEnabled,
        g_IsAdditionalEffectBusEnabled,
        g_IsAdditionalSubMixEnabled,
        g_IsCustomSubMixEnabled,
        g_CustomSubMixSubMixCount,
        g_CustomSubMixMixBufferCount
    );
    pOutValue->SetMemoryPoolAttachCheckEnabled( g_IsMemoryPoolAttachCheckEnabled );
}


} // namespace nn::atk
} // namespace nn

