﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_SoundHeap.h>
#include <nn/atk/atk_SoundThread.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/atk_Util.h>
#include <nn/atk/detail/atk_Macro.h>

namespace nn {
namespace atk {

/*--------------------------------------------------------------------------------*
  Name:         SoundHeap

  Description:  コンストラクタ

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
SoundHeap::SoundHeap() NN_NOEXCEPT
: m_FrameHeap()
, m_IsAutoMemoryPoolManagementEnabled(false)
{
}

/*--------------------------------------------------------------------------------*
  Name:         SoundHeap

  Description:  デストラクタ

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
SoundHeap::~SoundHeap() NN_NOEXCEPT
{
    Destroy();

//    m_CriticalSection.Finalize();
}

/*--------------------------------------------------------------------------------*
  Name:         Create

  Description:  サウンドヒープを作成する

  Arguments:    startAddress - メインメモリ上のヒープ開始アドレス
                size         - メインメモリ上のヒープのサイズ

  Returns:      成功したらtrue
 *--------------------------------------------------------------------------------*/
bool SoundHeap::Create(
    void* startAddress,
    size_t size
) NN_NOEXCEPT
{
    return Create(startAddress, size, true);
}

bool SoundHeap::Create(
    void* startAddress,
    size_t size,
    bool enableAutoMemoryPoolManagement
) NN_NOEXCEPT
{
    if (!atk::detail::Util::IsValidMemoryForDsp( startAddress, size ))
    {
        NN_ATK_WARNING(
            "the memory area (0x%08x - 0x%08x %dbyte) provided cross a 512 MB segment.",
            startAddress,
            util::BytePtr( startAddress, size ).Get(),
            size );
    }

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );


    void* frameHeapAddress = startAddress;
    size_t frameHeapSize = size;

    if ( enableAutoMemoryPoolManagement )
    {
        // メモリプールにアタッチするため、開始アドレスとサイズを補正する
        void* alignedStartAddress = util::BytePtr( startAddress ).AlignUp( nn::audio::MemoryPoolType::AddressAlignment ).Get();
        ptrdiff_t alignedSizeDistance = util::BytePtr( startAddress ).Distance( alignedStartAddress );

        if (static_cast<size_t>(alignedSizeDistance) > size)
        {
            // アライメントを取れるだけのサイズがなければ失敗
            return false;
        }

        frameHeapAddress = alignedStartAddress;
        // アライン後の残りサイズから SizeGranularity の倍数のサイズを確保する
        frameHeapSize = nn::util::align_down(size - alignedSizeDistance, nn::audio::MemoryPoolType::SizeGranularity);
    }

    bool result = m_FrameHeap.Create( frameHeapAddress, frameHeapSize );
    if ( result && enableAutoMemoryPoolManagement )
    {
        nn::atk::SoundSystem::AttachMemoryPool( &m_MemoryPool, frameHeapAddress, frameHeapSize );
        m_IsAutoMemoryPoolManagementEnabled = true;
    }
    return result;
}

/*--------------------------------------------------------------------------------*
  Name:         Destroy

  Description:  ヒープを破棄する

  Arguments:    無し

  Returns:      無し
 *--------------------------------------------------------------------------------*/
void SoundHeap::Destroy() NN_NOEXCEPT
{
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );

    if ( ! m_FrameHeap.IsValid() ) return;

    Clear();

    if ( m_IsAutoMemoryPoolManagementEnabled )
    {
        nn::atk::SoundSystem::DetachMemoryPool( &m_MemoryPool );
        m_IsAutoMemoryPoolManagementEnabled = false;
    }

    m_FrameHeap.Destroy();
}

/*--------------------------------------------------------------------------------*
  Name:         Alloc

  Description:  メインメモリ上のヒープからメモリ領域を割り当てます

  Arguments:    size        - メモリ領域のサイズ

  Returns:      割り当てたメモリ領域の先頭アドレス
 *--------------------------------------------------------------------------------*/
void* SoundHeap::Allocate( size_t size ) NN_NOEXCEPT
{
//    NN_DETAIL_ATK_INFO("SoundHeap::Allocate %d / %d\n",size,GetFreeSize());

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );

    void* buffer = m_FrameHeap.Alloc(
        size,
        DisposeCallbackFunc,
        nullptr,
        nullptr,
        nullptr
    );
    // NN_DETAIL_ATK_INFO("[%s] free(%d %dKB %.3fMB)\n", __FUNCTION__,
    //         GetFreeSize(), GetFreeSize()/1024, GetFreeSize()/(1024.f*1024));
    return buffer;
}

void* SoundHeap::Allocate(
    size_t size,
    SoundMemoryAllocatable::DisposeCallback heapCallback,
    void* heapCallbackArg ) NN_NOEXCEPT
{
    //    NN_DETAIL_ATK_INFO("SoundHeap::Alloc %d / %d\n",size,GetFreeSize());

    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );

    return m_FrameHeap.Alloc(
        size,
        DisposeCallbackFunc,
        nullptr,
        heapCallback,
        heapCallbackArg
    );
}

size_t SoundHeap::GetAllocateSize(size_t size, bool needMemoryPool) NN_NOEXCEPT
{
    NN_UNUSED(needMemoryPool);
    return size;
}

// 確保したメモリブロックを全て解放する
void SoundHeap::Clear() NN_NOEXCEPT
{
    if ( ! m_FrameHeap.IsValid() ) return;

    {
        detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );

        m_FrameHeap.Clear();
    }

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
    if ( cmdmgr.IsAvailable() ) {
        uint32_t tag = cmdmgr.FlushCommand( true );
        cmdmgr.WaitCommandReply( tag );
    }
}

// サウンドヒープの状態を保存する
int SoundHeap::SaveState() NN_NOEXCEPT
{
    detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );

    return m_FrameHeap.SaveState();
}

// サウンドヒープの状態を復元する
void SoundHeap::LoadState( int level ) NN_NOEXCEPT
{
    {
        detail::fnd::ScopedLock<detail::fnd::CriticalSection> lock( m_CriticalSection );

        m_FrameHeap.LoadState( level );

        detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();
        if ( cmdmgr.IsAvailable() ) {
            uint32_t tag = cmdmgr.FlushCommand( true );
            cmdmgr.WaitCommandReply( tag );
        }
    }

    // NN_DETAIL_ATK_INFO("[%s] free(%d %dKB %.3fMB)\n", __FUNCTION__,
    //         GetFreeSize(), GetFreeSize()/1024, GetFreeSize()/(1024.f*1024));
}

/* ========================================================================
 private function
 ======================================================================== */

void SoundHeap::DisposeCallbackFunc( void* mem, size_t size, void* arg ) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    detail::DriverCommand& cmdmgr = detail::DriverCommand::GetInstance();

    if ( cmdmgr.IsAvailable() ) {
        detail::DriverCommandInvalidateData* command =
            cmdmgr.AllocCommand<detail::DriverCommandInvalidateData>();
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        if (command == nullptr)
        {
            return; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
        }
#else
        NN_SDK_ASSERT_NOT_NULL(command);
#endif
        command->id = detail::DriverCommandId_InvalidateData;
        command->mem = mem;
        command->size = size;
        cmdmgr.PushCommand(command);
    }
}

} // namespace nn::atk
} // namespace nn

