﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_Sound3DEngine.h>

#include <nn/util/util_Vector.h>
#include <nn/atk/atk_BasicSound.h>      // nn::atk::SoundAmbientParam

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const uint32_t Sound3DEngine::UpdateVolume );
NN_DEFINE_STATIC_CONSTANT( const uint32_t Sound3DEngine::UpdatePriority );
NN_DEFINE_STATIC_CONSTANT( const uint32_t Sound3DEngine::UpdatePan );
NN_DEFINE_STATIC_CONSTANT( const uint32_t Sound3DEngine::UpdateSurroundPan );
NN_DEFINE_STATIC_CONSTANT( const uint32_t Sound3DEngine::UpdateFilter );
NN_DEFINE_STATIC_CONSTANT( const uint32_t Sound3DEngine::UpdatePitch );
NN_DEFINE_STATIC_CONSTANT( const uint32_t Sound3DEngine::UpdateStartPriority );

/*--------------------------------------------------------------------------------*
  Name:         Sound3DEngine

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
Sound3DEngine::Sound3DEngine() NN_NOEXCEPT
{
}

void Sound3DEngine::UpdateAmbientParam(
    SoundAmbientParam* pOutValue,
    const Sound3DManager* sound3DManager,
    const Sound3DParam* sound3DParam,
    uint32_t soundId,
    uint32_t updateFlag
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( sound3DManager );
    NN_SDK_ASSERT_NOT_NULL( sound3DParam );
    NN_SDK_ASSERT_NOT_NULL( pOutValue );

    (void)soundId;

    const Sound3DManager::ListenerList& listenerList = sound3DManager->GetListenerList();

    if ( updateFlag & UpdatePriority )
    {
        pOutValue->SetPriority( - sound3DManager->GetMaxPriorityReduction() );
    }
    if ( updateFlag & UpdateFilter )
    {
        pOutValue->SetBiquadFilterValue( 0.0f );
    }
    if ( updateFlag & UpdateVolume )
    {
        pOutValue->GetTvParam().SetVolume( 0.0f );
    }

    //  ▼デフォルト実装の挙動
    //
    //  ※ 本デフォルト実装では、volume, lpf, userData, outputLineFlag, {drc}FxSend
    //     は変更しない。
    //
    //  ----------------------------------------------------------
    //  SoundAmbientParam       Single      Multi(1)    Multi(*)
    //  ----------------------------------------------------------
    //      volume              -           -           -
    //      pitch               o           -           -
    //      lpf                 -           -           -
    //      biquadFilterValue   o           max         max
    //      biquadFilterType    o           o           o       (プレイヤーの値を参照)
    //      priority            o           max         max
    //      userData            -           -           -
    //      outputLineFlag      -           -           -
    //
    //      mainOutVolume       a1          b1          max
    //      pan                 a2          b2          -
    //      span                a3          b3          -
    //      fxSend              -           -           -
    //
    //      drcOutVolume        a1          b1          max
    //      drcPan              a2          b2          -
    //      drcSpan             a3          b3          -
    //      drcFxSend           -           -           -
    //  ----------------------------------------------------------
    //
    //  ※ 凡例
    //      Multi(1) : 当該出力先 (TV/DRC0/DRC1) にひとつしかリスナーが割り当てられていない
    //      Multi(*) :          〃                 複数のリスナーが割り当てられている
    //      - : 計算しない
    //      o : 計算する
    //      max : 計算して得られた値のうち、最大の値を採用する
    //      a1  : 同じ a1 のうちどちらか or どちらも計算する


    // マルチリスナー場合、ピッチ計算を行わない
    if ( listenerList.size() > 1 )
    {
        updateFlag &= ~UpdatePitch;
    }

    int destTv = 0;
    for( Sound3DManager::ListenerList::const_iterator itr = listenerList.begin() ;
         itr != listenerList.end() ; itr++ )
    {
        uint32_t flag = itr->GetOutputTypeFlag();
        if ( flag & Sound3DListener::OutputType_Tv )
        {
            destTv += 1;
        }
    }

    for( Sound3DManager::ListenerList::const_iterator itr = listenerList.begin() ;
         itr != listenerList.end() ; itr++ )
    {
        const Sound3DListener& listener = *itr;

        bool outTv = false;
        uint32_t flag = listener.GetOutputTypeFlag();
        if ( flag & Sound3DListener::OutputType_Tv )
        {
            outTv = true;
        }

        // 各 Calc で繰り返し計算される値を、先に計算しておく
        nn::util::Vector3fType actorPosition;
        float actorDistance = 0.0f;
        if ( updateFlag & ( UpdateVolume |
                            UpdatePriority |
                            UpdatePitch |
                            UpdateFilter ) )
        {
            nn::util::VectorSubtract( &actorPosition,
                               sound3DParam->position,
                               listener.GetPosition() );
            actorDistance = nn::util::VectorLength( actorPosition );
        }

        //------------------------------------------------------------------
        // 音量/プライオリティ計算
        if ( updateFlag & ( UpdateVolume | UpdatePriority ) )
        {
            float volume;
            int priority;
            Sound3DCalculator::CalculateVolumeAndPriority(
                &volume,
                &priority,
                *sound3DManager,
                listener,
                *sound3DParam,
                actorDistance
            );

            if ( updateFlag & UpdateVolume )
            {
                if ( outTv )
                {
                    // マルチリスナーの場合は、一番大きい値を採用する
                    pOutValue->GetTvParam().SetVolume(
                        std::max( volume, pOutValue->GetTvParam().GetVolume() ) );
                }
            }
            if ( updateFlag & UpdatePriority )
            {
                // マルチリスナーの場合は、一番大きい値を採用する
                pOutValue->SetPriority( std::max( priority, pOutValue->GetPriority() ) );
            }
        }

        //------------------------------------------------------------------
        // パン/サラウンドパン計算
        if ( updateFlag & ( UpdatePan | UpdateSurroundPan ))
        {
            // 当該リスナーの出力先に、ひとつのリスナーだけ割り当てられている場合に計算する
            bool isCalc = false;
            if ( outTv && destTv == 1 )
            {
                isCalc = true;
            }

            // 実際の計算
            if ( isCalc )
            {
                float pan;
                float span;

                Sound3DCalculator::CalculatePan(
                        &pan,
                        &span,
                        *sound3DManager,
                        listener,
                        *sound3DParam,
                        m_CalcPanParam
                        );

                if ( updateFlag & UpdatePan )
                {
                    if ( outTv )
                    {
                        pOutValue->GetTvParam().SetPan( pan );
                    }
                }
                if ( updateFlag & UpdateSurroundPan )
                {
                    if ( outTv )
                    {
                        pOutValue->GetTvParam().SetSurroundPan( span );
                    }
                }
            }
        }

        //------------------------------------------------------------------
        // ピッチ計算
        if ( updateFlag & UpdatePitch )
        {
            float pitch;

            Sound3DCalculator::CalculatePitch(
                &pitch,
                *sound3DManager,
                listener,
                *sound3DParam,
                actorPosition,
                actorDistance
            );

            pOutValue->SetPitch( pitch );
        }

        //------------------------------------------------------------------
        // Biquadフィルタ計算
        if ( updateFlag & UpdateFilter )
        {
            float biquadFilterValue;

            Sound3DCalculator::CalculateBiquadFilterValue(
                &biquadFilterValue,
                *sound3DManager,
                listener,
                *sound3DParam,  // 使わないが、今後の拡張性を考慮しておく
                actorDistance
            );

            pOutValue->SetBiquadFilterType( sound3DManager->GetBiquadFilterType() );
            // NN_DETAIL_ATK_INFO("(%.2f %.2f) ", biquadFilterValue, pOutValue->GetBiquadFilterValue() );

            // マルチリスナーの場合は、一番大きい値を採用する
            pOutValue->SetBiquadFilterValue(
                    std::max( biquadFilterValue, pOutValue->GetBiquadFilterValue() ) );
        }

        //------------------------------------------------------------------
        // NOTE:
        // 本関数をオーバーライドすることで、他のパラメータ (volume, lpf, userData,
        // outputLineFlag, fxSend, drcFxSend) を上書きすることができます。
    }
    // NN_DETAIL_ATK_INFO("=> %.2f\n", ambientParam->biquadFilterValue);
} // NOLINT(impl/function_size)

void Sound3DEngine::detail_UpdateAmbientParam(
    const Sound3DManager* sound3DManager,
    const Sound3DParam* sound3DParam,
    uint32_t soundId,
    SoundAmbientParam* ambientParam
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( sound3DManager );
    NN_SDK_ASSERT_NOT_NULL( sound3DParam );
    NN_SDK_ASSERT_NOT_NULL( ambientParam );

    uint32_t updateFlag = 0;
    if ( sound3DParam->controlFlag & SoundArchive::Sound3DInfo::FlagControl_Volume )
    {
        updateFlag |= UpdateVolume;
    }
    if ( sound3DParam->controlFlag & SoundArchive::Sound3DInfo::FlagControl_Priority )
    {
        updateFlag |= UpdatePriority;
    }
    if ( sound3DParam->controlFlag & SoundArchive::Sound3DInfo::FlagControl_Pan )
    {
        updateFlag |= UpdatePan;
    }
    if ( sound3DParam->controlFlag & SoundArchive::Sound3DInfo::FlagControl_SurroundPan )
    {
        updateFlag |= UpdateSurroundPan;
    }
    if ( sound3DParam->controlFlag & SoundArchive::Sound3DInfo::FlagControl_Filter )
    {
        updateFlag |= UpdateFilter;
    }
    if ( sound3DParam->dopplerFactor > 0 )
    {
        updateFlag |= UpdatePitch;
    }

    UpdateAmbientParam(
        ambientParam,
        sound3DManager,
        sound3DParam,
        soundId,
        updateFlag
    );
}

int Sound3DEngine::GetAmbientPriority(
    const Sound3DManager* sound3DManager,
    const Sound3DParam* sound3DParam,
    uint32_t soundId
) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( sound3DManager );
    NN_SDK_ASSERT_NOT_NULL( sound3DParam );

    uint32_t updateFlag = UpdateStartPriority ;
    if ( sound3DParam->controlFlag & SoundArchive::Sound3DInfo::FlagControl_Priority )
    {
        updateFlag |= UpdatePriority;
    }

    SoundAmbientParam ambientParam;
    UpdateAmbientParam(
        &ambientParam,
        sound3DManager,
        sound3DParam,
        soundId,
        updateFlag
    );

    return ambientParam.GetPriority();
}

} // namespace nn::atk
} // namespace nn

