﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_MemorySoundArchive.h>

#include <nn/atk/atk_GroupFileReader.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/detail/atk_MemoryFileStream.h>

namespace nn {
namespace atk {

/*--------------------------------------------------------------------------------*
  Name:         MemorySoundArchive

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
MemorySoundArchive::MemorySoundArchive() NN_NOEXCEPT
: m_pData( NULL )
{
}

/*--------------------------------------------------------------------------------*
  Name:         ~MemorySoundArchive

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
MemorySoundArchive::~MemorySoundArchive() NN_NOEXCEPT
{
}

/*--------------------------------------------------------------------------------*
  Name:         Initialize

  Description:  サウンドアーカイブを初期化する

  Arguments:    soundArchiveData - サウンドアーカイブデータ

  Returns:      成功したかどうか
 *--------------------------------------------------------------------------------*/
bool MemorySoundArchive::Initialize( const void* soundArchiveData ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( soundArchiveData );
    NN_SDK_ASSERT( util::is_aligned(reinterpret_cast<uintptr_t>(soundArchiveData), 4) );

    m_FileReader.Initialize( soundArchiveData );
    SoundArchive::Initialize( &m_FileReader );

    const unsigned long infoBlockOffset = m_FileReader.GetInfoBlockOffset();

    m_FileReader.SetInfoBlock(
        util::ConstBytePtr( soundArchiveData, infoBlockOffset ).Get()
    );

    const unsigned long stringBlockOffset = m_FileReader.GetStringBlockOffset();
    const unsigned long stringBlockSize = m_FileReader.GetStringBlockSize();

    // 文字列ブロックへのリファレンスは必ず含まれるが、
    // ブロック自体が含まれない場合、stringBlockOffset/stringBlockSize の両方共、
    // 0xffffffff が入っている。
    if ( stringBlockOffset == detail::SoundArchiveFileReader::InvalidOffset ||
         stringBlockSize == detail::SoundArchiveFileReader::InvalidSize )
    {
        // なにもしない
    }
    else
    {
        m_FileReader.SetStringBlock(
                util::ConstBytePtr( soundArchiveData, stringBlockOffset ).Get());
    }

    m_pData = soundArchiveData;

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Finalize

  Description:  サウンドアーカイブを閉じる

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void MemorySoundArchive::Finalize() NN_NOEXCEPT
{
    m_pData = NULL;
    m_FileReader.Finalize();
    SoundArchive::Finalize();
}

const void* MemorySoundArchive::detail_GetFileAddress( FileId fileId ) const NN_NOEXCEPT
{
    FileInfo fileInfo;
    bool result = detail_ReadFileInfo( fileId, &fileInfo );

    if ( result && fileInfo.offsetFromFileBlockHead != FileInfo::InvalidOffset )
    {
        return util::ConstBytePtr( m_pData,
                fileInfo.offsetFromFileBlockHead +
                m_FileReader.GetFileBlockOffset() ).Get();
    }
    else
    {
        // 「自動(共有)」の bxwsd/bxbnk は detail_ReadFileInfo で見つからない可能性
        // があるので、別途検索する (時間はかかるが仕方ない)
        uint32_t groupCount = GetGroupCount();
        for ( uint32_t i = 0; i < groupCount; i++ )
        {
            GroupInfo info;
            if ( ReadGroupInfo( &info, GetGroupIdFromIndex( i ) ) )
            {
                FileInfo groupFileInfo;
                if ( detail_ReadFileInfo( info.fileId, &groupFileInfo ) )
                {
                    const void* groupFile = util::ConstBytePtr(
                            m_pData,
                            groupFileInfo.offsetFromFileBlockHead +
                            m_FileReader.GetFileBlockOffset() ).Get();

                    detail::GroupFileReader reader(groupFile);
                    uint32_t groupItemCount = reader.GetGroupItemCount();
                    for ( uint32_t j = 0; j < groupItemCount; j++ )
                    {
                        detail::GroupItemLocationInfo locationInfo;
                        if ( reader.ReadGroupItemLocationInfo( &locationInfo, j ) )
                        {
                            if ( locationInfo.fileId == fileId &&
                                 locationInfo.address != NULL )
                            {
                                return locationInfo.address;
                            }
                        }
                    }
                }
            }
        }
        return NULL;
    }

    // 外部参照ファイルは対応しない
}

detail::fnd::FileStream* MemorySoundArchive::OpenStream( void* buffer, size_t size, detail::position_t begin, size_t length ) const NN_NOEXCEPT
{
    if ( m_pData == NULL ) return NULL;
    if ( size < sizeof( detail::MemoryFileStream ) ) return NULL;
    detail::MemoryFileStream* stream = new( buffer ) detail::MemoryFileStream(
        util::ConstBytePtr( m_pData, begin ).Get(), length );
    return stream;
}

detail::fnd::FileStream* MemorySoundArchive::OpenExtStream(
        void* buffer,
        size_t size,
        const char* extFilePath,
        void* cacheBuffer,
        size_t cacheSize ) const NN_NOEXCEPT
{
    NN_UNUSED(buffer);
    NN_UNUSED(size);
    NN_UNUSED(extFilePath);
    NN_UNUSED(cacheBuffer);
    NN_UNUSED(cacheSize);

    NN_ATK_WARNING("Cannot OpenExtStream for MemorySoundArchive");

    return NULL;
}

size_t MemorySoundArchive::detail_GetRequiredStreamBufferSize() const NN_NOEXCEPT
{
    return sizeof( detail::MemoryFileStream );
}

} // namespace nn::atk
} // namespace nn

