﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_FsSoundArchive.h>

#include <algorithm>
#include <nn/atk/atk_Util.h>
#include <nn/atk/detail/atk_Macro.h>
#include <nn/atk/fnd/io/atkfnd_FileStreamProxy.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

// #define NN_ATK_DEBUG_PRINT_ENABLE

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT( const int FsSoundArchive::BufferAlignSize );

/*--------------------------------------------------------------------------------*
  Name:         FsSoundArchive

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
FsSoundArchive::FsSoundArchive() NN_NOEXCEPT
: m_IsOpened( false )
, m_FileAccessMode( FileAccessMode_Always )
, m_FileAccessCount( 0 )
{
}

/*--------------------------------------------------------------------------------*
  Name:         ~FsSoundArchive

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
  *--------------------------------------------------------------------------------*/
FsSoundArchive::~FsSoundArchive() NN_NOEXCEPT
{
    Close();
}

bool FsSoundArchive::Open( const char* path ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(path);

    detail::fnd::FndResult openResult = m_FileStream.Open( path, detail::fnd::FileStream::AccessMode_Read );
    if ( openResult.IsFailed() )
    {
        return false;
    }
    m_FileAccessCount++;
    m_IsOpened = true;

    bool result = LoadFileHeader();
    if ( result == false )
    {
        NN_ATK_WARNING("Cannot load header");
        return false;
    }

    FileAccessEnd();

    char pathDir[FilePathMax];
    {
        pathDir[0] = '\0';
        int pathLen = static_cast<int>(std::strlen( path ));
        NN_SDK_ASSERT( pathLen < FilePathMax );
        if ( pathLen >= FilePathMax )
        {
            NN_ATK_WARNING("pathLen(%d) >= FilePathMax(%d)", pathLen, FilePathMax);
            return false;
        }

        for ( auto i = pathLen - 1; i >= 0; i-- )
        {
            const char ch = path[ i ] ;
            if ( ch == '/' || ch == '\\' )
            {
                util::Strlcpy( pathDir, path, std::min(FilePathMax, i + 1) );
                pathDir[i] = '\0';
                break;
            }
        }
    }

    SetExternalFileRoot( pathDir );

    // サウンドアーカイブのパスを保存
    {
        nn::util::SNPrintf(m_SoundArchiveFullPath, FilePathMax, "%s", path);
    }

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Close

  Description:  サウンドアーカイブを閉じる

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void FsSoundArchive::Close() NN_NOEXCEPT
{
    FileAccessBegin();

    if ( m_IsOpened )
    {
        m_FileStream.Close();
        m_IsOpened = false;
    }

    Finalize();
}

detail::fnd::FileStream*
FsSoundArchive::OpenStream( void* buffer, size_t size, detail::position_t begin, size_t length ) const NN_NOEXCEPT
{
    if ( ! m_IsOpened )
    {
        return NULL;
    }

    if ( size < sizeof( detail::fnd::FileStreamProxy ) )
    {
        return NULL;
    }

    detail::fnd::FileStream* stream = new ( buffer ) detail::fnd::FileStreamProxy( &m_FileStream, begin, length );

    //FsFileStream* stream = new( buffer ) FsFileStream( &m_FileStream, begin, length );

    return stream;
}

detail::fnd::FileStream*
FsSoundArchive::OpenExtStream(
    void* buffer,
    size_t size,
    const char* extFilePath,
    void* cacheBuffer,
    size_t cacheSize ) const NN_NOEXCEPT
{
    if ( ! m_IsOpened )
    {
        return NULL;
    }

    if ( size < sizeof( detail::fnd::FileStreamImpl ) )
    {
        return NULL;
    }

    detail::fnd::FileStream* fileStream = new( buffer ) detail::fnd::FileStreamImpl();
    NN_SDK_ASSERT_NOT_NULL(fileStream);
    fileStream->Open( extFilePath, detail::fnd::FileStream::AccessMode_Read );

    if ( !fileStream->IsOpened() )
    {
        return NULL;
    }

    if ((cacheBuffer != NULL) && (cacheSize > 0))
    {
        fileStream->EnableCache(cacheBuffer, cacheSize);
    }

    return fileStream;
}

size_t FsSoundArchive::detail_GetRequiredStreamBufferSize() const NN_NOEXCEPT
{
    return sizeof( detail::fnd::FileStream );
}

/*--------------------------------------------------------------------------------*
  Name:         LoadFileHeader

  Description:  サウンドアーカイブファイルのヘッダをロードする

  Arguments:    None.

  Returns:      成功したら true 失敗したら false
 *--------------------------------------------------------------------------------*/
bool FsSoundArchive::LoadFileHeader() NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsOpened );

    const int Align = 256;

    const size_t headerAlignSize =
        nn::util::align_up( sizeof(detail::SoundArchiveFile::FileHeader), Align );
    char headerArea[ sizeof(detail::SoundArchiveFile::FileHeader) + Align * 2 ];
    void* file = util::BytePtr( headerArea ).AlignUp( Align ).Get();

    size_t readSize = m_FileStream.Read( file, headerAlignSize );
    if ( readSize != headerAlignSize )
    {
        NN_ATK_WARNING(
            "FsSoundArchive::LoadFileHeader cannot read file.\n address(%08x) readSize(%d) != headerAlignSize(%d)",
            file, readSize, headerAlignSize
        );
        return false;
    }

    m_ArchiveReader.Initialize( file );
    Initialize( &m_ArchiveReader );

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         LoadHeader

  Description:  サウンドデータの情報テーブルをロードする

  Arguments:    buffer - ロードアドレス
                size - バッファサイズ

  Returns:      成功したら true 失敗したら false
 *--------------------------------------------------------------------------------*/
bool FsSoundArchive::LoadHeader( void* buffer, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsOpened );

    const int32_t infoChunkOffset = m_ArchiveReader.GetInfoBlockOffset();
    const uint32_t infoChunkSize = m_ArchiveReader.GetInfoBlockSize();

    if ( size < infoChunkSize )
    {
        NN_ATK_WARNING("FsSoundArchive::LoadHeader buffer size is too small.");
        return false;
    }

    //-----------------------------------------------------------------------------
    // 情報テーブルのロード

    FileAccessBegin();
    {
        m_FileStream.Seek( infoChunkOffset, detail::fnd::Stream::SeekOrigin::SeekOrigin_Begin );
        size_t readSize = m_FileStream.Read( buffer, infoChunkSize );

        if ( readSize != static_cast<size_t>(infoChunkSize) )
        {
            NN_ATK_WARNING("FsSoundArchive::LoadHeader cannot read file.");
            return false;
        }
    }
    FileAccessEnd();

    m_ArchiveReader.SetInfoBlock( buffer /*, infoChunkSize */ );

#ifdef NN_ATK_DEBUG_PRINT_ENABLE
    // デバッグ出力
    {
        // サウンド情報
        NN_DETAIL_ATK_INFO("### Sound INFO(%d)\n", m_ArchiveReader.GetSoundCount() );
        for ( uint32_t i = 0; i < m_ArchiveReader.GetSoundCount(); i++ )
        {
            // サウンド共通情報
            SoundArchive::ItemId soundId = GetSoundIdFromIndex( i );
            SoundArchive::SoundType type = m_ArchiveReader.GetSoundType( soundId );
            SoundArchive::SoundInfo info;
            bool ret = m_ArchiveReader.ReadSoundInfo( soundId, &info );
            NN_DETAIL_ATK_INFO("[%08X]?(%d) fileId(%d) playerId(0x%08X) plPrio(%d) actorId(%d) type(%d)\n",
                    soundId, ret, info.fileId, info.playerId, info.playerPriority, info.actorPlayerId, type );
            NN_DETAIL_ATK_INFO("  *common* volume(%d) panMode(%d) panCurve (%d)\n",
                    info.volume, info.panMode, info.panCurve );

            // ユーザーパラメータ
            {
                char buf[256]; buf[0] = '\0';
                for ( int i = 0; i <= SoundArchive::USER_PARAM_INDEX_MAX; i++ )
                {
                    uint32_t param;
                    bool result = m_ArchiveReader.ReadSoundUserParam( &param, soundId, i );
                    nn::util::SNPrintf( buf, 256, "%s [%d](0x%08x)", buf, result, param );
                }
                NN_DETAIL_ATK_INFO("  *userparam* %s\n", buf);
            }

            // 3D サウンド情報
            {
                SoundArchive::Sound3DInfo info3d;
                if ( ReadSound3DInfo( &info3d, soundId ) )
                {
                    uint32_t flag = info3d.flags;
                    NN_DETAIL_ATK_INFO("  *3D* ra(%.2f) cv(%d) df(%d) vol(%d) pr(%d) pa(%d) sp(%d) bqf(%d)\n",
                            info3d.decayRatio, info3d.decayCurve, info3d.dopplerFactor,
                            (flag & Sound3DInfo::FlagControl_Volume) > 0,
                            (flag & Sound3DInfo::FlagControl_Priority) > 0,
                            (flag & Sound3DInfo::FlagControl_Pan) > 0,
                            (flag & Sound3DInfo::FlagControl_SurroundPan) > 0,
                            (flag & Sound3DInfo::FlagControl_Filter) > 0 );
                }
                else
                {
                    NN_DETAIL_ATK_INFO("  *3D* data not found\n");
                }
            }

            // サウンド別個別情報
            switch ( type )
            {
            case SoundArchive::SoundType_Sequence:
            {
                SoundArchive::SequenceSoundInfo seqInfo;
                bool retDetail = m_ArchiveReader.ReadSequenceSoundInfo( soundId, &seqInfo );
                NN_DETAIL_ATK_INFO(" *SEQ* ret(%d) ofs(%d) bnk(0x%08X,0x%08X,0x%08X,0x%08X) trk(0x%x) chPrio(%d) rPrioFix(%d)\n",
                        retDetail,
                        seqInfo.startOffset,
                        seqInfo.bankIds[0], seqInfo.bankIds[1],
                        seqInfo.bankIds[2], seqInfo.bankIds[3],
                        seqInfo.allocateTrackFlags,
                        seqInfo.channelPriority, seqInfo.isReleasePriorityFix );
            }
            break;
            case SoundArchive::SoundType_Stream:
            {
                SoundArchive::StreamSoundInfo strmInfo;
                bool retDetail = m_ArchiveReader.ReadStreamSoundInfo( soundId, &strmInfo );
                NN_DETAIL_ATK_INFO("  *STRM* ret(%d) trk(0x%08x) channel(%d) pitch(%f) mainSend(%d) fxSend(%d,%d,%d)\n",
                        retDetail, strmInfo.allocateTrackFlags, strmInfo.allocateChannelCount,
                        strmInfo.pitch, strmInfo.mainSend, strmInfo.fxSend[AuxBus_A], strmInfo.fxSend[AuxBus_B], strmInfo.fxSend[AuxBus_C] );
                for (uint32_t i = 0; i < SoundArchive::StreamTrackCount; i++)
                {
                    NN_DETAIL_ATK_INFO("  trk[%d] vol(%3d) pan(%3d) span(%3d) flags(0x%04x) mainSend(%d) fxSend(%d,%d,%d) lpf(%d) bqf(%d,%d) chCount(%d) globalCh(%2d,%2d)\n",
                            i, strmInfo.trackInfo[i].volume, strmInfo.trackInfo[i].pan,
                            strmInfo.trackInfo[i].surroundPan, strmInfo.trackInfo[i].flags,
                            strmInfo.trackInfo[i].mainSend,
                            strmInfo.trackInfo[i].fxSend[0],
                            strmInfo.trackInfo[i].fxSend[1],
                            strmInfo.trackInfo[i].fxSend[2],
                            strmInfo.trackInfo[i].lowPassFilterFrequency,
                            strmInfo.trackInfo[i].biquadType,
                            strmInfo.trackInfo[i].biquadValue,
                            strmInfo.trackInfo[i].channelCount,
                            strmInfo.trackInfo[i].globalChannelIndex[0],
                            strmInfo.trackInfo[i].globalChannelIndex[1]);
                }
            }
            break;
            case SoundArchive::SoundType_Wave:
            {
                SoundArchive::WaveSoundInfo wsdInfo;
                bool retDetail = m_ArchiveReader.ReadWaveSoundInfo( soundId, &wsdInfo );
                NN_DETAIL_ATK_INFO("  *WSD* ret(%d) index(%d) trk(0x%x) chPrio(%d) rPrioFix(%d)\n",
                        retDetail,
                        wsdInfo.index, wsdInfo.allocateTrackCount,
                        wsdInfo.channelPriority, wsdInfo.isReleasePriorityFix );
            }
            break;
            case SoundArchive::SoundType_Invalid:
            {
                NN_DETAIL_ATK_INFO("Invalid SoundType (not STRM/WSD/SEQ)\n");
            }
            break;
            default:
                break;
            }
        }

        // バンク情報
        NN_DETAIL_ATK_INFO("### BANK Info(%d)\n", m_ArchiveReader.GetBankCount() );
        for ( uint32_t i = 0; i < m_ArchiveReader.GetBankCount(); i++ )
        {
            SoundArchive::ItemId bankId = GetBankIdFromIndex( i );
            SoundArchive::BankInfo info;
            bool ret = m_ArchiveReader.ReadBankInfo( bankId, &info );
            NN_DETAIL_ATK_INFO("[%08X]?(%d) fileId(%d)\n", bankId, ret, info.fileId );
        }

        // プレイヤー情報
        NN_DETAIL_ATK_INFO("### PLAYER Info(%d)\n", m_ArchiveReader.GetPlayerCount() );
        for ( uint32_t i = 0; i < m_ArchiveReader.GetPlayerCount(); i++ )
        {
            SoundArchive::ItemId playerId = GetPlayerIdFromIndex( i );
            SoundArchive::PlayerInfo info;
            bool ret = m_ArchiveReader.ReadPlayerInfo( playerId, &info );
            NN_DETAIL_ATK_INFO("[%08X]?(%d) max(%d) heapSize(%d)\n",
                    playerId, ret, info.playableSoundMax, info.playerHeapSize );
        }

        // サウンドグループ情報
        NN_DETAIL_ATK_INFO("### SOUND-GROUP Info(%d)\n", m_ArchiveReader.GetSoundGroupCount() );
        for ( uint32_t i = 0; i < m_ArchiveReader.GetSoundGroupCount(); i++ )
        {
            SoundArchive::ItemId soundGroupId = GetSoundGroupIdFromIndex( i );
            SoundArchive::SoundGroupInfo info;
            bool ret = m_ArchiveReader.ReadSoundGroupInfo( soundGroupId, &info );
            NN_DETAIL_ATK_INFO("[%08X]?(%d) startId(%08X) end(%08X) fileId:count(%d)",
                    soundGroupId, ret, info.startId, info.endId, info.fileIdTable->count );
            for ( uint32_t j = 0; j < info.fileIdTable->count; j++ )
            {
                NN_DETAIL_ATK_INFO(" [%08X]", info.fileIdTable->item[j]);
            }
            NN_DETAIL_ATK_INFO("\n");
        }

        // グループ情報
        NN_DETAIL_ATK_INFO("### GROUP Info(%d)\n", m_ArchiveReader.GetGroupCount() );
        for ( uint32_t i = 0; i < m_ArchiveReader.GetGroupCount(); i++ )
        {
            SoundArchive::ItemId groupId = GetGroupIdFromIndex( i );
            SoundArchive::GroupInfo info;
            bool ret = m_ArchiveReader.ReadGroupInfo( groupId, &info );
            NN_DETAIL_ATK_INFO("[%08X]?(%d) fileId(%4d) fileSize(%8d:0x%08x)\n",
                    groupId, ret, info.fileId, info.groupFileSize, info.groupFileSize );
        }

        // 波形アーカイブ情報
        NN_DETAIL_ATK_INFO("### WAVE-ARCHIVE Info(%d)\n", m_ArchiveReader.GetWaveArchiveCount() );
        for ( uint32_t i = 0; i < m_ArchiveReader.GetWaveArchiveCount(); i++ )
        {
            SoundArchive::ItemId warcId = GetWaveArchiveIdFromIndex( i );
            SoundArchive::WaveArchiveInfo info;
            bool ret = m_ArchiveReader.ReadWaveArchiveInfo( warcId, &info );
            NN_DETAIL_ATK_INFO("[%08X]?(%d) fileId(%d)\n", warcId, ret, info.fileId );
        }

        // ファイル情報
        NN_DETAIL_ATK_INFO("### FILE Info(%d)\n", m_ArchiveReader.GetFileCount() );
        for ( uint32_t i = 0; i < m_ArchiveReader.GetFileCount(); i++ )
        {
            SoundArchive::FileInfo info;
            bool ret = m_ArchiveReader.ReadFileInfo( i, &info );
            if ( info.externalFilePath != NULL )
            {
                NN_DETAIL_ATK_INFO("[%4d]?(%d) fileSize(%8d) ofs(%8d) path(%s)\n", i, ret,
                        info.fileSize, info.offsetFromFileBlockHead, info.externalFilePath );
            }
            else
            {
                NN_DETAIL_ATK_INFO("[%4d]?(%d) fileSize(%8d) ofs(%8d) path((null))\n", i, ret,
                        info.fileSize, info.offsetFromFileBlockHead );
            }
        }

        // サウンドアーカイブプレイヤー情報
        NN_DETAIL_ATK_INFO("### SOUND-ARCHIVE-PLAYER Info\n");
        {
            SoundArchive::SoundArchivePlayerInfo info;
            bool ret = m_ArchiveReader.ReadSoundArchivePlayerInfo( &info );
            NN_DETAIL_ATK_INFO("sequenceSoundMax (%2d)\n", info.sequenceSoundMax );
            NN_DETAIL_ATK_INFO("sequenceTrackMax (%2d)\n", info.sequenceTrackMax );
            NN_DETAIL_ATK_INFO("streamSoundMax   (%2d)\n", info.streamSoundMax );
            NN_DETAIL_ATK_INFO("streamTrackMax   (%2d)\n", info.streamTrackMax );
            NN_DETAIL_ATK_INFO("streamChannelMax (%2d)\n", info.streamChannelMax );
            NN_DETAIL_ATK_INFO("waveSoundMax     (%2d)\n", info.waveSoundMax );
            NN_DETAIL_ATK_INFO("waveTrackMax     (%2d)\n", info.waveTrackMax );
            NN_DETAIL_ATK_INFO("streamBufferTimes(%2d)\n", info.streamBufferTimes );
        }
    }
#endif /* NN_ATK_DEBUG_PRINT_ENABLE */

    return true;
} // NOLINT(impl/function_size)

/*--------------------------------------------------------------------------------*
  Name:         LoadLabelStringData

  Description:  ラベルデータをロードする

  Arguments:    buffer - ロードアドレス
                size - バッファサイズ

  Returns:      成功したら true 失敗したら false
 *--------------------------------------------------------------------------------*/
bool FsSoundArchive::LoadLabelStringData( void* buffer, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( m_IsOpened );

    const int32_t stringBlockOffset = m_ArchiveReader.GetStringBlockOffset();
    const uint32_t stringBlockSize = m_ArchiveReader.GetStringBlockSize();

    if ( stringBlockOffset == detail::Util::Reference::InvalidOffset )
    {
        // サウンドアーカイブの文字列ブロックが含まれていない
        return false;
    }

    if ( size < stringBlockSize )
    {
        NN_ATK_WARNING("FsSoundArchive::LoadLabelStringData buffer size is too small.");
        return false;
    }

    FileAccessBegin();
    {
        m_FileStream.Seek( stringBlockOffset, detail::fnd::Stream::SeekOrigin::SeekOrigin_Begin );
        size_t readSize = m_FileStream.Read( buffer, stringBlockSize );
        if ( readSize != stringBlockSize )
        {
            NN_ATK_WARNING("FsSoundArchive::LoadLabelStringData cannot read file.");
            return false;
        }
    }
    FileAccessEnd();

    m_ArchiveReader.SetStringBlock( buffer/*, stringBlockSize*/ );

#ifdef NN_ATK_DEBUG_PRINT_ENABLE
    // デバッグ出力
    {
        NN_DETAIL_ATK_INFO("### PATRICIA-TREE Info\n");
        m_ArchiveReader.DumpTree();

        NN_DETAIL_ATK_INFO("### LABEL => ID\n");
        int count = m_ArchiveReader.GetStringCount();
        for ( int i = 0; i < count; i++ )
        {
            const char* str = m_ArchiveReader.GetString( i );
            NN_DETAIL_ATK_INFO("[%02d] (%-16s) => ItemId(0x%08X)\n", i, str, GetItemId( str ) );
#if 0
            NN_DETAIL_ATK_INFO("     as Sound ID:       (0x%08X)\n", GetSoundId( str ) );
            NN_DETAIL_ATK_INFO("     as Bank ID:        (0x%08X)\n", GetBankId( str ) );
            NN_DETAIL_ATK_INFO("     as Player ID:      (0x%08X)\n", GetPlayerId( str ) );
            NN_DETAIL_ATK_INFO("     as SoundGroup ID:  (0x%08X)\n", GetSoundGroupId( str ) );
            NN_DETAIL_ATK_INFO("     as Group ID:       (0x%08X)\n", GetGroupId( str ) );
            NN_DETAIL_ATK_INFO("     as WaveArchive ID: (0x%08X)\n", GetWaveArchiveId( str ) );
#endif
        }

        NN_DETAIL_ATK_INFO("### ID => LABEL\n");
        NN_DETAIL_ATK_INFO("[Sound]\n");
        for ( uint32_t i = 0; i < GetSoundCount(); i++ )
        {
            uint32_t id = GetSoundIdFromIndex( i );
            NN_DETAIL_ATK_INFO("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NN_DETAIL_ATK_INFO("[Bank]\n");
        for ( uint32_t i = 0; i < GetBankCount(); i++ )
        {
            uint32_t id = GetBankIdFromIndex( i );
            NN_DETAIL_ATK_INFO("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NN_DETAIL_ATK_INFO("[Player]\n");
        for ( uint32_t i = 0; i < GetPlayerCount(); i++ )
        {
            uint32_t id = GetPlayerIdFromIndex( i );
            NN_DETAIL_ATK_INFO("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NN_DETAIL_ATK_INFO("[SoundGroup]\n");
        for ( uint32_t i = 0; i < GetSoundGroupCount(); i++ )
        {
            uint32_t id = GetSoundGroupIdFromIndex( i );
            NN_DETAIL_ATK_INFO("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NN_DETAIL_ATK_INFO("[Group]\n");
        for ( uint32_t i = 0; i < GetGroupCount(); i++ )
        {
            uint32_t id = GetGroupIdFromIndex( i );
            NN_DETAIL_ATK_INFO("  [%08X] (%s)\n", id, GetItemLabel(id) );
        }
        NN_DETAIL_ATK_INFO("[WaveArchive]\n");
        for ( uint32_t i = 0; i < GetWaveArchiveCount(); i++ )
        {
            uint32_t id = GetWaveArchiveIdFromIndex( i );
            const char* label = GetItemLabel( id );
            if ( label != NULL )
            {
                NN_DETAIL_ATK_INFO("  [%08X] (%s)\n", id, label );
            }
            else
            {
                NN_DETAIL_ATK_INFO("  [%08X] ((anonymous))\n", id );
            }
        }
    }
#endif /* NN_ATK_DEBUG_PRINT_ENABLE */

    return true;
}


void FsSoundArchive::FileAccessBegin() const NN_NOEXCEPT
{
    if ( m_FileAccessMode == FileAccessMode_InFunction )
    {
        //  複数スレッドから FileAccessBegin() が呼ばれたとき、
        //  m_FileStream.Open が 2 回呼ばれることがあるため排他制御します
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_FileOpenCloseLock );
        if ( m_FileAccessCount == 0 )
        {
            m_FileStream.Open( m_SoundArchiveFullPath, detail::fnd::FileStream::AccessMode_Read );
        }
        m_FileAccessCount++;
    }
}

void FsSoundArchive::FileAccessEnd() const NN_NOEXCEPT
{
    if ( m_FileAccessMode == FileAccessMode_InFunction )
    {
        nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_FileOpenCloseLock );
        if ( m_FileAccessCount == 1 )
        {
            m_FileStream.Close();
        }
        if ( m_FileAccessCount > 0 )
        {
            m_FileAccessCount--;
        }
    }
}

} // namespace nn::atk
} // namespace nn

