﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_ThreadCommon.h>

#include <nn/font/font_ScalableFont.h>
#include <nn/font/font_TextureCache.h>

#include <nn/ui2d/viewer/ui2d_Config.h>
#if defined(NN_UI2D_VIEWER_ENABLED)

#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ui2d.h>
#include <nn/font.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/ui2d/viewer/win/ui2d_DirResourceAccessorWin.h>
#include <nn/ui2d/viewer/win/ui2d_ScreenShotWin.h>
#else
#include <nn/ui2d/viewer/ui2d_FindableArcResourceAccessor.h>
#endif

#include <cstdarg>

namespace nn
{
namespace ui2d
{
//----------------------------------------------------------
// 注意：ステートマシン機能はライブラリ内部利用のため、利用しないでください。
//----------------------------------------------------------

//----------------------------------------------------------
// イベント種類
enum StateMachineEventKind
{
    StateMachineEventKind_None,
    StateMachineEventKind_Hit,
    StateMachineEventKind_Decided,
    StateMachineEventKind_StateTransitionCompleted,
    StateMachineEventKind_VariableChanged,
    StateMachineEventKind_StateChangeRequested,
    StateMachineEventKind_Max
};

//----------------------------------------------------------
// イベント
struct StateMachineEvent
{
    //! @brief 内部用機能のため使用禁止です。
    nn::util::IntrusiveListNode m_Link;

    uint32_t kind;
    uint64_t param1;
    uint64_t param2;
    uint32_t delayFrames;
};

typedef nn::util::IntrusiveList<StateMachineEvent, nn::util::IntrusiveListMemberNodeTraits<StateMachineEvent, &StateMachineEvent::m_Link> > StateMachineEventSet;

//----------------------------------------------------------
// イベントキュー
struct StateMachineEventQueue
{
    static const int Max = 10;

    StateMachineEventQueue()
        : m_pStateMachineEventArray(nullptr)
    {
    }

    //----------------------------------------------------------
    void Initialize()
    {
        m_pStateMachineEventArray = nn::ui2d::Layout::NewArray<StateMachineEvent>(Max);
        for (int i = 0; i < Max; i++)
        {
            m_FreeEvents.push_back(m_pStateMachineEventArray[i]);
        }

        Reset();
    }

    //----------------------------------------------------------
    void Finalize()
    {
        nn::ui2d::Layout::DeleteArray(m_pStateMachineEventArray, Max);
        m_pStateMachineEventArray = nullptr;
    }

    //----------------------------------------------------------
    void Reset()
    {
        for (auto iter = m_StateMachineEvents.begin(); iter != m_StateMachineEvents.end(); )
        {
            auto currIter = iter;
            iter++;

            m_FreeEvents.push_back(*currIter);
        }
    }

    //----------------------------------------------------------
    void Push(StateMachineEventKind kind, uint64_t param1, uint64_t param2, uint32_t delayFrames)
    {
        if (m_FreeEvents.empty())
        {
            NN_SDK_ASSERT(false);
            return;
        }

        // フリーリストから取って、後ろに追加
        auto iter = m_FreeEvents.begin();
        m_FreeEvents.pop_front();

        iter->kind = kind;
        iter->param1 = param1;
        iter->param2 = param2;
        iter->delayFrames = delayFrames;

        m_StateMachineEvents.push_back(*iter);
    }

    //----------------------------------------------------------
    StateMachineEvent* Peek()
    {
        if (m_StateMachineEvents.empty())
        {
            return nullptr;
        }

        return &m_StateMachineEvents.front();
    }

    //----------------------------------------------------------
    StateMachineEvent* Bottom()
    {
        if (m_StateMachineEvents.empty())
        {
            return nullptr;
        }

        return &m_StateMachineEvents.back();
    }

    //----------------------------------------------------------
    void Pop()
    {
        if (m_StateMachineEvents.empty())
        {
            return;
        }

        auto iter = m_StateMachineEvents.begin();
        m_StateMachineEvents.pop_front();
        m_FreeEvents.push_back(*iter);
    }

    //----------------------------------------------------------
    void MoveTopToTail()
    {
        StateMachineEvent* pHead = Peek();
        if (pHead != nullptr)
        {
            m_StateMachineEvents.pop_front();
            m_StateMachineEvents.push_back(*pHead);
        }
    }

    //----------------------------------------------------------
    StateMachineEvent*   m_pStateMachineEventArray;
    StateMachineEventSet m_StateMachineEvents;
    StateMachineEventSet m_FreeEvents;
};

//----------------------------------------------------------
struct FeatureParamaterType
{
    //----------------------------------------------------------
    FeatureParamaterType()
        : m_ResAnimationInfoType(0)
        , m_ParamTypeCount(0)
        , m_pParamTypes(nullptr)
    {
    }

    //----------------------------------------------------------
    uint32_t m_ResAnimationInfoType; //  AnimationTypeMaterialColor など。

    int      m_ParamTypeCount;
    uint8_t* m_pParamTypes; //  nn::ui2d::AnimTargetPaneColor(AnimTargetPaneColor_VertexLtR) など。
};

//----------------------------------------------------------
// 状態が持つ「特徴」パラメーター
struct FeatureParamater
{
public:

    //----------------------------------------------------------
    FeatureParamater()
        : m_pName(nullptr)
        , m_AnimContentType(0)
        , m_FeatureParamaterTypeCount(0)
        , m_pFeatureParamaterTypes(nullptr)
    {
    }

    //----------------------------------------------------------
    ~FeatureParamater()
    {
    }

    //----------------------------------------------------------
    // TODO★
    void Initialzie(const char*pName, AnimContentType contentType, uint32_t animationType, int countParamTypes, const uint8_t* pParamTypes)
    {
        const int size = nn::util::Strnlen(pName, CountOfStateMachineName) + 1;
        char* pBuf = static_cast<char*>(nn::ui2d::Layout::AllocateMemory(size));
        nn::util::Strlcpy(pBuf, pName, size);

        m_pName = static_cast<const char*>(pBuf);

        m_AnimContentType = static_cast<uint8_t>(contentType); // 例: nn::ui2d::AnimContentType_Material

        // 暫定で、Count = 1 固定で作っておく。
        m_FeatureParamaterTypeCount = 1;
        m_pFeatureParamaterTypes = nn::ui2d::Layout::NewArray<FeatureParamaterType>(1);

        // 暫定で、Count = 1 固定で作っておく。
        m_pFeatureParamaterTypes[0].m_ResAnimationInfoType = animationType; // 例: nn::ui2d::AnimationTypeMaterialColor
        m_pFeatureParamaterTypes[0].m_ParamTypeCount = countParamTypes;
        m_pFeatureParamaterTypes[0].m_pParamTypes = nn::ui2d::Layout::NewArray<uint8_t>(countParamTypes);
        for (int i = 0; i < countParamTypes; i++)
        {
            m_pFeatureParamaterTypes[0].m_pParamTypes[i] = pParamTypes[i]; // 例：AnimTargetMatColor_BufferR
        }
    }

    //----------------------------------------------------------
    void Finalize()
    {
        nn::ui2d::Layout::FreeMemory(const_cast<void*>(static_cast<const void*>(m_pName)));
        m_pName = nullptr;

        // m_pFeatureParamaterTypes
        {
            for (int i = 0; i < m_FeatureParamaterTypeCount; i++)
            {
                nn::ui2d::Layout::DeleteArray(m_pFeatureParamaterTypes[i].m_pParamTypes, 1);
                m_pFeatureParamaterTypes[i].m_pParamTypes = nullptr;
            }
            nn::ui2d::Layout::DeleteArray(m_pFeatureParamaterTypes, 1);
            m_pFeatureParamaterTypes = nullptr;
        }
    }

    //----------------------------------------------------------
    //! @brief 内部用機能のため使用禁止です。
    nn::util::IntrusiveListNode m_Link;

    const char* m_pName;

    uint8_t m_AnimContentType;

    int m_FeatureParamaterTypeCount;
    FeatureParamaterType* m_pFeatureParamaterTypes;
};

// 状態が持つ「特徴」パラメーター（複数束ねて、特徴となる）
typedef nn::util::IntrusiveList<FeatureParamater, nn::util::IntrusiveListMemberNodeTraits<FeatureParamater, &FeatureParamater::m_Link> > FeatureParamaterSet;

//----------------------------------------------------------
// 「特徴」の実際値
struct FeatureParamaterLeafStore
{
    //----------------------------------------------------------
    FeatureParamaterLeafStore()
        : paramValue(0.0f)
    {
    }

    //----------------------------------------------------------
    float paramValue;
};

//----------------------------------------------------------
// 「特徴」の実際値
struct FeatureParamaterLeafStoreSet
{
    //----------------------------------------------------------
    FeatureParamaterLeafStoreSet()
        : count(0)
        , pLeafs(nullptr)
    {
    }

    //----------------------------------------------------------
    int count;
    FeatureParamaterLeafStore* pLeafs;
};

//----------------------------------------------------------
// 「特徴」の実際値
// FeatureParamaterStore は、FeatureParamater と同じ数だけ、同じ順番で格納されていると想定しています。
struct FeatureParamaterStore
{
    nn::util::IntrusiveListNode m_Link;

    // 実際の値。
    int count;
    FeatureParamaterLeafStoreSet* pLeafSetArray;
};

// ここに、文字列情報を持つことのできるバリエーションを設けます。
// 子供のステートマシンに対して、任意のコマンドを発行できるようにします。

// 「特徴」の実際値（束ねたもの）
typedef nn::util::IntrusiveList<FeatureParamaterStore, nn::util::IntrusiveListMemberNodeTraits<FeatureParamaterStore, &FeatureParamaterStore::m_Link> > FeatureParamaterStoreList;

//----------------------------------------------------------
// 「特徴」の実際値
class FeatureParamaterStoreSet : public FeatureParamaterStoreList
{
public:
    //----------------------------------------------------------
    const FeatureParamaterLeafStore* GetAt(int contentIdx, int infoIdx, int targetIdx) const
    {
        int count = 0;
        for (auto iter = this->begin(); iter != this->end(); iter++)
        {
            if (count == contentIdx)
            {
                return &iter->pLeafSetArray[infoIdx].pLeafs[targetIdx];
            }

            count++;
        }

        return nullptr;
    }

    void Initialzie(const FeatureParamaterSet& featureParamaterSet)
    {
        for (auto iter = featureParamaterSet.begin(); iter != featureParamaterSet.end(); iter++)
        {
            FeatureParamaterStore* pStore = nn::ui2d::Layout::AllocateAndConstruct<FeatureParamaterStore>();
            pStore->count = iter->m_FeatureParamaterTypeCount;
            pStore->pLeafSetArray = nn::ui2d::Layout::NewArray<FeatureParamaterLeafStoreSet>(pStore->count);
            for (int i = 0; i < pStore->count; i++)
            {
                const int leafParamCount = iter->m_pFeatureParamaterTypes[i].m_ParamTypeCount;
                pStore->pLeafSetArray[i].count = leafParamCount;
                pStore->pLeafSetArray[i].pLeafs = nn::ui2d::Layout::NewArray<FeatureParamaterLeafStore>(leafParamCount);
            }

            this->push_back(*pStore);
        }
    }

    void Finalize()
    {
        auto endIter = this->end();
        for (auto iter = this->begin(); iter != endIter; )
        {
            auto currIter = iter;
            iter++;

            this->erase(currIter);
            for (int i = 0; i < currIter->count; i++)
            {
                const int leafParamCount = currIter->pLeafSetArray[i].count;
                nn::ui2d::Layout::DeleteArray(currIter->pLeafSetArray[i].pLeafs, leafParamCount);
            }
            nn::ui2d::Layout::DeleteArray(currIter->pLeafSetArray, currIter->count);

            nn::ui2d::Layout::DeleteObj(&(*currIter));
        }
    }

    void SetData(int paramIndex, int storeCount, int elementIdx, float value)
    {
        int index = 0;
        for (auto iter = this->begin(); iter != this->end(); iter++)
        {
            if (paramIndex == index)
            {
                NN_SDK_ASSERT(storeCount < iter->count);
                NN_SDK_ASSERT(elementIdx < iter->pLeafSetArray[storeCount].count);

                iter->pLeafSetArray[storeCount].pLeafs[elementIdx].paramValue = value;
            }

            index++;
        }
    }
};

//----------------------------------------------------------
// ステート
struct State
{
    //----------------------------------------------------------
    State()
        : m_pName(nullptr)
    {
    }

    //----------------------------------------------------------
    ~State()
    {
    }

    //----------------------------------------------------------
    void Initialzie(const char*pName, const FeatureParamaterSet& featureParamaterSet)
    {
        const int size = nn::util::Strnlen(pName, CountOfStateMachineName) + 1;
        auto pBuf = nn::ui2d::Layout::AllocateMemory(size);
        nn::util::Strlcpy(static_cast<char*>(pBuf), pName, size);
        m_pName = static_cast<const char*>(pBuf);

        m_FeatureParamaterStoreSet.Initialzie(featureParamaterSet);
    }

    //----------------------------------------------------------
    void Finalize()
    {
        nn::ui2d::Layout::FreeMemory(const_cast<void*>(static_cast<const void*>(m_pName)));
        m_pName = nullptr;

        m_FeatureParamaterStoreSet.Finalize();
    }

    //----------------------------------------------------------
    void SetTestDataToStore(int paramIndex, int storeCount, float* pStore)
    {
        // テスト用に適当な値を入れています
        int index = 0;
        for (auto iter = m_FeatureParamaterStoreSet.begin(); iter != m_FeatureParamaterStoreSet.end(); iter++)
        {
            if (paramIndex == index)
            {
                for (int i = 0; i < iter->count; i++)
                {
                    for (int j = 0; j < iter->pLeafSetArray[i].count; j++)
                    {
                        float val = (storeCount > j) ? pStore[j] : 0.0f;
                        iter->pLeafSetArray[i].pLeafs[j].paramValue = val;
                    }
                }
            }

            index++;
        }
    }

    //----------------------------------------------------------
    nn::util::IntrusiveListNode m_Link;

    const char* m_pName;

    FeatureParamaterStoreSet m_FeatureParamaterStoreSet;
};

typedef nn::util::IntrusiveList<State, nn::util::IntrusiveListMemberNodeTraits<State, &State::m_Link> > StateSet;

//----------------------------------------------------------
// ステート変数型
enum StateMachineVariableType
{
    StateMachineVariableType_Bool,
    StateMachineVariableType_Float,
    StateMachineVariableType_Uint32,
    StateMachineVariableType_Uint64,
    StateMachineVariableType_Max,
};

//----------------------------------------------------------
// ステート演算
enum StateMachineOperator
{
    StateMachineOperator_GreaterThan,
    StateMachineOperator_GreaterEqual,
    StateMachineOperator_Equal,
    StateMachineOperator_NotEqual,
    StateMachineOperator_LessEqual,
    StateMachineOperator_LessThan,
    StateMachineOperator_AlwaysPass,
    StateMachineOperator_AlwaysFail,
    StateMachineOperator_Max
};

//----------------------------------------------------------
// ステート変数
struct StateMachineVariableContent
{
    //----------------------------------------------------------
    union
    {
        bool     boolValue;
        float    floatValue;
        uint32_t uint32Value;
        uint64_t uint64Value;
    } m_Value;

    //----------------------------------------------------------
    StateMachineVariableContent()
        : m_Type(StateMachineVariableType_Bool)
    {
        m_Value.uint64Value = 0;
    }

    //----------------------------------------------------------
    static bool Set(StateMachineVariableContent* pContent, StateMachineVariableType type, void* pValue)
    {
        pContent->m_Type = type;
        switch (pContent->m_Type)
        {
        case StateMachineVariableType_Bool:
            return SetAsBool(pContent, *reinterpret_cast<bool*>(pValue));
        case StateMachineVariableType_Float:
            return SetAsFloat(pContent, *reinterpret_cast<float*>(pValue));
        default:
            return false;
        }
    }

    //----------------------------------------------------------
    static bool SetAsFloat(StateMachineVariableContent* pContent, float newValue)
    {
        NN_SDK_ASSERT(pContent->m_Type == StateMachineVariableType_Float);

        if (newValue != pContent->m_Value.floatValue)
        {
            pContent->m_Value.floatValue = newValue;
            return true;
        }

        return false;
    }

    //----------------------------------------------------------
    static bool SetAsBool(StateMachineVariableContent* pContent, bool newValue)
    {
        NN_SDK_ASSERT(pContent->m_Type == StateMachineVariableType_Bool);

        if (newValue != pContent->m_Value.boolValue)
        {
            pContent->m_Value.boolValue = newValue;
            return true;
        }

        return false;
    }

    //----------------------------------------------------------
    template<class T> static bool Operator_GreaterThan_(T valA, T valB)
    {
        return valA > valB;
    }

    //----------------------------------------------------------
    template<class T> static bool Operator_LessThan_(T valA, T valB)
    {
        return valA < valB;
    }

    //----------------------------------------------------------
    static bool GetOperationResult(StateMachineOperator op, const StateMachineVariableContent& valA, const StateMachineVariableContent& valB)
    {
        // 型は一致している想定
        NN_SDK_ASSERT(valA.m_Type == valB.m_Type);

        switch (op)
        {
        case StateMachineOperator_GreaterThan:
            switch (valA.m_Type)
            {
            case StateMachineVariableType_Bool: return Operator_GreaterThan_(valA.m_Value.boolValue, valB.m_Value.boolValue);
            case StateMachineVariableType_Float: return Operator_GreaterThan_(valA.m_Value.floatValue, valB.m_Value.floatValue);
            default:
                break;
            }
            break;
        case StateMachineOperator_LessThan:
            switch (valA.m_Type)
            {
            case StateMachineVariableType_Bool: return Operator_LessThan_(valA.m_Value.boolValue, valB.m_Value.boolValue);
            case StateMachineVariableType_Float: return Operator_LessThan_(valA.m_Value.floatValue, valB.m_Value.floatValue);
            default:
                break;
            }
            break;
        default:
            break;
        }

        return false;
    }

    //----------------------------------------------------------
    StateMachineVariableType m_Type;
};

//----------------------------------------------------------
// ステート変数
struct StateMachineVariable
{
    nn::util::IntrusiveListNode m_Link;
    char                        m_Name[nn::ui2d::ResourceNameStrMax + 1];
    StateMachineVariableContent m_Content;
};

typedef nn::util::IntrusiveList<StateMachineVariable, nn::util::IntrusiveListMemberNodeTraits<StateMachineVariable, &StateMachineVariable::m_Link> > StateMachineVariableList;

//----------------------------------------------------------
struct StateMachineVariableManager
{
    //----------------------------------------------------------
    void Initialize(StateMachineEventQueue* pEventQueue)
    {
        m_pEventQueue = pEventQueue;
    }

    //----------------------------------------------------------
    void InitialzieVariable_(StateMachineVariable* pVariable, const char* pName, StateMachineVariableType contenType)
    {
        nn::util::Strlcpy(pVariable->m_Name, pName, sizeof(pVariable->m_Name));
        pVariable->m_Content.m_Type = contenType;
        pVariable->m_Content.m_Value.uint64Value = 0;
    }

    //----------------------------------------------------------
    void Finalize()
    {
        auto endIter = m_StateMachineVariableList.end();
        for (auto iter = m_StateMachineVariableList.begin(); iter != endIter; )
        {
            auto currIter = iter;
            iter++;

            m_StateMachineVariableList.erase(currIter);
            nn::ui2d::Layout::DeleteObj(&(*currIter));
        }
    }

    //----------------------------------------------------------
    bool RegisterNewVariable(const char* pName, StateMachineVariableType contenType)
    {
        if (FindByName_(pName) != nullptr)
        {
            return false;
        }

        auto* pNewValuable = nn::ui2d::Layout::AllocateAndConstruct<StateMachineVariable>();
        NN_SDK_ASSERT_NOT_NULL(pNewValuable);

        InitialzieVariable_(pNewValuable, pName, contenType);

        m_StateMachineVariableList.push_back(*pNewValuable);

        return true;
    }

    //----------------------------------------------------------
    // 目当ての変数が存在するかを調査することが可能。
    bool Contanis(const char* pName)
    {
        return FindByName_(pName) != nullptr;
    }

    //----------------------------------------------------------
    // StateMachineVariable は外に出さない。すべての編集操作をラップする。
    StateMachineVariable* FindByName_(const char* pName)
    {
        for (auto iter = m_StateMachineVariableList.begin(); iter != m_StateMachineVariableList.end(); iter++)
        {
            if (strcmp(iter->m_Name, pName) == 0)
            {
                return &(*iter);
            }
        }

        return nullptr;
    }

    //----------------------------------------------------------
    void PushModifyEvent_(const char* pName, StateMachineVariable* pVariable)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pEventQueue);

        uint64_t param1 = reinterpret_cast<uint64_t>(pName);
        uint64_t param2 = reinterpret_cast<uint64_t>(pVariable);

        m_pEventQueue->Push(StateMachineEventKind_VariableChanged, param1, param2, 1);
    }

    //----------------------------------------------------------
    void SetBoolValue(const char* pName, bool newValue)
    {
        auto* pVar = FindByName_(pName);

        // 存在すること、型が一致していること。。。を前提とする。
        NN_SDK_ASSERT_NOT_NULL(pVar);

        if (newValue != pVar->m_Content.m_Value.boolValue)
        {
            pVar->m_Content.m_Value.boolValue = newValue;
            PushModifyEvent_(pVar->m_Name, pVar);
        }
    }

    //----------------------------------------------------------
    void SetFloatValue(const char* pName, float newValue)
    {
        auto* pVar = FindByName_(pName);

        // 存在すること、型が一致していること。。。を前提とする。
        NN_SDK_ASSERT_NOT_NULL(pVar);

        if (newValue != pVar->m_Content.m_Value.floatValue)
        {
            pVar->m_Content.m_Value.floatValue = newValue;
            PushModifyEvent_(pVar->m_Name, pVar);
        }
    }

    //----------------------------------------------------------
    bool GetValueAsBool(const char* pName)
    {
        auto* pVar = FindByName_(pName);

        // 存在すること、型が一致していること。。。を前提とする。
        NN_SDK_ASSERT_NOT_NULL(pVar);
        NN_SDK_ASSERT(pVar->m_Content.m_Type == StateMachineVariableType_Bool);

        return pVar->m_Content.m_Value.boolValue;
    }

    //----------------------------------------------------------
    float GetValueAsFloat(const char* pName)
    {
        auto* pVar = FindByName_(pName);

        // 存在すること、型が一致していること。。。を前提とする。
        NN_SDK_ASSERT_NOT_NULL(pVar);
        NN_SDK_ASSERT(pVar->m_Content.m_Type == StateMachineVariableType_Float);

        return pVar->m_Content.m_Value.floatValue;
    }

    //----------------------------------------------------------
    StateMachineEventQueue*  m_pEventQueue;
    StateMachineVariableList m_StateMachineVariableList;
};

//----------------------------------------------------------
// 状態遷移条件
struct TransitionConditionBase
{
    //----------------------------------------------------------
    TransitionConditionBase()
    {
    }

    //----------------------------------------------------------
    virtual ~TransitionConditionBase()
    {
    }

    //----------------------------------------------------------
    // 状態遷移のトリガー条件を満たすか調査します。
    virtual bool IsTriggered(const StateMachineEvent& queue) = 0;

    //----------------------------------------------------------
    uint8_t  m_Type; // 種類
};

//----------------------------------------------------------
// 状態遷移条件
struct TransitionConditionNone : public TransitionConditionBase
{
    //----------------------------------------------------------
    TransitionConditionNone()
    {
    }

    //----------------------------------------------------------
    virtual ~TransitionConditionNone() NN_OVERRIDE
    {
        TransitionConditionBase::~TransitionConditionBase();
    }

    //----------------------------------------------------------
    virtual bool IsTriggered(const StateMachineEvent& eventObj) NN_OVERRIDE
    {
        NN_UNUSED(eventObj);
        return false;
    }
};

//----------------------------------------------------------
// 状態遷移条件
struct TransitionConditionStateChangeRequested : public TransitionConditionBase
{
    //----------------------------------------------------------
    explicit TransitionConditionStateChangeRequested(const char* pTargetStateName)
        : m_pTargetStateName(pTargetStateName)
    {
    }

    //----------------------------------------------------------
    virtual ~TransitionConditionStateChangeRequested() NN_OVERRIDE
    {
        TransitionConditionBase::~TransitionConditionBase();
    }

    //----------------------------------------------------------
    virtual bool IsTriggered(const StateMachineEvent& eventObj) NN_OVERRIDE
    {
        if (eventObj.kind == StateMachineEventKind_StateChangeRequested)
        {
            return std::strcmp(reinterpret_cast<const char*>(eventObj.param2), m_pTargetStateName) == 0;
        }

        return false;
    }

    const char* m_pTargetStateName;
};
//----------------------------------------------------------
// 状態遷移条件
struct TransitionConditionIsStateBegin : public TransitionConditionBase
{
    //----------------------------------------------------------
    TransitionConditionIsStateBegin()
        : pName(nullptr)
    {
    }

    //----------------------------------------------------------
    virtual ~TransitionConditionIsStateBegin() NN_OVERRIDE
    {
        TransitionConditionBase::~TransitionConditionBase();
    }

    //----------------------------------------------------------
    virtual bool IsTriggered(const StateMachineEvent& eventObj) NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(pName);

        if (eventObj.kind != StateMachineEventKind_StateTransitionCompleted)
        {
            return false;
        }

        if (strcmp(pName, reinterpret_cast<const char*>(eventObj.param1)) != 0)
        {
            return false;
        }

        return true;
    }

    //----------------------------------------------------------
    const char* pName;
};

//----------------------------------------------------------
// 状態遷移条件
struct TransitionConditionIsHit : public TransitionConditionBase
{
    TransitionConditionIsHit()
    {
    }

    virtual ~TransitionConditionIsHit() NN_OVERRIDE
    {
        TransitionConditionBase::~TransitionConditionBase();
    }

    virtual bool IsTriggered(const StateMachineEvent& eventObj) NN_OVERRIDE
    {
        return eventObj.kind == StateMachineEventKind_Hit;
    }
};

//----------------------------------------------------------
// 状態遷移条件
struct TransitionConditionIsDecided : public TransitionConditionBase
{
    TransitionConditionIsDecided()
    {
    }

    virtual ~TransitionConditionIsDecided() NN_OVERRIDE
    {
        TransitionConditionBase::~TransitionConditionBase();
    }

    virtual bool IsTriggered(const StateMachineEvent& eventObj) NN_OVERRIDE
    {
        return eventObj.kind == StateMachineEventKind_Decided;
    }
};

//----------------------------------------------------------
// 状態遷移条件：ステートマシン変数の変更
struct TransitionConditionVariableChanged : public TransitionConditionBase
{
    //----------------------------------------------------------
    TransitionConditionVariableChanged()
        : m_StateMachineOperator(StateMachineOperator_GreaterThan)
    {
    }

    //----------------------------------------------------------
    virtual ~TransitionConditionVariableChanged() NN_OVERRIDE
    {
        TransitionConditionBase::~TransitionConditionBase();
    }

    //----------------------------------------------------------
    void Initialzie(const char* pTargetName, StateMachineOperator stateMachineOperator)
    {
        nn::util::Strlcpy(m_Name, pTargetName, sizeof(m_Name));
        m_StateMachineOperator = stateMachineOperator;
    }

    void SetOperatorParamater(StateMachineVariableType type, void* pContent)
    {
        StateMachineVariableContent::Set(&m_Content, type, pContent);
    }

    //----------------------------------------------------------
    virtual bool IsTriggered(const StateMachineEvent& eventObj) NN_OVERRIDE
    {
        if (eventObj.kind != StateMachineEventKind_VariableChanged)
        {
            return false;
        }

        const char* pName = reinterpret_cast<const char*>(eventObj.param1);
        if (strcmp(pName, m_Name) != 0)
        {
            return false;
        }

        const StateMachineVariable* pVal = reinterpret_cast<const StateMachineVariable*>(eventObj.param2);
        NN_SDK_ASSERT_NOT_NULL(pVal);
        if (!StateMachineVariableContent::GetOperationResult(m_StateMachineOperator, pVal->m_Content, m_Content))
        {
            return false;
        }

        return true;
    }

    //----------------------------------------------------------

    char m_Name[nn::ui2d::ResourceNameStrMax + 1];

    StateMachineOperator                m_StateMachineOperator;
    StateMachineVariableContent         m_Content;
};

//----------------------------------------------------------
struct TransitionTiming
{
    //----------------------------------------------------------
    TransitionTiming()
        : offset(0.0f)
        , duration(0.0f)
        , easingType(0)
        , easingExtraParamater(0.0f)
    {
    }

    //----------------------------------------------------------
    float offset;
    float duration;

    uint8_t easingType;

    float easingExtraParamater;
};

//----------------------------------------------------------
struct TransitionTrackEvent
{
    float                 delay;
    StateMachineEventKind kind;

    const void* pParam1;
    const void* pParam2;
};

//----------------------------------------------------------
struct TransitionSection
{
    //----------------------------------------------------------
    TransitionSection()
        : isUsedForEvent(false)
    {
    }

    //----------------------------------------------------------
    TransitionTiming            timing;

    union
    {
        // キーフレームとして利用する場合
        FeatureParamaterLeafStore   store[4];
        // イベントとして利用する場合
        TransitionTrackEvent        trackEvent;
    };

    bool                        isUsedForEvent;
};

//----------------------------------------------------------
// トラック
struct TransitionTimeLineTrack
{
public:
    //----------------------------------------------------------
    TransitionTimeLineTrack()
        : m_TransitionSectionCount(0)
        , m_pTransitionSections(nullptr)
    {
    }

    void Initialize(int kyeCount)
    {
        NN_SDK_ASSERT(m_pTransitionSections == nullptr);
        m_TransitionSectionCount = kyeCount;
        m_pTransitionSections = nn::ui2d::Layout::NewArray<TransitionSection>(kyeCount);
    }

    void Finalize()
    {
        nn::ui2d::Layout::DeleteArray(m_pTransitionSections, m_TransitionSectionCount);
        m_pTransitionSections = nullptr;
        m_TransitionSectionCount = 0;
    }

    int CalculateEventCount() const
    {
        int eventKeyCount = 0;
        for (int sectionIdx = 0; sectionIdx < this->m_TransitionSectionCount; sectionIdx++)
        {
            if (this->m_pTransitionSections[sectionIdx].isUsedForEvent)
            {
                eventKeyCount++;
            }
        }

        return eventKeyCount;
    }

    //----------------------------------------------------------
    int GetSectionIndexFromTime(float time) const
    {
        int idx = -1;
        for (int i = 0; i < this->m_TransitionSectionCount; i++)
        {
            const auto& timing = this->m_pTransitionSections[i].timing;
            if (timing.offset <= time)
            {
                idx = i;
                if (timing.offset + timing.duration > time)
                {
                    break;
                }
            }
        }

        // 終端キーの最後より後ろかどうか
        if (idx != -1 && idx == this->m_TransitionSectionCount - 1)
        {
            if (m_Timing.offset + m_Timing.duration <= time)
            {
                idx++;
            }
        }

        return idx;
    }

    //----------------------------------------------------------
    const TransitionSection* GetSectionByIndex(int paramIdx) const
    {
        if (paramIdx == -1 || this->m_TransitionSectionCount <= paramIdx)
        {
            return nullptr;
        }

        return &m_pTransitionSections[paramIdx];
    }

    //----------------------------------------------------------
    void SetupParametrizedAnimationEvent(nn::ui2d::ResParameterizedAnimParameter* pParam, int sectionIdex) const
    {
        if (m_TransitionSectionCount == 0)
        {
            return;
        }

        pParam->parameterizedAnimType = ParameterizedAnimType_Event;

        // キーがある場合
        if (sectionIdex < 0)
        {
            // 先頭のキーより前
            pParam->duration = m_pTransitionSections[0].timing.offset - m_Timing.offset;
            pParam->offset = m_Timing.offset;

            pParam->pEvent->pParam1 = nullptr;
            pParam->pEvent->pParam2 = nullptr;
        }
        else if (sectionIdex >= m_TransitionSectionCount)
        {
            // 最後のキーより後ろ
            pParam->duration = (m_Timing.offset + m_Timing.duration) - m_pTransitionSections[sectionIdex - 1].timing.offset;
            pParam->offset = m_pTransitionSections[sectionIdex - 1].timing.offset;

            pParam->pEvent->pParam1 = nullptr;
            pParam->pEvent->pParam2 = nullptr;
        }
        else
        {
            // 中間区間
            const TransitionSection* pCurr = &m_pTransitionSections[sectionIdex];
            const TransitionSection* pNext = (sectionIdex + 1 < m_TransitionSectionCount) ? &m_pTransitionSections[sectionIdex + 1] : nullptr;

            const float endFrame = (pNext != nullptr) ? pNext->timing.offset : (m_Timing.offset + m_Timing.duration);

            pParam->duration = endFrame - pCurr->timing.offset;
            pParam->offset = pCurr->timing.offset;

            pParam->pEvent->pParam1 = pCurr->trackEvent.pParam1;
            pParam->pEvent->pParam2 = pCurr->trackEvent.pParam2;
        }
    }

    //----------------------------------------------------------
    void SetupParametrizedAnimation(nn::ui2d::ResParameterizedAnimParameter* pParam, int targetIdx, int sectionIdex) const
    {
        const TransitionTiming*      pTiming = nullptr;
        if (m_TransitionSectionCount == 0)
        {
            // キーが無い場合
            pParam->duration = m_Timing.duration;
            pParam->offset = m_Timing.offset;
            pParam->parameterizedAnimType = m_Timing.easingType;

            pTiming = &m_Timing;
        }
        else
        {
            // キーがある場合
            if (sectionIdex < 0)
            {
                // 先頭のキーより前
                pParam->duration = m_pTransitionSections[0].timing.offset - m_Timing.offset;
                pParam->offset = m_Timing.offset;
                pParam->parameterizedAnimType = m_Timing.easingType;

                // pParam->startValue は、Transition の Start を採用
                pParam->value.targetValue = m_pTransitionSections[0].store[targetIdx].paramValue;

                pTiming = &m_Timing;
            }
            else if (sectionIdex >= m_TransitionSectionCount)
            {
                // 最後のキーより後ろ
                pParam->duration = (m_Timing.offset + m_Timing.duration) - m_pTransitionSections[sectionIdex - 1].timing.offset;
                pParam->offset = m_pTransitionSections[sectionIdex - 1].timing.offset;
                pParam->parameterizedAnimType = m_pTransitionSections[sectionIdex - 1].timing.easingType;

                pParam->value.startValue = m_pTransitionSections[sectionIdex - 1].store[targetIdx].paramValue;
                // pParam->targetValue は、Transition の Traget を採用

                pTiming = &m_pTransitionSections[sectionIdex - 1].timing;
            }
            else
            {
                // 中間区間
                const TransitionSection* pCurr = &m_pTransitionSections[sectionIdex];
                const TransitionSection* pNext = (sectionIdex + 1 < m_TransitionSectionCount) ? &m_pTransitionSections[sectionIdex + 1] : nullptr;

                const float endFrame = (pNext != nullptr) ? pNext->timing.offset : (m_Timing.offset + m_Timing.duration);

                pParam->duration = endFrame - pCurr->timing.offset;
                pParam->offset = pCurr->timing.offset;
                pParam->parameterizedAnimType = pCurr->timing.easingType;

                pParam->value.startValue = pCurr->store[targetIdx].paramValue;
                if (pNext != nullptr)
                {
                    pParam->value.targetValue = pNext->store[targetIdx].paramValue;
                }

                pTiming = &pCurr->timing;
            }
        }

        NN_SDK_ASSERT_NOT_NULL(pTiming);
        if (IsUseExtraParamaterParameterizedAnimType(static_cast<ParameterizedAnimType>(pTiming->easingType)))
        {
            pParam->value.targetValue = pTiming->easingExtraParamater;
        }
    }

    //----------------------------------------------------------
    TransitionTiming      m_Timing;
    int                   m_TransitionSectionCount;
    TransitionSection*    m_pTransitionSections;
};


//----------------------------------------------------------
// パラメーター毎に個別のタイミングで状態遷移する
struct TransitionTimeLine
{
    //----------------------------------------------------------
    TransitionTimeLine()
        : m_TotalLength(0.0f)
        , m_TransitionTimeLineTrackCount(0)
        , m_pTransitionTimeLineTracks(nullptr)
    {
    }

    //----------------------------------------------------------
    ~TransitionTimeLine()
    {
    }

    //----------------------------------------------------------
    float GetTotalLength()
    {
        return m_TotalLength;
    }

    void SetupParametrizedAnimationFromTime(nn::ui2d::ResParameterizedAnimParameter* pParam,
        int contentIdx,
        int infoIdx,
        int targetIdx,
        float time,
        const FeatureParamaterStoreSet& paramStoreSetStart, const FeatureParamaterStoreSet& paramStoreSetTarget)
    {
        SetupParametrizedAnimation(pParam, contentIdx, infoIdx, targetIdx, GetSectionIndexFromTime(contentIdx, time), paramStoreSetStart, paramStoreSetTarget);
    }

    //----------------------------------------------------------
    void Initialize(int trackCount)
    {
        NN_SDK_ASSERT(m_pTransitionTimeLineTracks == nullptr);

        m_TransitionTimeLineTrackCount = trackCount;
        m_pTransitionTimeLineTracks = nn::ui2d::Layout::NewArray<TransitionTimeLineTrack>(trackCount);
    }

    //----------------------------------------------------------
    void Finalize()
    {
        if (m_pTransitionTimeLineTracks != nullptr)
        {
            for (int i = 0; i < m_TransitionTimeLineTrackCount; i++)
            {
                m_pTransitionTimeLineTracks[i].Finalize();
            }
        }

        nn::ui2d::Layout::DeleteArray(m_pTransitionTimeLineTracks, m_TransitionTimeLineTrackCount);
        m_pTransitionTimeLineTracks = nullptr;
        m_TransitionTimeLineTrackCount = 0;
    }

    //----------------------------------------------------------
    const TransitionTimeLineTrack* GetTrackByFeatureParamaterIndex(int targetIdx) const
    {
        if (m_TransitionTimeLineTrackCount <= targetIdx)
        {
            return nullptr;
        }

        return &m_pTransitionTimeLineTracks[targetIdx];
    }

    //----------------------------------------------------------
    int GetSectionIndexFromTime(int contentIdx, float time) const
    {
        const TransitionTimeLineTrack* pTrack = GetTrackByFeatureParamaterIndex(contentIdx);
        if (pTrack == nullptr)
        {
            return -1;
        }

        return pTrack->GetSectionIndexFromTime(time);
    }

    //----------------------------------------------------------
    bool SetupParametrizedAnimation(nn::ui2d::ResParameterizedAnimParameter* pParam,
        int contentIdx,
        int infoIdx,
        int targetIdx,
        int paramIdx,
        const FeatureParamaterStoreSet& paramStoreSetStart, const FeatureParamaterStoreSet& paramStoreSetTarget)
    {
        const TransitionTimeLineTrack* pTrack = GetTrackByFeatureParamaterIndex(contentIdx);
        pParam->value.startValue = paramStoreSetStart.GetAt(contentIdx, infoIdx, targetIdx)->paramValue;
        pParam->value.targetValue = paramStoreSetTarget.GetAt(contentIdx, infoIdx, targetIdx)->paramValue;
        if (pTrack != nullptr)
        {
            pTrack->SetupParametrizedAnimation(pParam, targetIdx, paramIdx);
        }

        return true;
    }

    //----------------------------------------------------------
    void SetupParametrizedAnimationEvent(nn::ui2d::ResParameterizedAnimParameter* pParam, int contentIdx, int paramIdx)
    {
        const TransitionTimeLineTrack* pTrack = GetTrackByFeatureParamaterIndex(contentIdx);
        if (pTrack != nullptr)
        {
            pTrack->SetupParametrizedAnimationEvent(pParam, paramIdx);
        }
    }

    //----------------------------------------------------------
    float m_TotalLength;
    //----------------------------------------------------------
    int                      m_TransitionTimeLineTrackCount;
    TransitionTimeLineTrack* m_pTransitionTimeLineTracks;
};

//----------------------------------------------------------
// 状態遷移
struct Transition
{
    //----------------------------------------------------------
    Transition()
        : m_pPrevStateName(nullptr)
        , m_pNextStateName(nullptr)
        , m_IsCancelable(false)
        , m_IsLoop(false)
        , m_IsNotPushCompletedEvent(false)
        , m_pCondition(nullptr)
        , m_pTimeLine(nullptr)
    {
    }

    //----------------------------------------------------------
    ~Transition()
    {
    }

    //----------------------------------------------------------
    void Initialzie(const char* pPrev, const char* pNext, const FeatureParamaterSet& featureParamaterSet)
    {
        {
            const int size = nn::util::Strnlen(pPrev, CountOfStateMachineName) + 1;
            auto pBuf = nn::ui2d::Layout::AllocateMemory(size);
            nn::util::Strlcpy(static_cast<char*>(pBuf), pPrev, size);
            m_pPrevStateName = static_cast<const char*>(pBuf);
        }

        {
            const int size = nn::util::Strnlen(pNext, CountOfStateMachineName) + 1;
            auto pBuf = nn::ui2d::Layout::AllocateMemory(size);
            nn::util::Strlcpy(static_cast<char*>(pBuf), pNext, size);
            m_pNextStateName = static_cast<const char*>(pBuf);
        }

        m_FeatureParamaterStoreSetOfStartValue.Initialzie(featureParamaterSet);
    }

    //----------------------------------------------------------
    void Finalize()
    {
        m_FeatureParamaterStoreSetOfStartValue.Finalize();

        nn::ui2d::Layout::FreeMemory(const_cast<void*>(static_cast<const void*>(m_pPrevStateName)));
        m_pPrevStateName = nullptr;
        nn::ui2d::Layout::FreeMemory(const_cast<void*>(static_cast<const void*>(m_pNextStateName)));
        m_pNextStateName = nullptr;

        nn::ui2d::Layout::DeleteObj(m_pCondition);
        m_pCondition = nullptr;

        m_pTimeLine->Finalize();
        nn::ui2d::Layout::DeleteObj(m_pTimeLine);
        m_pTimeLine = nullptr;
    }

    //----------------------------------------------------------
    bool IsCancelable() const
    {
        return m_IsLoop || m_IsCancelable;
    }

    //----------------------------------------------------------

    nn::util::IntrusiveListNode m_Link;

    // 遷移元
    const char* m_pPrevStateName;
    // 遷移先
    const char* m_pNextStateName;

    // 状態遷移をキャンセル可能かどうか
    bool m_IsCancelable;

    // ループする状態遷移かどうか？ ループ時は、強制的にキャンセル可能になる。
    bool m_IsLoop;

    // ループする状態遷移かどうか？ ループ時は、強制的にキャンセル可能になる。
    bool m_IsNotPushCompletedEvent;

    // 条件
    TransitionConditionBase* m_pCondition;

    // 変化タイミングの情報
    TransitionTimeLine* m_pTimeLine;

    // カスタムのスタート値（遷移の途中で、遷移した場合に更新される） 現在のTransition に切り替わった時の、値。
    // TODO:Transition ではなく、StateLayer に一つ持っておけば十分。メモリがもったいないので移動する。
    FeatureParamaterStoreSet m_FeatureParamaterStoreSetOfStartValue;
};

//----------------------------------------------------------
// アニメーターを管理するもの
struct AnimatorSlot
{
    //----------------------------------------------------------
    AnimatorSlot()
        : m_pGroup(nullptr)
        , m_pLayout(nullptr)
        , m_pAnimResFile(nullptr)
        , m_pAnimator(nullptr)
    {
    }

    //----------------------------------------------------------
    void Initialize(nn::ui2d::Layout* pLayout, const FeatureParamaterSet& featureParamaterSet)
    {
        m_pLayout = pLayout;

        if (featureParamaterSet.size() <= 0)
        {
            return;
        }

        // m_Group
        {
            NN_SDK_ASSERT(featureParamaterSet.size() > 0);
            m_pGroup = nn::ui2d::Layout::AllocateAndConstruct<Group>();
            for (auto iter = featureParamaterSet.begin(); iter != featureParamaterSet.end(); iter++)
            {
                auto pPane = pLayout->GetRootPane()->FindPaneByName(iter->m_pName);
                NN_SDK_ASSERT_NOT_NULL(pPane);
                m_pGroup->AppendPane(pPane);
            }
        }
    }

    //----------------------------------------------------------
    void Finalzie()
    {
        // m_Group
        {
            nn::ui2d::Layout::DeleteObj(m_pGroup);
            m_pGroup = nullptr;
        }

        m_pLayout = nullptr;
    }

    //----------------------------------------------------------
    void Bind(nn::gfx::Device* pDevice, void* pAnimResFile)
    {
        NN_SDK_ASSERT(m_pAnimator == nullptr);
        NN_SDK_ASSERT(m_pAnimResFile == nullptr);

        NN_SDK_ASSERT(m_pLayout != nullptr);

        m_pAnimResFile = pAnimResFile;

        nn::ui2d::GroupAnimator* pPaneAnimatior = m_pLayout->CreateAnimTransform<nn::ui2d::GroupAnimator>(pDevice, pAnimResFile);
        pPaneAnimatior->Setup(m_pGroup, true);
        pPaneAnimatior->PlayAuto(1.0f);

        m_pAnimator = pPaneAnimatior;
    }

    //----------------------------------------------------------
    void* Unbind()
    {
        if (m_pAnimator != nullptr)
        {
            m_pAnimator->UnbindAll();
            m_pLayout->DeleteAnimTransform(m_pAnimator);

            m_pAnimator = nullptr;
        }

        void* pCurrentRes = m_pAnimResFile;
        m_pAnimResFile = nullptr;

        return pCurrentRes;
    }

    //----------------------------------------------------------
    const void* GetAnimResource() const
    {
        return m_pAnimResFile;
    }

    //----------------------------------------------------------
    const float GetAnimationFrame() const
    {
        if (m_pAnimator == nullptr)
        {
            return 0.0f;
        }

        return m_pAnimator->GetFrame();
    }

    //----------------------------------------------------------
    const float GetAnimationFrameMax() const
    {
        if (m_pAnimator == nullptr)
        {
            return 0.0f;
        }

        return m_pAnimator->GetFrameMax();
    }

    //----------------------------------------------------------
    nn::ui2d::Group*            m_pGroup;
    nn::ui2d::Layout*           m_pLayout;
    void*                       m_pAnimResFile;
    nn::ui2d::Animator*         m_pAnimator;// アニメーター
};

typedef nn::util::IntrusiveList<Transition, nn::util::IntrusiveListMemberNodeTraits<Transition, &Transition::m_Link> > TransitionSet;

//----------------------------------------------------------
// ステートレイヤー
// 共通の「特徴」を持つステートの集まり
struct StateLayer
{
    StateLayer()
        : m_pLayout(nullptr)
        , m_pName(nullptr)
        , m_pCurrentState(nullptr)
        , m_pActiveTransition(nullptr)
        , m_IsCurrentTransitionCancelable(false)
        , m_LastIsStateTransitionCompleted(false)
        , m_IsStateTransitionJustCompleted(false)
    {
        ResetStateFlags_();
    }

    //----------------------------------------------------------
    bool IsInitialized() const
    {
        if (m_pLayout == nullptr)
        {
            return false;
        }

        if (m_pCurrentState == nullptr)
        {
            return false;
        }

        return true;
    }

    //----------------------------------------------------------
    bool IsAllTargetFound_(const nn::ui2d::Layout* pLayout) const
    {
        for (auto iter = m_FeatureParamaterSet.begin(); iter != m_FeatureParamaterSet.end(); iter++)
        {
            // 初期化失敗
            if (pLayout->GetRootPane()->FindPaneByName(iter->m_pName) == nullptr)
            {
                return false;
            }
        }

        return true;
    }

    //----------------------------------------------------------
    // 初期化
    void Initialize(nn::gfx::Device* pDevice, nn::ui2d::Layout* pLayout, const char* pName)
    {
        NN_UNUSED(pDevice);

        if (!IsAllTargetFound_(pLayout))
        {
            return;
        }

        m_pLayout = pLayout;

        const int size = nn::util::Strnlen(pName, CountOfStateMachineName) + 1;
        auto pBuf = static_cast<char*>( nn::ui2d::Layout::AllocateMemory(size));
        nn::util::Strlcpy(pBuf, pName, size);
        m_pName = static_cast<const char*>(pBuf);

        m_AnimatorSlot.Initialize(pLayout, m_FeatureParamaterSet);

        m_pCurrentState = nullptr;
        ResetStateFlags_();
    }

    //----------------------------------------------------------
    void SetToBase()
    {
        m_pCurrentState = &(*m_StateSet.begin());
    }

    //----------------------------------------------------------
    // 終了
    void Finalize()
    {
        nn::ui2d::Layout::FreeMemory(const_cast<void*>(static_cast<const void*>(m_pName)));
        m_pName = nullptr;

        m_pCurrentState = nullptr;
        ResetStateFlags_();

        // FeatureParamaterSet
        {
            auto endIter = m_FeatureParamaterSet.end();
            for (auto iter = m_FeatureParamaterSet.begin(); iter != endIter; )
            {
                auto currIter = iter;
                iter++;

                currIter->Finalize();
                m_FeatureParamaterSet.erase(currIter);
                nn::ui2d::Layout::DeleteObj(&(*currIter));
            }
        }

        // m_TransitionSet
        {
            auto endIter = m_TransitionSet.end();
            for (auto iter = m_TransitionSet.begin(); iter != endIter; )
            {
                auto currIter = iter;
                iter++;

                currIter->Finalize();
                m_TransitionSet.erase(currIter);
                nn::ui2d::Layout::DeleteObj(&(*currIter));
            }
        }

        // State
        for (auto iter = m_StateSet.begin(); iter != m_StateSet.end(); )
        {
            auto currIter = iter;
            iter++;

            currIter->Finalize();
            m_StateSet.erase(currIter);
            nn::ui2d::Layout::DeleteObj(&(*currIter));
        }

        UnbindAnimatorSlot_();

        m_AnimatorSlot.Finalzie();
    }

    //----------------------------------------------------------
    State* FindStateByName(const char* pName)
    {
        auto endIter = m_StateSet.end();
        for (auto iter = m_StateSet.begin(); iter != endIter; iter++)
        {
            if (strcmp(iter->m_pName, pName) == 0)
            {
                return &(*iter);
            }
        }

        return nullptr;
    }

    //----------------------------------------------------------
    const State* FindStateByName(const char* pName) const
    {
        auto endIter = m_StateSet.end();
        for (auto iter = m_StateSet.begin(); iter != endIter; iter++)
        {
            if (strcmp(iter->m_pName, pName) == 0)
            {
                return &(*iter);
            }
        }

        return nullptr;
    }

    //----------------------------------------------------------
    int FindFeatureParamaterByName(const char* pName, int contentType) const
    {
        int count = 0;
        auto endIter = m_FeatureParamaterSet.end();
        for (auto iter = m_FeatureParamaterSet.begin(); iter != endIter; iter++)
        {
            if (strcmp(iter->m_pName, pName) == 0)
            {
                if (iter->m_AnimContentType == contentType)
                {
                    return count;
                }
            }
            count++;
        }

        return -1;
    }

    //----------------------------------------------------------
    void Update()
    {
        // TransitionCompleted 関連
        {
            bool current = IsStateTransitionCompleted();
            m_IsStateTransitionJustCompleted = m_LastIsStateTransitionCompleted != current;
            m_LastIsStateTransitionCompleted = current;
        }
    }

    //----------------------------------------------------------
    void ResetStateFlags_()
    {
        m_IsStateTransitionJustCompleted = false;
        m_LastIsStateTransitionCompleted = false;
    }

    //----------------------------------------------------------
    bool IsStateTransitionCompleted() const
    {
        if (m_AnimatorSlot.m_pAnimator == nullptr)
        {
            return true;
        }

        if (!m_AnimatorSlot.m_pAnimator->IsEnabled())
        {
            return true;
        }

        if (m_AnimatorSlot.m_pAnimator->IsEndFrame())
        {
            return true;
        }

        return false;
    }

    //----------------------------------------------------------
    bool IsStateTransitionJustCompleted()
    {
        return m_IsStateTransitionJustCompleted;
    }

    //----------------------------------------------------------
    const Transition* GetActiveTransition() const
    {
        if (IsStateTransitionCompleted())
        {
            return nullptr;
        }

        return m_pActiveTransition;
    }

    //----------------------------------------------------------
    void BindSlot_(nn::gfx::Device* pDevice, Transition* pNextTransition, const Transition* pPrevTransition, float prevAnimFrame)
    {
        auto pAnimRes = BuildAnimationResource(*this, pNextTransition, pPrevTransition, prevAnimFrame);
        m_AnimatorSlot.Bind(pDevice, pAnimRes);

        m_pActiveTransition = pNextTransition;
    }

    //----------------------------------------------------------
    void UnbindAnimatorSlot_()
    {
        auto pCurrentRes = m_AnimatorSlot.Unbind();
        if (pCurrentRes != nullptr)
        {
            nn::ui2d::Layout::FreeMemory(const_cast<void*>(pCurrentRes));
        }

        m_pActiveTransition = nullptr;
    }

    //----------------------------------------------------------
    void ChangeState(nn::gfx::Device* pDevice, Transition& transition)
    {
        // ステート変更を行う
        auto pNextState = FindStateByName(transition.m_pNextStateName);
        NN_SDK_ASSERT(pNextState != nullptr);

        m_pCurrentState = pNextState;

        ResetStateFlags_();
        m_IsCurrentTransitionCancelable = transition.IsCancelable();

        // 動的に構築する
        {
            const Transition* pPrevTransition = m_pActiveTransition;
            const float prevAnimFrame = m_AnimatorSlot.GetAnimationFrame();

            UnbindAnimatorSlot_();

            BindSlot_(pDevice, &transition, pPrevTransition, prevAnimFrame);
        }
    }

    //----------------------------------------------------------
    static void* BuildAnimationResource(StateLayer& stateLayer, Transition* pPrevTransition, const Transition* pNextTansition, float prevTransitionFrame);

    //----------------------------------------------------------
    nn::util::IntrusiveListNode m_Link;

    nn::ui2d::Layout*    m_pLayout;
    const char*          m_pName;
    State*               m_pCurrentState;

    StateSet             m_StateSet;
    FeatureParamaterSet  m_FeatureParamaterSet;

    AnimatorSlot         m_AnimatorSlot;

    TransitionSet        m_TransitionSet;
    const Transition*    m_pActiveTransition;

    bool                 m_IsCurrentTransitionCancelable;
    bool                 m_LastIsStateTransitionCompleted;
    bool                 m_IsStateTransitionJustCompleted;
};

// ステートレイヤー（の束）
typedef nn::util::IntrusiveList<StateLayer, nn::util::IntrusiveListMemberNodeTraits<StateLayer, &StateLayer::m_Link> > StateLayerSet;

//----------------------------------------------------------
// ステートマシン
struct StateMachine
{
    //----------------------------------------------------------
    StateMachine()
        : m_Layout(nullptr)
        , m_isDown(false)
    {
    }

    //----------------------------------------------------------
    void Initialize(nn::ui2d::Layout* pLayout)
    {
        m_Layout = pLayout;
        m_StateMachineEventQueue.Initialize();
        m_StateMachineVariableManager.Initialize(&m_StateMachineEventQueue);
    }

    //----------------------------------------------------------
    bool IsInitialized() const
    {
        for (auto iter = m_StateLayerSet.begin(); iter != m_StateLayerSet.end(); iter++)
        {
            if(!iter->IsInitialized())
            {
                return false;
            }
        }

        return true;
    }

    //----------------------------------------------------------
    void Finalize()
    {
        for (auto iter = m_StateLayerSet.begin(); iter != m_StateLayerSet.end(); )
        {
            auto currIter = iter;
            iter++;

            m_StateLayerSet.erase(currIter);

            currIter->Finalize();
            nn::ui2d::Layout::DeleteObj(&(*currIter));
        }

        m_Layout = nullptr;
        m_StateMachineEventQueue.Finalize();
        m_StateMachineVariableManager.Finalize();
    }

    //----------------------------------------------------------
    void PostEvent(StateMachineEventKind kind, uint64_t param1, uint64_t param2, uint32_t delayFrames)
    {
        m_StateMachineEventQueue.Push(kind, param1, param2, delayFrames);
    }

    //----------------------------------------------------------
    void UpdateStateLayerTransitions_(nn::gfx::Device* pDevice, StateLayer* pStateLayer, const StateMachineEvent& eventMsg)
    {
        if (!pStateLayer->IsInitialized())
        {
            return;
        }

        if (!pStateLayer->m_IsCurrentTransitionCancelable)
        {
            if (!pStateLayer->IsStateTransitionCompleted())
            {
                return;
            }
        }

        for (auto iter = pStateLayer->m_TransitionSet.begin(); iter != pStateLayer->m_TransitionSet.end(); iter++)
        {
            // 現在のステートを対象としているトリガーかどうか？
            if (strcmp(iter->m_pPrevStateName, pStateLayer->m_pCurrentState->m_pName) != 0)
            {
                continue;
            }

            // 状態遷移するか調査
            if (iter->m_pCondition != nullptr && iter->m_pCondition->IsTriggered(eventMsg))
            {
                // 状態を遷移する
                pStateLayer->ChangeState(pDevice, *iter);
                break;
            }
        }
    }

    //----------------------------------------------------------
    // StateLayer の更新（StateLayerのメソッドに引っ越します）
    void UpdateStateLayer_(StateLayer* pStateLayer)
    {
        if (!pStateLayer->IsInitialized())
        {
            return;
        }

        // 更新します
        pStateLayer->Update();

        // ステート完了イベントの送信
        if (pStateLayer->IsStateTransitionJustCompleted())
        {
            const bool doNotPushCompleteEvent = pStateLayer->m_pActiveTransition != nullptr && pStateLayer->m_pActiveTransition->m_IsNotPushCompletedEvent;
            if (!doNotPushCompleteEvent)
            {
                uint64_t param = reinterpret_cast<uint64_t>(pStateLayer->m_pCurrentState->m_pName);
                m_StateMachineEventQueue.Push(StateMachineEventKind_StateTransitionCompleted, param, 0, 0);
            }
        }
    }

    //----------------------------------------------------------
    // ユーザーの入力を与えて、内部メッセージを m_StateMachineEventQueue に生成します。(TODO)
    void UpdateUserInput(const nn::util::Float2* pPos, bool isDown, bool isRelease = false)
    {
        NN_UNUSED(pPos);
        NN_UNUSED(isRelease);

        if (isDown && !m_isDown)
        {
            m_StateMachineEventQueue.Push(StateMachineEventKind_Decided, 0, 0, 0);
        }

        m_isDown = isDown;
    }

    //----------------------------------------------------------
    void Update(nn::gfx::Device* pDevice)
    {
        // 更新
        for (auto iter = m_StateLayerSet.begin(); iter != m_StateLayerSet.end(); iter++)
        {
            UpdateStateLayer_(&(*iter));
        }

        // メッセージキューを舐め、StateLayer にメッセージを振り分ける。。。
        StateMachineEvent* pEventMsg = m_StateMachineEventQueue.Peek();
        StateMachineEvent* pEndEventMsg = m_StateMachineEventQueue.Bottom();

        while (pEventMsg != nullptr)
        {
            // イベントのディレイフレーム対応
            if (pEventMsg->delayFrames > 0)
            {
                // ディレイフレームが完了するまで、処理しません。
                pEventMsg->delayFrames--;
                m_StateMachineEventQueue.MoveTopToTail();
            }
            else
            {
                for (auto iter = m_StateLayerSet.begin(); iter != m_StateLayerSet.end(); iter++)
                {
                    // テスト用に適当な値を入れています。
                    UpdateStateLayerTransitions_(pDevice, &(*iter), *pEventMsg);
                }

                m_StateMachineEventQueue.Pop();
            }

            // 終端まで走査したら抜ける
            if (pEventMsg == pEndEventMsg)
            {
                break;
            }

            pEventMsg = m_StateMachineEventQueue.Peek();
        }
    }

    //----------------------------------------------------------
    // 対象としている、レイアウト
    nn::ui2d::Layout*               m_Layout;

    StateLayerSet                   m_StateLayerSet;

    // イベントキュー
    StateMachineEventQueue          m_StateMachineEventQueue;

    StateMachineVariableManager     m_StateMachineVariableManager;

    bool                            m_isDown;
};

//--------------------------------------------------------------------------------------------------
// 生成クラス（TODO:最終的にはリソースクラスを入力に受け取ってデータドリブンで構築されるようにします）
class StateMachineFactory
{
public:

    //----------------------------------------------------------
    StateMachineFactory(nn::gfx::Device* pDevice, nn::ui2d::Layout* pLayout)
        : m_pDevice(pDevice)
        , m_pLayout(pLayout)
    {
    }

    //----------------------------------------------------------

    void DoRelocate_(ResStateMachine* pResStateMachine, void* pBinFile)
    {
        if (pResStateMachine->isRelocated)
        {
            return;
        }

        pResStateMachine->isRelocated = true;

        pResStateMachine->pLayers.Relocate(pBinFile);
        pResStateMachine->pVariables.Relocate(pBinFile);

        for (int i = 0; i < pResStateMachine->layerCount; i++)
        {
            ResStateLayer* pLayer = pResStateMachine->pLayers.Get() + i;

            pLayer->pStates.Relocate(pBinFile);
            for (int ii = 0; ii < pLayer->stateCount; ii++)
            {
                ResState* pState = pLayer->pStates.Get() + ii;

                pState->pParamaters.Relocate(pBinFile);
            }

            pLayer->pTransitions.Relocate(pBinFile);
            for (int ij = 0; ij < pLayer->transitionCount; ij++)
            {
                ResStateTransition* pTransition = pLayer->pTransitions.Get() + ij;

                pTransition->pTracks.Relocate(pBinFile);
                for (int iji = 0; iji < pTransition->trackCount; iji++)
                {
                    ResStateTransitionTrack* pTrack = pTransition->pTracks.Get() + iji;

                    pTrack->pKies.Relocate(pBinFile);
                }

                pTransition->trigger.pVariableArray.Relocate(pBinFile);
            }
        }
    }

    //----------------------------------------------------------
    // ステートマシンを構築します。
    void Build(StateMachine* pStateMachine, void* pResStateMachineFile)
    {
        nn::font::detail::BinaryBlockHeader* pBinFile = reinterpret_cast<nn::font::detail::BinaryBlockHeader*>((char*)pResStateMachineFile);

        ResStateMachine* pResStateMachine = (ResStateMachine*)pBinFile;
        DoRelocate_(pResStateMachine, pBinFile);

        for (int i = 0; i < pResStateMachine->layerCount; i++)
        {
            ResStateLayer* pLayer = pResStateMachine->pLayers.Get() + i;

            auto pStateLayer = BuildStateLayer(pLayer);
            pStateMachine->m_StateLayerSet.push_back(*pStateLayer);
        }

        // ステートマシン変数を追加
        const bool result = pStateMachine->m_StateMachineVariableManager.RegisterNewVariable("TestFloat", StateMachineVariableType_Float);
        NN_SDK_ASSERT(result);
        NN_UNUSED(result);
    }

    //----------------------------------------------------------

    void BuildFeatureParamaters_(StateLayer* pStateLayer, ResStateLayer* pResStateLayer)
    {
        // 先頭は Base ステート。
        ResState* pResBaseState = pResStateLayer->pStates.Get() + 0;
        for (int paramIdx = 0; paramIdx < pResBaseState->paramaterCount; paramIdx++)
        {
            ResStateParamaterStore* pResParamStore = pResBaseState->pParamaters.Get() + paramIdx;
            switch (pResParamStore->type)
            {
            case StateMachineFeatureParamaterKind_Position:
            {
                FeatureParamater* param1 = nn::ui2d::Layout::AllocateAndConstruct<FeatureParamater>();
                uint8_t param[] = { AnimTargetPane_TranslateX, AnimTargetPane_TranslateY, AnimTargetPane_TranslateZ };
                param1->Initialzie(pResParamStore->targetName, nn::ui2d::AnimContentType_Pane, nn::ui2d::AnimationTypePaneSrt, 3, param);
                pStateLayer->m_FeatureParamaterSet.push_back(*param1);
            }
            break;
            case StateMachineFeatureParamaterKind_Scale:
            {
                FeatureParamater* param1 = nn::ui2d::Layout::AllocateAndConstruct<FeatureParamater>();
                uint8_t param[] = { AnimTargetPane_ScaleX, AnimTargetPane_ScaleY };
                param1->Initialzie(pResParamStore->targetName, nn::ui2d::AnimContentType_Pane, nn::ui2d::AnimationTypePaneSrt, 2, param);
                pStateLayer->m_FeatureParamaterSet.push_back(*param1);
            }
            break;
            case StateMachineFeatureParamaterKind_RotateZ:
            {
                FeatureParamater* param1 = nn::ui2d::Layout::AllocateAndConstruct<FeatureParamater>();
                uint8_t param[] = { AnimTargetPane_RotateX, AnimTargetPane_RotateY, AnimTargetPane_RotateZ };
                param1->Initialzie(pResParamStore->targetName, nn::ui2d::AnimContentType_Pane, nn::ui2d::AnimationTypePaneSrt, 3, param);
                pStateLayer->m_FeatureParamaterSet.push_back(*param1);
            }
            break;
            case StateMachineFeatureParamaterKind_BlackColor:
            case StateMachineFeatureParamaterKind_WhiteColor:
            {
                FeatureParamater* param1 = nn::ui2d::Layout::AllocateAndConstruct<FeatureParamater>();

                if (pResParamStore->type == StateMachineFeatureParamaterKind_BlackColor)
                {
                    uint8_t param[] = { AnimTargetMatColor_BufferR, AnimTargetMatColor_BufferG, AnimTargetMatColor_BufferB, AnimTargetMatColor_BufferA };
                    param1->Initialzie(pResParamStore->targetName, nn::ui2d::AnimContentType_Material, nn::ui2d::AnimationTypeMaterialColor, 4, param);
                }
                else
                {
                    uint8_t param[] = { AnimTargetMatColor_Konst0R, AnimTargetMatColor_Konst0G, AnimTargetMatColor_Konst0B, AnimTargetMatColor_Konst0A };
                    param1->Initialzie(pResParamStore->targetName, nn::ui2d::AnimContentType_Material, nn::ui2d::AnimationTypeMaterialColor, 4, param);
                }

                pStateLayer->m_FeatureParamaterSet.push_back(*param1);
            }
            break;
            case StateMachineFeatureParamaterKind_None:
            {
                // 暫定的にダミーを挿入しておきます。
                FeatureParamater* param1 = nn::ui2d::Layout::AllocateAndConstruct<FeatureParamater>();
                uint8_t param[] = { AnimTargetPane_TranslateX };
                param1->Initialzie(pResParamStore->targetName, nn::ui2d::AnimContentType_Pane, nn::ui2d::AnimationTypePaneSrt, 1, param);
                pStateLayer->m_FeatureParamaterSet.push_back(*param1);
            }
            break;
            case StateMachineFeatureParamaterKind_StateMachineEvent:
            {
                FeatureParamater* param1 = nn::ui2d::Layout::AllocateAndConstruct<FeatureParamater>();
                uint8_t param[] = { AnimTargetStateMachineEvent_PostToChild };
                param1->Initialzie(pResParamStore->targetName, nn::ui2d::AnimContentType_StateMachine, nn::ui2d::AnimationTypeStateMachine, 1, param);
                pStateLayer->m_FeatureParamaterSet.push_back(*param1);
            }
            break;
            case StateMachineFeatureParamaterKind_Max:
            default:
            {
                NN_SDK_ASSERT(false);
                return;
            }
            break;
            }
        }
    }

    //----------------------------------------------------------

    void BuildStates_(StateLayer* pStateLayer, ResStateLayer* pResStateLayer)
    {
        for (int stateIdx = 0; stateIdx < pResStateLayer->stateCount; stateIdx++)
        {
            ResState* pResState = pResStateLayer->pStates.Get() + stateIdx;

            State* pState1 = nn::ui2d::Layout::AllocateAndConstruct<State>();
            {
                pState1->Initialzie(pResState->name, pStateLayer->m_FeatureParamaterSet);

                for (int paramIdx = 0; paramIdx < pResState->paramaterCount; paramIdx++)
                {
                    ResStateParamaterStore* pStore = pResState->pParamaters.Get() + paramIdx;
                    float value[] = { pStore->x, pStore->y, pStore->z, pStore->w };
                    pState1->SetTestDataToStore(paramIdx, 4, value);
                }
            }
            pStateLayer->m_StateSet.push_back(*pState1);
        }
    }

    //----------------------------------------------------------

    void BuildTransitionTrigger_(Transition* pTransition, ResStateTransition* pResTransition)
    {
        {
            ResStateTransitionTrigger* pResTrigger = &pResTransition->trigger;
            switch (pResTrigger->triggerKind)
            {
            case StateMachineTriggerKind_IsTick:
            {
                TransitionConditionNone* pCondition = nn::ui2d::Layout::AllocateAndConstruct<TransitionConditionNone>();
                pTransition->m_pCondition = pCondition;
            }
            break;
            case StateMachineTriggerKind_IsHit:
            {
                TransitionConditionIsHit* pCondition = nn::ui2d::Layout::AllocateAndConstruct<TransitionConditionIsHit>();
                pTransition->m_pCondition = pCondition;
            }
            break;
            case StateMachineTriggerKind_IsDecided:
            {
                TransitionConditionIsDecided* pCondition = nn::ui2d::Layout::AllocateAndConstruct<TransitionConditionIsDecided>();
                pTransition->m_pCondition = pCondition;
            }
            break;
            case StateMachineTriggerKind_IsStateTransitionCompleted:
            {
                TransitionConditionIsStateBegin* pCondition = nn::ui2d::Layout::AllocateAndConstruct<TransitionConditionIsStateBegin>();

                pCondition->pName = pResTransition->startName;

                pTransition->m_pCondition = pCondition;
            }
            break;
            case StateMachineTriggerKind_IsVariableChanged:
            {
                TransitionConditionVariableChanged* pCondition = nn::ui2d::Layout::AllocateAndConstruct<TransitionConditionVariableChanged>();

                NN_SDK_ASSERT(pResTrigger->variableCount == 1);

                ResStateVariable* pResStateVariable = pResTrigger->pVariableArray.Get();

                nn::util::Strlcpy(pCondition->m_Name, pResStateVariable->name, sizeof(pCondition->m_Name));
                pCondition->m_StateMachineOperator = static_cast<StateMachineOperator>(pResTrigger->stateMachineOperator);
                StateMachineVariableContent::SetAsFloat(&pCondition->m_Content, pResStateVariable->value0);

                pTransition->m_pCondition = pCondition;
            }
            break;
            case StateMachineTriggerKind_IsStateChangeRequested:
            {
                struct TransitionConditionStateChangeRequested* pCondition = nn::ui2d::Layout::AllocateAndConstruct<struct TransitionConditionStateChangeRequested>(pResTransition->endName);
                pTransition->m_pCondition = pCondition;
            }
            break;
            case StateMachineTriggerKind_Max:
            default:
            {
                NN_SDK_ASSERT(false);
                return;
            }
            break;
            }
        }
    }

    //----------------------------------------------------------

    void BuildTransitions_(StateLayer* pStateLayer, ResStateLayer* pResStateLayer)
    {
        for (int transitionIdx = 0; transitionIdx < pResStateLayer->transitionCount; transitionIdx++)
        {
            ResStateTransition* pResTransition = pResStateLayer->pTransitions.Get() + transitionIdx;

            Transition* pTransition = nn::ui2d::Layout::AllocateAndConstruct<Transition>();
            pTransition->Initialzie(pResTransition->startName, pResTransition->endName, pStateLayer->m_FeatureParamaterSet);

            TransitionTimeLine* pTimeLine = nn::ui2d::Layout::AllocateAndConstruct<TransitionTimeLine>();
            pTransition->m_pTimeLine = pTimeLine;

            pTimeLine->m_TotalLength = pResTransition->totalDuration;

            // Track
            pTimeLine->Initialize(pResTransition->trackCount);
            TransitionTimeLineTrack* pTracks = pTimeLine->m_pTransitionTimeLineTracks;

            for (int trackIdx = 0; trackIdx < pResTransition->trackCount; trackIdx++)
            {
                const ResStateTransitionTrack* pResTrack = pResTransition->pTracks.Get() + trackIdx;

                pTracks[trackIdx].m_Timing.offset = pResTrack->offset;
                pTracks[trackIdx].m_Timing.duration = pResTrack->duration;
                pTracks[trackIdx].m_Timing.easingType = pResTrack->easingType;
                pTracks[trackIdx].m_Timing.easingExtraParamater = pResTrack->easingExtraParamater;

                // Key
                {
                    pTracks[trackIdx].Initialize(pResTrack->keyCount);
                    TransitionSection* pKies = pTracks[trackIdx].m_pTransitionSections;

                    for (int keyIdx = 0; keyIdx < pResTrack->keyCount; keyIdx++)
                    {
                        const ResStateTransitionTrackKey* pResKey = pResTrack->pKies.Get() + keyIdx;

                        pKies[keyIdx].timing.offset = pResKey->offset;
                        pKies[keyIdx].timing.duration = 1;
                        pKies[keyIdx].timing.easingType = pResKey->easingType;
                        pKies[keyIdx].timing.easingExtraParamater = pResKey->easingExtraParamater;

                        pKies[keyIdx].isUsedForEvent = pResKey->keyEvent.kind != StateMachineEventKind_None;
                        if (pKies[keyIdx].isUsedForEvent)
                        {
                            // イベントの場合
                            pKies[keyIdx].trackEvent.delay = pResKey->keyEvent.delay;
                            pKies[keyIdx].trackEvent.kind = static_cast<StateMachineEventKind>(pResKey->keyEvent.kind);
                            pKies[keyIdx].trackEvent.pParam1 = pResKey->keyEvent.param1;
                            pKies[keyIdx].trackEvent.pParam2 = pResKey->keyEvent.param2;
                        }
                        else
                        {
                            // キーの場合
                            pKies[keyIdx].store[0].paramValue = pResKey->keyParamater.x;
                            pKies[keyIdx].store[1].paramValue = pResKey->keyParamater.y;
                            pKies[keyIdx].store[2].paramValue = pResKey->keyParamater.z;
                            pKies[keyIdx].store[3].paramValue = pResKey->keyParamater.w;
                        }
                    }
                }
            }

            pTransition->m_IsCancelable = pResTransition->isCancelable == 1;
            pTransition->m_IsLoop = pResTransition->isLoop == 1;

            pStateLayer->m_TransitionSet.push_back(*pTransition);

            BuildTransitionTrigger_(pTransition, pResTransition);
        }
    }

    //----------------------------------------------------------

    StateLayer* BuildStateLayer(ResStateLayer* pResStateLayer)
    {
        StateLayer* pStateLayer = nn::ui2d::Layout::AllocateAndConstruct<StateLayer>();
        NN_SDK_ASSERT(pResStateLayer->stateCount > 0);

        // FeatureParamaterSet の初期化
        {
            BuildFeatureParamaters_(pStateLayer, pResStateLayer);
            pStateLayer->Initialize(m_pDevice, m_pLayout, pResStateLayer->name);
        }

        // state の初期化
        BuildStates_(pStateLayer, pResStateLayer);

        // Transition の初期化
        BuildTransitions_(pStateLayer, pResStateLayer);

        //// 初期ステートを設定します。
        pStateLayer->SetToBase();

        return pStateLayer;
    }

    nn::gfx::Device* m_pDevice;
    nn::ui2d::Layout* m_pLayout;
};

//----------------------------------------------------------
class ControlBase
{
public:
    ControlBase()
    {
    }

    virtual ~ControlBase()
    {
    }

    virtual void Initialize(nn::gfx::Device* pDevice, nn::ui2d::Layout* pLayout)
    {
        NN_UNUSED(pDevice);
        NN_UNUSED(pLayout);
    }

    virtual void Finalize(nn::gfx::Device* pDevice)
    {
        NN_UNUSED(pDevice);
    }

    virtual void Update(float step)
    {
        NN_UNUSED(step);
    }

    virtual void UpdateUserInput(const nn::util::Float2* pPos, bool isDown, bool isRelease = false)
    {
        NN_UNUSED(pPos);
        NN_UNUSED(isDown);
        NN_UNUSED(isRelease);
    }

    nn::util::IntrusiveListNode m_Link;
};

typedef nn::util::IntrusiveList<ControlBase, nn::util::IntrusiveListMemberNodeTraits<ControlBase, &ControlBase::m_Link> > ControlList;

//----------------------------------------------------------
// ステートマシン
// ひとまず、確認のために、ButtonBase の派生クラスとしますが、ControlBase を新設します。
// また、ButtonBase や ボタンクラス一式は ControlBase からの派生したものに、丸丸差し替えようと思います。
class StateMachineControl : public ControlBase
{

public:
    //----------------------------------------------------------
    StateMachineControl()
        : m_pDevice(nullptr)
    {
    }

    //----------------------------------------------------------
    virtual ~StateMachineControl() NN_OVERRIDE
    {
        ControlBase::~ControlBase();
    }

    //----------------------------------------------------------
    virtual void UpdateUserInput(const nn::util::Float2* pPos, bool isDown, bool isRelease = false) NN_OVERRIDE
    {
        m_pStateMachine->UpdateUserInput(pPos, isDown, isRelease);
    }

    //----------------------------------------------------------
    virtual void Update(float step) NN_OVERRIDE
    {
        NN_UNUSED(step);
        m_pStateMachine->Update(m_pDevice);
    }

    //----------------------------------------------------------
    virtual void Initialize(nn::gfx::Device* pDevice, nn::ui2d::Layout* pLayout) NN_OVERRIDE
    {
        NN_UNUSED(pLayout);
        NN_SDK_ASSERT_NOT_NULL(pLayout);

        m_pDevice = pDevice;
    }

    //----------------------------------------------------------
    virtual void Finalize(nn::gfx::Device* pDevice) NN_OVERRIDE
    {
        NN_UNUSED(pDevice);

        m_pDevice = nullptr;
    }

    //----------------------------------------------------------
    StateMachine& GetStateMachine()
    {
        return *m_pStateMachine;
    }

    void SetStateMachine(StateMachine* pStateMachine)
    {
        m_pStateMachine = pStateMachine;
    }

    //----------------------------------------------------------
    nn::gfx::Device*        m_pDevice;
    StateMachine*           m_pStateMachine;
};

//----------------------------------------------------------
class DefaultControlCreatorEx : public nn::ui2d::DefaultControlCreator
{
public:

    //----------------------------------------------------------
    // コンストラクタ
    DefaultControlCreatorEx(ButtonGroup* pButtonGroup, ControlList* pControlList)
        : nn::ui2d::DefaultControlCreator(pButtonGroup),
        m_pControlList(pControlList)
    {
    }

    //----------------------------------------------------------
    //
    virtual void CreateControl(nn::gfx::Device* pDevice, nn::ui2d::Layout* pLayout, const ui2d::ControlSrc& controlSrc) NN_OVERRIDE
    {
        if (m_pControlList == nullptr)
        {
            return;
        }

        // レイアウトが、構築された ステートマシンを持っているかどうかを調べて、コントロールを生成するように。

        ControlBase* pControl = nullptr;
        StateMachine* pStateMachine = pLayout->GetStateMachine();
        if (pStateMachine != nullptr)
        {
            auto pStateMachieCtrl = nn::ui2d::Layout::AllocateAndConstruct<StateMachineControl>();
            pStateMachieCtrl->SetStateMachine(pStateMachine);
            pStateMachieCtrl->Initialize(pDevice, pLayout);

            pControl = pStateMachieCtrl;

            if (pControl)
            {
                m_pControlList->push_back(*pControl);
            }
        }

        nn::ui2d::DefaultControlCreator::CreateControl(pDevice, pLayout, controlSrc);
    }

    //----------------------------------------------------------
    ControlList*         m_pControlList;
};

} // namespace ui2d
} // namespace nn

#endif // NN_UI2D_VIEWER_ENABLED
