﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/nn_Common.h>
#include <nn/os/os_MessageQueueTypes.h>

namespace nn { namespace atk2 { namespace detail {

    // @brief コマンドを表す構造体です
    struct Command
    {
        struct Command* next;
        uint32_t id;
        uint32_t tag;
        uintptr_t memory_next;
    };

    // @brief コマンド処理に使用するバッファを表すクラスです
    class CommandBuffer
    {
    public:
        // @brief 関数の処理結果を表す列挙体です
        enum Result
        {
            Result_Success = 0,        //!< 成功です
            Result_ErrorUnknown        //!< 不明なエラーが発生しました
        };
    public:
        CommandBuffer() NN_NOEXCEPT;
        ~CommandBuffer() NN_NOEXCEPT;

        void Initialize( void* commandBuffer, size_t commandBufferSize ) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        void* AllocMemory( size_t size ) NN_NOEXCEPT;
        void FreeMemory( Command* lastCommand ) NN_NOEXCEPT;

        size_t GetCommandBufferSize() const NN_NOEXCEPT;
        size_t GetAllocatableCommandSize() const NN_NOEXCEPT;
        size_t GetAllocatedCommandBufferSize() const NN_NOEXCEPT;

    private:
        uint32_t* m_CommandMemoryArea;
        size_t m_CommandMemoryAreaSize;

        uintptr_t m_CommandMemoryAreaBegin;
        uintptr_t m_CommandMemoryAreaEnd;

        // m_CommandMemoryAreaBegin == m_CommandMemoryAreaBegin の時、
        // エリアサイズがゼロなのか、最大なのかを判定するフラグ
        bool m_CommandMemoryAreaZeroFlag;
    };

    // @brief 非同期での変更の反映を行うためのコマンド処理を管理します
    class CommandManager
    {
    public:
        CommandManager() NN_NOEXCEPT;
        ~CommandManager() NN_NOEXCEPT;

        bool IsAvailable() const NN_NOEXCEPT { return m_Available; }

        void Initialize( void* commandBuffer, size_t commandBufferSize ) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        //void SetRequestProcessCommandFunc( RequestProcessCommandFunc func ) NN_NOEXCEPT { m_pRequestProcessCommandFunc = func; }

        template< typename CommandType >
        CommandType* AllocCommand() NN_NOEXCEPT
        {
            return reinterpret_cast<CommandType*>( AllocMemory( sizeof(CommandType), true ) );
        }
        template< typename CommandType >
        CommandType* AllocCommand( bool forceProcessCommandFlag ) NN_NOEXCEPT
        {
            return reinterpret_cast<CommandType*>( AllocMemory( sizeof(CommandType), forceProcessCommandFlag ) );
        }

        uint32_t PushCommand( Command* command ) NN_NOEXCEPT;

        uint32_t FlushCommand( bool forceFlag ) NN_NOEXCEPT;
        uint32_t FlushCommand( bool forceFlag, bool forceProcessCommandFlag ) NN_NOEXCEPT;
        void RecvCommandReply() NN_NOEXCEPT;
        Command* RecvCommandReplySync() NN_NOEXCEPT;

        void WaitCommandReply( uint32_t tag ) NN_NOEXCEPT;
        bool IsFinishCommand( uint32_t tag ) const NN_NOEXCEPT;

        size_t GetCommandBufferSize() const NN_NOEXCEPT;
        size_t GetAllocatableCommandSize() const NN_NOEXCEPT;
        size_t GetAllocatedCommandBufferSize() const NN_NOEXCEPT;
        int GetAllocatedCommandCount() const NN_NOEXCEPT;
        uint32_t GetCommandListCount() const NN_NOEXCEPT
        {
            return static_cast<const uint32_t>(m_CommandListCount);
        }

        bool ProcessCommand() NN_NOEXCEPT;

    private:
        static const int SendCommandQueueCount = 32;
        static const int RecvCommandQueueCount = SendCommandQueueCount + 1;

        static const uint32_t InvalidCommand = 0xffffffff;

    private:
        void* AllocMemory( size_t size, bool forceProcessCommandFlag ) NN_NOEXCEPT;
        void* TryAllocMemory( size_t size ) NN_NOEXCEPT;
        void FinalizeCommandList( Command* command ) NN_NOEXCEPT;

        virtual void ProcessCommandList(Command* commandList) = 0;
    private:
        bool m_Available;
        //RequestProcessCommandFunc m_pRequestProcessCommandFunc;

        nn::os::MessageQueueType   m_SendCommandQueue;
        uintptr_t m_SendCommandQueueBuffer[ SendCommandQueueCount ];
        bool m_IsInitializedSendMessageQueue;
        nn::os::MessageQueueType   m_RecvCommandQueue;
        uintptr_t m_RecvCommandQueueBuffer[ RecvCommandQueueCount ];
        bool m_IsInitializedRecvMessageQueue;

        Command* m_CommandListBegin;
        Command* m_CommandListEnd;

        std::atomic<int> m_CommandListCount;

        uint32_t m_CommandTag;
        uint32_t m_FinishCommandTag;

        CommandBuffer m_CommandBuffer;

        // 充填状況のモニタリング用
        int m_AllocatedCommandCount;
    };

}}}

