﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/atk/atk_Config.h>
#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/atk_SoundStartable.h>
#include <nn/atk/viewer/atk_Result.h>

namespace nn {
namespace atk {

class SoundHandle;

namespace viewer {

struct SoundParameters;

//---------------------------------------------------------------------------
//! @brief  【β版】サウンドを制御するクラスです。
//---------------------------------------------------------------------------
class SoundController
{
public: // 型の定義
    //---------------------------------------------------------------------------
    //! @brief  サウンドの状態を示します。
    //---------------------------------------------------------------------------
    enum State
    {
        State_Disabled,     //!< 正しいサウンドIDが設定されていない等、無効な状態です。
        State_Stopped,      //!< 停止中です。
        State_Playing,      //!< 再生中です。
        State_Paused        //!< 一時停止中です。
    };

protected: // コンストラクタ
    //! @internal
    SoundController() NN_NOEXCEPT { }

    //! @internal
    virtual ~SoundController() NN_NOEXCEPT { }

public: // メソッド
    //===========================================================================
    //! @name 状態の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief   初期化の有無を取得します。
    //!
    //! @return  初期化済みの場合は true、初期化されていない場合は false を返します。
    //---------------------------------------------------------------------------
    virtual bool IsInitialized() const NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドハンドルを取得します。
    //!
    //! @return  サウンドハンドルを返します。
    //---------------------------------------------------------------------------
    virtual const SoundHandle* GetSoundHandle() const NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドハンドルを取得します。
    //!
    //! @return  サウンドハンドルを返します。
    //---------------------------------------------------------------------------
    virtual SoundHandle* GetSoundHandle() NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   対象のサウンドIDを取得します。
    //!
    //! @return  対象のサウンドIDを返します。
    //---------------------------------------------------------------------------
    virtual SoundArchive::ItemId GetSoundId() const NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   対象のサウンドラベルを取得します。
    //!
    //! @return  対象のサウンドラベルを返します。
    //---------------------------------------------------------------------------
    virtual const char* GetLabel() const NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドの状態を取得します。
    //!
    //! @return  サウンドの状態を返します。
    //---------------------------------------------------------------------------
    virtual State GetState() const NN_NOEXCEPT = 0;

    //@}
    // 状態の取得
    //===========================================================================

    //===========================================================================
    //! @name パラメータ制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief   サウンドのパラメータを取得します。
    //!
    //! @return  サウンドのパラメータを返します。
    //---------------------------------------------------------------------------
    virtual const SoundParameters& GetParameters() const NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドのパラメータを取得します。
    //!
    //! @return  サウンドのパラメータを返します。
    //---------------------------------------------------------------------------
    virtual SoundParameters& GetParameters() NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドパラメータの変更状態を設定します。
    //!
    //! @param[in] isDirty 変更ありなら true を、無しなら false を指定します。
    //---------------------------------------------------------------------------
    virtual void SetParameterDirty(bool isDirty) NN_NOEXCEPT = 0;

    //@}
    // パラメータ取得
    //===========================================================================

    //===========================================================================
    //! @name 再生制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief   サウンドを再生します。
    //!
    //! @param   pStartInfo 詳細な再生パラメータです。
    //!
    //! @return  処理結果を返します。
    //---------------------------------------------------------------------------
    virtual viewer::Result Play(const SoundStartable::StartInfo * pStartInfo = NULL) NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドを停止します。
    //!
    //! @return  処理結果を返します。
    //---------------------------------------------------------------------------
    virtual viewer::Result Stop() NN_NOEXCEPT = 0;

    //---------------------------------------------------------------------------
    //! @brief   サウンドを一時停止します。
    //!
    //! @return  処理結果を返します。
    //---------------------------------------------------------------------------
    virtual viewer::Result Pause() NN_NOEXCEPT = 0;

    //@}
    // 再生制御
    //===========================================================================
};

} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
