﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using SdkEnvironmentCheckerLibrary.Properties;

namespace SdkEnvironmentCheckerLibrary.Checkers
{
    public class InstalledProgramVersionChecker
    {
        private static readonly ProgramInstallInfo[] Installed = InstalledPrograms.EnumerateProgramInfos(InstalledPrograms.InstallType.LocalMachine).ToArray();

        public static CheckerResult Check(InstalledProgramVersionInfo requiredProgram)
        {
            if (requiredProgram == null) throw new ArgumentNullException(nameof(requiredProgram));

            var name = requiredProgram.DisplayName;

            if (string.IsNullOrWhiteSpace(name))
            {
                return CheckerResult.ConfirmFailed("Unknown Program", "'DisplayName' is required");
            }

            var checkerResult = default(CheckerResult);

            try
            {
                foreach (var installedProgram in Installed)
                {
                    if (!requiredProgram.IsTarget(installedProgram.DisplayName)) continue;

                    var current = requiredProgram.CreateResult(installedProgram.Version);

                    // 複数ヒットした場合に、NewVersionInstalled → Installed → NotInstalled → ConfirmFailed の優先順
                    if (checkerResult == null || checkerResult.Status < current.Status)
                    {
                        checkerResult = current;
                    }
                }
            }
            catch (Exception e)
            {
                return CheckerResult.ConfirmFailed(name, e.Message);
            }

            return checkerResult ?? CheckerResult.NotInstalled(name);
        }
    }

    public class InstalledProgramVersionInfo
    {
        private readonly Regex regex;

        public InstalledProgramVersionInfo(string displayName, IEnumerable<RequiredVersionInfo> requiredVersions = null) : this(displayName, null, requiredVersions)
        {
        }

        public InstalledProgramVersionInfo(string displayName, string searchPattern, IEnumerable<RequiredVersionInfo> requiredVersions = null)
        {
            DisplayName = displayName ?? throw new ArgumentNullException(nameof(displayName));

            if (!string.IsNullOrWhiteSpace(searchPattern))
            {
                regex = new Regex(searchPattern);
            }

            RequiredVersions = requiredVersions?
                .Where(r => r != null)
                .OrderBy(r => r.Version)
                .ToArray() ?? new RequiredVersionInfo[0];
        }

        public string DisplayName { get; }

        public IReadOnlyList<RequiredVersionInfo> RequiredVersions { get; }

        public bool IsTarget(string targetName)
        {
            if (regex != null) return regex.IsMatch(targetName ?? string.Empty);
            return targetName?.StartsWith(DisplayName) ?? false;
        }

        public InstallStatus Match(Version version)
        {
            // 空の場合は Installed 扱い
            if (!RequiredVersions.Any()) return InstallStatus.Installed;

            // バージョンが取得できない場合はチェック失敗にする
            if (version == null) return InstallStatus.ConfirmFailed;

            // 最小バージョンの完全一致のみ先にチェック
            if (RequiredVersions.First().Version == version) return InstallStatus.Installed;

            // MatchLevel に応じた部分一致チェック
            if (RequiredVersions.Any(t => t.IsSupported(version))) return InstallStatus.NewVersionInstalled;

            return InstallStatus.NotInstalled;
        }

        public CheckerResult CreateResult(Version version)
        {
            switch (Match(version))
            {
                case InstallStatus.Installed: return CheckerResult.Installed(DisplayName);
                case InstallStatus.NewVersionInstalled: return CheckerResult.NewVersionInstalled(DisplayName);
                case InstallStatus.NotInstalled: return CheckerResult.NotInstalled(DisplayName);
                default: return CheckerResult.ConfirmFailed(DisplayName, Resources.VersionReadFailed);
            }
        }
    }

    public class RequiredVersionInfo
    {
        public RequiredVersionInfo(Version version, MatchLevel level)
        {
            Version = version ?? throw new ArgumentNullException(nameof(version));
            MatchLevel = level;
        }

        public Version Version { get; }

        public MatchLevel MatchLevel { get; }

        public bool IsSupported(Version current)
        {
            if (current == null) throw new ArgumentNullException(nameof(current));

            if (MatchLevel == MatchLevel.None) return Version <= current;

            // Major 一致必須
            if (Version.Major != current.Major) return false;
            if (MatchLevel == MatchLevel.Major) return Version <= current;

            // Minor 一致必須
            if (Version.Minor != current.Minor) return false;
            if (MatchLevel == MatchLevel.Minor) return Version <= current;

            // Build 一致必須
            if (Version.Build != current.Build) return false;
            if (MatchLevel == MatchLevel.Build) return Version <= current;

            // 完全一致必須
            return Version.Revision == current.Revision;
        }
    }

    public enum MatchLevel
    {
        /// <summary>
        /// 完全一致を要求しないことを示します。
        /// Major, Minor, Build, Revision は指定値以上の値が入力された場合に許容されます。
        /// </summary>
        None = 0,

        /// <summary>
        /// Major が完全一致することを要求します。
        /// Minor, Build, Revision は指定値以上の値が入力された場合に許容されます。
        /// </summary>
        Major = 1,

        /// <summary>
        /// Major, Minor が完全一致することを要求します。
        /// Build, Revision は指定値以上の値が入力された場合に許容されます。
        /// </summary>
        Minor = 2,

        /// <summary>
        /// Major, Minor, Build が完全一致することを要求します。
        /// Revision は指定値以上の値が入力された場合に許容されます。
        /// </summary>
        Build = 3,

        /// <summary>
        /// Major, Minor, Build, Revision が完全一致することを要求します。
        /// </summary>
        All = 4,

        /// <summary>
        /// Major, Minor, Build, Revision が完全一致することを要求します。
        /// </summary>
        Revision = All,
    }
}
