﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using System.IO;
using System.Globalization;
using YamlDotNet;
using CommandUtility;
using RTellerService;
using RepairCommon;

namespace RequestToRepairService
{
    public enum RequestType
    {
        Backup,
        Restore
    }

    public class RequestToRepairService
    {
        [CommandLineOption("input",
            Description = "input request file.",
            IsRequired = true)]
        public string Input { get; set; }

        [CommandLineOption("output",
            Description = "output result file.",
            IsRequired = true)]
        public string Output { get; set; }

        [CommandLineOption("type",
            Description = "request type.",
            IsRequired = true)]
        public RequestType RequestType { get; set; }

        [CommandLineOption("config",
            Description = "config file.")]
        public string Config { get; set; }

        [CommandLineOption("get-default-config",
            Description = "get default config file.")]
        public bool DefaultConfig { get; set; }

        class RequestToRepairServiceConfig
        {
            public string ServiceUrl { get; set; }
            public string BackupProfileName { get; set; }
            public string RestoreProfileName { get; set; }

            public static RequestToRepairServiceConfig GetDefault()
            {
                return new RequestToRepairServiceConfig()
                {
                    ServiceUrl = "https://10.100.20.41/rts-api/services/1.0/RTellerService",
                    BackupProfileName = "test-secblob-01-backup",
                    RestoreProfileName = "test-secblob-01-restore"
                };
            }
        }

        public int Run()
        {
            if (DefaultConfig)
            {
                using (var stringWriter = new StringWriter())
                {
                    var serializer = new YamlDotNet.Serialization.Serializer(YamlDotNet.Serialization.SerializationOptions.EmitDefaults);
                    serializer.Serialize(stringWriter, RequestToRepairServiceConfig.GetDefault());
                    stringWriter.Flush();

                    Console.WriteLine(stringWriter.ToString());

                    return 0;
                }
            }

            RequestToRepairServiceConfig config = LoadConfig(this.Config);

            CommandUtility.RetryUtility.Do(() =>
            {
                System.Net.ServicePointManager.ServerCertificateValidationCallback =
                    ((sender, certificate, chain, sslPolicyErrors) => true);

                var service = new RTellerService.RTellerService();

                if (config.ServiceUrl != null)
                {
                    service.Url = config.ServiceUrl;
                    Console.WriteLine("[RequestToRepairService] Use Custom Url: {0}", service.Url);
                }
                else
                {
                    Console.WriteLine("[RequestToRepairService] Use Default Url: {0}", service.Url);
                }

                switch (RequestType)
                {
                    case RequestType.Backup:
                        RequestBackup(service, config);
                        break;
                    case RequestType.Restore:
                        RequestRestore(service, config);
                        break;
                    default:
                        throw new Exception("unknown request type.");
                }
            },
            (e) => {
                Console.WriteLine("RequestToRepairService: {0}", e.Message);
            },
            5,
            TimeSpan.FromSeconds(1));

            return 0;
        }

        private void RequestBackup(RTellerService.RTellerService service, RequestToRepairServiceConfig config)
        {
            var request = File.ReadAllBytes(this.Input);

            if (request.Length != RepairConstant.BACKUP_REQUEST_COMMAND_FILE_SIZE)
            {
                throw new Exception(string.Format("Invalid request format. Size = {0}", request.Length));
            }

            var blobRequest = new RTellerService.EncryptKeyBlobRequest()
            {
                profileName = config.BackupProfileName,
                signature = request.Take(256).ToArray(),
                sessionId = request.Skip(256).Take(256).ToArray(),
                keyBlob = request.Skip(256).Skip(256).Take(256).ToArray(),
                id = new byte [] { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }
            };

            // device id
            char [] idstr = new char[16];
            Array.Copy(request.Skip(256).Skip(256).Skip(256).Take(16).ToArray(), idstr, 16);
            int j = 0;
            for (int i = 0; i < 16; i += 2)
            {
                char [] hexstr = new char[3];
                Array.Copy(idstr, i, hexstr, 0, 2);
                blobRequest.id[j++] = (byte)int.Parse(new string(hexstr) , System.Globalization.NumberStyles.HexNumber);
            }

            Console.WriteLine("[RequestToRepairService] Request:");
            Console.WriteLine("[RequestToRepairService]   Url: {0}", service.Url);
            Console.WriteLine("[RequestToRepairService]   Profile: {0}", blobRequest.profileName);
            Console.WriteLine("[RequestToRepairService]   KeyBlob: {0}\n{1}", blobRequest.keyBlob.Length, MakeHexExpression(blobRequest.keyBlob));
            Console.WriteLine("[RequestToRepairService]   Signature: {0}\n{1}", blobRequest.signature.Length, MakeHexExpression(blobRequest.signature));

            var result = service.EncryptKeyBlob(blobRequest);

            if (result.resultCode == RTellerService.ResultCode.OK)
            {
                var resultBytes = result.signature.Concat(result.initialVector).Concat(result.sessionId).Concat(result.keyBlob).ToArray();
                File.WriteAllBytes(this.Output, resultBytes);
                Console.WriteLine("[RequestToRepairService] Response Received");
                Console.WriteLine("[RequestToRepairService] ResponseData(Size={0}):\n{1}", resultBytes.Length, MakeHexExpression(resultBytes));
            }
            else
            {
                Console.WriteLine("RequestToRepairService: {0}\nError Message: {1}", result.resultCode.ToString(), result.errorMessage);
                throw new Exception(string.Format("RequestToRepairService: {0}\nError Message: {1}", result.resultCode.ToString(), result.errorMessage));
            }
        }

        private void RequestRestore(RTellerService.RTellerService service, RequestToRepairServiceConfig config)
        {
            var request = File.ReadAllBytes(this.Input);

            if (request.Length != RepairConstant.RESTORE_REQUEST_COMMAND_FILE_SIZE)
            {
                throw new Exception(string.Format("Invalid request format. Size = {0}", request.Length));
            }

            var blobRequest = new RTellerService.DecryptKeyBlobRequest()
            {
                profileName = config.RestoreProfileName,
                signature =     request.Take(256).ToArray(),
                initialVector = request.Skip(256).Take(16).ToArray(),
                sessionId =     request.Skip(256).Skip(16).Take(256).ToArray(),
                keyBlob =       request.Skip(256).Skip(16).Skip(256).Take(256).ToArray(),
                id = new byte[] { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }
            };

            // device id
            char[] idstr = new char[16];
            Array.Copy(request.Skip(256).Skip(16).Skip(256).Skip(256).Take(16).ToArray(), idstr, 16);
            int j = 0;
            for (int i = 0; i < 16; i += 2)
            {
                char[] hexstr = new char[3];
                Array.Copy(idstr, i, hexstr, 0, 2);
                blobRequest.id[j++] = (byte)int.Parse(new string(hexstr), System.Globalization.NumberStyles.HexNumber);
            }

            Console.WriteLine("[RequestToRepairService] Request:");
            Console.WriteLine("[RequestToRepairService]   Url: {0}", service.Url);
            Console.WriteLine("[RequestToRepairService]   Profile: {0}", blobRequest.profileName);
            Console.WriteLine("[RequestToRepairService]   IV: {0}\n{1}", blobRequest.initialVector.Length, MakeHexExpression(blobRequest.initialVector));
            Console.WriteLine("[RequestToRepairService]   signature: {0}\n{1}", blobRequest.signature.Length, MakeHexExpression(blobRequest.signature));
            Console.WriteLine("[RequestToRepairService]   keyblob: {0}\n{1}", blobRequest.keyBlob.Length, MakeHexExpression(blobRequest.keyBlob));

            var result = service.DecryptKeyBlob(blobRequest);

            if (result.resultCode == RTellerService.ResultCode.OK)
            {
                var resultBytes = result.signature.Concat(result.sessionId).Concat(result.keyBlob).ToArray();
                File.WriteAllBytes(this.Output, resultBytes);
                Console.WriteLine("[RequestToRepairService] Response Received");
                Console.WriteLine("[RequestToRepairService] ResponseData(Size={0}):\n{1}", resultBytes, MakeHexExpression(resultBytes));
            }
            else
            {
                Console.WriteLine("RequestToRepairService: {0}\nError Message: {1}", result.resultCode.ToString(), result.errorMessage);
                throw new Exception(string.Format("RequestToRepairService: {0}\nError Message: {1}", result.resultCode.ToString(), result.errorMessage));
            }
        }

        private byte[] EncryptKey(byte[] key, byte[] sessionKey)
        {
            return CommandUtility.CryptUtility.EncryptAesEcb128(key, sessionKey);
        }

        private RequestToRepairServiceConfig LoadConfig(string configPath)
        {
            if (configPath == null)
            {
                return RequestToRepairServiceConfig.GetDefault();
            }
            else
            {
                return CommandUtility.ConvertUtility.LoadYaml<RequestToRepairServiceConfig>(File.ReadAllText(configPath));
            }
        }

        private string MakeHexExpression(byte[] data)
        {
            return string.Join("", data.Select((v,i)=> v.ToString("X2") + ( ((i+1) % 16) == 0 ? "\n" : " " )));
        }
    }

    class Program
    {
        static void Main(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

            try
            {
            RequestToRepairService parsed;
            var parser = new Nintendo.Foundation.IO.CommandLineParser();

            if (false == parser.ParseArgs<RequestToRepairService>(args, out parsed))
            {
                System.Environment.Exit(1);
            }

            System.Environment.Exit(parsed.Run());
            }
            catch (Exception exception)
            {
                PrintException(exception);
                System.Environment.Exit(1);
            }
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Console.Error.WriteLine(string.Format("StackTrace: {0}", exception.StackTrace));
        }
    }
}
