﻿using System;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;
using CommandUtility;
using System.IO;
using RepairCommon;

namespace BackupRestoreTool
{
    class Utility
    {
        public static void WaitFile(FileInfo file, int fileSize, int timeout)
        {
            for (int i = 0; i < timeout; i++)
            {
                if (File.Exists(file.FullName))
                {
                    var len = new FileInfo(file.FullName).Length;
                    Console.WriteLine("File found. size={0}", len);

                    if (len == fileSize)
                    {
                        break;
                    }
                }

                Task.Delay(1000).Wait();
                Console.WriteLine("Wait {0}/{1}, path = {2}", i + 1, timeout, file.FullName);
            }
        }
    }

    abstract class BackupRestoreToolCommon : SubCommandCommon
    {
        [CommandLineOption("workdir",
            Description = "set working directory.")]
        public string WorkingDirectoryPath { get; set; }

        [CommandLineOption("config",
            Description = "config file.",
            IsRequired = true)]
        public string Config { get; set; }

        [CommandLineOption("timeout",
            Description = "set timeout.",
            IsRequired = false,
            DefaultValue = 360)]

        public int Timeout { get; set; }

        [CommandLineOption("usb_device_path",
            Description = "USB device patht.",
            IsRequired = false,
            DefaultValue = "")]

        public string UsbDevicePath { get; set; }
        [CommandLineOption("autoexit",
            Description = "automatically exit application.",
            IsRequired = false)]
        public bool IsAutoExit { get; set; }

        [CommandLineOption("autostart",
            Description = "automatically start application without waiting user input.",
            IsRequired = false)]
        public bool IsAutoStart { get; set; }


        protected FileInfo RequestFile;
        protected FileInfo ResponseFile;
        protected const string BackupToolProgramId = "0x0100000000002057";
        protected const string RestoreToolProgramId = "0x0100000000002059";

        // USB デバイスの GUID
        protected static readonly Guid UsbDeviceGuid = new Guid("97FFFD48-2D1D-47A0-85A3-07FDE6FA0143");

        protected void RequestToRepairServer(string type, int requestFileSize)
        {
            Utility.WaitFile(RequestFile, requestFileSize, Timeout);

            SdkTool.Execute("RequestToRepairService.exe",
                "--type", type,
                "--input", RequestFile.FullName,
                "--output", ResponseFile.FullName,
                "--config", Config);
        }
    }

    class BackupCommand : BackupRestoreToolCommon
    {
        public override int Run()
        {
            using (var temporaryHolder = new TemporaryFileHolder("BackupRepairTool", WorkingDirectoryPath))
            {
                RequestFile = temporaryHolder.CreateTemporaryFilePath("request");
                ResponseFile = temporaryHolder.CreateTemporaryFilePath("response");

                TaskUtility.RunTasks(() =>
                {
                    RunBackupToolOnTarget();
                },
                () =>
                {
                    var RequestSize = RepairConstant.BACKUP_REQUEST_COMMAND_FILE_SIZE;
                    RequestToRepairServer("Backup", RequestSize);
                });

                return 0;
            }
        }

        private void RunBackupToolOnTarget()
        {
#if false
            SdkTool.Execute("RunOnTarget.exe",
                Path.Combine(SdkPath.FindSdkRoot(), "Programs/Iris/Outputs/NX-NXFP2-a64/TargetTools/RepairBackup/Develop/RepairBackup.nsp"));
#else
            Console.WriteLine("Execute the backup tool.");
            var argument = SdkTool.MakeArgumentString(new string[] {
                    "--request", RequestFile.FullName,
                    "--response", ResponseFile.FullName });
            if (IsAutoExit)
            {
                argument += " --autoexit";
            }
            SdkTool.Execute("RunOnTargetFromNand.exe",
                "--timeout", Timeout.ToString(),
                "--id", BackupToolProgramId,
                "--usb_device_path", UsbDevicePath,
                "--args", argument
            );
#endif
        }
    }

    class RestoreCommand : BackupRestoreToolCommon
    {
        public override int Run()
        {
            using (var temporaryHolder = new TemporaryFileHolder("BackupRepairTool", WorkingDirectoryPath))
            {
                RequestFile = temporaryHolder.CreateTemporaryFilePath("request");
                ResponseFile = temporaryHolder.CreateTemporaryFilePath("response");

                TaskUtility.RunTasks(() =>
                {
                    RunRestoreToolOnTarget();
                },
                () =>
                {
                    var RequestSize = RepairConstant.RESTORE_REQUEST_COMMAND_FILE_SIZE;
                    RequestToRepairServer("Restore", RequestSize);
                });

                return 0;
            }
        }

        private void RunRestoreToolOnTarget()
        {
#if false
            SdkTool.Execute("RunOnTarget.exe",
                Path.Combine(SdkPath.FindSdkRoot(), "Programs/Iris/Outputs/NX-NXFP2-a64/TargetTools/RepairRestore/Develop/RepairRestore.nsp"));
#else
            Console.WriteLine("Execute the restore tool.");
            var argument = SdkTool.MakeArgumentString(new string[] {
                    "--request", RequestFile.FullName,
                    "--response", ResponseFile.FullName });
            if (IsAutoExit)
            {
                argument += " --autoexit";
            }
            if (IsAutoStart)
            {
                argument += " --autostart";
            }
            SdkTool.Execute("RunOnTargetFromNand.exe",
                "--timeout", Timeout.ToString(),
                "--id", RestoreToolProgramId,
                "--usb_device_path", UsbDevicePath,
                "--args", argument
            );
#endif
        }
    }

    class BackupRestoreTool
    {
        [CommandLineSubCommand("backup", Description = "Backup")]
        public BackupCommand BackupCommand { get; set; }

        [CommandLineSubCommand("restore", Description = "Restore")]
        public RestoreCommand RestoreCommand { get; set; }
    }

    class Program
    {
        public static void Main(string[] args)
        {
            if(MultiCommandRunner<BackupRestoreTool>.Run(args) != 0)
            {
                System.Environment.Exit(1);
            }
        }
    }
}
