﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VibrationConverterConsole.BnvibIO
{
    public enum BnvibFileFormatId
    {
        Reserved = 0,
        AmFm5bitVer1 = 1,
        AmFm5bitVer2 = 2,
        PcmVer1 = 3,
    }

    public class BnvibFileInfo
    {
        public const int DefaultMetaDataSize = MetaDataSizeBasic;
        public const BnvibFileFormatId DefaultFormatId = BnvibFileFormatId.PcmVer1;
        public const int DefaultSamplingRate = 200;

        public const int MetaDataSizeBasic = 4;
        public const int MetaDataSizeWithLoop = 12;
        public const int MetaDataSizeWithLoopInterval = 16;

        public BnvibFileInfo()
        {
            MetaDataSize = DefaultMetaDataSize;
            FormatId = DefaultFormatId;
            SamplingRate = DefaultSamplingRate;
            DataSize = 0;
            SampleLength = 0;
            LoopStart = 0;
            LoopEnd = 0;
            LoopInterval = 0;
        }

        public int MetaDataSize { get; set; }
        public bool HasLoop
        {
            get { return (MetaDataSize >= MetaDataSizeWithLoop); }
            set
            {
                if (value)
                {
                    if (MetaDataSize < MetaDataSizeWithLoop)
                    {
                        MetaDataSize = MetaDataSizeWithLoop;
                    }
                }
                else
                {
                    MetaDataSize = MetaDataSizeBasic;
                }
            }
        }
        public bool HasLoopInterval
        {
            get { return (MetaDataSize >= MetaDataSizeWithLoopInterval); }
            set
            {
                if (value)
                {
                    if (MetaDataSize < MetaDataSizeWithLoopInterval)
                    {
                        MetaDataSize = MetaDataSizeWithLoopInterval;
                    }
                }
                else
                {
                    MetaDataSize = MetaDataSizeWithLoop;
                }
            }
        }
        public BnvibFileFormatId FormatId { get; set; }
        public int SamplingRate { get; set; }
        public int DataSize { get; set; }
        public int SampleLength { get; set; }
        public int LoopStart { get; set; }
        public int LoopEnd { get; set; }
        public int LoopInterval { get; set; }
    }

    public static class AmplitudeConverter
    {
        public static float GetAmplitude(byte b)
        {
            return (float)b / (float)(byte.MaxValue);
        }
        public static byte GetByte(float a)
        {
            double v = Math.Round(a * byte.MaxValue);
            if (v < 0.0) return 0;
            if (v > byte.MaxValue) return byte.MaxValue;
            return (byte)v;
        }
    }

    public static class FrequencyConverter
    {
        private const int FreqTableLength = 256;
        private const double MinFreq = 10.0;
        private const double FreqResolution = 32.0;
        private static float[] s_FreqTable = null;
        public static float GetFrequency(byte b)
        {
            if (s_FreqTable == null)
            {
                s_FreqTable = new float[FreqTableLength];
                for (int i = 0; i < FreqTableLength; i++)
                {
                    s_FreqTable[i] = (float)(MinFreq * Math.Pow(2.0, i / FreqResolution));
                }
            }
            return s_FreqTable[b];
        }
        public static byte GetByte(float f)
        {
            if (f <= 0.0f) return 0;
            double v = Math.Round(Math.Log(f / MinFreq, 2.0) * FreqResolution);
            if (v < 0.0) return 0;
            if (v > 255.0) return 255;
            return (byte)v;
        }
    }

    public struct VibrationValue
    {
        public float amplitudeLow;
        public float frequencyLow;
        public float amplitudeHigh;
        public float frequencyHigh;

        public const int RawDataSize = 4;
        public const float DefaultFrequencyLow = 160.0f;
        public const float DefaultFrequencyHigh = 320.0f;

        public static VibrationValue Make()
        {
            return Make(0.0f, DefaultFrequencyLow, 0.0f, DefaultFrequencyHigh);
        }
        public static VibrationValue Make(float amplitudeLow, float frequencyLow, float amplitudeHigh, float frequencyHigh)
        {
            VibrationValue v;
            v.amplitudeLow = amplitudeLow;
            v.frequencyLow = frequencyLow;
            v.amplitudeHigh = amplitudeHigh;
            v.frequencyHigh = frequencyHigh;
            return v;
        }

        public static VibrationValue FromRawData(byte[] rawData)
        {
            VibrationValue v;
            v.amplitudeLow = AmplitudeConverter.GetAmplitude(rawData[0]);
            v.frequencyLow = FrequencyConverter.GetFrequency(rawData[1]);
            v.amplitudeHigh = AmplitudeConverter.GetAmplitude(rawData[2]);
            v.frequencyHigh = FrequencyConverter.GetFrequency(rawData[3]);
            return v;
        }

        public byte[] ToRawData()
        {
            byte[] buf = new byte[4];
            buf[0] = AmplitudeConverter.GetByte(amplitudeLow);
            buf[1] = FrequencyConverter.GetByte(frequencyLow);
            buf[2] = AmplitudeConverter.GetByte(amplitudeHigh);
            buf[3] = FrequencyConverter.GetByte(frequencyHigh);
            return buf;
        }
    }

    public struct AmFmCodePair
    {
        public const int AmFmCodeBitWidth = 5;
        public const int AmFmCodePairBitWidth = AmFmCodeBitWidth * 2;
        public const int AmFmCodeValueMin = 0;
        public const int AmFmCodeValueMax = (1 << AmFmCodeBitWidth) - 1;
        public const UInt32 AmFmCodeLowMask = AmFmCodeHighMask << AmFmCodeBitWidth;
        public const UInt32 AmFmCodeHighMask = ((UInt32)1 << AmFmCodeBitWidth) - 1;

        public int amfmCodeLow;
        public int amfmCodeHigh;

        public UInt32 ToRawData()
        {
            return ((UInt32)amfmCodeLow << AmFmCodeBitWidth) | (UInt32)amfmCodeHigh;
        }
        static public AmFmCodePair FromRawData(UInt32 rawData)
        {
            AmFmCodePair pair;
            pair.amfmCodeLow = (int)((rawData & AmFmCodeLowMask) >> AmFmCodeBitWidth);
            pair.amfmCodeHigh = (int)(rawData & AmFmCodeHighMask);
            return pair;
        }
    }

    public struct PackedAmFmCodes
    {
        private const int PairCountBitShift = 30;
        public const int RawDataSize = sizeof(UInt32);

        private UInt32 m_RawData;

        public UInt32 ToRawData()
        {
            return m_RawData;
        }
        static public PackedAmFmCodes FromRawData(UInt32 rawData)
        {
            PackedAmFmCodes pack;
            pack.m_RawData = rawData;
            return pack;
        }

        private static int GetAmFmCodePairBitShift(int index)
        {
            return PairCountBitShift - AmFmCodePair.AmFmCodePairBitWidth * (index + 1);
        }
        public AmFmCodePair[] GetAmFmCodePairArray()
        {
            int pairCount = (int)(m_RawData >> PairCountBitShift);
            AmFmCodePair[] pairArray = new AmFmCodePair[pairCount];

            for (int i = 0; i < pairCount; i++)
            {
                int bitShift = GetAmFmCodePairBitShift(i);
                pairArray[i] = AmFmCodePair.FromRawData(m_RawData >> bitShift);
            }
            return pairArray;
        }
        static public PackedAmFmCodes FromAmFmCodePairArray(AmFmCodePair[] pairArray)
        {
            PackedAmFmCodes pack;
            pack.m_RawData = ((UInt32)pairArray.Length << PairCountBitShift);

            for (int i = 0; i < pairArray.Length; i++)
            {
                int bitShift = GetAmFmCodePairBitShift(i);
                pack.m_RawData |= (pairArray[i].ToRawData() << bitShift);
            }
            return pack;
        }
    }

    public class BnvibSampleData
    {
        private List<byte> m_RawByteList;

        public BnvibSampleData()
        {
            m_RawByteList = new List<byte>();
        }

        public int SampleCount
        {
            get { return m_RawByteList.Count() / 4 ; }
        }

        public byte[] ToRawData()
        {
            return m_RawByteList.ToArray();
        }

        public VibrationValue GetVibrationValueAt(int index)
        {
            if (0 <= index && index < SampleCount)
            {
                byte[] byteArray = m_RawByteList.GetRange(index * 4, 4).ToArray();
                return VibrationValue.FromRawData(byteArray);
            }
            else
            {
                return VibrationValue.Make();
            }
        }

        public void SetVibrationValueAt(int index, VibrationValue v)
        {
            if (0 <= index && index < SampleCount)
            {
                byte[] byteArray = v.ToRawData();
                for (int i = 0; i < 4; i++)
                {
                    m_RawByteList[index * 4 + i] = byteArray[i];
                }
            }
        }

        public void AddVibrationValue(VibrationValue v)
        {
            byte[] byteArray = v.ToRawData();
            m_RawByteList.AddRange(byteArray);
        }

        public static BnvibSampleData FromRawData(byte[] rawData)
        {
            BnvibSampleData data = new BnvibSampleData();
            data.m_RawByteList.AddRange(rawData);
            return data;
        }
    }

    public class BnvibFile
    {
        public BnvibFile()
        {
            Info = new BnvibFileInfo();
            Data = new BnvibSampleData();
        }
        public BnvibFileInfo Info { get; set; }
        public BnvibSampleData Data { get; set; }
    }
}
