﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Net;
    using System.Net.Sockets;
    using System.Text;
    using System.Threading;
    using System.Threading.Tasks;
    using System.Xml.Linq;
    using Microsoft.Win32;

    /// <summary>
    /// 開発機を管理します。
    /// </summary>
    internal sealed class TargetManager
    {
        private const string HostName = "localhost";

        private const string PortName = "iywys@$hid";

        private const int RetryInterval = 250;

        private const int UpdateInterval = 1000;

        private const int DefaultBufferSize = 1024;

        private static readonly Encoding Encoding = new UTF8Encoding(false);

        private readonly object taskSyncObject = new object();

        private CancellationTokenSource tokenSource = null;

        private Task task = null;

        private readonly IDictionary<string, TargetInfo> portMap;

        private class TargetManagerException : Exception
        {
            public TargetManagerException(string message) :
                base(message)
            {
            }
        }

        /// <summary>
        /// TargetManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal TargetManager()
        {
            this.portMap = new Dictionary<string, TargetInfo>();
        }

        /// <summary>
        /// 開発機が接続されると発生します。
        /// </summary>
        internal event EventHandler<TargetInfo> TargetConnected;

        /// <summary>
        /// 開発機が切断されると発生します。
        /// </summary>
        internal event EventHandler<TargetInfo> TargetDisconnected;

        /// <summary>
        /// 開発機の管理を開始します。
        /// </summary>
        internal void Start()
        {
            lock (this.taskSyncObject)
            {
                this.tokenSource = new CancellationTokenSource();

                CancellationToken token = this.tokenSource.Token;

                this.task = Task.Run(() => this.Monitor(token));
            }
        }

        /// <summary>
        /// 開発機の管理を終了します。
        /// </summary>
        internal void Stop()
        {
            lock (this.taskSyncObject)
            {
                this.tokenSource.Cancel();

                this.task.Wait();
                this.task.Dispose();
                this.task = null;

                this.tokenSource.Dispose();
                this.tokenSource = null;
            }
        }

        private static StreamReader CreateStreamReader(TcpClient client)
        {
            NetworkStream stream = client.GetStream();

            var useBom = false;

            var leaveOpen = true;

            return new StreamReader(
                stream, Encoding, useBom, DefaultBufferSize, leaveOpen);
        }

        private static void RequestSystemPortMapping(
            TcpClient client, CancellationToken token)
        {
            NetworkStream stream = client.GetStream();

            var leaveOpen = true;

            using (var sw = new StreamWriter(
                stream, Encoding, DefaultBufferSize, leaveOpen))
            {
                var xelement = new XElement("RequestSystemPortMapping");

                var value = xelement.ToString(SaveOptions.DisableFormatting);

                using (Task task = sw.WriteLineAsync(value))
                {
                    task.Wait(token);
                }
            }
        }

        private static IDictionary<string, TargetInfo> ParsePortMap(
            XElement htcsInfo)
        {
            var portMap = new Dictionary<string, TargetInfo>();

            foreach (XElement item in htcsInfo.Descendants(Tag.PortMapItem))
            {
                if (item.Element(Tag.HtcsPortName).Value != PortName)
                {
                    continue;
                }

                string portNumber = item.Element(Tag.TcpPortNumber).Value;

                int port = 0;

                if (!int.TryParse(portNumber, out port))
                {
                    continue;
                }

                string peerName = item.Element(Tag.HtcsPeerName).Value;

                string ipAddress = item.Element(Tag.IPAddress).Value;

                var info = new TargetInfo(peerName, ipAddress, port);

                portMap[peerName] = info;
            }

            return portMap;
        }

        private bool IsHtcsControlPort(int port)
        {
            try
            {
                using (var socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp))
                {
                    socket.Connect(new IPEndPoint(IPAddress.Loopback, port));
                    using (var socketStream = new NetworkStream(socket))
                    using (var socketReader = new StreamReader(socketStream))
                    {
                        Task<string> str = socketReader.ReadLineAsync();
                        if (str.Wait(3000))
                        {
                            return (str.Result.IndexOf("<HtcsInfo><TargetList>") == 0);
                        }
                        return false;
                    }
                }
            }
            catch
            {
                return false;
            }
        }

        private int GetHtcsControlPort()
        {
            bool isRegistryFound = false;
            int port = 8003;
            try
            {
                RegistryKey regkey = Registry.CurrentUser.OpenSubKey(@"Software\Nintendo\NX\Oasis\Ports", false);

                if (regkey?.GetValue("HtcsControl") != null)
                {
                    port = (int)regkey.GetValue("HtcsControl");
                    isRegistryFound = true;
                }
                regkey.Close();
            }
            catch (Exception e)
            {
                Debug.WriteLine("Could not read Registry: Exception {0}", e.ToString());
            }

            if (!IsHtcsControlPort(port))
            {
                if (isRegistryFound)
                {
                    throw new TargetManagerException("Target Manager is in an invalid state. Please restart Target Manager.");
                }
                port = 7974;
            }
            return port;
        }

        private void Monitor(CancellationToken token)
        {
            TcpClient client = null;

            var isRequested = false;

            StreamReader reader = null;

            while (!token.IsCancellationRequested)
            {
                if (client == null)
                {
                    try
                    {
                        client = new TcpClient(HostName, GetHtcsControlPort());
                    }
                    catch
                    {
                        token.WaitHandle.WaitOne(
                            TimeSpan.FromMilliseconds(RetryInterval));

                        continue;
                    }
                }

                if (!isRequested)
                {
                    try
                    {
                        RequestSystemPortMapping(client, token);

                        isRequested = true;
                    }
                    catch
                    {
                        client.Close();
                        client = null;

                        continue;
                    }
                }

                if (reader == null)
                {
                    reader = CreateStreamReader(client);
                }

                IDictionary<string, TargetInfo> newPortMap = null;

                try
                {
                    string line = null;

                    using (Task<string> task = reader.ReadLineAsync())
                    {
                        task.Wait(token);
                        line = task.GetAwaiter().GetResult();
                    }

                    newPortMap = ParsePortMap(XElement.Parse(line));
                }
                catch
                {
                    this.ClearPortMap();

                    reader.Close();
                    reader = null;

                    isRequested = false;

                    client.Close();
                    client = null;

                    continue;
                }

                this.UpdatePortMap(newPortMap);

                token.WaitHandle.WaitOne(
                    TimeSpan.FromMilliseconds(UpdateInterval));
            }

            this.ClearPortMap();

            if (reader != null)
            {
                reader.Close();
            }

            if (client != null)
            {
                client.Close();
            }
        }

        private void UpdatePortMap(IDictionary<string, TargetInfo> newPortMap)
        {
            ICollection<string> oldKeys = this.portMap.Keys;

            ICollection<string> newKeys = newPortMap.Keys;

            foreach (string key in oldKeys.Except(newKeys))
            {
                this.TargetDisconnected(this, this.portMap[key]);
            }

            foreach (string key in newKeys.Except(oldKeys))
            {
                this.TargetConnected(this, newPortMap[key]);
            }

            this.portMap.Clear();

            foreach (KeyValuePair<string, TargetInfo> pair in newPortMap)
            {
                this.portMap.Add(pair);
            }
        }

        private void ClearPortMap()
        {
            foreach (KeyValuePair<string, TargetInfo> pair in this.portMap)
            {
                this.TargetDisconnected(this, pair.Value);
            }

            this.portMap.Clear();
        }

        private static class Tag
        {
            internal const string PortMapItem = "PortMapItem";

            internal const string HtcsPeerName = "HtcsPeerName";

            internal const string HtcsPortName = "HtcsPortName";

            internal const string IPAddress = "IPAddress";

            internal const string TcpPortNumber = "TcpPortNumber";
        }
    }
}
