﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Runtime.InteropServices;

    /// <summary>
    /// 構造体のバイナリ変換を扱うクラスです。
    /// </summary>
    internal static class StructConverter
    {
        /// <summary>
        /// 構造体をバイト配列に変換します。
        /// </summary>
        /// <typeparam name="T">構造体の型です。</typeparam>
        /// <param name="value">構造体の値です。</param>
        /// <returns>バイト配列です。</returns>
        internal static byte[] ToBytes<T>(T value) where T : struct
        {
            int size = Marshal.SizeOf<T>();

            IntPtr ptr = Marshal.AllocHGlobal(size);

            Marshal.StructureToPtr(value, ptr, false);

            var bytes = new byte[size];

            Marshal.Copy(ptr, bytes, 0, size);

            Marshal.FreeHGlobal(ptr);

            AdjustEndian<T>(bytes);

            return bytes;
        }

        /// <summary>
        /// バイト配列を構造体に変換します。
        /// </summary>
        /// <typeparam name="T">構造体の型です。</typeparam>
        /// <param name="bytes">バイト配列です。</param>
        /// <returns>構造体の値です。</returns>
        internal static T FromBytes<T>(byte[] bytes) where T : struct
        {
            AdjustEndian<T>(bytes);

            GCHandle gch = GCHandle.Alloc(bytes, GCHandleType.Pinned);

            var value = Marshal.PtrToStructure<T>(gch.AddrOfPinnedObject());

            gch.Free();

            return value;
        }

        private static void AdjustEndian<T>(byte[] bytes) where T : struct
        {
            foreach (var field in typeof(T).GetFields())
            {
                var offset = Marshal.OffsetOf<T>(field.Name).ToInt32();

                var fieldType = field.FieldType;

                Array.Reverse(bytes, offset, Marshal.SizeOf(fieldType));
            }
        }
    }
}
