﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.IO;
    using System.IO.MemoryMappedFiles;
    using System.Threading;

    /// <summary>
    /// メモリマップドファイルを扱うクラスです。
    /// </summary>
    internal sealed class Mmap : IDisposable
    {
        private const string MutexSuffix = "_MUTEX";

        private bool isDisposed = false;

        private readonly string name;

        private readonly long size;

        private readonly Mutex mutex;

        private MemoryMappedFile file = null;

        private MemoryMappedViewAccessor accessor = null;

        /// <summary>
        /// Mmap クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="name">ファイル名です。</param>
        /// <param name="size">ファイルサイズです。</param>
        internal Mmap(string name, long size)
        {
            this.name = name;

            this.size = size;

            this.mutex = new Mutex(false, name + MutexSuffix);
        }

        ~Mmap()
        {
            this.Dispose(false);
        }

        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// メモリマップドファイル上で排他的にデリゲートを実行します。
        /// </summary>
        /// <param name="action">デリゲートです。</param>
        internal void Invoke(Action<UnmanagedMemoryAccessor, bool> action)
        {
            try
            {
                this.mutex.WaitOne();

                var exists = true;

                if (this.accessor == null)
                {
                    try
                    {
                        this.file = MemoryMappedFile.OpenExisting(this.name);
                    }
                    catch (FileNotFoundException)
                    {
                        exists = false;

                        this.file = MemoryMappedFile.CreateNew(this.name, this.size);
                    }

                    this.accessor = this.file.CreateViewAccessor();
                }

                action(this.accessor, exists);
            }
            finally
            {
                this.mutex.ReleaseMutex();
            }
        }

        private void Dispose(bool isDisposing)
        {
            if (!this.isDisposed)
            {
                if (isDisposing)
                {
                    if (this.accessor != null)
                    {
                        this.accessor.Dispose();
                    }

                    if (this.file != null)
                    {
                        this.file.Dispose();
                    }

                    this.mutex.Dispose();
                }

                this.isDisposed = true;
            }
        }
    }
}
