﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Collections.Generic;
    using System.Reflection;
    using System.Runtime.InteropServices;
    using System.Windows.Forms;

    /// <summary>
    /// キーボードを管理します。
    /// </summary>
    internal sealed class KeyboardManager : IDisposable
    {
        private const int WH_KEYBOARD_LL = 13;

        private const int WM_KEYDOWN = 0x0100;
        private const int WM_KEYUP = 0x0101;
        private const int WM_SYSKEYDOWN = 0x0104;
        private const int WM_SYSKEYUP = 0x0105;

        private bool disposed = false;

        private object syncObject = new object();

        private bool isSuppressionEnabled = false;

        private readonly IList<HookProc> hookProcs = new List<HookProc>();

        private readonly IntPtr hHook;

        /// <summary>
        /// KeyboardManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal KeyboardManager()
        {
            // ガベージコレクションから保護
            this.hookProcs.Add(new HookProc(this.KeyboardHookProc));

            this.hHook = SetKeyboardHook(this.hookProcs[0]);
        }

        ~KeyboardManager()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// キーボード入力イベントの伝搬を隠蔽するか否かを表す値を取得または設定します。
        /// </summary>
        internal bool IsSuppressionEnabled
        {
            get
            {
                lock (this.syncObject)
                {
                    return this.isSuppressionEnabled;
                }
            }

            set
            {
                lock (this.syncObject)
                {
                    this.isSuppressionEnabled = value;
                }
            }
        }

        /// <summary>
        /// キーボードのキーが押されると発生します。
        /// </summary>
        internal event EventHandler<KeyboardEventArgs> KeyDown = delegate {};

        /// <summary>
        /// キーボードのキーが離されると発生します。
        /// </summary>
        internal event EventHandler<KeyboardEventArgs> KeyUp = delegate {};

        /// <summary>
        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        private void Dispose(bool isDisposing)
        {
            if (!this.disposed)
            {
                this.disposed = true;

                UnhookKeyboardHook(this.hHook);

                if (isDisposing)
                {
                    this.hookProcs.Clear();
                }
            }
        }

        private KeyboardEventArgs CreateKeyboardEventArgs(
            IntPtr wParam, IntPtr lParam)
        {
            var message = wParam.ToInt32();

            var state = (KBDLLHOOKSTRUCT)
                Marshal.PtrToStructure(lParam, typeof(KBDLLHOOKSTRUCT));

            var key = (Keys)state.vkCode;

            var scanCode = MapVirtualKeyToScanCode(key);

            var usageId = MapScanCodeToHidUsageId(scanCode);

            return new KeyboardEventArgs(key, usageId);
        }

        [DllImport("user32.dll")]
        private static extern IntPtr SetWindowsHookEx(
            int idHook, HookProc lpfn, IntPtr hMod, int dwThreadId);

        [DllImport("user32.dll")]
        private static extern bool UnhookWindowsHookEx(IntPtr hHook);

        [DllImport("user32.dll")]
        private static extern IntPtr CallNextHookEx(
            IntPtr hHook, int nCode, IntPtr wParam, IntPtr lParam);

        [DllImport("user32.dll")]
        private static extern uint MapVirtualKey(uint uCode, uint uMapType);

        private static IntPtr SetKeyboardHook(HookProc hookProc)
        {
            return SetWindowsHookEx(
                WH_KEYBOARD_LL,
                hookProc,
                Marshal.GetHINSTANCE(
                    Assembly.GetExecutingAssembly().GetModules()[0]),
                0);
        }

        private static void UnhookKeyboardHook(IntPtr hHook)
        {
            if (hHook != IntPtr.Zero)
            {
                UnhookWindowsHookEx(hHook);
            }
        }

        private static uint MapVirtualKeyToScanCode(Keys key)
        {
            switch (key)
            {
                case Keys.Prior:
                    return 0xE049;
                case Keys.Next:
                    return 0xE051;
                case Keys.End:
                    return 0xE04F;
                case Keys.Home:
                    return 0xE047;
                case Keys.Left:
                    return 0xE04B;
                case Keys.Up:
                    return 0xE048;
                case Keys.Right:
                    return 0xE04D;
                case Keys.Down:
                    return 0xE050;
                case Keys.Snapshot:
                    return 0xE037;
                case Keys.Insert:
                    return 0xE052;
                case Keys.Delete:
                    return 0xE053;
                default:
                    return MapVirtualKey((uint)key, 0x04);
            }
        }

        private static uint MapScanCodeToHidUsageId(uint scanCode)
        {
            var usageIdMap = new[] {
                0x00, 0x29, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23,
                0x24, 0x25, 0x26, 0x27, 0x2D, 0x2E, 0x2A, 0x2B,
                0x14, 0x1A, 0x08, 0x15, 0x17, 0x1C, 0x18, 0x0C,
                0x12, 0x13, 0x2F, 0x30, 0x28, 0xE0, 0x04, 0x16,
                0x07, 0x09, 0x0A, 0x0B, 0x0D, 0x0E, 0x0F, 0x33,
                0x34, 0x35, 0xE1, 0x31, 0x1D, 0x1B, 0x06, 0x19,
                0x05, 0x11, 0x10, 0x36, 0x37, 0x38, 0xE5, 0x55,
                0xE2, 0x2C, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E,
                0x3F, 0x40, 0x41, 0x42, 0x43, 0x53, 0x47, 0x5F,
                0x60, 0x61, 0x56, 0x5C, 0x5D, 0x5E, 0x57, 0x59,
                0x5A, 0x5B, 0x62, 0x63, 0x00, 0x00, 0x64, 0x44,
                0x45, 0x67, 0x00, 0x00, 0x8C, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00, 0x68, 0x69, 0x6A, 0x6B,
                0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72, 0x00,
                0x88, 0x00, 0x00, 0x87, 0x00, 0x00, 0x73, 0x93,
                0x92, 0x8A, 0x00, 0x8B, 0x00, 0x89, 0x85, 0x00
            };

            if (0 <= scanCode && scanCode < usageIdMap.Length)
            {
                return (uint)usageIdMap[scanCode];
            }

            switch (scanCode)
            {
                case 0x00F1: return 0x91;
                case 0x00F2: return 0x90;
                case 0xE01C: return 0x58;
                case 0xE01D: return 0xE4;
                case 0xE035: return 0x54;
                case 0xE037: return 0x46;
                case 0xE038: return 0xE6;
                case 0xE046: return 0x48;
                case 0xE047: return 0x4A;
                case 0xE048: return 0x52;
                case 0xE049: return 0x4B;
                case 0xE04B: return 0x50;
                case 0xE04D: return 0x4F;
                case 0xE04F: return 0x4D;
                case 0xE050: return 0x51;
                case 0xE051: return 0x4E;
                case 0xE052: return 0x49;
                case 0xE053: return 0x4C;
                case 0xE05B: return 0xE3;
                case 0xE05C: return 0xE7;
                case 0xE05D: return 0x65;
                case 0xE05E: return 0x66;
                case 0xE11D: return 0x48;
                default: return 0x00;
            }
        }

        private IntPtr KeyboardHookProc(
            int nCode, IntPtr wParam, IntPtr lParam)
        {
            bool isSuppressionEnabled = this.IsSuppressionEnabled;

            bool isKeyUpEvent = false;

            if (nCode == 0)
            {
                KeyboardEventArgs args =
                    CreateKeyboardEventArgs(wParam, lParam);

                switch (wParam.ToInt32())
                {
                    case WM_KEYDOWN:
                    case WM_SYSKEYDOWN:
                        this.KeyDown(this, args);
                        break;

                    case WM_KEYUP:
                    case WM_SYSKEYUP:
                        isKeyUpEvent = true;
                        this.KeyUp(this, args);
                        break;

                    default:
                        break;
                }
            }

            if ((isSuppressionEnabled || this.IsSuppressionEnabled) &&
                !isKeyUpEvent)
            {
                return (IntPtr)1;
            }
            else
            {
                return CallNextHookEx(this.hHook, nCode, wParam, lParam);
            }
        }

        private delegate IntPtr HookProc(
            int nCode, IntPtr wParam, IntPtr lParam);

        [StructLayout(LayoutKind.Sequential)]
        private struct KBDLLHOOKSTRUCT
        {
            internal uint vkCode;
            internal uint scanCode;
            internal uint flags;
            internal uint time;
            internal IntPtr dwExtraInfo;
        }
    }
}
