﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Runtime.InteropServices;
    using System.Text;
    using Resources;

    /// <summary>
    /// HidShell のポートアクセスを扱うクラスです。
    /// </summary>
    internal sealed class HidShellPortAccessor : IDisposable
    {
        /// <summary>
        /// 既定のポートです。
        /// </summary>
        internal const uint DefaultPort = 0u;

        /// <summary>
        /// 入力方向です。
        /// </summary>
        internal const uint In = 0u;

        /// <summary>
        /// 出力方向です。
        /// </summary>
        internal const uint Out = 1u;

        private bool isDisposed = false;

        /// <summary>
        /// HidShellPortAccessor クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal HidShellPortAccessor()
        {
            var handle = IntPtr.Zero;

            var result = (HidShellResult)
                Native.AquireHidShellExclusiveRight(out handle);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }

            this.Handle = handle;
        }

        ~HidShellPortAccessor()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// ポートのハンドルを返します。
        /// </summary>
        internal IntPtr Handle { get; private set; }

        /// <summary>
        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// 指定された開発機をポートをアタッチします。
        /// </summary>
        /// <param name="targetName">開発機の名前です。</param>
        /// <returns>ポートのアタッチに成功したか否かを表す値です。</returns>
        internal bool AttachPort(string targetName)
        {
            PortName name = CreatePortName(targetName);

            var result = (HidShellResult)
                Native.AttachHidShellPort(this.Handle, ref name);

            switch (result)
            {
                case HidShellResult.Success:
                    return true;

                case HidShellResult.NoAvailablePort:
                    return false;

                default:
                    throw new HidShellException(result);
            }
        }

        /// <summary>
        /// 指定された開発機をポートからデタッチします。
        /// </summary>
        /// <param name="targetName">開発機の名前です。</param>
        internal void DetachPort(string targetName)
        {
            PortName name = CreatePortName(targetName);

            var result = (HidShellResult)
                Native.DetachHidShellPort(this.Handle, ref name);

            switch (result)
            {
                case HidShellResult.Success:
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        /// <summary>
        /// 指定された開発機が登録されたポートを返します。
        /// </summary>
        /// <param name="outPort">ポートを格納します。</param>
        /// <param name="targetName">開発機の名前です。</param>
        /// <returns>ポートが見つかったか否かを表す値を返します。</returns>
        internal bool GetPort(out uint outPort, string targetName)
        {
            PortName name = CreatePortName(targetName);

            var result = (HidShellResult)
                Native.GetHidShellPort(this.Handle, out outPort, ref name);

            switch (result)
            {
                case HidShellResult.Success:
                    return true;

                case HidShellResult.PortNotFound:
                    return false;

                default:
                    throw new HidShellException(result);
            }
        }

        private static PortName CreatePortName(string targetName)
        {
            var name = new PortName { bytes = new byte[128] };

            if (name.bytes.Length <= Encoding.UTF8.GetByteCount(targetName))
            {
                throw new HidShellException(string.Format(
                    Strings.HidShellErrorTargetInvalidName, targetName));
            }

            Encoding.UTF8.GetBytes(
                targetName, 0, targetName.Length, name.bytes, 0);

            return name;
        }

        private void Dispose(bool isDisposing)
        {
            if (this.isDisposed)
            {
                return;
            }

            Native.ReleaseHidShellExclusiveRight(this.Handle);

            this.isDisposed = true;
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct PortName
        {
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 128)]
            internal byte[] bytes;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int AquireHidShellExclusiveRight(
                out IntPtr outHandle);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int ReleaseHidShellExclusiveRight(
                IntPtr handle);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int AttachHidShellPort(
                IntPtr handle, ref PortName name);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int DetachHidShellPort(
                IntPtr handle, ref PortName name);


            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellPort(
                IntPtr handle, out uint outPort, ref PortName name);
        }
    }
}
