﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>

//!< HidShell の結果を表す列挙体です。
enum class HidShellResult : int32_t
{
    Success,                    //!< 成功しました。
    NullPortPointer,            //!< ポートの書き込み先が NULL 値でした。
    InvalidPort,                //!< ポートが不正でした。
    PortNotFound,               //!< ポートが見つかりませんでした。
    NoAvailablePort,            //!< ポートに空きがありませんでした。
    NullPortListPointer,        //!< ポートリストの書き込み先が NULL 値でした。
    InvalidPortDirection,       //!< ポートのデータ転送方向が不正でした。
    NullHandlePointer,          //!< ハンドルの書き込み先が NULL 値でした。
    InvalidHandle,              //!< ハンドルが不正でした。
    InvalidPortName,            //!< ポート名が不正でした。
    NullStatePointer,           //!< 入力状態の書き込み先が NULL 値でした。
    StateNotSet,                //!< 入力状態は設定されていませんでした。
    FailedToCreateMutex,        //!< ミューテックスの作成に失敗しました。
    FailedToCreateSharedMemory, //!< 共有メモリの作成に失敗しました。
    NullAddressPointer,         //!< アドレスの書き込み先が NULL 値でした。
    NullExistenceFlagPointer,   //!< 存在フラグの書き込み先が NULL 値でした。
    InvalidCaptureButtonState,  //!< 撮影ボタンの入力状態が不正でした。
    InvalidDebugPadState,       //!< デバッグパッドの入力状態が不正でした。
    InvalidHomeButtonState,     //!< ホームボタンの入力状態が不正でした。
    InvalidTouchScreenState,    //!< タッチパネルの入力状態が不正でした。
    InvalidBasicXpadState,      //!< 基本的な構成を持つ Xpad の入力状態が不正でした。
    InvalidBasicXpadId,         //!< 基本的な構成を持つ Xpad の ID が不正でした。
    InvalidAbstractedPadState,  //!< AbstractedPad の入力状態が不正でした。
    InvalidAbstractedPadId,     //!< AbstractedPad の ID が不正でした。
};

//!< HidShell のハンドルを表す型です。
typedef uint64_t HidShellHandle;

//!< HidShell のポート名を表す型です。
struct HidShellPortName final
{
    char string[128];   //!< ポート名の文字列表現（NULL 終端）
};

//!< HidShell のポートリストを表す型です。
struct HidShellPortList final
{
    uint32_t count;         //!< ポートの数
    uint32_t ports[1 + 32]; //!< ポート
};

//!< HidShell のポートのデータ転送方向を表す列挙体です。
enum class HidShellPortDirection : uint32_t
{
    In,     //!< 入力
    Out,    //!< 出力
};

//!< HidShell のスティックの入力状態を表す構造体です。
struct HidShellStickState final
{
    int32_t x;  //!< x 軸座標
    int32_t y;  //!< y 軸座標
};

//!< HidShell の撮影ボタンの定義です。
struct HidShellCaptureButton final
{
    const uint64_t Active = 0x00000001ull;  //!< アクティブな撮影ボタン
};

//!< HidShell の撮影ボタンの入力状態を表す構造体です。
struct HidShellCaptureButtonState final
{
    uint64_t buttons;   //!< 撮影ボタンの状態
};

//!< HidShell のデバッグパッドの属性定義です。
struct HidShellDebugPadAttribute final
{
    const uint32_t IsConnected = 0x00000001u;   //!< 接続状態にあるか否か
};

//!< HidShell のデバッグパッドのデジタルボタン定義です。
struct HidShellDebugPadButton final
{
    const uint32_t A = 0x00000001u;         //!< a ボタン
    const uint32_t B = 0x00000002u;         //!< b ボタン
    const uint32_t X = 0x00000004u;         //!< X ボタン
    const uint32_t Y = 0x00000008u;         //!< Y ボタン
    const uint32_t L = 0x00000010u;         //!< L ボタン
    const uint32_t R = 0x00000020u;         //!< R ボタン
    const uint32_t ZL = 0x00000040u;        //!< ZL ボタン
    const uint32_t ZR = 0x00000080u;        //!< ZR ボタン
    const uint32_t Start = 0x00000100u;     //!< Start ボタン
    const uint32_t Select = 0x00000200u;    //!< Select ボタン
    const uint32_t Left = 0x00000400u;      //!< 十字ボタン 左
    const uint32_t Up = 0x00000800u;        //!< 十字ボタン 上
    const uint32_t Right = 0x00001000u;     //!< 十字ボタン 右
    const uint32_t Down = 0x00002000u;      //!< 十字ボタン 下
};

//!< HidShell のデバッグパッドの入力状態を表す構造体です。
struct HidShellDebugPadState final
{
    uint32_t attributes;        //!< 入力状態の属性
    uint32_t buttons;           //!< デジタルボタンの状態
    HidShellStickState stickL;  //!< 左スティックの状態
    HidShellStickState stickR;  //!< 右スティックの状態
};

//!< HidShell のホームボタンの定義です。
struct HidShellHomeButton final
{
    const uint64_t Active = 0x00000001ull;  //!< アクティブなホームボタン
};

//!< HidShell のホームボタンの入力状態を表す構造体です。
struct HidShellHomeButtonState final
{
    uint64_t buttons;   //!< ホームボタンの状態
};

//!< HidShell のタッチの属性定義です。
struct HidShellTouchAttribute final
{
    const uint32_t Start = 0x00000001u; //!< タッチが開始されたか否か
    const uint32_t End = 0x00000002u;   //!< タッチが解放されたか否か
};

const size_t HidShellTouchStateCountMax = 16;   //!< HidShell のタッチの最大数

//!< HidShell のタッチの入力状態を表す構造体です。
struct HidShellTouchState final
{
    int64_t deltaTime;      //!< 最後にサンプリングされてからの経過時間
    uint32_t attributes;    //!< 入力状態の属性
    int32_t fingerId;       //!< 識別子
    int32_t x;              //!< x 軸座標
    int32_t y;              //!< y 軸座標
    int32_t diameterX;      //!< 楕円で表された接触範囲の x 軸直径
    int32_t diameterY;      //!< 楕円で表された接触範囲の y 軸直径
    int32_t rotationAngle;  //!< 接触範囲の時計回りを正とする回転の度数
    int32_t _padding;
};

//!< HidShell のタッチパネルの入力状態を表す構造体です。
struct HidShellTouchScreenState final
{
    int32_t count;                                          //!< タッチの数
    int32_t _padding;
    HidShellTouchState touches[HidShellTouchStateCountMax]; //!< タッチ
};

//!< HidShell の Xpad のバッテリー残量を表す列挙体です。
enum class HidShellXpadPowerState : int32_t
{
    Disconnected,   //!< デバイスは切断されています。
    OnBattery,      //!< デバイスは切断されておりバッテリーで動作しています。
    NoBattery,      //!< デバイスは接続されておりバッテリーを持っていません。
    Charging,       //!< デバイスは接続されており充電中です。
    Charged,        //!< デバイスは接続されており充電済みです。
};

//!< HidShell の Xpad のバッテリー残量を表す列挙体です。
enum class HidShellXpadBatteryLevel : int32_t
{
    Empty,      //!< バッテリー残量はゼロです。
    Critical,   //!< バッテリー残量がほとんど残っていません。
    Low,        //!< バッテリー残量が少なくなっています。
    Medium,     //!< バッテリー残量は約半分です。
    High,       //!< バッテリー残量は十分です。
    Full,       //!< バッテリー残量は最大です。
};

const size_t HidShellBasicXpadCountMax = 8; //!< HidShell の基本的な構成を持つ Xpad の最大数です。

//!< HidShell の基本的な構成を持つ Xpad のデジタルボタン定義です。
struct HidShellBasicXpadButton final
{
    const uint32_t A = 0x00000001u;         //!< a ボタン
    const uint32_t B = 0x00000002u;         //!< b ボタン
    const uint32_t X = 0x00000004u;         //!< x ボタン
    const uint32_t Y = 0x00000008u;         //!< y ボタン
    const uint32_t StickL = 0x00000010u;    //!< L スティックボタン
    const uint32_t StickR = 0x00000020u;    //!< R スティックボタン
    const uint32_t L = 0x00000040u;         //!< L ボタン
    const uint32_t R = 0x00000080u;         //!< R ボタン
    const uint32_t ZL = 0x00000100u;        //!< ZL ボタン
    const uint32_t ZR = 0x00000200u;        //!< ZR ボタン
    const uint32_t Start = 0x00000400u;     //!< Start ボタン
    const uint32_t Select = 0x00000800u;    //!< Select ボタン
    const uint32_t Left = 0x00001000u;      //!< 十字ボタン 左
    const uint32_t Up = 0x00002000u;        //!< 十字ボタン 上
    const uint32_t Right = 0x00004000u;     //!< 十字ボタン 右
    const uint32_t Down = 0x00008000u;      //!< 十字ボタン 下
};

//!< HidShell の基本的な構成を持つ Xpad の入力状態を表す構造体です。
struct HidShellBasicXpadState final
{
    HidShellXpadPowerState powerState;      //!< 電源状態
    HidShellXpadBatteryLevel batteryLevel;  //!< バッテリー残量
    uint32_t buttons;                       //!< デジタルボタンの状態
    HidShellStickState stickL;              //!< 左スティックの状態
    HidShellStickState stickR;              //!< 右スティックの状態
};

const size_t HidShellAbstractedPadCountMax = 8; //!< HidShell の基本的な構成を持つ AbstractedPad の最大数です。

//!< HidShell の基本的な構成を持つ AbstractedPad のデジタルボタン定義です。
struct HidShellAbstractedPadButton final
{
    const uint32_t A            = 0x00000001u;  //!< コントローラーの a ボタン
    const uint32_t B            = 0x00000002u;  //!< コントローラーの b ボタン
    const uint32_t X            = 0x00000004u;  //!< コントローラーの x ボタン
    const uint32_t Y            = 0x00000008u;  //!< コントローラーの y ボタン
    const uint32_t StickL       = 0x00000010u;  //!< コントローラーの L スティックボタン
    const uint32_t StickR       = 0x00000020u;  //!< コントローラーの R スティックボタン
    const uint32_t L            = 0x00000040u;  //!< コントローラーの L ボタン
    const uint32_t R            = 0x00000080u;  //!< コントローラーの R ボタン
    const uint32_t ZL           = 0x00000100u;  //!< コントローラーの ZL ボタン
    const uint32_t ZR           = 0x00000200u;  //!< コントローラーの ZR ボタン
    const uint32_t Start        = 0x00000400u;  //!< コントローラーの Start ボタン
    const uint32_t Select       = 0x00000800u;  //!< コントローラーの Select ボタン
    const uint32_t Left         = 0x00001000u;  //!< コントローラーの 方向ボタン 左
    const uint32_t Up           = 0x00002000u;  //!< コントローラーの 方向ボタン 上
    const uint32_t Right        = 0x00004000u;  //!< コントローラーの 方向ボタン 右
    const uint32_t Down         = 0x00008000u;  //!< コントローラーの 方向ボタン 下
    const uint32_t SL           = 0x00010000u;  //!< コントローラーの SL ボタン
    const uint32_t SR           = 0x00020000u;  //!< コントローラーの SR ボタン
    const uint32_t Home         = 0x00040000u;  //!< コントローラーの Home ボタン
    const uint32_t Shot         = 0x00080000u;  //!< コントローラーの Shot ボタン
    const uint32_t StickLLeft   = 0x00100000u;  //!< 左スティックによる十字ボタンエミュレーション 左
    const uint32_t StickLUp     = 0x00200000u;  //!< 左スティックによる十字ボタンエミュレーション 上
    const uint32_t StickLRight  = 0x00400000u;  //!< 左スティックによる十字ボタンエミュレーション 右
    const uint32_t StickLDown   = 0x00800000u;  //!< 左スティックによる十字ボタンエミュレーション 下
    const uint32_t StickRLeft   = 0x01000000u;  //!< 右スティックによる十字ボタンエミュレーション 左
    const uint32_t StickRUp     = 0x02000000u;  //!< 右スティックによる十字ボタンエミュレーション 上
    const uint32_t StickRRight  = 0x04000000u;  //!< 右スティックによる十字ボタンエミュレーション 右
    const uint32_t StickRDown   = 0x08000000u;  //!< 右スティックによる十字ボタンエミュレーション 下
    const uint32_t Palma        = 0x10000000u;  //!< Palma 専用ボタン
};

//!< 仮想デバイスのデバイス色を表す構造体です。
struct HidShellAbstractedPadColor final
{
    uint8_t main[4];        //!< メイン色
    uint8_t sub[4];         //!< サブ色
};

//!< HidShell の AbstractedPad のバッテリー残量を表す列挙体です。
enum class HidShellAbstractedPadBatteryLevel : uint8_t
{
    Empty,      //!< バッテリー残量はゼロです。
    Critical,   //!< バッテリー残量がほとんど残っていません。
    Low,        //!< バッテリー残量が少なくなっています。
    Medium,     //!< バッテリー残量は約半分です。
    High,       //!< バッテリー残量は十分です。
};

//!< 仮想デバイスの電源状態です。
struct PowerInfo final
{
    uint8_t isPowered;
    uint8_t isCharging;
    uint8_t padding[1];
    HidShellAbstractedPadBatteryLevel batteryLevel;
};

//!< HidShell の仮想デバイスの入力状態を表す構造体です。
struct HidShellAbstractedPadState final
{
    uint8_t                     deviceType;     //!< デバイスタイプ
    uint32_t                    attribute;      //!< AbstractedPad の属性
    HidShellAbstractedPadColor  color;          //!< デバイス色
    uint8_t                     interfaceType;  //!< デバイスインターフェース
    PowerInfo                   powerInfo;      //!< 電源状態
    uint32_t                    buttons;        //!< digitalボタンの状態
    HidShellStickState          stickL;         //!< 左スティックの状態
    HidShellStickState          stickR;         //!< 右スティックの状態
};

#ifdef _WINDLL
#define HID_SHELL_DLLAPI extern "C" __declspec(dllexport)
#else
#define HID_SHELL_DLLAPI extern "C" __declspec(dllimport)
#endif

HID_SHELL_DLLAPI
int32_t AquireHidShellExclusiveRight(void** pOutHandle);

HID_SHELL_DLLAPI
int32_t ReleaseHidShellExclusiveRight(void* handle);

HID_SHELL_DLLAPI
int32_t AttachHidShellPort(void* handle, const void* pName);

HID_SHELL_DLLAPI
int32_t DetachHidShellPort(void* handle, const void* pName);

HID_SHELL_DLLAPI
int32_t GetHidShellPort(void* handle, uint32_t* pOutPort, const void* pName);

HID_SHELL_DLLAPI
int32_t GetHidShellPortList(void* handle, void* pOutList);

HID_SHELL_DLLAPI
int32_t GetHidShellCaptureButtonState(
    void* handle, void* pOutValue, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t SetHidShellCaptureButtonState(
    void* handle, const void* pValue, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t GetHidShellDebugPadState(
    void* handle, void* pOutValue, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t SetHidShellDebugPadState(
    void* handle, const void* pValue, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t GetHidShellHomeButtonState(
    void* handle, void* pOutValue, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t SetHidShellHomeButtonState(
    void* handle, const void* pValue, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t GetHidShellTouchScreenState(
    void* handle, void* pOutValue, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t SetHidShellTouchScreenState(
    void* handle, const void* pValue, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t GetHidShellBasicXpadState(
    void* handle,
    void* pOutValue, uint32_t id, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t SetHidShellBasicXpadState(
    void* handle,
    const void* pValue, uint32_t id, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t GetHidShellAbstractedPadState(
    void* handle,
    void* pOutValue, uint32_t id, uint32_t port, uint32_t direction);

HID_SHELL_DLLAPI
int32_t SetHidShellAbstractedPadState(
    void* handle,
    const void* pValue, uint32_t id, uint32_t port, uint32_t direction);

//!< HidShell の占有権を取得します。
inline HidShellResult AquireHidShellExclusiveRight(HidShellHandle* pOutHandle)
{
    return static_cast<HidShellResult>(
        AquireHidShellExclusiveRight(reinterpret_cast<void**>(pOutHandle)));
}

//!< HidShell の占有権を開放します。
inline HidShellResult ReleaseHidShellExclusiveRight(HidShellHandle handle)
{
    return static_cast<HidShellResult>(
        ReleaseHidShellExclusiveRight(reinterpret_cast<void*>(handle)));
}

//!< HidShell のポートをアタッチします。
inline HidShellResult AttachHidShellPort(
    HidShellHandle handle, const HidShellPortName& name)
{
    return static_cast<HidShellResult>(
        AttachHidShellPort(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<const void*>(&name)));
}

//!< HidShell のポートをデタッチします。
inline HidShellResult DetachHidShellPort(
    HidShellHandle handle, const HidShellPortName& name)
{
    return static_cast<HidShellResult>(
        DetachHidShellPort(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<const void*>(&name)));
}

//!< HidShell のポートを取得します。
inline HidShellResult GetHidShellPort(
    HidShellHandle handle, uint32_t* pOutPort, const HidShellPortName& name)
{
    return static_cast<HidShellResult>(
        GetHidShellPort(
            reinterpret_cast<void*>(handle),
            pOutPort, reinterpret_cast<const void*>(&name)));
}

//!< HidShell のポートリストを取得します。
inline HidShellResult GetHidShellPortList(
    HidShellHandle handle, HidShellPortList* pOutList)
{
    return static_cast<HidShellResult>(
        GetHidShellPortList(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<void*>(pOutList)));
}

//!< HidShell の撮影ボタンの入力状態を取得します。
inline HidShellResult GetHidShellCaptureButtonState(
    HidShellHandle handle,
    HidShellCaptureButtonState* pOutValue,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        GetHidShellCaptureButtonState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<void*>(pOutValue),
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell の撮影ボタンの入力状態を設定します。
inline HidShellResult SetHidShellCaptureButtonState(
    HidShellHandle handle,
    const HidShellCaptureButtonState& value,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        SetHidShellCaptureButtonState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<const void*>(&value),
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell のデバッグパッドの入力状態を取得します。
inline HidShellResult GetHidShellDebugPadState(
    HidShellHandle handle,
    HidShellDebugPadState* pOutValue,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        GetHidShellDebugPadState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<void*>(pOutValue),
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell のデバッグパッドの入力状態を設定します。
inline HidShellResult SetHidShellDebugPadState(
    HidShellHandle handle,
    const HidShellDebugPadState& value,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        SetHidShellDebugPadState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<const void*>(&value),
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell のホームボタンの入力状態を取得します。
inline HidShellResult GetHidShellHomeButtonState(
    HidShellHandle handle,
    HidShellHomeButtonState* pOutValue,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        GetHidShellHomeButtonState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<void*>(pOutValue),
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell のホームボタンの入力状態を設定します。
inline HidShellResult SetHidShellHomeButtonState(
    HidShellHandle handle,
    const HidShellHomeButtonState& value,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        SetHidShellHomeButtonState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<const void*>(&value),
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell のタッチパネルの入力状態を取得します。
inline HidShellResult GetHidShellTouchScreenState(
    HidShellHandle handle,
    HidShellTouchScreenState* pOutValue,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        GetHidShellTouchScreenState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<void*>(pOutValue),
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell のタッチパネルの入力状態を設定します。
inline HidShellResult SetHidShellTouchScreenState(
    HidShellHandle handle,
    const HidShellTouchScreenState& value,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        SetHidShellTouchScreenState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<const void*>(&value),
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell の基本的な構成を持つ Xpad の入力状態を取得します。
inline HidShellResult GetHidShellBasicXpadState(
    HidShellHandle handle,
    HidShellBasicXpadState* pOutValue, uint32_t id,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        GetHidShellBasicXpadState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<void*>(pOutValue), id,
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell の基本的な構成を持つ Xpad の入力状態を設定します。
inline HidShellResult SetHidShellBasicXpadState(
    HidShellHandle handle,
    const HidShellBasicXpadState& value, uint32_t id,
    uint32_t port, HidShellPortDirection direction)
{
    return static_cast<HidShellResult>(
        SetHidShellBasicXpadState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<const void*>(&value), id,
            port, static_cast<uint32_t>(direction)));
}
//!< HidShell の基本的な構成を持つ AbstractedPad の入力状態を取得します。
inline HidShellResult GetHidShellAbstractedPadState(
    HidShellHandle handle,
    HidShellAbstractedPadState* pOutValue, uint32_t id, uint32_t port, uint32_t direction)
{
    return static_cast<HidShellResult>(
        GetHidShellAbstractedPadState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<void*>(pOutValue), id,
            port, static_cast<uint32_t>(direction)));
}

//!< HidShell の基本的な構成を持つ AbstractedPad の入力状態を設定します。
inline HidShellResult SetHidShellAbstractedPadState(
    HidShellHandle handle,
    const HidShellAbstractedPadState* value, uint32_t id, uint32_t port, uint32_t direction)
{
    return static_cast<HidShellResult>(
        SetHidShellAbstractedPadState(
            reinterpret_cast<void*>(handle),
            reinterpret_cast<const void*>(&value), id,
            port, static_cast<uint32_t>(direction)));
}
