﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace HidShell
{
    using System;
    using System.Collections.Generic;
    using Nintendo.ToolFoundation.CommandLine;
    using Properties;

    /// <summary>
    /// コマンドライン引数パーサ向けの引数定義です。
    /// </summary>
    internal sealed class ProgramArgument
    {
        /// <summary>
        /// ProgramArgument クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal ProgramArgument()
        {
            this.SubCommandArgumentForGetEvent =
                new SubCommandArgumentForGetEvent();

            this.SubCommandArgumentForSendEvent =
                new SubCommandArgumentForSendEvent();

            this.SubCommandArgumentForWait =
                new SubCommandArgumentForWait();
        }

        /// <summary>
        /// サブコマンド get-event の引数定義です。
        /// </summary>
        internal SubCommandArgumentForGetEvent SubCommandArgumentForGetEvent
        {
            get; private set;
        }

        /// <summary>
        /// サブコマンド send-event の引数定義です。
        /// </summary>
        internal SubCommandArgumentForSendEvent SubCommandArgumentForSendEvent
        {
            get; private set;
        }

        /// <summary>
        /// サブコマンド wait の引数定義です。
        /// </summary>
        internal SubCommandArgumentForWait SubCommandArgumentForWait
        {
            get; private set;
        }

        /// <summary>
        /// コマンドラインパーサに自身を登録します。
        /// </summary>
        /// <param name="parser">コマンドラインパーサです。</param>
        internal void RegiserWith(DelegateBasedCommandLineParser parser)
        {
            this.SubCommandArgumentForGetEvent.RegiserWith(parser);

            this.SubCommandArgumentForSendEvent.RegiserWith(parser);

            this.SubCommandArgumentForWait.RegiserWith(parser);
        }
    }

    /// <summary>
    /// サブコマンド get-event の引数定義です。
    /// </summary>
    internal sealed class SubCommandArgumentForGetEvent
    {
        /// <summary>
        /// SubCommandArgumentForGetEvent クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal SubCommandArgumentForGetEvent()
        {
        }

        /// <summary>
        /// コマンドが選択されたか否かを表す値を返します。
        /// </summary>
        internal bool IsSelected { get; private set; }

        /// <summary>
        /// 開発機の名前を取得します。
        /// </summary>
        internal string TargetName { get; private set; }

        /// <summary>
        /// タイムアウト時間（秒）を取得します。
        /// </summary>
        internal int Timeout { get; private set; }

        /// <summary>
        /// デバイス種別を取得します。
        /// </summary>
        internal string DeviceType { get; private set; }

        /// <summary>
        /// デバイス識別子を取得します。
        /// </summary>
        internal string DeviceId { get; private set; }

        /// <summary>
        /// コマンドラインパーサに自身を登録します。
        /// </summary>
        /// <param name="parser">コマンドラインパーサです。</param>
        internal void RegiserWith(DelegateBasedCommandLineParser parser)
        {
            var context = parser
                .AddCommand("get-event", () => this.IsSelected = true)
                .SetDescription(Resources.GetEventCommand);

            context
                .AddValueOption('t', "target", x => this.TargetName = x)
                .SetDescription(Resources.CommonOptionTarget);

            context
                .AddValueOption('w', "timeout", this.AcceptTimeoutOption)
                .SetDescription(Resources.CommonOptionTimeout);

            context
                .AddValue(x => this.DeviceType = x, 1)
                .SetValueName("device")
                .SetDescription(Resources.CommonValueDevice);

            context
                .AddValue(x => this.DeviceId = x, 1)
                .SetValueName("id")
                .SetDescription(Resources.CommonValueId);
        }

        private void AcceptTimeoutOption(string timeout)
        {
            int result = 0;

            if (int.TryParse(timeout, out result))
            {
                this.Timeout = result;
            }
            else
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorTimeoutInvalid, timeout));
            }
        }
    }

    /// <summary>
    /// サブコマンド send-event の引数定義です。
    /// </summary>
    internal sealed class SubCommandArgumentForSendEvent
    {
        /// <summary>
        /// SubCommandArgumentForSendEvent クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal SubCommandArgumentForSendEvent()
        {
            this.EventArguments = new List<string>();
        }

        /// <summary>
        /// コマンドが選択されたか否かを表す値を返します。
        /// </summary>
        internal bool IsSelected { get; private set; }

        /// <summary>
        /// 開発機の名前を取得します。
        /// </summary>
        internal string TargetName { get; private set; }

        /// <summary>
        /// タイムアウト時間（秒）を取得します。
        /// </summary>
        internal int Timeout { get; private set; }

        /// <summary>
        /// イベントログのファイルパスを取得します。
        /// </summary>
        internal string EventLogPath { get; private set; }

        /// <summary>
        /// イベントログの繰り返し回数を取得します。
        /// </summary>
        internal int EventLogRepeatCount { get; private set; } = 1;

        /// <summary>
        /// デバイス種別を取得します。
        /// </summary>
        internal string DeviceType { get; private set; }

        /// <summary>
        /// デバイス識別子を取得します。
        /// </summary>
        internal string DeviceId { get; private set; }

        /// <summary>
        /// イベント種別を取得します。
        /// </summary>
        internal string EventType { get; private set; }

        /// <summary>
        /// イベント引数を取得します。
        /// </summary>
        internal List<string> EventArguments { get; private set; }

        /// <summary>
        /// コマンドラインパーサに自身を登録します。
        /// </summary>
        /// <param name="parser">コマンドラインパーサです。</param>
        internal void RegiserWith(DelegateBasedCommandLineParser parser)
        {
            var context = parser
                .AddCommand("send-event", () => this.IsSelected = true)
                .SetDescription(Resources.SendEventCommand);

            context
                .AddValueOption('t', "target", x => this.TargetName = x)
                .SetDescription(Resources.CommonOptionTarget);

            context
                .AddValueOption('w', "timeout", this.AcceptTimeoutOption)
                .SetDescription(Resources.CommonOptionTimeout);

            context
                .AddValueOption('f', "file", x => this.EventLogPath = x)
                .SetDescription(Resources.SendEventOptionFile);

            context
                .AddValueOption("file-repeat", this.AcceptFileRepeatOption)
                .SetDescription(Resources.SendEventOptionFileRepeat);

            context
                .AddValue(x => this.DeviceType = x, 1)
                .SetValueName("device")
                .SetDescription(Resources.CommonValueDevice);

            context
                .AddValue(x => this.DeviceId = x, 1)
                .SetValueName("id")
                .SetDescription(Resources.CommonValueId);

            context
                .AddValue(x => this.EventType = x, 1)
                .SetValueName("event")
                .SetDescription(Resources.SendEventValueEvent);

            context
                .AddValue(x => this.EventArguments.Add(x), -1)
                .SetValueName("arg")
                .SetDescription(Resources.SendEventValueArg);
        }

        private void AcceptTimeoutOption(string timeout)
        {
            int result = 0;

            if (int.TryParse(timeout, out result))
            {
                this.Timeout = result;
            }
            else
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorTimeoutInvalid, timeout));
            }
        }

        private void AcceptFileRepeatOption(string count)
        {
            int result = 0;

            if (int.TryParse(count, out result))
            {
                this.EventLogRepeatCount = result;
            }
            else
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorFileRepeatInvalid, count));
            }
        }
    }

    /// <summary>
    /// サブコマンド wait の引数定義です。
    /// </summary>
    internal sealed class SubCommandArgumentForWait
    {
        /// <summary>
        /// コマンドが選択されたか否かを表す値を返します。
        /// </summary>
        internal bool IsSelected { get; private set; }

        /// <summary>
        /// 待ち時間を取得します。
        /// </summary>
        internal TimeSpan TimeSpan { get; private set; }

        /// <summary>
        /// コマンドラインパーサに自身を登録します。
        /// </summary>
        /// <param name="parser">コマンドラインパーサです。</param>
        internal void RegiserWith(DelegateBasedCommandLineParser parser)
        {
            var context = parser
                .AddCommand("wait", () => this.IsSelected = true)
                .SetDescription(Resources.WaitCommand);

            context
                .AddValue(this.AcceptTimeSpanOption, 1)
                .SetValueName("time")
                .SetDescription(Resources.WaitValueTimeSpan)
                .SetRequired();
        }

        private void AcceptTimeSpanOption(string time)
        {
            long result = 0;

            if (long.TryParse(time, out result))
            {
                this.TimeSpan = TimeSpan.FromMilliseconds(result);
            }
            else
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorTimeSpanInvalid, time));
            }
        }
    }
}
