﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace HidShell
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition.Hosting;
    using System.Diagnostics;
    using System.Globalization;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Text.RegularExpressions;
    using System.Threading;
    using System.Threading.Tasks;
    using Nintendo.ToolFoundation.CommandLine;
    using Properties;

    /// <summary>
    /// エントリーポイントを定義します。
    /// </summary>
    internal static class Program
    {
        private static readonly Dictionary<string, uint> portCache;

        /// <summary>
        /// Program クラスの静的メンバを初期化します。
        /// </summary>
        static Program()
        {
            portCache = new Dictionary<string, uint>();
        }

        /// <summary>
        /// エントリーポイントです。
        /// </summary>
        internal static void Main(string[] args)
        {
            try
            {
                if (File.Exists(GetResourceLocalizerPath()))
                {
                    var cultureInfo = new CultureInfo("en", false);

                    Thread.CurrentThread.CurrentUICulture = cultureInfo;
                }

                List<List<string>> commands = SplitCommands(args);

                if (commands.Count == 0) { commands.Add(new List<string>()); }

                var arguments = new List<ProgramArgument>();

                foreach (List<string> command in commands)
                {
                    var parser = new DelegateBasedCommandLineParser();

                    parser.AddHelp(Console.Error.WriteLine);

                    var argument = new ProgramArgument();

                    argument.RegiserWith(parser);

                    if (!parser.ParseArgs(command, Console.Error.WriteLine))
                    {
                        Environment.Exit(1);
                    }

                    arguments.Add(argument);
                }

                Assembly assembly = typeof(IDevice).Assembly;

                using (var catalog = new AssemblyCatalog(assembly))
                using (var container = new CompositionContainer(catalog))
                {
                    foreach (ProgramArgument argument in arguments)
                    {
                        DispatchCommand(container, argument);
                    }
                }
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine(ex.Message);

                Environment.Exit(1);
            }
        }

        private static string GetResourceLocalizerPath()
        {
            return Path.Combine(
                Path.GetDirectoryName(Assembly.GetEntryAssembly().Location),
                "en", "HidShell.resources.dll");
        }

        private static List<List<string>> SplitCommands(string[] args)
        {
            var commands = new List<List<string>>();

            var command = new List<string>();

            foreach (string arg in args)
            {
                if (arg != "--")
                {
                    command.Add(arg);
                }
                else
                {
                    if (command.Count != 0)
                    {
                        commands.Add(command);
                    }

                    command = new List<string>();
                }
            }

            if (command.Count != 0)
            {
                commands.Add(command);
            }

            return commands;
        }

        private static void DispatchCommand(
            CompositionContainer container, ProgramArgument argument)
        {
            if (argument.SubCommandArgumentForGetEvent.IsSelected)
            {
                ExecuteGetEvent(
                    argument.SubCommandArgumentForGetEvent,
                    container.GetExportedValues<IDevice>());

                return;
            }

            if (argument.SubCommandArgumentForSendEvent.IsSelected)
            {
                ExecuteSendEvent(
                    argument.SubCommandArgumentForSendEvent,
                    container.GetExportedValues<IDevice>());

                return;
            }

            if (argument.SubCommandArgumentForWait.IsSelected)
            {
                Thread.Sleep(argument.SubCommandArgumentForWait.TimeSpan);

                return;
            }
        }

        private static void ExecuteGetEvent(
            SubCommandArgumentForGetEvent argument,
            IEnumerable<IDevice> devices)
        {
            var port = GetPort(argument.TargetName, argument.Timeout);

            var logger = new Logger();

            foreach (var device in devices)
            {
                device.SetPort(port);

                device.SetLogger(logger);
            }

            var deviceType = argument.DeviceType;

            if (deviceType == null)
            {
                Parallel.ForEach(
                    devices, device => { device.DumpEvent(null); });
            }
            else
            {
                var device =
                    devices.FirstOrDefault(x => x.Supports(deviceType));

                if (device == null)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorDeviceTypeUnexpected, deviceType));
                }

                device.DumpEvent(argument.DeviceId);
            }
        }

        private static void ExecuteSendEvent(
            SubCommandArgumentForSendEvent argument,
            IEnumerable<IDevice> devices)
        {
            var port = GetPort(argument.TargetName, argument.Timeout);

            foreach (var device in devices)
            {
                device.SetPort(port);
            }

            var path = argument.EventLogPath;

            if (path != null)
            {
                for (var i = 0; i < argument.EventLogRepeatCount; ++i)
                {
                    ExecuteSendEvent(devices, path);
                }
            }
            else
            {
                ExecuteSendEvent(
                    devices,
                    argument.DeviceType,
                    argument.DeviceId,
                    argument.EventType,
                    argument.EventArguments);
            }
        }

        private static uint GetPort(string targetName, int timeout)
        {
            var port = PortAccessor.DefaultPort;

            if (targetName == null)
            {
                if (0 < timeout)
                {
                    throw new HidShellException(
                        Resources.ErrorTargetNotSpecified);
                }

                using (var accessor = new PortAccessor())
                {
                    uint[] ports = accessor.GetPorts();

                    if (ports.Length > 1)
                    {
                        return ports[1];
                    }
                }

                return port;
            }
            else
            {
                if (portCache.TryGetValue(targetName, out port))
                {
                    return port;
                }

                timeout *= 1000;

                var stopWatch = new Stopwatch();

                stopWatch.Start();

                bool retried = false;

                do
                {
                    bool found = false;

                    using (var accessor = new PortAccessor())
                    {
                        if (accessor.GetPort(out port, targetName))
                        {
                            found = true;
                        }
                    }

                    if (found)
                    {
                        portCache[targetName] = port;

                        if (retried)
                        {
                            timeout -= (int)stopWatch.ElapsedMilliseconds;

                            Thread.Sleep(Math.Max(0, Math.Min(timeout, 3000)));
                        }

                        return port;
                    }

                    retried = true;

                    Thread.Sleep(50);
                }
                while (stopWatch.ElapsedMilliseconds < timeout);

                throw new HidShellException(string.Format(
                    Resources.ErrorTargetNotFound, targetName));
            }
        }

        private static void ExecuteSendEvent(
            IEnumerable<IDevice> devices,
            string eventLogPath)
        {
            using (var file = File.OpenText(eventLogPath))
            {
                var pattern = @"^\[ *(\d+)\.(\d+)\] (\w+) +(\w+) +(\w+)";

                var regex = new Regex(pattern, RegexOptions.Compiled);

                var stopWatch = new Stopwatch();

                stopWatch.Start();

                while (true)
                {
                    var line = file.ReadLine();

                    if (line == null)
                    {
                        break;
                    }

                    line = line.Split(new[] { '#' }, 2).First().TrimEnd();

                    if (string.IsNullOrEmpty(line))
                    {
                        continue;
                    }

                    var match = regex.Match(line);

                    if (!match.Success)
                    {
                        throw new HidShellException(string.Format(
                            Resources.ErrorFileLineUnexpected, line));
                    }

                    line = line.Substring(match.Length);

                    var time =
                        long.Parse(match.Groups[1].Value) * 1000 +
                        long.Parse(match.Groups[2].Value);

                    var span = time - stopWatch.ElapsedMilliseconds;

                    if (span > 0)
                    {
                        Thread.Sleep((int)span);
                    }

                    ExecuteSendEvent(
                        devices,
                        match.Groups[3].Value,
                        match.Groups[4].Value,
                        match.Groups[5].Value,
                        line.Split().Where(x => !string.IsNullOrEmpty(x)));
                }
            }
        }

        private static void ExecuteSendEvent(
            IEnumerable<IDevice> devices,
            string deviceType,
            string deviceId,
            string eventType,
            IEnumerable<string> eventArgs)
        {
            if (deviceType == null)
            {
                throw new HidShellException(
                    Resources.ErrorDeviceTypeNotSpecified);
            }

            if (deviceId == null)
            {
                throw new HidShellException(
                    Resources.ErrorDeviceIdNotSpecified);
            }

            if (eventType == null)
            {
                throw new HidShellException(
                    Resources.ErrorEventTypeNotSpecified);
            }

            eventArgs = eventArgs ?? new string[0];

            var device = devices.FirstOrDefault(x => x.Supports(deviceType));

            if (device == null)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceTypeUnexpected, deviceType));
            }

            device.SendEvent(deviceId, eventType, eventArgs.ToArray());
        }
    }
}
