﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace HidShell.Devices
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Threading.Tasks;
    using Properties;

    /// <summary>
    /// CaptureButton を扱うクラスです。
    /// </summary>
    [PartCreationPolicy(CreationPolicy.NonShared)]
    internal sealed class CaptureButton : IDevice
    {
        private const string DeviceType = "CaptureButton";

        private const int CaptureButtonCountMax = 1;

        private readonly List<string> deviceIds = new List<string>();

        private readonly List<EventDumper> eventDumpers;

        private readonly List<EventSender> eventSenders;

        private uint port = PortAccessor.DefaultPort;

        private Logger logger = null;

        internal CaptureButton()
        {
            this.eventDumpers = new List<EventDumper>();

            this.eventSenders = new List<EventSender>();

            for (int i = 0; i < CaptureButtonCountMax; ++i)
            {
                var deviceId = i.ToString();

                this.deviceIds.Add(deviceId);

                this.eventSenders.AddRange(new []
                {
                    new EventSender(
                        deviceId, EventType.ButtonDown,
                        GetButtonDownEventSender(i)),
                    new EventSender(
                        deviceId, EventType.ButtonUp,
                        GetButtonUpEventSender(i)),
                });
            }
        }

        /// <summary>
        /// ポートを設定します。
        /// </summary>
        /// <param name="port">ポートです。</param>
        public void SetPort(uint port)
        {
            this.port = port;
        }

        /// <summary>
        /// ロガーを設定します。
        /// </summary>
        /// <param name="logger">ロガーです。</param>
        public void SetLogger(Logger logger)
        {
            this.logger = logger;
        }

        /// <summary>
        /// 指定されたデバイス種別をサポートするか否かを表す値を返します。
        /// </summary>
        /// <param name="deviceType">デバイス種別を表す文字列です。</param>
        /// <returns>指定されたデバイス種別をサポートするか否かを表す値です。</returns>
        public bool Supports(string deviceType)
        {
            return deviceType == DeviceType;
        }

        /// <summary>
        /// デバイス識別子で指定されたデバイスについてイベントをダンプします。
        /// </summary>
        /// <param name="deviceId">デバイス識別子です。</param>
        public void DumpEvent(string deviceId)
        {
            var dumpers = deviceId == null
                ? this.eventDumpers
                : this.eventDumpers.Where(x => x.SupportsDeviceId(deviceId));

            if (deviceId != null && dumpers.Count() == 0)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceIdUnexpected, deviceId));
            }

            Parallel.ForEach(dumpers, dumper => dumper.Invoke(this));
        }

        /// <summary>
        /// デバイス識別子で指定されたデバイスに対してイベントを発行します。
        /// </summary>
        /// <param name="deviceId">デバイス識別子です。</param>
        /// <param name="eventType">イベント種別です。</param>
        /// <param name="eventArgs">イベント引数です。</param>
        public void SendEvent(
            string deviceId, string eventType, string[] eventArgs)
        {
            var senders =
                this.eventSenders.Where(x => x.SupportsDeviceId(deviceId));

            if (senders.Count() == 0)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceIdUnexpected, deviceId));
            }

            var sender =
                senders.FirstOrDefault(x => x.SupportsEventType(eventType));

            if (sender == null)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorEventTypeUnexpected, eventType));
            }

            sender.Invoke(this, eventArgs);
        }

        private static void GetCaptureButtonAutoPilotState(
            IntPtr handle, out CaptureButtonState outState, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellCaptureButtonState(
                    handle, out outState, port, PortAccessor.In);

            switch (result)
            {
                case HidShellResult.Success:
                    return;

                case HidShellResult.StateNotSet:
                    outState = new CaptureButtonState();
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        private static void SetCaptureButtonAutoPilotState(
            IntPtr handle, ref CaptureButtonState state, uint port)
        {
            var result = (HidShellResult)
                Native.SetHidShellCaptureButtonState(
                    handle, ref state, port, PortAccessor.In);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static Action<CaptureButton, string[]> GetButtonDownEventSender(
            int index)
        {
            var state = new CaptureButtonState();

            return (CaptureButton that, string[] args) =>
            {
                if (args.Count() != 0)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.ButtonDown));
                }

                using (var accessor = new PortAccessor())
                {
                    GetCaptureButtonAutoPilotState(
                        accessor.Handle, out state, that.port);

                    state.Buttons |= 1ul << index;

                    SetCaptureButtonAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private static Action<CaptureButton, string[]> GetButtonUpEventSender(
            int index)
        {
            var state = new CaptureButtonState();

            return (CaptureButton that, string[] args) =>
            {
                if (args.Count() != 0)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.ButtonUp));
                }

                using (var accessor = new PortAccessor())
                {
                    GetCaptureButtonAutoPilotState(
                        accessor.Handle, out state, that.port);

                    state.Buttons &= ~(1ul << index);

                    SetCaptureButtonAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private struct CaptureButtonState
        {
            internal ulong Buttons;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellCaptureButtonState(
                IntPtr handle,
                out CaptureButtonState outState, uint port, uint direction);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int SetHidShellCaptureButtonState(
                IntPtr handle,
                ref CaptureButtonState state, uint port, uint direction);
        }

        private static class EventType
        {
            internal const string ButtonDown = "ButtonDown";

            internal const string ButtonUp = "ButtonUp";
        }

        private sealed class EventDumper
        {
            private string deviceId;

            private Action<CaptureButton> dumper;

            internal EventDumper(string deviceId, Action<CaptureButton> dumper)
            {
                this.deviceId = deviceId;

                this.dumper = dumper;
            }

            internal bool SupportsDeviceId(string deviceId)
            {
                return deviceId == this.deviceId;
            }

            internal void Invoke(CaptureButton that)
            {
                this.dumper(that);
            }
        }

        private sealed class EventSender
        {
            private string deviceId;

            private string eventType;

            private Action<CaptureButton, string[]> sender;

            internal EventSender(
                string deviceId,
                string eventType,
                Action<CaptureButton, string[]> sender)
            {
                this.deviceId = deviceId;

                this.eventType = eventType;

                this.sender = sender;
            }

            internal bool SupportsDeviceId(string deviceId)
            {
                return deviceId == this.deviceId;
            }

            internal bool SupportsEventType(string eventType)
            {
                return eventType == this.eventType;
            }

            internal void Invoke(CaptureButton that, string[] eventArgs)
            {
                this.sender(that, eventArgs);
            }
        }
    }
}
