﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml;

namespace Nintendo.NX.KuinaFirmwareUpdater
{
    class Config
    {
        /// <summary>
        /// 最近使用したファイルの最大数
        /// </summary>
        const int RecentFileCountMax = 1;

        /// <summary>
        /// 唯一のインスタンス
        /// </summary>
        private static Config instance = null;

        /// <summary>
        /// 設定ファイル名
        /// </summary>
        private const string ConfigFileName = "data/config.xml";

        /// <summary>
        /// 一時設定ファイル名
        /// </summary>
        private const string TempConfigFileName = "data/config.$$$";

        /// <summary>
        /// 要求するFWバージョン
        /// </summary>
        public byte FwMajorVersion { get; private set; } = 0xff;
        public byte FwMinorVersion { get; private set; } = 0xff;
        public byte FwBuildVersion { get; private set; } = 0xff;
        /// <summary>
        /// 設定ファイルのパス
        /// </summary>
        private string ConfigFilePath
        {
            get { return Path.Combine(Application.StartupPath, ConfigFileName); }
        }

        /// <summary>
        /// 一時設定ファイルのパス
        /// </summary>
        private string TempConfigFilePath
        {
            get { return Path.Combine(Application.StartupPath, TempConfigFileName); }
        }

        private Config()
        {

        }

        /// <summary>
        /// インスタンスの取得
        /// </summary>
        /// <returns></returns>
        public static Config GetInstance()
        {
            if (instance == null)
            {
                instance = new Config();
            }

            return instance;
        }

        private void CopyFrom(Config config)
        {
            FwMajorVersion = config.FwMajorVersion;
            FwMinorVersion = config.FwMinorVersion;
            FwBuildVersion = config.FwBuildVersion;
        }

        /// <summary>
        /// 設定の読み込み
        /// </summary>
        public void Load()
        {
            if (!File.Exists(ConfigFilePath))
            {
                return;
            }

            var tempConfig = new Config();

            var document = new XmlDocument();
            document.Load(ConfigFilePath);

            var root = document.DocumentElement;
            for (int i = 0; i < root.ChildNodes.Count; i++)
            {
                var node = root.ChildNodes[i];
                switch (node.Name)
                {
                case "FwMajorVersion":
                    {
                        byte version;
                        if (byte.TryParse(node.InnerText, out version))
                        {
                            tempConfig.FwMajorVersion = version;
                        }
                    }
                    break;
                case "FwMinorVersion":
                    {
                        byte version;
                        if (byte.TryParse(node.InnerText, out version))
                        {
                            tempConfig.FwMinorVersion = version;
                        }
                    }
                    break;
                case "FwBuildVersion":
                    {
                        byte version;
                        if (byte.TryParse(node.InnerText, out version))
                        {
                            tempConfig.FwBuildVersion = version;
                        }
                    }
                    break;
                }
            }

            CopyFrom(tempConfig);
        }

        /// <summary>
        /// 設定の保存
        /// </summary>
        public void Save()
        {
            var settings = new XmlWriterSettings();
            settings.Encoding = Encoding.UTF8;

            XmlWriter writer = null;
            using (writer = XmlWriter.Create(TempConfigFilePath, settings))
            {
                writer.WriteStartDocument();
                writer.WriteStartElement("UsbFirmwareUpdater");

                writer.WriteElementString("FwMajorVersion", FwMajorVersion.ToString());
                writer.WriteElementString("FwMinorVersion", FwMinorVersion.ToString());
                writer.WriteElementString("FwBuildVersion", FwBuildVersion.ToString());

                writer.WriteEndElement();
                writer.WriteEndDocument();
                writer.Close();
            }

            // 保存に成功したら一時ファイルを本ファイルに上書き
            File.Copy(TempConfigFilePath, ConfigFilePath, true);
            File.Delete(TempConfigFilePath);
        }
    }
}
