﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace InputCapture
{
    using System;
    using System.Diagnostics;
    using System.Drawing;
    using System.Threading;
    using System.Windows.Forms;
    using InputCapture.Communication;
    using InputCapture.Interop;
    using InputCapture.Windows.Forms;

    /// <summary>
    /// キャプチャマネージャクラスです。
    /// </summary>
    public sealed class CaptureManager : IDisposable
    {
        private const int ReceiverInterval = 10;
        private const int ScreenMargin = 5;

        private readonly Connecter connecter;
        private readonly MessageSender messageSender;
        private readonly Thread receiverThread;
        private readonly UserSettings userSettings;
        private readonly MouseHook mouseHook = new MouseHook();
        private readonly KeyboardHook keyboardHook = new KeyboardHook();

        private bool isCapturing = false;
        private bool isCaptureEnabled = true;
        private bool isPointerOnEdge = false;                       // ポインタが画面端にあるか。
        private Rectangle screenRect = new Rectangle(0, 0, 0, 0);   // 全スクリーン矩形（マルチスクリーン全体を包含）
        private Point pointerPos;                                   // PC のポインタ位置を保存

        private MouseButton mouseButtons = MouseButton.None;
        private bool isMouseButtonDown = false;
        private bool isDoubleClicked = false;
        private MouseHookEventArgs.MouseButtons mouseDownButtonFirst = MouseHookEventArgs.MouseButtons.None;
        private DateTime mouseDownTime = DateTime.MaxValue;
        private Size doubleClickDelta = Size.Empty;

        private bool isStartedByHotkey = false;
        private bool isKeyDown = false;
        private KeyModifier hotKeyModifier = KeyModifier.None;
        private KeyModifier keyModifier = KeyModifier.None;
        private bool isShiftPressed = false;
        private bool isControlPressed = false;
        private bool isMenuPressed = false;

        private InfoDialog infoDialog = null;

        ///-----------------------------------------------------------------

        /// <summary>
        /// キャプチャ実行中かを取得します。
        /// </summary>
        public bool IsCapturing
        {
            get
            {
                return this.isCapturing;
            }

            private set
            {
                this.isCapturing = value;
            }
        }

        /// <summary>
        /// キャプチャ開始可能かを取得します。
        /// </summary>
        public bool IsCaptureEnabled
        {
            get
            {
                return this.isCaptureEnabled;
            }

            set
            {
                this.isCaptureEnabled = value;
            }
        }

        /// <summary>
        /// 接続中かを取得します。
        /// </summary>
        public bool IsConnected
        {
            get
            {
                return this.connecter.GetConnectionState() == Connecter.ConnectionState.Connected;
            }
        }

        /// <summary>
        /// ユーザ設定を取得します。
        /// </summary>
        public UserSettings UserSettings
        {
            get
            {
                return this.userSettings;
            }
        }

        ///-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public CaptureManager()
        {
            this.screenRect = Utility.CalcScreenSize();

            this.connecter = new Connecter();
            this.messageSender = new MessageSender(this.connecter);

            this.userSettings = UserSettings.LoadAppSettings();
            this.SetConnecterSettings();
            this.SetHotKey();

            this.mouseHook.MouseMove += OnHookMouseMove;
            this.mouseHook.MouseDown += OnHookMouseDown;
            this.mouseHook.MouseUp += OnHookMouseUp;
            this.mouseHook.MouseWheel += OnHookMouseWheel;
            this.mouseHook.Start();

            this.keyboardHook.KeyDown += OnKeyDown;
            this.keyboardHook.KeyUp += OnKeyUp;
            this.keyboardHook.SystemKeyDown += OnKeyDown;
            this.keyboardHook.SystemKeyUp += OnKeyUp;
            this.keyboardHook.Start();

            this.receiverThread = new Thread(new ThreadStart(this.ReceiverThreadProc));
            this.receiverThread.Start();
        }

        /// <summary>
        /// 破棄の処理です。
        /// </summary>
        public void Dispose()
        {
            this.keyboardHook.Stop();
            this.mouseHook.Stop();

            this.receiverThread.Abort();
            this.receiverThread.Join();

            this.connecter.Dispose();
        }

        ///-----------------------------------------------------------------

        /// <summary>
        /// 手動で接続を開始します。
        /// </summary>
        public void ManualStartConnect()
        {
            this.connecter.StartConnect();
            this.connecter.ManualDisconnect = false;
        }

        /// <summary>
        /// 手動で接続を停止します。
        /// </summary>
        public void ManualStopConnect()
        {
            if (this.IsCapturing)
            {
                this.StopCapture();
            }

            this.connecter.StopConnect();
            this.connecter.ManualDisconnect = true;
        }

        /// <summary>
        /// 設定を反映させます。
        /// </summary>
        public void SetConnecterSettings()
        {
            this.connecter.AutoConnectEnabled = this.userSettings.IsAutoConnect;
            this.connecter.AutoDisconnectEnabled = this.userSettings.IsAutoDisconnect;
            this.connecter.CheckAliveInterval = this.userSettings.CheckAliveInterval;
            this.connecter.PollingInterval = (int)this.userSettings.PollingInterval;
        }

        /// <summary>
        /// ホットキーを設定します。
        /// </summary>
        public void SetHotKey()
        {
            this.hotKeyModifier = KeyModifier.None;

            if ((this.userSettings.HotKeyModifier & Keys.Shift) == Keys.Shift)
            {
                this.hotKeyModifier |= KeyModifier.Shift;
            }
            if ((this.userSettings.HotKeyModifier & Keys.Control) == Keys.Control)
            {
                this.hotKeyModifier |= KeyModifier.Control;
            }
            if ((this.userSettings.HotKeyModifier & Keys.Alt) == Keys.Alt)
            {
                this.hotKeyModifier |= KeyModifier.Alt;
            }
        }

        //-----------------------------------------------------------------

        private void ReceiverThreadProc()
        {
            // キャプチャ受信スレッド
            try
            {
                while (true)
                {
                    if (this.IsCapturing && this.connecter.IsMouseOut)
                    {
                        this.MouseOut();
                    }

                    if (!this.IsConnected)
                    {
                        // 切断状態になっているときにはキャプチャを止める。
                        this.StopCapture();
                    }

                    Thread.Sleep(ReceiverInterval);
                }
            }
            catch (System.Threading.ThreadAbortException)
            {
                Debug.WriteLine("Thread Abort: ReceiverThreadProc");
            }
        }

        private void StartCapture(bool captureOnMousePosition, BorderType borderType)
        {
            this.mouseHook.Stop();
            this.keyboardHook.Stop();

            this.isShiftPressed = false;
            this.isControlPressed = false;
            this.isMenuPressed = false;
            this.keyModifier = KeyModifier.None;

            // 現在の PC ポインタ位置を保存
            this.pointerPos = Cursor.Position;

            CaptureEventArg eventArg = new CaptureEventArg();
            eventArg.borderType = borderType;
            eventArg.captureOnMousePosition = captureOnMousePosition;
            eventArg.pointerPos = this.pointerPos;
            eventArg.borderLength = 0;
            if (borderType == BorderType.Right || borderType == BorderType.Left)
            {
                eventArg.borderLength = (uint)(this.screenRect.Bottom - this.screenRect.Top);
            }
            else if (borderType == BorderType.Top || borderType == BorderType.Bottom)
            {
                eventArg.borderLength = (uint)(this.screenRect.Right - this.screenRect.Left);
            }

            this.IsCapturing = this.messageSender.WriteInputData(MessageType.CaptureStart, eventArg);

            this.mouseHook.Start();
            this.keyboardHook.Start();

            if (this.IsCapturing)
            {
                // 消えたままになる恐れがあるので Cursor.Hide は使用しない。
                // MouseHookEventArgs による移動量の計算のため、カーソルは (0, 0) に固定する
                Cursor.Position = new Point(0, 0);

                this.ShowInfoDialog();

                Debug.WriteLine("CaptureStart");
            }
        }

        private void StopCapture()
        {
            this.HideInfoDialog();

            this.IsCapturing = false;
            Debug.WriteLine("CaptureStop");
        }

        private bool SendStopCaptureMsg()
        {
            // 実機にキャプチャ停止メッセージを送信
            CaptureEventArg eventArg = new CaptureEventArg();
            if (this.messageSender.WriteInputData(MessageType.CaptureStop, eventArg))
            {
                Debug.WriteLine("SendStopCaptureMsg");
                return true;
            }

            return false;
        }

        ///-----------------------------------------------------------------

        private void OnHookMouseMove(object sender, MouseHookEventArgs e)
        {
            if (this.IsCapturing)
            {
                // Point dist = new Point(e.X - this.originPos.X, e.Y - this.originPos.Y);

                // Cursor.Position と MouseHookEventArgs で座標系がずれている場合がある
                // マウスは (0, 0) に固定し、原点からの差分のみで移動量を計算する
                Point dist = new Point(e.X, e.Y);

                if (dist.X != 0 || dist.Y != 0)
                {
                    PointerEventArg eventArg = new PointerEventArg();
                    eventArg.point = dist;
                    if (this.messageSender.WriteInputData(MessageType.MouseMove, eventArg))
                    {
                        Debug.WriteLine("Mouse Move : " + eventArg.ToInfoStr());
                    }
                }

                // ポーリングレートが高いマウスだとイベント発行数が多くなり、
                // 通信路の負荷が高くなるので、一定時間処理を遅延させて発行数を減らす
                Thread.Sleep(5);

                e.IsCanceled = true;
            }
            else
            {
                this.isPointerOnEdge = this.CalcIsPointerOnEdge(Cursor.Position);

                if (this.userSettings.CaptureBorderSwitchType == BorderSwitchType.None &&
                    this.isPointerOnEdge && !this.isMouseButtonDown)
                {
                    if (this.StartCaptureOnMouse())
                    {
                        e.IsCanceled = true;
                    }
                }
            }
        }

        private void OnHookMouseDown(object sender, MouseHookEventArgs e)
        {
            if (e.Button != MouseHookEventArgs.MouseButtons.None)
            {
                this.isMouseButtonDown = true;
            }

            this.isDoubleClicked = this.CalcIsDoubleClicked(e.Button);

            if (this.IsConnected)
            {
                if (this.IsCapturing)
                {
                    this.UpdateMouseButtons(e.Button, true);

                    MouseEventArg eventArg = new MouseEventArg();
                    eventArg.button = this.mouseButtons;

                    if (this.messageSender.WriteInputData(MessageType.MouseDown, eventArg))
                    {
                        Debug.WriteLine("Mouse Down : " + eventArg.ToInfoStr());
                    }

                    e.IsCanceled = true;
                }
                else if (this.isPointerOnEdge && e.Button == MouseHookEventArgs.MouseButtons.Left)
                {
                    if ((this.isDoubleClicked && this.userSettings.CaptureBorderSwitchType == BorderSwitchType.LDoubleClick) ||
                        (!this.isDoubleClicked && this.userSettings.CaptureBorderSwitchType == BorderSwitchType.LClick))
                    {
                        e.IsCanceled = true;
                    }
                }
            }
        }

        private void OnHookMouseUp(object sender, MouseHookEventArgs e)
        {
            if (this.IsConnected)
            {
                if (this.IsCapturing)
                {
                    this.UpdateMouseButtons(e.Button, false);

                    MouseEventArg eventArg = new MouseEventArg();
                    eventArg.button = this.mouseButtons;

                    if (this.messageSender.WriteInputData(MessageType.MouseUp, eventArg))
                    {
                        Debug.WriteLine("Mouse Up   : " + eventArg.ToInfoStr());
                    }

                    e.IsCanceled = true;
                }
                else if (this.isPointerOnEdge && e.Button == MouseHookEventArgs.MouseButtons.Left)
                {
                    if ((this.isDoubleClicked && this.userSettings.CaptureBorderSwitchType == BorderSwitchType.LDoubleClick) ||
                        this.userSettings.CaptureBorderSwitchType == BorderSwitchType.LClick)
                    {
                        if (this.StartCaptureOnMouse())
                        {
                            e.IsCanceled = true;
                        }
                    }
                }
            }

            this.isMouseButtonDown = false;
            this.isDoubleClicked = false;
        }

        private void OnHookMouseWheel(object sender, MouseHookEventArgs e)
        {
            if (this.IsConnected)
            {
                if (this.IsCapturing)
                {
                    MouseEventArg eventArg = new MouseEventArg();
                    eventArg.wheel = (short)e.Delta;
                    eventArg.scrollLines = (short)Utility.GetWheelScrollLines(e.Delta);

                    if (this.messageSender.WriteInputData(MessageType.MouseWheel, eventArg))
                    {
                        Debug.WriteLine("Mouse Wheel: " + eventArg.ToInfoStr());
                    }

                    e.IsCanceled = true;
                }
            }
        }

        private bool StartCaptureOnMouse()
        {
            //マウスポインタが境界に来たときにキャプチャを開始
            if (this.userSettings.IsCaptureOnMousePosition &&
                this.IsConnected && this.IsCaptureEnabled && !this.isKeyDown)
            {
                this.StartCapture(true, this.userSettings.CaptureBorderType);

                if (this.IsCapturing)
                {
                    return true;
                }
            }

            return false;
        }

        private void MouseOut()
        {
            // マウス移動でキャプチャを終了
            {
                // PC 側マウスカーソル位置を調整します。
                Point mouseOutPos = this.connecter.MouseOutPos;

                // スクリーン幅が 1200 の場合、
                // screenRect は Left=0, Right=1200 を返すが Cursor.Position は 0 - 1199 を返す。
                // Right, Bottom の場合に 1 を引く必要がある
                int margin = 1;
                if (this.userSettings.CaptureBorderType == BorderType.Right)
                {
                    mouseOutPos.X = this.screenRect.Right - (margin + 1);
                }
                else if (this.userSettings.CaptureBorderType == BorderType.Left)
                {
                    mouseOutPos.X = this.screenRect.Left + margin;
                }
                else if (this.userSettings.CaptureBorderType == BorderType.Top)
                {
                    mouseOutPos.Y = this.screenRect.Top + margin;
                }
                else if (this.userSettings.CaptureBorderType == BorderType.Bottom)
                {
                    mouseOutPos.Y = this.screenRect.Bottom - (margin + 1);
                }

                Cursor.Position = mouseOutPos;
            }

            this.StopCapture();
            this.connecter.ResetMouseOut();
        }

        ///-----------------------------------------------------------------

        private void OnKeyDown(object sender, KeyboardHookEventArgs e)
        {
            this.isKeyDown = true;
            this.isStartedByHotkey = false;
            this.UpdateKeyModifier(e.KeyCode, true);

            if (!this.IsConnected)
            {
                return;
            }

            if (!this.IsCapturing)
            {
                if (this.userSettings.IsCaptureOnHotKey &&
                    this.IsCaptureEnabled &&
                    (this.keyModifier == this.hotKeyModifier) && (e.KeyCode == (int)this.userSettings.HotKey))
                {
                    this.StartCapture(false, this.userSettings.CaptureBorderType);

                    if (this.IsCapturing)
                    {
                        this.isStartedByHotkey = true;
                    }
                }
            }
            else
            {
                bool stopCapturing = false;

                // Ctrl+Alt+Del が押されたらキャプチャを停止
                stopCapturing |= (this.keyModifier == (KeyModifier.Control | KeyModifier.Alt)) && (e.KeyCode == (int)Keys.Delete);

                // ホットキーでキャプチャを停止
                stopCapturing |= this.userSettings.IsCaptureOnHotKey &&
                    (this.keyModifier == this.hotKeyModifier) && (e.KeyCode == (int)this.userSettings.HotKey);

                if (stopCapturing)
                {
                    this.SendStopCaptureMsg();
                    this.StopCapture();

                    Cursor.Position = this.pointerPos;
                }
                else
                {
                    this.SendKeyDownMsg(e.KeyCode);
                    this.UpdateAndSendModifierKey(true);
                }

                e.IsCanceled = true;
            }
        }

        private void OnKeyUp(object sender, KeyboardHookEventArgs e)
        {
            if (this.IsConnected)
            {
                if (this.IsCapturing && !this.isStartedByHotkey)
                {
                    this.SendKeyUpMsg(e.KeyCode);
                    this.UpdateAndSendModifierKey(false);

                    e.IsCanceled = true;
                }
            }

            this.UpdateKeyModifier(e.KeyCode, false);
            this.isKeyDown = false;
        }

        private void UpdateAndSendModifierKey(bool onKeyDown)
        {
            // VK_SHIFT = 0x10、VK_CONTROL = 0x11、VK_MENU = 0x12 は
            // キーのフック処理で取得できないため、別途処理・送信を行う。
            // （VK_LSHIFT = 0xA0 などはフックで取得できる。）
            bool shiftPressed = Utility.IsKeyPressed(Utility.VirtKey.VK_SHIFT);
            bool controlPressed = Utility.IsKeyPressed(Utility.VirtKey.VK_CONTROL);
            bool menuPressed = Utility.IsKeyPressed(Utility.VirtKey.VK_MENU);

            if (onKeyDown)
            {
                if (shiftPressed)
                {
                    SendKeyDownMsg((int)Utility.VirtKey.VK_SHIFT);
                }
                if (controlPressed)
                {
                    SendKeyDownMsg((int)Utility.VirtKey.VK_CONTROL);
                }
                if (menuPressed)
                {
                    SendKeyDownMsg((int)Utility.VirtKey.VK_MENU);
                }
            }
            else
            {
                if (this.isShiftPressed && !shiftPressed)
                {
                    SendKeyUpMsg((int)Utility.VirtKey.VK_SHIFT);
                }
                if (this.isControlPressed && !controlPressed)
                {
                    SendKeyUpMsg((int)Utility.VirtKey.VK_CONTROL);
                }
                if (this.isMenuPressed && !menuPressed)
                {
                    SendKeyUpMsg((int)Utility.VirtKey.VK_MENU);
                }
            }

            this.isShiftPressed = shiftPressed;
            this.isControlPressed = controlPressed;
            this.isMenuPressed = menuPressed;
        }

        private void SendKeyDownMsg(int keyCode)
        {
            Debug.Assert(this.IsConnected);
            Debug.Assert(this.IsCapturing);

            KeyEventArg eventArg = Utility.CreateKeyEventArg(keyCode, this.keyModifier);
            if (eventArg == null)
            {
                Debug.WriteLine(string.Format("Key Down  : Unhandled {0:X2}", keyCode));
                return;
            }

            if (this.messageSender.WriteInputData(MessageType.KeyDown, eventArg))
            {
                Debug.WriteLine("Key Down  : " + eventArg.ToInfoStr());
            }
        }

        private void SendKeyUpMsg(int keyCode)
        {
            Debug.Assert(this.IsConnected);
            Debug.Assert(this.IsCapturing);

            KeyEventArg eventArg = Utility.CreateKeyEventArg(keyCode, this.keyModifier);
            if (eventArg == null)
            {
                Debug.WriteLine(string.Format("Key Up    : Unhandled {0:X2}", keyCode));
                return;
            }

            if (this.messageSender.WriteInputData(MessageType.KeyUp, eventArg))
            {
                Debug.WriteLine("Key Up    : " + eventArg.ToInfoStr());
            }
        }

        ///-----------------------------------------------------------------

        private void ShowInfoDialog()
        {
            if (this.infoDialog == null)
            {
                this.infoDialog = new InfoDialog(this.userSettings);
                this.infoDialog.Show();
            }
        }

        private void HideInfoDialog()
        {
            if (this.infoDialog != null)
            {
                if (this.infoDialog.InvokeRequired)
                {
                    this.infoDialog.Invoke(new MethodInvoker(() => { this.infoDialog.Close(); }));
                }
                else
                {
                    this.infoDialog.Close();
                }
            }

            this.infoDialog = null;
        }

        private bool CalcIsPointerOnEdge(Point position)
        {
            int edgeWidth = this.userSettings.CaptureBorderSwitchType == BorderSwitchType.None ? 0 : ScreenMargin;

            // スクリーン幅が 1200 の場合、
            // screenRect は Left=0, Right=1200 を返すが Cursor.Position は 0 - 1199 を返す。
            // Right, Bottom の場合に screenRect から 1 を引く必要がある
            switch (this.userSettings.CaptureBorderType)
            {
                case BorderType.Right:
                    return position.X >= this.screenRect.Right - (edgeWidth + 1);

                case BorderType.Left:
                    return position.X <= this.screenRect.Left + edgeWidth;

                case BorderType.Top:
                    return position.Y <= this.screenRect.Top + edgeWidth;

                case BorderType.Bottom:
                    return position.Y >= this.screenRect.Bottom - (edgeWidth + 1);

                default:
                    break;
            }

            return false;
        }

        private bool CalcIsDoubleClicked(MouseHookEventArgs.MouseButtons mouseDownButton)
        {
            if (this.mouseDownButtonFirst != mouseDownButton)
            {
                this.ResetDoubleClickParameters(mouseDownButton);
                return false;
            }

            TimeSpan delta = DateTime.Now - this.mouseDownTime;
            if (delta < TimeSpan.Zero || delta.TotalMilliseconds > SystemInformation.DoubleClickTime)
            {
                this.ResetDoubleClickParameters(mouseDownButton);
            }
            else
            {
                bool isAcceptable = true;
                if (Math.Abs(this.doubleClickDelta.Width) > SystemInformation.DoubleClickSize.Width)
                {
                    isAcceptable = false;
                }
                if (Math.Abs(this.doubleClickDelta.Height) > SystemInformation.DoubleClickSize.Height)
                {
                    isAcceptable = false;
                }

                if (isAcceptable)
                {
                    this.mouseDownTime = DateTime.MaxValue;
                    this.mouseDownButtonFirst = MouseHookEventArgs.MouseButtons.None;

                    return true;
                }
                else
                {
                    this.ResetDoubleClickParameters(mouseDownButton);
                }
            }

            return false;
        }

        private void ResetDoubleClickParameters(MouseHookEventArgs.MouseButtons mouseDownButton)
        {
            this.mouseDownButtonFirst = mouseDownButton;
            this.mouseDownTime = DateTime.Now;
            this.doubleClickDelta = Size.Empty;
        }

        private void UpdateMouseButtons(MouseHookEventArgs.MouseButtons mouseButton, bool isMouseDown)
        {
            if (isMouseDown)
            {
                this.mouseButtons |= this.ToMouseButton(mouseButton);
            }
            else
            {
                this.mouseButtons &= ~this.ToMouseButton(mouseButton);
            }
        }

        private MouseButton ToMouseButton(MouseHookEventArgs.MouseButtons mouseButton)
        {
            switch (mouseButton)
            {
                case MouseHookEventArgs.MouseButtons.Left:
                    return MouseButton.Left;

                case MouseHookEventArgs.MouseButtons.Right:
                    return MouseButton.Right;

                case MouseHookEventArgs.MouseButtons.Middle:
                    return MouseButton.Middle;

                case MouseHookEventArgs.MouseButtons.XButton1:
                    return MouseButton.X1;

                case MouseHookEventArgs.MouseButtons.XButton2:
                    return MouseButton.X2;
            }

            return MouseButton.None;
        }

        private void UpdateKeyModifier(int keyCode, bool isDown)
        {
            Keys key = (Keys)keyCode;

            if (isDown)
            {
                this.keyModifier |= this.ToKeyModifier(key);
            }
            else
            {
                this.keyModifier &= ~this.ToKeyModifier(key);
            }
        }

        private KeyModifier ToKeyModifier(Keys key)
        {
            switch (key)
            {
                case Keys.ShiftKey:
                case Keys.LShiftKey:
                case Keys.RShiftKey:
                    return KeyModifier.Shift;

                case Keys.ControlKey:
                case Keys.LControlKey:
                case Keys.RControlKey:
                    return KeyModifier.Control;

                case Keys.Menu:
                case Keys.LMenu:
                case Keys.RMenu:
                    return KeyModifier.Alt;
            }

            return KeyModifier.None;
        }
    }
}
