﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include "NXUtility.h"

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
//! @brief DLL から関数を取得するためのマクロです。
//=============================================================================
#define PROC_ADDRESS(handle, name)                                   \
*reinterpret_cast<void**>(&name) = GetProcAddress(handle, #name);    \
    if (name == nullptr)                                             \
    {                                                                \
        cerr << "Error: Cannot find function: " << #name << endl;    \
    }                                                                \

//-----------------------------------------------------------------------------
//! @brief 初期化します。
//-----------------------------------------------------------------------------
RStatus NXUtility::Initialize(const char* dllPath)
{
    //-----------------------------------------------------------------------------
    // 初期化済みなら終了処理をします。
    if (IsInitialized())
    {
        Finalize();
    }

    //-----------------------------------------------------------------------------
    // DLL ファイルのパスを設定します。
    if (dllPath != nullptr)
    {
        m_DllPath = dllPath;
    }
    else
    {
        dllPath = m_DllPath.c_str();
    }
    //cerr << "NXUtility: initialize: " << dllPath << endl;

    //-----------------------------------------------------------------------------
    // DLL をロードします。
    m_hDll = LoadLibraryEx(dllPath, nullptr, LOAD_WITH_ALTERED_SEARCH_PATH);
    if (m_hDll == nullptr)
    {
        return RStatus(RStatus::FAILURE, "Cannot load dll: " + std::string(dllPath) + // RShowError
            " (" + GetWindowsLastErrorMessage() + ")");
    }

    //-----------------------------------------------------------------------------
    // 関数アドレスを取得します。
    PROC_ADDRESS(m_hDll, ConvertTiling);
    PROC_ADDRESS(m_hDll, ReleaseTexture);
    PROC_ADDRESS(m_hDll, GetDataSizeNX);
    PROC_ADDRESS(m_hDll, GetMemoryPoolAlignment);

    if (ConvertTiling          == nullptr ||
        ReleaseTexture         == nullptr ||
        GetDataSizeNX          == nullptr ||
        GetMemoryPoolAlignment == nullptr)
    {
        Finalize();
        return RStatus(RStatus::FAILURE, "Cannot find function: " + std::string(dllPath)); // RShowError
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 終了します。
//-----------------------------------------------------------------------------
void NXUtility::Finalize()
{
    //cerr << "NXUtility: finalize: " << m_hDll << endl;

    //-----------------------------------------------------------------------------
    // DLL を解放します。
    if (m_hDll != nullptr)
    {
        FreeLibrary(m_hDll);
        m_hDll = nullptr;
    }

    //-----------------------------------------------------------------------------
    // 関数アドレスをクリアします。
    ConvertTiling          = nullptr;
    ReleaseTexture         = nullptr;
    GetDataSizeNX          = nullptr;
    GetMemoryPoolAlignment = nullptr;
}

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

