﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include "Convert.h"

using namespace nn::gfx::tool::texcvtr;

#define DLLEXPORT extern "C" __declspec(dllexport)

//=============================================================================
// variables
//=============================================================================
namespace {

GfxTexConverter* g_pConverter = nullptr;

} // namespace

//-----------------------------------------------------------------------------
//! @brief 構造化例外の変換関数です。
//-----------------------------------------------------------------------------
void TranslateSe(unsigned int code, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; // 標準 C++ の例外を発生させます。
    R_UNUSED_VARIABLE(code);
}

//-----------------------------------------------------------------------------
//! @brief DLL のメイン関数です。
//-----------------------------------------------------------------------------
BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
    switch (fdwReason)
    {
        case DLL_PROCESS_ATTACH:
            _set_se_translator(TranslateSe); // このスレッドでのみ有効です。
            g_pConverter = new GfxTexConverter(hinstDLL);
            break;

        case DLL_PROCESS_DETACH:
            if (g_pConverter != nullptr)
            {
                delete g_pConverter;
                g_pConverter = nullptr;
            }
            break;

        case DLL_THREAD_ATTACH:
            break;

        case DLL_THREAD_DETACH:
            break;

        default:
            break;
    }
    R_UNUSED_VARIABLE(lpvReserved);
    return TRUE;
}

//-----------------------------------------------------------------------------
//! @brief コンバーターのバージョンを返します。
//-----------------------------------------------------------------------------
DLLEXPORT int GetCvtrVersion()
{
    return g_pConverter->GetCvtrVersion();
}

//-----------------------------------------------------------------------------
//! @brief コンバーターの使用方法を表示します。
//-----------------------------------------------------------------------------
DLLEXPORT void ShowUsage(
    std::ostream& os,
    const wchar_t* cvtrName,
    const wchar_t* cmdName,
    const bool isVersionOnly
)
{
    g_pConverter->ShowUsage(os, cvtrName, cmdName, isVersionOnly);
}

//-----------------------------------------------------------------------------
//! @brief 現在の環境で GPU によるエンコーディングが可能か判定します。
//-----------------------------------------------------------------------------
DLLEXPORT bool CheckGpuEncoding()
{
    return g_pConverter->CheckGpuEncoding();
}

//-----------------------------------------------------------------------------
//! @brief コンバーターに COM のモードを設定します。
//-----------------------------------------------------------------------------
DLLEXPORT void SetComMode(const bool initializesCom, const int coInit)
{
    g_pConverter->SetComMode(initializesCom, coInit);
}

//-----------------------------------------------------------------------------
//! @brief 引数がグローバルオプションなら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool IsGlobalOptionArg(const wchar_t* arg)
{
    return g_pConverter->IsGlobalOptionArg(arg);
}

//-----------------------------------------------------------------------------
//! @brief 指定した引数の次の引数がグローバルオプションの値なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool IsNextArgGlobalOptionValue(const wchar_t* arg)
{
    return g_pConverter->IsNextArgGlobalOptionValue(arg);
}

//-----------------------------------------------------------------------------
//! @brief 指定した引数の次の引数がジョブオプションの値なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool IsNextArgJobOptionValue(const wchar_t* arg)
{
    return g_pConverter->IsNextArgJobOptionValue(arg);
}

//-----------------------------------------------------------------------------
//! @brief グローバルオプションを設定します。
//-----------------------------------------------------------------------------
DLLEXPORT bool SetOptions(const wchar_t* options[])
{
    return g_pConverter->SetOptions(options);
}

//-----------------------------------------------------------------------------
//! @brief メモリーをクリアし、ジョブオプションを初期化します。
//!        元画像、出力データ、エラー文字列などをすべてクリアします。
//-----------------------------------------------------------------------------
DLLEXPORT void Clear()
{
    g_pConverter->Clear();
}

//-----------------------------------------------------------------------------
//! @brief 入力ファイルをリードします。
//-----------------------------------------------------------------------------
DLLEXPORT bool ReadInputFile(const wchar_t* paths[], const wchar_t* options[])
{
    return g_pConverter->ReadInputFile(paths, options);
}

//-----------------------------------------------------------------------------
//! @brief メモリー上の ftx ファイルのデータをリードします。
//-----------------------------------------------------------------------------
DLLEXPORT bool ReadFtxData(
    const void* pData,
    const size_t dataSize,
    const wchar_t* path,
    const wchar_t* options[]
)
{
    return g_pConverter->ReadFtxData(pData, dataSize, path, options);
}

//-----------------------------------------------------------------------------
//! @brief メモリー上のビットマップデータをリードします。
//-----------------------------------------------------------------------------
DLLEXPORT bool ReadBitmapData(
    const void* pData,
    const int width,
    const int height,
    const int depth,
    const bool hasAlpha,
    const bool isFloat,
    const wchar_t* paths[],
    const wchar_t* originalPaths[],
    const wchar_t* options[]
)
{
    return g_pConverter->ReadBitmapData(pData, width, height, depth, hasAlpha, isFloat,
        paths, originalPaths, options);
}

//-----------------------------------------------------------------------------
//! @brief 変換して出力ファイルをセーブします。
//!        リード時のジョブオプションで指定した出力パスにセーブします。
//-----------------------------------------------------------------------------
DLLEXPORT bool ConvertToFile()
{
    return g_pConverter->ConvertToFile();
}

//-----------------------------------------------------------------------------
//! @brief 変換して出力ファイルをメモリーに格納します。
//!        リード時のジョブオプションで指定した出力パスによって
//!        出力ファイル形式が決定されます。
//-----------------------------------------------------------------------------
DLLEXPORT bool ConvertToData(const void** ppData, size_t* pDataSize)
{
    return g_pConverter->ConvertToData(ppData, pDataSize);
}

//-----------------------------------------------------------------------------
//! @brief エラーのユニコード文字列を返します。
//-----------------------------------------------------------------------------
DLLEXPORT const wchar_t* GetErrorString()
{
    return g_pConverter->GetErrorString();
}

//-----------------------------------------------------------------------------
//! @brief エラーの Shift JIS 文字列を返します。
//-----------------------------------------------------------------------------
DLLEXPORT const char* GetErrorShiftJisString()
{
    return g_pConverter->GetErrorShiftJisString();
}

//-----------------------------------------------------------------------------
//! @brief 作成情報を設定します。
//-----------------------------------------------------------------------------
DLLEXPORT bool SetCreateInfo(const wchar_t* createInfo)
{
    return g_pConverter->SetCreateInfo(createInfo);
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータを設定します。
//-----------------------------------------------------------------------------
DLLEXPORT bool SetUserData(const void* pPackedUserData, const size_t packedUserDataSize)
{
    return g_pConverter->SetUserData(pPackedUserData, packedUserDataSize);
}

//-----------------------------------------------------------------------------
//! @brief 編集用コメントラベルを設定します。
//-----------------------------------------------------------------------------
DLLEXPORT bool SetCommentLabel(const void* pCommentLabel, const size_t commentLabelSize)
{
    return g_pConverter->SetCommentLabel(pCommentLabel, commentLabelSize);
}

//-----------------------------------------------------------------------------
//! @brief 編集用コメントカラー文字列を設定します。
//-----------------------------------------------------------------------------
DLLEXPORT bool SetCommentColor(const void* pCommentCol, const size_t commentColSize)
{
    return g_pConverter->SetCommentColor(pCommentCol, commentColSize);
}

//-----------------------------------------------------------------------------
//! @brief ツールデータを設定します。
//-----------------------------------------------------------------------------
DLLEXPORT bool SetToolData(const void* pToolData, const size_t toolDataSize)
{
    return g_pConverter->SetToolData(pToolData, toolDataSize);
}

//-----------------------------------------------------------------------------
//! @brief ユーザーツールデータを設定します。
//-----------------------------------------------------------------------------
DLLEXPORT bool SetUserToolData(const void* pUserToolData, const size_t userToolDataSize)
{
    return g_pConverter->SetUserToolData(pUserToolData, userToolDataSize);
}

