﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// GetExrInfo ParseExrInfo
// ReadExrAsFloat16 ReadExrAsFloat32 ParseExrAsFloat16 ParseExrAsFloat32
// WriteExrFloat16 WriteExrFloat32
// ConvertYcaToRgba32

// ExrIMemoryStream

//=============================================================================
// include
//=============================================================================
#pragma warning(push)
    #pragma warning(disable: 4100) // 引数は関数の本体部で 1 度も参照されません。
    #pragma warning(disable: 4244) // '=' : 'int' から 'unsigned short' への変換です。
    #pragma warning(disable: 4996) // 'strncpy': This function or variable may be unsafe.
    #include <ImfVersion.h>
    #include <ImfInputFile.h>
    #include <ImfOutputFile.h>
    #include <ImfChannelList.h>
    #include <ImfRgbaFile.h>
    #include <ImfRgbaYca.h>            // YCA から RGBA に変換するなら必要
    #include <ImfStandardAttributes.h> // YCA から RGBA に変換するなら必要
    #include <ImfIO.h> // メモリからリードするなら必要
    #include <ImfTiledInputFile.h>  // ミップマップをリードするなら必要
    //#include <ImfArray.h>
#pragma warning(pop)

#include <cstdint>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>

#include <windows.h>
#undef min
#undef max

using namespace std;

#define DLLEXPORT extern "C" __declspec(dllexport)

//=============================================================================
//! @brief 未使用変数の警告を抑止するためのマクロです。
//=============================================================================
#define R_UNUSED_VARIABLE(Variable) (void)(&Variable);

//-----------------------------------------------------------------------------
//! @brief 構造化例外の変換関数です。
//-----------------------------------------------------------------------------
void TranslateSe(unsigned int code, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; // 標準 C++ の例外を発生させます。
    R_UNUSED_VARIABLE(code);
}

//-----------------------------------------------------------------------------
//! @brief DLL のメイン関数です。
//-----------------------------------------------------------------------------
BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
    switch (fdwReason)
    {
        case DLL_PROCESS_ATTACH:
            _set_se_translator(TranslateSe); // このスレッドでのみ有効です。
            break;

        case DLL_PROCESS_DETACH:
            break;

        case DLL_THREAD_ATTACH:
            break;

        case DLL_THREAD_DETACH:
            break;

        default:
            break;
    }
    R_UNUSED_VARIABLE(hinstDLL);
    R_UNUSED_VARIABLE(lpvReserved);
    return TRUE;
}

//=============================================================================
// 定数の定義です。
//=============================================================================

//-----------------------------------------------------------------------------
// EXR
namespace ROpenExr
{

//! @brief 各種フラグを表す列挙型です。
enum Flags
{
    FLAGS_UINT    = (1 <<  0), //!< ピクセルタイプが UINT なら ON です。
    FLAGS_HALF    = (1 <<  1), //!< ピクセルタイプが HALF なら ON です。
    FLAGS_YC      = (1 <<  2), //!< YC カラーなら ON です。
    FLAGS_MIPMAP  = (1 <<  3), //!< ミップマップなら ON です。
    FLAGS_CUBEMAP = (1 <<  4), //!< キューブマップなら ON です。
    FLAGS_TILED   = (1 <<  5), //!< タイル形式なら ON です。
};

} // namespace ROpenExr

namespace
{

//-----------------------------------------------------------------------------
// color
const int R_RGB_COUNT  = 3; //!< RGB の成分数です。
const int R_RGBA_COUNT = 4; //!< RGBA の成分数です。

const int R_RGB_BYTES  = 3; //!< RGB 各 8 ビット整数のピクセルのバイト数です。
const int R_RGBA_BYTES = 4; //!< RGBA 各 8 ビット整数のピクセルのバイト数です。

const int R_RGB_FLOAT16_BYTES  = 3 * sizeof(uint16_t); //!< RGB 各 16 ビット浮動小数点数のピクセルのバイト数です。
const int R_RGBA_FLOAT16_BYTES = 4 * sizeof(uint16_t); //!< RGBA 各 16 ビット浮動小数点数のピクセルのバイト数です。

const int R_RGB_FLOAT_BYTES  = 3 * sizeof(float); //!< RGB 各 32 ビット浮動小数点数のピクセルのバイト数です。
const int R_RGBA_FLOAT_BYTES = 4 * sizeof(float); //!< RGBA 各 32 ビット浮動小数点数のピクセルのバイト数です。

//=============================================================================
// variables
//=============================================================================
std::string g_ErrorString; // エラーの Shift JIS 文字列です。
std::wstring g_ErrorWString; // エラーのユニコード文字列です。

//=============================================================================
// メモリ関連の関数です。
//=============================================================================

//! @brief 2 つの値を交換します。
template <typename T>
inline void RSwapValue(T& r1, T& r2)
{
    T tmpVal = r1;
    r1 = r2;
    r2 = tmpVal;
}

//! @brief メモリからリトルエンディアンの値を取得します。
template <typename T>
void RGetMemLittle(T& value, const void* mem)
{
    memcpy(&value, mem, sizeof(T));
    #if 0
    // swap for big endian
    uint8_t* p1 = reinterpret_cast<uint8_t*>(&value);
    uint8_t* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int ibyte = 0; ibyte < byteSizeHalf; ++ibyte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//=============================================================================
//! @brief EXR 入力メモリストリームのクラスです。
//=============================================================================
class ExrIMemoryStream : public Imf::IStream
{
public:
    //! コンストラクタです。
    ExrIMemoryStream(const void* pData, const Imf::Int64 dataSize)
    : Imf::IStream("memory"),
      m_pData(reinterpret_cast<const uint8_t*>(pData)),
      m_DataSize(dataSize),
      m_Pos(0)
    {
    }

    //! デストラクタです。
    virtual ~ExrIMemoryStream()
    {
    }

    //! メモリ上にデータが存在するなら true を返します。
    //! true なら呼び出し側はデータをコピーしないで直接アクセスします。
    virtual bool isMemoryMapped() const
    {
        return true;
    }

    //! n バイトリードしてポインタ c で指定されたバッファにコピーします。
    //! ファイルの最後までリードしたなら false、まだデータが残っていれば true を返します。
    virtual bool read(char c[/*n*/], int n)
    {
        //cerr << "read: " << n << endl;
        memcpy(c, m_pData + m_Pos, n);
        m_Pos += n;
        return (m_Pos < m_DataSize);
    }

    //! 現在のデータのポインタを返し、位置を n バイト進めます。
    virtual char* readMemoryMapped(int n)
    {
        //cerr << "readMemoryMapped: " << n << endl;
        char* pRet = reinterpret_cast<char*>(const_cast<uint8_t*>(m_pData + m_Pos));
        m_Pos += n;
        return pRet;
    }

    //! 現在の位置を返します。
    virtual Imf::Int64 tellg()
    {
        return m_Pos;
    }

    //! 現在の位置を設定します。
    virtual void seekg (Imf::Int64 pos)
    {
        m_Pos = pos;
    }

protected:
    const uint8_t* m_pData; //!< データの先頭ポインタです。
    Imf::Int64 m_DataSize; //!< データのサイズです。
    Imf::Int64 m_Pos; //!< 現在の位置です。
};

//-----------------------------------------------------------------------------
//! @brief 16-bit float のビットマップの RGBA 値を表示します。
//!
//! @param[in] pBitmapF16 16-bit float のビットマップへのポインタです。
//! @param[in] imageW ビットマップの幅です。
//! @param[in] imageH ビットマップの高さです。
//-----------------------------------------------------------------------------
inline void DisplayBitmapF16(const void* pBitmapF16, const int imageW, const int imageH)
{
    const Imf::Rgba* pixels = reinterpret_cast<const Imf::Rgba*>(pBitmapF16);
    for (int iy = 0; iy < imageH; ++iy)
    {
        for (int ix = 0; ix < imageW; ++ix)
        {
            const Imf::Rgba& pixel = pixels[ix + iy * imageW];
            cerr << "(" << ix << "," << iy << "): "
                 << pixel.r << " "
                 << pixel.g << " "
                 << pixel.b << " "
                 << pixel.a << endl;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 32-bit float のビットマップの RGBA 値を標準エラーに出力します。
//!
//! @param[in] pBitmapF32 32-bit float のビットマップへのポインタです。
//! @param[in] imageW ビットマップの幅です。
//! @param[in] imageH ビットマップの高さです。
//-----------------------------------------------------------------------------
inline void DisplayBitmapF32(const void* pBitmapF32, const int imageW, const int imageH)
{
    const float* pixels = reinterpret_cast<const float*>(pBitmapF32);
    for (int iy = 0; iy < imageH; ++iy)
    {
        for (int ix = 0; ix < imageW; ++ix)
        {
            const float* pPix = &pixels[(ix + iy * imageW) * R_RGBA_COUNT];
            cerr << "(" << ix << "," << iy << "): "
                 << pPix[0] << " "
                 << pPix[1] << " "
                 << pPix[2] << " "
                 << pPix[3] << endl;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief get Unicode (UCS-2) from Shift JIS
//-----------------------------------------------------------------------------
std::wstring RGetUnicodeFromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    const int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::wstring();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);
    std::wstring dst(wcharBuf, wcslen(wcharBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief get Shift JIS from Unicode (UCS-2)
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUnicode(const std::wstring& src)
{
    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    const int sjisSize = WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, NULL, 0, NULL, NULL);
    if (sjisSize == 0)
    {
        return std::string();
    }

    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, sjisBuf, sjisSize, NULL, NULL);
    std::string dst(sjisBuf, strlen(sjisBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief エラー文字列を設定します。
//!
//! @param[in] str エラーの Shift JIS 文字列です。
//-----------------------------------------------------------------------------
void SetErrorString(const std::string& str)
{
    g_ErrorString = str;
    g_ErrorWString = RGetUnicodeFromShiftJis(str);
    #ifdef EXR_DEBUG_PRINT
    cerr << "Error: " << str << endl;
    #endif
}

//-----------------------------------------------------------------------------
//! @brief RGBA 32-bit float のフレームバッファを設定します。
//!
//! @param[in,out] fbuf フレームバッファです。
//! @param[in] pBaseF32 32-bit float のビットマップへのポインタです。
//!                     幅 x 高さ x 4 x 4 バイトのメモリを確保しておきます。
//! @param[in] channelPrefix チャンネル名のプレフィックスです。
//! @param[in] imageW ビットマップの幅です。
//-----------------------------------------------------------------------------
void SetRgbaFloat32FrameBuffer(
    Imf::FrameBuffer& fbuf,
    void* pBaseF32,
    const std::string& channelPrefix,
    const int imageW
)
{
    char* pBase = reinterpret_cast<char*>(pBaseF32);
    const size_t xst = R_RGBA_COUNT * sizeof(float);
    const size_t yst = xst * imageW;
    fbuf.insert((channelPrefix + "R").c_str(), Imf::Slice(Imf::FLOAT, pBase + 0 * sizeof(float), xst, yst, 1, 1, 0.0));
    fbuf.insert((channelPrefix + "G").c_str(), Imf::Slice(Imf::FLOAT, pBase + 1 * sizeof(float), xst, yst, 1, 1, 0.0));
    fbuf.insert((channelPrefix + "B").c_str(), Imf::Slice(Imf::FLOAT, pBase + 2 * sizeof(float), xst, yst, 1, 1, 0.0));
    fbuf.insert((channelPrefix + "A").c_str(), Imf::Slice(Imf::FLOAT, pBase + 3 * sizeof(float), xst, yst, 1, 1, 1.0));
}

//-----------------------------------------------------------------------------
//! @brief YCA 32-bit float のフレームバッファを設定します。
//!
//! @param[in,out] fbuf フレームバッファです。
//! @param[in] pBaseF32 32-bit float のビットマップへのポインタです。
//!                     幅 x 高さ x 4 x 4 バイトのメモリを確保しておきます。
//! @param[in] channelPrefix チャンネル名のプレフィックスです。
//! @param[in] imageW ビットマップの幅です。
//! @param[in] isYuv422 YUV422 なら true です。
//-----------------------------------------------------------------------------
void SetYcaFloat32FrameBuffer(
    Imf::FrameBuffer& fbuf,
    void* pBaseF32,
    const std::string& channelPrefix,
    const int imageW,
    const bool isYuv422
)
{
    char* pBase = reinterpret_cast<char*>(pBaseF32);
    const size_t xst = R_RGBA_COUNT * sizeof(float);
    const size_t yst = xst * imageW;
    const size_t uvxst = (isYuv422) ? xst * 2 : xst;
    const size_t uvyst = (isYuv422) ? yst * 2 : yst;
    const int uvs      = (isYuv422) ? 2 : 1;
    fbuf.insert((channelPrefix + "Y" ).c_str(), Imf::Slice(Imf::FLOAT, pBase + 0 * sizeof(float), xst  , yst  ,   1,   1, 0.0));
    fbuf.insert((channelPrefix + "BY").c_str(), Imf::Slice(Imf::FLOAT, pBase + 1 * sizeof(float), uvxst, uvyst, uvs, uvs, 0.0));
    fbuf.insert((channelPrefix + "RY").c_str(), Imf::Slice(Imf::FLOAT, pBase + 2 * sizeof(float), uvxst, uvyst, uvs, uvs, 0.0));
    fbuf.insert((channelPrefix + "A" ).c_str(), Imf::Slice(Imf::FLOAT, pBase + 3 * sizeof(float), xst  , yst  ,   1,   1, 1.0));
}

//-----------------------------------------------------------------------------
//! @brief YCA 32-bit float を RGBA 32-bit float に変換します。
//!
//! @param[in,out] pBitmapF32 32-bit float のビットマップへのポインタです。
//! @param[in] imageW ビットマップの幅です。
//! @param[in] imageH ビットマップの高さです。
//! @param[in] channelCount チャンネル数です。
//! @param[in] isYuv422 YUV422 なら true です。
//! @param[in] header ヘッダです。
//-----------------------------------------------------------------------------
void ConvertYcaToRgba32(
    void* pBitmapF32,
    const int imageW,
    const int imageH,
    const int channelCount,
    const bool isYuv422,
    const Imf::Header& header
)
{
    float* pDstTop = reinterpret_cast<float*>(pBitmapF32);
    float* pDst = pDstTop;
    const int pixCount = imageW * imageH;
    if (channelCount <= 2) // Y or YA
    {
        //-----------------------------------------------------------------------------
        // Y 成分のみなら R = G = B = Y にします。
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            pDst[1] = pDst[2] = pDst[0];
            pDst += R_RGBA_COUNT;
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // YUV422 なら補間して YUV444 にします。
        if (isYuv422)
        {
            // 水平方向に補間します。
            for (int iy = 0; iy < imageH; iy += 2)
            {
                const float* pL = pDstTop + iy * imageW * R_RGBA_COUNT;
                const float* pR = (imageW >= 3) ? pL + R_RGBA_COUNT * 2 : pL;
                for (int ix = 1; ix < imageW; ix += 2)
                {
                    pDst[4 + 1] = (pL[1] + pR[1]) * 0.5f;
                    pDst[4 + 2] = (pL[2] + pR[2]) * 0.5f;
                    pL = pR;
                    pR = (ix + 1 + 2 < imageW) ? pL + R_RGBA_COUNT * 2 : pL;
                    pDst += R_RGBA_COUNT * 2;
                }
                pDst += imageW * R_RGBA_COUNT;
            }

            // 垂直方向に補間します。
            pDst = pDstTop + imageW * R_RGBA_COUNT;
            for (int iy = 1; iy < imageH; iy += 2)
            {
                const int y0 = iy - 1;
                const int y1 = (y0 + 2 < imageH) ? y0 + 2 : y0;
                const float* pU = pDstTop + y0 * imageW * R_RGBA_COUNT;
                const float* pD = pDstTop + y1 * imageW * R_RGBA_COUNT;
                for (int ix = 0; ix < imageW; ++ix)
                {
                    pDst[1] = (pU[1] + pD[1]) * 0.5f;
                    pDst[2] = (pU[2] + pD[2]) * 0.5f;
                    pU += R_RGBA_COUNT;
                    pD += R_RGBA_COUNT;
                    pDst += R_RGBA_COUNT;
                }
                pDst += imageW * R_RGBA_COUNT;
            }
        }

        //-----------------------------------------------------------------------------
        // YUV を RGB に変換します。
        Imf::Chromaticities cr;
        if (Imf::hasChromaticities(header))
        {
            cr = Imf::chromaticities(header);
        }
        const Imath::V3f yw = Imf::RgbaYca::computeYw(cr);

        pDst = pDstTop;
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            const float y = pDst[0];
            const float u = pDst[1];
            const float v = pDst[2];
            if (u == 0.0f && v == 0.0f)
            {
                pDst[1] = pDst[2] = y;
            }
            else
            {
                pDst[0] = (v + 1.0f) * y;
                pDst[2] = (u + 1.0f) * y;
                pDst[1] = (y - pDst[0] * yw.x - pDst[2] * yw.z) / yw.y;
            }
            pDst += R_RGBA_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 32-bit float のビットマップを取得する共通処理です。
//!
//! @param[out] pBitmapF32 32-bit float のビットマップへのポインタです。
//!                        幅 x 高さ x 4 x 4 バイトのメモリを確保しておきます。
//! @param[in] pData メモリ上の EXR ファイルデータのポインタです。
//! @param[in] dataSize メモリ上の EXR ファイルデータのサイズです。
//! @param[in] channelCount チャンネル数です。
//! @param[in] flags 各種フラグです。
//! @param[in] filePath EXR ファイルのパスです（デバッグ用）。ファイルからリードしていない場合は NULL を指定します。
//-----------------------------------------------------------------------------
void GetAsFloat32Common(
    void* pBitmapF32,
    const void* pData,
    const size_t dataSize,
    const int channelCount,
    const uint32_t flags,
    const char* filePath
)
{
    //-----------------------------------------------------------------------------
    // InputFile オブジェクトを生成します。
    // InputFile は、EXR ファイルの pixelType と
    // フレームバッファの pixelType が異なる場合でもリード可能です。
    // （ImfMisc.cpp の copyIntoFrameBuffer で変換）
    ExrIMemoryStream ims(pData, dataSize);
    Imf::InputFile file(ims);
    R_UNUSED_VARIABLE(filePath);

    //-----------------------------------------------------------------------------
    // フレームバッファを設定します。
    const Imath::Box2i dw = file.header().dataWindow();
    const int imageW = dw.max.x - dw.min.x + 1;
    const int imageH = dw.max.y - dw.min.y + 1;
    R_UNUSED_VARIABLE(imageH);
    #ifdef EXR_DEBUG_PRINT
    cerr << "Read as float32 ("
         << ((filePath != NULL) ? "by InputFile" : "from memory")
         << "): " << imageW << " x " << imageH << endl;
    #endif
    Imf::FrameBuffer fbuf;
    const int baseOfs = (dw.min.x + dw.min.y * imageW) * R_RGBA_COUNT;
    float* pBaseF32 = reinterpret_cast<float*>(pBitmapF32) - baseOfs;
        // ↑フレームバッファの先頭アドレスはウィンドウ空間の左上なので
        //   アドレスを調整します。
    const bool isYc = ((flags & ROpenExr::FLAGS_YC) != 0);
    bool isYuv422 = true;
    if (!isYc)
    {
        SetRgbaFloat32FrameBuffer(fbuf, pBaseF32, "", imageW);
    }
    else
    {
        SetYcaFloat32FrameBuffer(fbuf, pBaseF32, "", imageW, isYuv422);
    }
    file.setFrameBuffer(fbuf);

    //-----------------------------------------------------------------------------
    // リードします。
    file.readPixels(dw.min.y, dw.max.y);

    //-----------------------------------------------------------------------------
    // 必要なら YCA から RGBA に変換します。
    if (isYc)
    {
        ConvertYcaToRgba32(pBitmapF32, imageW, imageH, channelCount, isYuv422, file.header());
    }

    //-----------------------------------------------------------------------------
    // ミップマップデータを取得します。
    #if 0
    if ((flags & ROpenExr::FLAGS_TILED ) != 0 &&
        (flags & ROpenExr::FLAGS_MIPMAP) != 0)
    {
        ims.seekg(0);
        Imf::TiledInputFile tiled(ims);
        const int mipLevel = tiled.numLevels();
        #ifdef EXR_DEBUG_PRINT
        cerr << "mipmap: " << mipLevel << " levels" << endl;
        #endif
        for (int level = 1; level < mipLevel; ++level)
        {
            #ifdef EXR_DEBUG_PRINT
            cerr << "level " << level << ": " << tiled.levelWidth(level) << " x " << tiled.levelHeight(level) << " ("
                 << tiled.numXTiles(level) << " x " << tiled.numYTiles(level) << ")" << endl;
            #endif
            tiled.setFrameBuffer(fbuf);
            tiled.readTiles(0, tiled.numXTiles(level) - 1, 0, tiled.numYTiles(level) - 1, level);
        }
    }
    #endif
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief メモリ上の EXR ファイルデータから情報を取得します。
//!
//! @param[out] pWidth 幅を格納する変数へのポインタです。
//! @param[out] pHeight 高さを格納する変数へのポインタです。
//! @param[out] pDepth 奥行きを格納する変数へのポインタです。
//! @param[out] pChannelCount チャンネル数を格納する変数へのポインタです。
//! @param[out] pFlags 各種フラグを格納する変数へのポインタです。
//! @param[in] pData メモリ上の EXR ファイルデータのポインタです。
//! @param[in] dataSize メモリ上の EXR ファイルデータのサイズです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool ParseExrInfo(
    int* pWidth,
    int* pHeight,
    int* pDepth,
    int* pChannelCount,
    uint32_t* pFlags,
    const void* pData,
    const size_t dataSize
)
{
    //-----------------------------------------------------------------------------
    // 初期値を設定します。
    *pWidth  = 0;
    *pHeight = 0;
    *pDepth  = 1;
    *pChannelCount = 0;
    *pFlags = 0;

    //-----------------------------------------------------------------------------
    // ヘッダの識別子をチェックします。
    const char* pHeader = reinterpret_cast<const char*>(pData);
    if (dataSize < 8      ||
        pHeader[0] != 0x76 ||
        pHeader[1] != 0x2F ||
        pHeader[2] != 0x31 ||
        pHeader[3] != 0x01)
    {
        SetErrorString(std::string("EXR file is wrong"));
        return false;
    }

    //-----------------------------------------------------------------------------
    // バージョンを解析します。
    const char* pCur = pHeader + 4;
    uint32_t version;
    RGetMemLittle(version, pCur);
    if ((version & Imf::TILED_FLAG) != 0)
    {
        *pFlags |= ROpenExr::FLAGS_TILED;
    }
    pCur += 4;

    //-----------------------------------------------------------------------------
    // ヘッダを解析します。
    const char* pEnd = pHeader + dataSize;
    bool isChannelsFound = false;
    bool isWindowFound = false;
    while (pCur < pEnd && *pCur != 0)
    {
        // アトリビュート名を取得します。
        const std::string attrName(pCur);
        pCur += attrName.size() + 1;
        if (pCur >= pEnd)
        {
            break;
        }

        // アトリビュートタイプを取得します。
        const std::string attrType(pCur);
        pCur += attrType.size() + 1;
        if (pCur >= pEnd)
        {
            break;
        }

        // アトリビュートのサイズを取得します。
        uint32_t attrSize;
        RGetMemLittle(attrSize, pCur);
        #ifdef EXR_DEBUG_PRINT
        cerr << "attr: " << attrName << " (" << attrType << ") " << attrSize << endl;
        #endif
        pCur += sizeof(attrSize);
        if (pCur + attrSize > pEnd)
        {
            break;
        }

        if (attrName == "channels") // chlist
        {
            // チャンネル情報を取得します。
            const char* pChan = pCur;
            int iChan = 0;
            while (pChan < pCur + attrSize && *pChan != 0)
            {
                const std::string chanName(pChan);
                pChan += chanName.size() + 1;
                int chanPixelType; // 0=UINT, 1=HALF, 2=FLOAT
                RGetMemLittle(chanPixelType, pChan);
                pChan += 4;
                pChan += 1 + 3; // skip pLinear & reserved
                pChan += 8; // skip xSampling & ySampling
                #ifdef EXR_DEBUG_PRINT
                cerr << "  chan" << iChan << ": " << chanName << ": " << chanPixelType << endl;
                #endif
                ++iChan;
                const bool isRgba = (
                    chanName == "R" ||
                    chanName == "G" ||
                    chanName == "B" ||
                    chanName == "A");
                const bool isYc = (
                    chanName == "Y"  ||
                    chanName == "BY" ||
                    chanName == "RY");
                if (isRgba || isYc)
                {
                    ++*pChannelCount;
                    if (chanPixelType == Imf::UINT)
                    {
                        *pFlags |= ROpenExr::FLAGS_UINT;
                    }
                    else if (chanPixelType == Imf::HALF)
                    {
                        *pFlags |= ROpenExr::FLAGS_HALF;
                    }

                    if (isYc)
                    {
                        *pFlags |= ROpenExr::FLAGS_YC;
                    }
                }
            }
            isChannelsFound = true;
        }
        else if (attrName == "dataWindow")
        {
            // データウィンドウを取得します。
            if (attrType == "box2i")
            {
                int minX, minY, maxX, maxY;
                RGetMemLittle(minX, pCur +  0);
                RGetMemLittle(minY, pCur +  4);
                RGetMemLittle(maxX, pCur +  8);
                RGetMemLittle(maxY, pCur + 12);
                *pWidth  = maxX - minX + 1;
                *pHeight = maxY - minY + 1;
            }
            isWindowFound = true;
        }
        else if (attrName == "compression")
        {
            // 圧縮モードを取得します。
            #ifdef EXR_DEBUG_PRINT
            static const char* const compressionStrs[] =
            {
                "NONE", // 0
                "RLE",  // 1
                "ZIPS", // 2
                "ZIP",  // 3
                "PIZ",  // 4
                "PXR24",// 5
                "B44",  // 6
                "B44A", // 7
            };
            cerr << "  " << compressionStrs[pCur[0]] << " (" << static_cast<int>(pCur[0]) << ")" << endl;
            #endif
        }
        else if (attrName == "envmap")
        {
            // 環境マップモードを取得します。
            if (pCur[0] == 1)
            {
                *pFlags |= ROpenExr::FLAGS_CUBEMAP;
            }
            #ifdef EXR_DEBUG_PRINT
            static const char* const envmapStrs[] =
            {
                "LATLONG", // 0
                "CUBE",    // 1
            };
            cerr << "  " << envmapStrs[pCur[0]] << " (" << static_cast<int>(pCur[0]) << ")" << endl;
            #endif
        }
        else if (attrName == "tiles")
        {
            // タイル情報を取得します。
            int xSize, ySize;
            RGetMemLittle(xSize, pCur + 0);
            RGetMemLittle(ySize, pCur + 4);
            const int levelMode    = (pCur[8]     ) & 0x0f;
            const int roundingMode = (pCur[8] >> 4) & 0x0f;
            if (levelMode == 1)
            {
                *pFlags |= ROpenExr::FLAGS_MIPMAP;
            }
            R_UNUSED_VARIABLE(roundingMode);

            #ifdef EXR_DEBUG_PRINT
            static const char* const levelModeStrs[] =
            {
                "ONE",    // 0
                "MIPMAP", // 1
                "RIPMAP", // 2
            };
            static const char* const roundingModeStrs[] =
            {
                "ROUND_DOWN", // 0
                "ROUND_UP",   // 1
            };
            cerr << "  " << xSize << " x " << ySize << " " << levelModeStrs[levelMode] << " " << roundingModeStrs[roundingMode] << endl;
            #endif
        }
        else if (attrName == "multiView")
        {
            // ビュー名配列を取得します。
            #ifdef EXR_DEBUG_PRINT
            const char* pView = pCur;
            while (pView < pCur + attrSize)
            {
                uint32_t len;
                RGetMemLittle(len, pView);
                pView += sizeof(uint32_t);
                cerr << "  " << std::string(pView, len) << endl;
                pView += len;
            }
            #endif
        }
        else if (attrType == "string")
        {
            #ifdef EXR_DEBUG_PRINT
            cerr << "  " << std::string(pCur, attrSize) << endl;
            #endif
        }
        else if (attrType == "preview")
        {
            // プレビュー画像データを取得します。
            // uint32_t 型の幅と高さの後に RGBA データ（1 ピクセル 4 バイト）が続きます。
            // A 成分は透明なら 0、不透明なら 127 が格納されています。
            uint32_t previewW;
            uint32_t previewH;
            RGetMemLittle(previewW, pCur + 0);
            RGetMemLittle(previewH, pCur + 4);
            #ifdef EXR_DEBUG_PRINT
            cerr << "  " << previewW << " x " << previewH << endl;
            OutTgaFile("preview.tga", pCur + 8, previewW, previewH, true);
            #endif
        }

        pCur += attrSize;
    }

    if (!isChannelsFound || !isWindowFound)
    {
        SetErrorString(std::string("EXR file is wrong"));
        return false;
    }

    // キューブマップなら高さと奥行きを調整します。
    if ((*pFlags & ROpenExr::FLAGS_CUBEMAP) != 0)
    {
        if (*pHeight >= 6)
        {
            *pDepth = 6;
            *pHeight /= *pDepth;
        }
    }

    return true;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief EXR ファイルのヘッダをリードして情報を取得します。
//!
//! @param[out] pWidth 幅を格納する変数へのポインタです。
//! @param[out] pHeight 高さを格納する変数へのポインタです。
//! @param[out] pDepth 奥行きを格納する変数へのポインタです。
//! @param[out] pChannelCount チャンネル数を格納する変数へのポインタです。
//! @param[out] pFlags 各種フラグを格納する変数へのポインタです。
//! @param[in] filePath EXR ファイルのパスです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool GetExrInfo(
    int* pWidth,
    int* pHeight,
    int* pDepth,
    int* pChannelCount,
    uint32_t* pFlags,
    const wchar_t* filePath
)
{
    //-----------------------------------------------------------------------------
    // ファイルの先頭部分をリードします。
    const std::string sjisPath = RGetShiftJisFromUnicode(filePath);
    std::ifstream ifs(sjisPath.c_str(), ios_base::binary);
    if (!ifs)
    {
        SetErrorString(std::string("Can't open the file: ") + sjisPath);
        return false;
    }
    const int HEADER_BUF_SIZE = 1024;
    char* pHeader = new char[HEADER_BUF_SIZE];
    ifs.read(pHeader, HEADER_BUF_SIZE);
    const size_t readSize = static_cast<int>(ifs.gcount());
    #ifdef EXR_DEBUG_PRINT
    cerr << "Read EXR header: " << readSize << endl;
    #endif
    if (readSize > 0)
    {
        pHeader[readSize - 1] = 0x00; // 文字列取得に成功するために最後に 0 を代入します。
    }
    ifs.close();

    //-----------------------------------------------------------------------------
    // ファイルの先頭部分から情報を取得します。
    const bool success = ParseExrInfo(pWidth, pHeight, pDepth, pChannelCount, pFlags, pHeader, readSize);
    delete[] pHeader;
    return success;
}

//-----------------------------------------------------------------------------
//! @brief EXR ファイルをリードして 16-bit float のビットマップを取得します。
//!
//! @param[out] pBitmapF16 16-bit float のビットマップへのポインタです。
//!                        幅 x 高さ x 4 x 2 バイトのメモリを確保しておきます。
//! @param[in] filePath EXR ファイルのパスです。
//! @param[in] channelCount チャンネル数です。
//! @param[in] flags 各種フラグです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool ReadExrAsFloat16(
    void* pBitmapF16,
    const wchar_t* filePath,
    const int channelCount,
    const uint32_t flags
)
{
    try
    {
        const std::string sjisPath = RGetShiftJisFromUnicode(filePath);
        Imf::RgbaInputFile file(sjisPath.c_str());
        const Imath::Box2i dw = file.dataWindow();
        const int imageW = dw.max.x - dw.min.x + 1;
        const int imageH = dw.max.y - dw.min.y + 1;
        R_UNUSED_VARIABLE(imageH);
        #ifdef EXR_DEBUG_PRINT
        cerr << "Read as f16: " << imageW << " x " << imageH << endl;
        #endif
        const int baseOfs = dw.min.x + dw.min.y * imageW;
        file.setFrameBuffer(reinterpret_cast<Imf::Rgba*>(pBitmapF16) - baseOfs, 1, imageW);
            // ↑フレームバッファの先頭アドレスはウィンドウ空間の左上なので
            //   アドレスを調整します。
        file.readPixels(dw.min.y, dw.max.y);
        R_UNUSED_VARIABLE(channelCount);
        R_UNUSED_VARIABLE(flags);
    }
    catch (const std::exception& exc)
    {
        SetErrorString(exc.what());
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief メモリ上の EXR ファイルデータから 16-bit float のビットマップを取得します。
//!
//! @param[out] pBitmapF16 16-bit float のビットマップへのポインタです。
//!                        幅 x 高さ x 4 x 2 バイトのメモリを確保しておきます。
//! @param[in] pData メモリ上の EXR ファイルデータのポインタです。
//! @param[in] dataSize メモリ上の EXR ファイルデータのサイズです。
//! @param[in] channelCount チャンネル数です。
//! @param[in] flags 各種フラグです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool ParseExrAsFloat16(
    void* pBitmapF16,
    const void* pData,
    const size_t dataSize,
    const int channelCount,
    const uint32_t flags
)
{
    ExrIMemoryStream ims(pData, dataSize);
    try
    {
        Imf::RgbaInputFile file(ims);
        const Imath::Box2i dw = file.dataWindow();
        const int imageW = dw.max.x - dw.min.x + 1;
        const int imageH = dw.max.y - dw.min.y + 1;
        R_UNUSED_VARIABLE(imageH);
        #ifdef EXR_DEBUG_PRINT
        cerr << "Read as f16 (from memory): " << imageW << " x " << imageH << endl;
        #endif
        const int baseOfs = dw.min.x + dw.min.y * imageW;
        file.setFrameBuffer(reinterpret_cast<Imf::Rgba*>(pBitmapF16) - baseOfs, 1, imageW);
            // ↑フレームバッファの先頭アドレスはウィンドウ空間の左上なので
            //   アドレスを調整します。
        file.readPixels(dw.min.y, dw.max.y);
        R_UNUSED_VARIABLE(channelCount);
        R_UNUSED_VARIABLE(flags);
    }
    catch (const std::exception& exc)
    {
        SetErrorString(exc.what());
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief EXR ファイルをリードして 32-bit float のビットマップを取得します。
//!
//! @param[out] pBitmapF32 32-bit float のビットマップへのポインタです。
//!                        幅 x 高さ x 4 x 4 バイトのメモリを確保しておきます。
//! @param[in] filePath EXR ファイルのパスです。
//! @param[in] channelCount チャンネル数です。
//! @param[in] flags 各種フラグです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool ReadExrAsFloat32(
    void* pBitmapF32,
    const wchar_t* filePath,
    const int channelCount,
    const uint32_t flags
)
{
    //-----------------------------------------------------------------------------
    // EXR ファイルをメモリ上にすべてリードします。
    const std::string sjisPath = RGetShiftJisFromUnicode(filePath);
    std::ifstream ifs(sjisPath.c_str(), ios_base::binary);
    if (!ifs)
    {
        SetErrorString(std::string("Can't open the file: ") + sjisPath);
        return false;
    }
    ifs.seekg(0L, ios_base::end);
    const size_t dataSize = static_cast<size_t>(ifs.tellg());
    ifs.seekg(0L, ios_base::beg);
    uint8_t* pData = new uint8_t[dataSize];
    ifs.read(reinterpret_cast<char*>(pData), dataSize);
    ifs.close();

    //-----------------------------------------------------------------------------
    // リードしたデータからビットマップを取得します。
    try
    {
        GetAsFloat32Common(pBitmapF32, pData, dataSize, channelCount, flags, sjisPath.c_str());
    }
    catch (const std::exception& exc)
    {
        SetErrorString(exc.what());
        delete[] pData;
        return false;
    }
    delete[] pData;
    return true;
}

//-----------------------------------------------------------------------------
//! @brief メモリ上の EXR ファイルデータから 32-bit float のビットマップを取得します。
//!
//! @param[out] pBitmapF32 32-bit float のビットマップへのポインタです。
//!                        幅 x 高さ x 4 x 4 バイトのメモリを確保しておきます。
//! @param[in] pData メモリ上の EXR ファイルデータのポインタです。
//! @param[in] dataSize メモリ上の EXR ファイルデータのサイズです。
//! @param[in] channelCount チャンネル数です。
//! @param[in] flags 各種フラグです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool ParseExrAsFloat32(
    void* pBitmapF32,
    const void* pData,
    const size_t dataSize,
    const int channelCount,
    const uint32_t flags
)
{
    try
    {
        GetAsFloat32Common(pBitmapF32, pData, dataSize, channelCount, flags, NULL);
    }
    catch (const std::exception& exc)
    {
        SetErrorString(exc.what());
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 16-bit float のビットマップを EXR ファイルにライトします。
//!
//! @param[in] filePath EXR ファイルのパスです。
//! @param[in] pBitmapF16 16-bit float のビットマップへのポインタです。
//! @param[in] imageW ビットマップの幅です。
//! @param[in] imageH ビットマップの高さです。
//! @param[in] imageD ビットマップの奥行きです。
//! @param[in] flags 各種フラグです。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool WriteExrFloat16(
    const wchar_t* filePath,
    const void* pBitmapF16,
    const int imageW,
    const int imageH,
    const int imageD,
    const uint32_t flags,
    const bool hasAlpha
)
{
    try
    {
        const std::string sjisPath = RGetShiftJisFromUnicode(filePath);
        #ifdef EXR_DEBUG_PRINT
        cerr << "Output: " << sjisPath.c_str() << endl;
        #endif
        Imf::Header header(imageW, imageH);
        //header.compression() = Imf::PIZ_COMPRESSION; // default is ZIP_COMPRESSION
        const Imf::RgbaChannels rgbaChannels = (hasAlpha) ? Imf::WRITE_RGBA : Imf::WRITE_RGB;
        Imf::RgbaOutputFile file(sjisPath.c_str(), header, rgbaChannels);
        file.setFrameBuffer(reinterpret_cast<const Imf::Rgba*>(pBitmapF16), 1, imageW);
        file.writePixels(imageH);
        R_UNUSED_VARIABLE(imageD);
        R_UNUSED_VARIABLE(flags);
    }
    catch (const std::exception& exc)
    {
        SetErrorString(exc.what());
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 32-bit float のビットマップを EXR ファイルにライトします。
//!
//! @param[in] filePath EXR ファイルのパスです。
//! @param[in] pBitmapF32 32-bit float のビットマップへのポインタです。
//! @param[in] imageW ビットマップの幅です。
//! @param[in] imageH ビットマップの高さです。
//! @param[in] imageD ビットマップの奥行きです。
//! @param[in] flags 各種フラグです。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool WriteExrFloat32(
    const wchar_t* filePath,
    const void* pBitmapF32,
    const int imageW,
    const int imageH,
    const int imageD,
    const uint32_t flags,
    const bool hasAlpha
)
{
    try
    {
        const std::string sjisPath = RGetShiftJisFromUnicode(filePath);
        #ifdef EXR_DEBUG_PRINT
        cerr << "Output: " << sjisPath.c_str() << endl;
        #endif
        Imf::Header header(imageW, imageH);
        //header.compression() = Imf::PIZ_COMPRESSION; // default is ZIP_COMPRESSION
        header.channels().insert("R", Imf::Channel(Imf::FLOAT));
        header.channels().insert("G", Imf::Channel(Imf::FLOAT));
        header.channels().insert("B", Imf::Channel(Imf::FLOAT));
        if (hasAlpha)
        {
            header.channels().insert("A", Imf::Channel(Imf::FLOAT));
        }

        Imf::OutputFile file(sjisPath.c_str(), header);
        Imf::FrameBuffer fbuf;
        SetRgbaFloat32FrameBuffer(fbuf, const_cast<void*>(pBitmapF32), "", imageW);
        file.setFrameBuffer(fbuf);
        file.writePixels(imageH);
        R_UNUSED_VARIABLE(imageD);
        R_UNUSED_VARIABLE(flags);
    }
    catch (const std::exception& exc)
    {
        SetErrorString(exc.what());
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 32-bit float のビットマップを 16-bit float 形式で EXR ファイルにライトします。
//!
//! @param[in] filePath EXR ファイルのパスです。
//! @param[in] pBitmapF32 32-bit float のビットマップへのポインタです。
//! @param[in] imageW ビットマップの幅です。
//! @param[in] imageH ビットマップの高さです。
//! @param[in] imageD ビットマップの奥行きです。
//! @param[in] flags 各種フラグです。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool WriteExrFloat32AsFloat16(
    const wchar_t* filePath,
    const void* pBitmapF32,
    const int imageW,
    const int imageH,
    const int imageD,
    const uint32_t flags,
    const bool hasAlpha
)
{
    uint8_t* pBitmapF16 = NULL;
    try
    {
        //-----------------------------------------------------------------------------
        // convert to 16-bit float
        pBitmapF16 = new uint8_t[imageW * imageH * R_RGBA_COUNT * sizeof(half)];
        const float* pSrc = reinterpret_cast<const float*>(pBitmapF32);
        half* pDst = reinterpret_cast<half*>(pBitmapF16);
        const int valCount = imageW * imageH * R_RGBA_COUNT;
        for (int iVal = 0; iVal < valCount; ++iVal)
        {
            *pDst++ = static_cast<half>(*pSrc++);
        }
        WriteExrFloat16(filePath, pBitmapF16, imageW, imageH, imageD, flags, hasAlpha);
        delete[] pBitmapF16;
    }
    catch (const std::exception& exc)
    {
        if (pBitmapF16 != NULL)
        {
            delete[] pBitmapF16;
        }
        SetErrorString(exc.what());
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief エラーのユニコード文字列を返します。
//-----------------------------------------------------------------------------
DLLEXPORT const wchar_t* GetErrorString()
{
    return g_ErrorWString.c_str();
}

//-----------------------------------------------------------------------------
//! @brief エラーの Shift JIS 文字列を返します。
//-----------------------------------------------------------------------------
DLLEXPORT const char* GetErrorShiftJisString()
{
    return g_ErrorString.c_str();
}

