﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DllMain NvttConvert ConvertCore
// NvttIsGpuEncodingAvailable
// SetSurfaceData SetCompressionOptions SetBatchList

// MemoryOutputHandler

//=============================================================================
// include
//=============================================================================
#include "Process.h"

#include <nvtt.h>

#include "../Encoder/Encoder.h"

using namespace std;
using namespace nn::gfx::tool::texenc;

#define DLLEXPORT extern "C" __declspec(dllexport)

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
//! @brief メモリ出力ハンドラのクラスです。
//=============================================================================
class MemoryOutputHandler : public nvtt::OutputHandler
{
protected:
    void* m_pMemory; //!< 出力先のメモリへのポインタです。
    size_t m_Size; //!< 出力先のサイズです。
    size_t m_Offset; //!< 出力先のメモリ先頭からの現在のオフセットです。

public:
    MemoryOutputHandler(void* pMemory, const size_t size)
    : m_pMemory(pMemory),
      m_Size(size),
      m_Offset(0)
    {
    }

    virtual ~MemoryOutputHandler()
    {
    }

    void beginImage(int size, int width, int height, int depth, int face, int miplevel)
    {
        //cout << "begin image: " << size << ", " << width << " x " << height << " x " << depth << ", " << face << ", " << miplevel << endl;
        NVTTDLL_UNUSED_VARIABLE(size);
        NVTTDLL_UNUSED_VARIABLE(width);
        NVTTDLL_UNUSED_VARIABLE(height);
        NVTTDLL_UNUSED_VARIABLE(depth);
        NVTTDLL_UNUSED_VARIABLE(face);
        NVTTDLL_UNUSED_VARIABLE(miplevel);
    }

    bool writeData(const void* data, int size)
    {
        //cout << "write data: " << m_Offset + size << " / " << m_Size << endl;
        if (m_Offset + size <= m_Size)
        {
            memcpy(reinterpret_cast<uint8_t*>(m_pMemory) + m_Offset, data, size);
            m_Offset += size;
            return true;
        }
        else
        {
            return false;
        }
    }

    void endImage()
    {
        //cout << "end image" << endl;
    }
};

//-----------------------------------------------------------------------------
//! @brief オブジェクト群を解放し、ポインタ配列を空にします。
//!
//! @tparam T オブジェクトの型です。
//!
//! @param[in,out] ppObjects オブジェクトへのポインタ配列へのポインタです。
//-----------------------------------------------------------------------------
template <typename T>
void DeleteObjectArray(std::vector<T*>* ppObjects)
{
    std::vector<T*>& pObjects = *ppObjects;
    for (size_t objIdx = 0; objIdx < pObjects.size(); ++objIdx)
    {
        delete pObjects[objIdx];
    }
    pObjects.clear();
}

//-----------------------------------------------------------------------------
//! @brief ワイド文字列（Unicode）から ANSI 文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetAnsiFromUnicode(const std::wstring& src)
{
    std::string dst;
    const int ansiSize = WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, nullptr, 0, nullptr, nullptr);
    if (ansiSize != 0)
    {
        char* ansiBuf = new char[ansiSize];
        WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, ansiBuf, ansiSize, nullptr, nullptr);
        dst = std::string(ansiBuf, strlen(ansiBuf));
        delete[] ansiBuf;
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief プレーンな RGBA フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return プレーンな RGBA フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPlainRgbaFormat(const std::string& formatStr)
{
    return (formatStr == "unorm_8_8_8_8"     ||
            formatStr == "snorm_8_8_8_8"     ||
            formatStr == "srgb_8_8_8_8"      ||
            formatStr == "float_32_32_32_32");
}

//-----------------------------------------------------------------------------
//! @brief 浮動小数点数型フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 浮動小数点数型フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsFloatFormat(const std::string& formatStr)
{
    return (formatStr.find("float_" ) == 0 ||
            formatStr.find("ufloat_") == 0);
}

//-----------------------------------------------------------------------------
//! @brief snorm フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return snorm フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSnormFormat(const std::string& formatStr)
{
    return (formatStr.find("snorm_") == 0);
}

//-----------------------------------------------------------------------------
//! @brief BC1 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsBc1Format(const std::string& formatStr)
{
    return (formatStr == "unorm_bc1" || formatStr == "srgb_bc1");
}

//-----------------------------------------------------------------------------
//! @brief BC1/BC2/BC3 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsBc123Format(const std::string& formatStr)
{
    return (formatStr == "unorm_bc1" ||
            formatStr == "unorm_bc2" ||
            formatStr == "unorm_bc3" ||
            formatStr == "srgb_bc1"  ||
            formatStr == "srgb_bc2"  ||
            formatStr == "srgb_bc3");
}

//-----------------------------------------------------------------------------
//! @brief BC6/BC7 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsBc67Format(const std::string& formatStr)
{
    return (formatStr == "ufloat_bc6" ||
            formatStr == "float_bc6"  ||
            formatStr == "unorm_bc7"  ||
            formatStr == "srgb_bc7");
}

//-----------------------------------------------------------------------------
//! @brief BC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsBcFormat(const std::string& formatStr)
{
    return (formatStr.find("unorm_bc" ) == 0 ||
            formatStr.find("snorm_bc" ) == 0 ||
            formatStr.find("srgb_bc"  ) == 0 ||
            formatStr.find("ufloat_bc") == 0 ||
            formatStr.find("float_bc" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief ASTC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ASTC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsAstcFormat(const std::string& formatStr)
{
    return (formatStr.find("unorm_astc") == 0 ||
            formatStr.find("srgb_astc" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief ASTC フォーマットなら true を返します。
//!
//! @param[in] format nvtt フォーマットです。
//!
//! @return ASTC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsAstcFormat(const nvtt::Format format)
{
    switch (format)
    {
    case nvtt::Format_ASTC_LDR_4x4:
    case nvtt::Format_ASTC_LDR_5x4:
    case nvtt::Format_ASTC_LDR_5x5:
    case nvtt::Format_ASTC_LDR_6x5:
    case nvtt::Format_ASTC_LDR_6x6:
    case nvtt::Format_ASTC_LDR_8x5:
    case nvtt::Format_ASTC_LDR_8x6:
    case nvtt::Format_ASTC_LDR_8x8:
    case nvtt::Format_ASTC_LDR_10x5:
    case nvtt::Format_ASTC_LDR_10x6:
    case nvtt::Format_ASTC_LDR_10x8:
    case nvtt::Format_ASTC_LDR_10x10:
    case nvtt::Format_ASTC_LDR_12x10:
    case nvtt::Format_ASTC_LDR_12x12:
        return true;

    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief 圧縮ブロックの幅を取得します。
//!
//! @param[in] format nvtt フォーマットです。
//!
//! @return 圧縮ブロックの幅を返します。
//-----------------------------------------------------------------------------
int GetBlockWidth(const nvtt::Format format)
{
    switch (format)
    {
    case nvtt::Format_RGBA:
        return 1;

    case nvtt::Format_ASTC_LDR_5x4:
    case nvtt::Format_ASTC_LDR_5x5:
        return 5;

    case nvtt::Format_ASTC_LDR_6x5:
    case nvtt::Format_ASTC_LDR_6x6:
        return 6;

    case nvtt::Format_ASTC_LDR_8x5:
    case nvtt::Format_ASTC_LDR_8x6:
    case nvtt::Format_ASTC_LDR_8x8:
        return 8;

    case nvtt::Format_ASTC_LDR_10x5:
    case nvtt::Format_ASTC_LDR_10x6:
    case nvtt::Format_ASTC_LDR_10x8:
    case nvtt::Format_ASTC_LDR_10x10:
        return 10;

    case nvtt::Format_ASTC_LDR_12x10:
    case nvtt::Format_ASTC_LDR_12x12:
        return 12;

    default:
        return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief 圧縮ブロックの高さを取得します。
//!
//! @param[in] format nvtt フォーマットです。
//!
//! @return 圧縮ブロックの高さを返します。
//-----------------------------------------------------------------------------
int GetBlockHeight(const nvtt::Format format)
{
    switch (format)
    {
    case nvtt::Format_RGBA:
        return 1;

    case nvtt::Format_ASTC_LDR_5x5:
    case nvtt::Format_ASTC_LDR_6x5:
    case nvtt::Format_ASTC_LDR_8x5:
    case nvtt::Format_ASTC_LDR_10x5:
        return 5;

    case nvtt::Format_ASTC_LDR_6x6:
    case nvtt::Format_ASTC_LDR_8x6:
    case nvtt::Format_ASTC_LDR_10x6:
        return 6;

    case nvtt::Format_ASTC_LDR_8x8:
    case nvtt::Format_ASTC_LDR_10x8:
        return 8;

    case nvtt::Format_ASTC_LDR_10x10:
    case nvtt::Format_ASTC_LDR_12x10:
        return 10;

    case nvtt::Format_ASTC_LDR_12x12:
        return 12;

    default:
        return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャのデータサイズ（バイト数）を取得します。
//!
//! @param[out] pLevelOffsets 各ミップマップデータのイメージデータ先頭からのオフセット配列を格納します。
//!                           nullptr なら格納しません。
//! @param[in] format nvtt フォーマットです。
//! @param[in] dimension 次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipCount ミップマップのレベル数です。
//!
//! @return テクスチャのデータサイズを返します。
//-----------------------------------------------------------------------------
size_t GetTextureDataSize(
    std::vector<size_t>* pLevelOffsets,
    const nvtt::Format format,
    const int dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount
)
{
    if (pLevelOffsets != nullptr)
    {
        pLevelOffsets->clear();
    }
    const int blockW = GetBlockWidth(format);
    const int blockH = GetBlockHeight(format);
    const bool isCompressed = (blockW != 1 || blockH != 1);
    size_t unitBytes;
    if (!isCompressed)
    {
        unitBytes = 4 * sizeof(uint8_t);
    }
    else if (format == nvtt::Format_BC1  ||
             format == nvtt::Format_BC1a ||
             format == nvtt::Format_BC4  ||
             format == nvtt::Format_BC4S)
    {
        unitBytes = 8;
    }
    else
    {
        unitBytes = 16;
    }

    size_t dataSize = 0;
    for (int level = 0; level < mipCount; ++level)
    {
        if (pLevelOffsets != nullptr)
        {
            pLevelOffsets->push_back(dataSize);
        }
        const int levelW = std::max(imageW >> level, 1);
        const int levelH = std::max(imageH >> level, 1);
        const int levelD = (dimension == ImageDimension_3d) ?
            std::max(imageD >> level, 1) : imageD;
        int unitW = levelW;
        int unitH = levelH;
        int unitD = levelD;
        if (isCompressed)
        {
            unitW = (levelW + blockW - 1) / blockW;
            unitH = (levelH + blockH - 1) / blockH;
        }
        const size_t levelSize = unitBytes * unitW * unitH * unitD;
        dataSize += levelSize;
    }
    return dataSize;
}

//-----------------------------------------------------------------------------
//! @brief エンコード品質文字列からエンコードレベルを取得します。
//!
//! @param[in] qualityStr エンコード品質文字列です。
//!
//! @return エンコードレベルを返します。
//-----------------------------------------------------------------------------
int GetQualityLevel(const std::string& qualityStr)
{
    // エンコード品質文字列の先頭が数値なら、それをエンコードレベルとします。
    int digitCount = 0;
    for (size_t charIdx = 0; charIdx < qualityStr.size(); ++charIdx)
    {
        const char c = qualityStr[charIdx];
        if ('0' <= c && c <= '9')
        {
            ++digitCount;
        }
        else
        {
            break;
        }
    }
    return (digitCount > 0) ? atoi(qualityStr.substr(0, digitCount).c_str()) : 1;
}

//-----------------------------------------------------------------------------
//! @brief サーフェスデータを設定します。
//!
//! @param[in,out] pSurface サーフェスへのポインタです。
//! @param[in] pPixels ピクセルデータです。
//!                    1 ピクセルあたり 4 または 16 バイトで、RGBA の順に格納します。画像の左上が先頭です。
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool SetSurfaceData(
    nvtt::Surface* pSurface,
    const void* pPixels,
    const std::string& formatStr,
    const int imageW,
    const int imageH,
    const int imageD
)
{
    //-----------------------------------------------------------------------------
    // サーフェスにピクセルデータを設定します。
    if (!IsPlainRgbaFormat(formatStr))
    {
        return false;
    }
    const bool isFloat = IsFloatFormat(formatStr);
    const bool isSignedNorm = IsSnormFormat(formatStr);

    nvtt::Surface& surface = *pSurface;
    const nvtt::InputFormat inputFormat =
        (isFloat     ) ? nvtt::InputFormat_RGBA_32F :
        (isSignedNorm) ? nvtt::InputFormat_BGRA_8SB :
        nvtt::InputFormat_BGRA_8UB; // 整数型は BGRA の順なので注意
    if (!surface.setImage(inputFormat, imageW, imageH, imageD, pPixels))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // 元のピクセルデータが整数型なら R 成分と B 成分を交換します。
    // 同時に A 成分がすべて 1.0 かチェックします。
    const size_t pixelCount = surface.width() * surface.height() * surface.depth();
    float* pR = surface.channel(0);
    float* pG = surface.channel(1);
    float* pB = surface.channel(2);
    float* pA = surface.channel(3);
    bool hasAlpha = false;
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        if (!isFloat)
        {
            const float r = *pR;
            *pR = *pB;
            *pB = r;
        }
        if (*pA != 1.0f)
        {
            hasAlpha = true;
        }
        ++pR;
        ++pG;
        ++pB;
        ++pA;
    }
    surface.setAlphaMode((hasAlpha) ? nvtt::AlphaMode_Transparency : nvtt::AlphaMode_None);

    return true;
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのアルファを 2 値化（0 or 1）します。
//!
//! @param[in,out] pSurface サーフェスへのポインタです。
//! @param[in] alphaThreshold アルファがこの値以上なら 1、この値未満なら 0 にします。
//-----------------------------------------------------------------------------
void BinalizeAlpha(nvtt::Surface* pSurface, const float alphaThreshold)
{
    nvtt::Surface& surface = *pSurface;
    if (surface.alphaMode() != nvtt::AlphaMode_None)
    {
        const size_t pixelCount = surface.width() * surface.height() * surface.depth();
        float* pA = surface.channel(3);
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            *pA = (*pA >= alphaThreshold) ? 1.0f : 0.0f;
            ++pA;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief サーフェスを LA 形式の法線マップに変換します。
//!        R 成分が X 成分、G 成分が Y 成分となっているサーフェスを、
//!        R / G / B 成分がすべて X 成分、A 成分が Y 成分となるように変換します。
//!
//! @param[in,out] pSurface サーフェスへのポインタです。
//-----------------------------------------------------------------------------
void ConvertToNormalMapLa(nvtt::Surface* pSurface)
{
    nvtt::Surface& surface = *pSurface;
    const size_t pixelCount = surface.width() * surface.height() * surface.depth();
    float* pR = surface.channel(0);
    float* pG = surface.channel(1);
    float* pB = surface.channel(2);
    float* pA = surface.channel(3);
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const float g = *pG;
        *pG = *pB = *pR;
        *pA = g;
        ++pR;
        ++pG;
        ++pB;
        ++pA;
    }
    surface.setAlphaMode(nvtt::AlphaMode_Transparency);
}

//-----------------------------------------------------------------------------
//! @brief nvtt に渡すフォーマットを取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return フォーマットを返します。
//!         対応するフォーマットが存在しなければ Format_Count を返します。
//-----------------------------------------------------------------------------
nvtt::Format GetNvttFormat(const std::string& formatStr)
{
    if (IsBcFormat(formatStr))
    {
        return
            (formatStr.find("bc1") != std::string::npos) ? nvtt::Format_BC1a :
            (formatStr.find("bc2") != std::string::npos) ? nvtt::Format_BC2  :
            (formatStr.find("bc3") != std::string::npos) ? nvtt::Format_BC3  :
            (formatStr == "unorm_bc4"                  ) ? nvtt::Format_BC4  :
            (formatStr == "snorm_bc4"                  ) ? nvtt::Format_BC4S :
            (formatStr == "unorm_bc5"                  ) ? nvtt::Format_BC5  :
            (formatStr == "snorm_bc5"                  ) ? nvtt::Format_BC5S :
            (formatStr.find("bc6") != std::string::npos) ? nvtt::Format_BC6  :
            (formatStr.find("bc7") != std::string::npos) ? nvtt::Format_BC7  :
            nvtt::Format_Count;
    }
    else if (IsAstcFormat(formatStr))
    {
        return
            (formatStr.find("4x4"  ) != std::string::npos) ? nvtt::Format_ASTC_LDR_4x4   :
            (formatStr.find("5x4"  ) != std::string::npos) ? nvtt::Format_ASTC_LDR_5x4   :
            (formatStr.find("5x5"  ) != std::string::npos) ? nvtt::Format_ASTC_LDR_5x5   :
            (formatStr.find("6x5"  ) != std::string::npos) ? nvtt::Format_ASTC_LDR_6x5   :
            (formatStr.find("6x6"  ) != std::string::npos) ? nvtt::Format_ASTC_LDR_6x6   :
            (formatStr.find("8x5"  ) != std::string::npos) ? nvtt::Format_ASTC_LDR_8x5   :
            (formatStr.find("8x6"  ) != std::string::npos) ? nvtt::Format_ASTC_LDR_8x6   :
            (formatStr.find("8x8"  ) != std::string::npos) ? nvtt::Format_ASTC_LDR_8x8   :
            (formatStr.find("10x5" ) != std::string::npos) ? nvtt::Format_ASTC_LDR_10x5  :
            (formatStr.find("10x6" ) != std::string::npos) ? nvtt::Format_ASTC_LDR_10x6  :
            (formatStr.find("10x8" ) != std::string::npos) ? nvtt::Format_ASTC_LDR_10x8  :
            (formatStr.find("10x10") != std::string::npos) ? nvtt::Format_ASTC_LDR_10x10 :
            (formatStr.find("12x10") != std::string::npos) ? nvtt::Format_ASTC_LDR_12x10 :
            (formatStr.find("12x12") != std::string::npos) ? nvtt::Format_ASTC_LDR_12x12 :
            nvtt::Format_Count;
    }
    else
    {
        return nvtt::Format_Count;
    }
}

//-----------------------------------------------------------------------------
//! @brief nvtt に渡すエンコード品質を取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//! @param[in] qualityStr エンコード品質文字列です。
//!
//! @return エンコード品質を返します。
//-----------------------------------------------------------------------------
nvtt::Quality GetNvttQuality(
    const std::string& formatStr,
    const std::string& qualityStr
)
{
    if (qualityStr.find("fastest") == 0)
    {
        return nvtt::Quality_Fastest;
    }
    else if (qualityStr.find("normal") == 0)
    {
        return nvtt::Quality_Normal;
    }
    else if (qualityStr.find("production") == 0)
    {
        return nvtt::Quality_Production;
    }
    else if (qualityStr.find("highest") == 0)
    {
        return nvtt::Quality_Highest;
    }
    else
    {
        const int qualityLevel = GetQualityLevel(qualityStr) +
            (IsAstcFormat(formatStr) ? 1 : 0); // astc フォーマットはデフォルトを Production にします。
        return
            (qualityLevel == 0) ? nvtt::Quality_Fastest    :
            (qualityLevel == 1) ? nvtt::Quality_Normal     :
            (qualityLevel == 2) ? nvtt::Quality_Production :
            (qualityLevel >= 3) ? nvtt::Quality_Highest    :
            nvtt::Quality_Normal;
    }
}

//-----------------------------------------------------------------------------
//! @brief 圧縮オプションを設定します。
//!
//! @param[in,out] pCompOpts 圧縮オプションへのポインタです。
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//! @param[in] format nvtt フォーマットです。
//! @param[in] qualityStr エンコード品質文字列です。
//! @param[in] encodeFlag エンコードフラグです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool SetCompressionOptions(
    nvtt::CompressionOptions* pCompOpts,
    const std::string& formatStr,
    const nvtt::Format format,
    const std::string& qualityStr,
    const int encodeFlag
)
{
    //-----------------------------------------------------------------------------
    // フォーマットを設定します。
    nvtt::CompressionOptions& compOpts = *pCompOpts;
    compOpts.setFormat(format);

    //-----------------------------------------------------------------------------
    // ピクセルタイプを設定します。
    const nvtt::PixelType pixelType = // 現在 BC6 のエンコードにのみ影響（他の圧縮フォーマットでは無視される）
        (IsSnormFormat(formatStr) ) ? nvtt::PixelType_SignedNorm    :
        (formatStr == "ufloat_bc6") ? nvtt::PixelType_UnsignedFloat :
        (formatStr == "float_bc6" ) ? nvtt::PixelType_Float         :
        nvtt::PixelType_UnsignedNorm;
    compOpts.setPixelType(pixelType);

    //-----------------------------------------------------------------------------
    // エンコード品質を設定します。
    const nvtt::Quality quality = GetNvttQuality(formatStr, qualityStr);
    compOpts.setQuality(quality);

    //-----------------------------------------------------------------------------
    // RGBA の誤差に対する重み付けを設定します。
    if (IsBc123Format(formatStr) &&
        qualityStr.find("perceptual") != std::string::npos)
    {
        // bc1 / bc2 / bc3 フォーマットは Normal 以上の品質なら重み付けが反映されます。
        compOpts.setColorWeights(0.2126f, 0.7152f, 0.0722f); // NTSC luma
    }
    else if (IsAstcFormat(format)                       &&
             format != nvtt::Format_ASTC_LDR_4x4        &&
             (encodeFlag & EncodeFlag_NormalMapLa) == 0)
    {
        // 4x4 以外の astc フォーマットでは A 成分の重み付けを大きくします。
        // 4x4 では A 成分が 0 のピクセルを維持する特別処理が有効なので、重み付けは 1 にします。
        float weightA = 2.0f;
        const std::string customWeightAStr =
            GetEnvVariable("NINTENDO_TEXTURE_CONVERTER_NVTT_ASTC_WEIGHT_A"); // 環境変数は非公開機能
        if (!customWeightAStr.empty())
        {
            const float customWeightA = strtof(customWeightAStr.c_str(), nullptr);
            if (customWeightA > 0.0f)
            {
                weightA = customWeightA;
            }
        }
        compOpts.setColorWeights(1.0f, 1.0f, 1.0f, weightA);
    }
    else
    {
        compOpts.setColorWeights(1.0f, 1.0f, 1.0f);
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief バッチリストを設定します。
//!
//! @param[out] pList バッチリストへのポインタです。
//! @param[out] ppHandlers メモリ出力ハンドラへのポインタ配列へのポインタです。
//! @param[out] ppOutOptss 出力オプションへのポインタ配列へのポインタです。
//! @param[in] pDstData 変換後のデータへのポインタです。
//! @param[in] pSurfaces サーフェスへのポインター配列です。
//! @param[in] format nvtt フォーマットです。
//! @param[in] dimension 次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipCount ミップマップのレベル数です。
//-----------------------------------------------------------------------------
void SetBatchList(
    nvtt::BatchList* pList,
    std::vector<MemoryOutputHandler*>* ppHandlers,
    std::vector<nvtt::OutputOptions*>* ppOutOptss,
    void* pDstData,
    const std::vector<nvtt::Surface*>& pSurfaces,
    const nvtt::Format format,
    const int dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount
)
{
    std::vector<size_t> levelOfss;
    const size_t encodedSize = GetTextureDataSize(&levelOfss, format, dimension,
        imageW, imageH, imageD, mipCount);

    pList->Clear();
    std::vector<MemoryOutputHandler*>& pHandlers = *ppHandlers;
    std::vector<nvtt::OutputOptions*>& pOutOptss = *ppOutOptss;
    uint8_t* pLevelDst = reinterpret_cast<uint8_t*>(pDstData);
    for (int level = 0; level < mipCount; ++level)
    {
        const size_t nextOfs = (level < mipCount - 1) ? levelOfss[level + 1] : encodedSize;
        const size_t levelSize = nextOfs - levelOfss[level];
        MemoryOutputHandler* pHandler = new MemoryOutputHandler(pLevelDst, levelSize);
        nvtt::OutputOptions* pOutOpts = new nvtt::OutputOptions();
        pOutOpts->setOutputHandler(pHandler);
        pList->Append(pSurfaces[level], 0, 0, pOutOpts);
        pHandlers.push_back(pHandler);
        pOutOptss.push_back(pOutOpts);
        pLevelDst += levelSize;
    }
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 開いているメモ帳のウィンドウにメッセージを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void NoteTrace(const char* format, ...)
{
    HWND hWnd = FindWindow("Notepad", nullptr);
    if (hWnd != nullptr)
    {
        static const POINT pt = { 0, 0 };
        hWnd = ChildWindowFromPoint(hWnd, pt);
        if (hWnd != nullptr)
        {
            char strBuf[1024];
            va_list args;
            va_start(args, format);
            const int size = vsprintf_s(strBuf, format, args);
            va_end(args);
            SendMessage(hWnd, EM_REPLACESEL, 0, reinterpret_cast<LPARAM>(strBuf));
            SendMessage(hWnd, EM_REPLACESEL, 0, reinterpret_cast<LPARAM>("\r\n"));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//-----------------------------------------------------------------------------
std::string GetEnvVariable(const char* name)
{
    std::string ret;
    char* pValue;
    size_t length;
    if (_dupenv_s(&pValue, &length, name) == 0)
    {
        // 環境変数が見つからない場合は、pValue が nullptr に
        // length が 0、戻り値が 0（成功）となります。
        if (pValue != nullptr)
        {
            ret = pValue;
            free(pValue);
        }
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief Windows のシステムメッセージ文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetWindowsSystemMessage(const int messageId)
{
    void* pMessage = nullptr;
    FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS, nullptr, messageId,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), reinterpret_cast<LPSTR>(&pMessage), 0, nullptr);
    std::string message;
    if (pMessage != nullptr)
    {
        message = reinterpret_cast<const char*>(pMessage);
        LocalFree(pMessage);
        const size_t spaceIdx = message.find_last_not_of(" \t\r\n"); // 末尾の空白と改行を削除します。
        message = (spaceIdx != std::string::npos) ? message.substr(0, spaceIdx + 1) : "";
    }
    else
    {
        char strBuf[32];
        sprintf_s(strBuf, "0x%08X", messageId);
        message = strBuf;
    }
    return message;
}

//-----------------------------------------------------------------------------
//! @brief Windows の最後のエラーのメッセージ文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetWindowsLastErrorMessage()
{
    return GetWindowsSystemMessage(GetLastError());
}

//-----------------------------------------------------------------------------
//! @brief 構造化例外の変換関数です。
//-----------------------------------------------------------------------------
void TranslateSe(unsigned int code, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; // 標準 C++ の例外を発生させます。
    NVTTDLL_UNUSED_VARIABLE(code);
}

//-----------------------------------------------------------------------------
//! @brief DLL のメイン関数です。
//-----------------------------------------------------------------------------
BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
    static char s_ExePath[NvttDllMaxPath] = { 0 }; // DLL を使用している EXE ファイルのパスです。

    switch (fdwReason)
    {
        case DLL_PROCESS_ATTACH:
        {
            GetModuleFileNameA(nullptr, s_ExePath, sizeof(s_ExePath));
            //NoteTrace("process attach: %s", s_ExePath);
            _set_se_translator(TranslateSe); // このスレッドでのみ有効です。
            break;
        }

        case DLL_THREAD_ATTACH:
            break;

        case DLL_THREAD_DETACH:
            break;

        case DLL_PROCESS_DETACH:
            //NoteTrace("process detach: %s", s_ExePath);
            break;

        default:
            break;
    }
    NVTTDLL_UNUSED_VARIABLE(hinstDLL);
    NVTTDLL_UNUSED_VARIABLE(lpvReserved);
    return TRUE;
}

//-----------------------------------------------------------------------------
//! @brief 現在の DLL ファイルのパスを取得します。
//-----------------------------------------------------------------------------
std::string GetCurrentDllPath()
{
    char path[NvttDllMaxPath];
    HMODULE hDll = nullptr;
    GetModuleHandleExA(GET_MODULE_HANDLE_EX_FLAG_FROM_ADDRESS |
        GET_MODULE_HANDLE_EX_FLAG_UNCHANGED_REFCOUNT,
        reinterpret_cast<LPCSTR>(&DllMain), &hDll);
    GetModuleFileNameA(hDll, path, static_cast<DWORD>(sizeof(path)));
        // バッファ長が足りない場合、返り値はコピーしたサイズですが、
        // GetLastError() は 122 (ERROR_INSUFFICIENT_BUFFER) になります。
    return std::string(path);
}

//-----------------------------------------------------------------------------
//! @brief nvtt で画像のフォーマットを変換する処理のコア関数です。
//-----------------------------------------------------------------------------
bool ConvertCore(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const std::string& srcFormatStr,
    const std::string& qualityStr,
    const int encodeFlag,
    const int dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount
)
{
    //-----------------------------------------------------------------------------
    // フォーマットをチェックします。
    const nvtt::Format nvttFormat = GetNvttFormat(dstFormatStr);
    if (nvttFormat == nvtt::Format_Count)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // コンテキストを初期化します。
    bool usesCuda = (encodeFlag & EncodeFlag_GpuEncoding) != 0;
    NvttDllTimeMeasure tm;
    nvtt::Context context(usesCuda);
    //cerr << "nvtt::Context(): " << tm.GetMilliSecond() << "ms" << endl;
    //NoteTrace("nvtt::Context(): %fms", tm.GetMilliSecond());
    if (usesCuda && !context.isCudaSupported())
    {
        if ((encodeFlag & EncodeFlag_GpuAuto) == 0 &&
            GetEnvVariable("NINTENDO_TEXTURE_CONVERTER_GPU_ENCODING_AUTO") != "1") // 環境変数は非公開機能
        {
            return false; // GPU エンコーディングが不可能な場合にエラーにするならこの行を有効にします。
        }
        usesCuda = false;
    }
    context.enableCudaAcceleration(usesCuda);

    //-----------------------------------------------------------------------------
    // 変換オプションを設定します。
    nvtt::CompressionOptions compOpts;
    if (!SetCompressionOptions(&compOpts, dstFormatStr, nvttFormat, qualityStr, encodeFlag))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // 変換元のサーフェス群を設定します。
    const bool binalizeAlpha = IsBc1Format(dstFormatStr);
    const bool isNormalMapLa = (IsAstcFormat(dstFormatStr) &&
        (encodeFlag & EncodeFlag_NormalMapLa) != 0);
    const size_t srcPixelBytes = (IsFloatFormat(srcFormatStr)) ?
        4 * sizeof(float) : 4 * sizeof(uint8_t);
    const uint8_t* pLevelSrc = reinterpret_cast<const uint8_t*>(pSrc);

    std::vector<nvtt::Surface*> pSurfaces;
    for (int level = 0; level < mipCount; ++level)
    {
        // サーフェスデータを設定します。
        nvtt::Surface* pSurface = new nvtt::Surface();
        pSurfaces.push_back(pSurface);
        const int levelW = std::max(imageW >> level, 1);
        const int levelH = std::max(imageH >> level, 1);
        const int levelD = (dimension == ImageDimension_3d) ?
            std::max(imageD >> level, 1) : imageD;
        if (!SetSurfaceData(pSurface, pLevelSrc, srcFormatStr, levelW, levelH, levelD))
        {
            DeleteObjectArray(&pSurfaces);
            return false;
        }

        // 必要ならサーフェスのピクセルデータを加工します。
        if (binalizeAlpha)
        {
            BinalizeAlpha(pSurface, 0.5f);
        }
        if (isNormalMapLa)
        {
            ConvertToNormalMapLa(pSurface);
        }
        pLevelSrc += srcPixelBytes * levelW * levelH * levelD;
    }

    //-----------------------------------------------------------------------------
    // エンコードします。
    bool isSucceeded = false;
    if (mipCount == 1)
    {
        const nvtt::Surface& surface = *pSurfaces[0];
        const size_t encodedSize = context.estimateSize(surface, mipCount, compOpts);
        MemoryOutputHandler handler(pDst, encodedSize);
        nvtt::OutputOptions outOpts;
        outOpts.setOutputHandler(&handler);

        const int faceIdx = 0;
        const int mipLevel = 0;
        isSucceeded = context.compress(surface, faceIdx, mipLevel, compOpts, outOpts);
    }
    else
    {
        nvtt::BatchList list;
        std::vector<MemoryOutputHandler*> pHandlers;
        std::vector<nvtt::OutputOptions*> pOutOptss;
        SetBatchList(&list, &pHandlers, &pOutOptss, pDst, pSurfaces, nvttFormat,
            dimension, imageW, imageH, imageD, mipCount);

        isSucceeded = context.compress(list, compOpts);
        DeleteObjectArray(&pHandlers);
        DeleteObjectArray(&pOutOptss);
    }
    DeleteObjectArray(&pSurfaces);
    return isSucceeded;
}

//-----------------------------------------------------------------------------
//! @brief nvtt で画像のフォーマットを変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormat 変換後のフォーマット文字列です。
//! @param[in] srcFormat 変換前のフォーマット文字列です。
//! @param[in] quality エンコード品質文字列です。
//! @param[in] encodeFlag エンコードフラグです。
//! @param[in] dimension 次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipCount ミップマップのレベル数です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool NvttConvert(
    void* pDst,
    const void* pSrc,
    const wchar_t* dstFormat,
    const wchar_t* srcFormat,
    const wchar_t* quality,
    const int encodeFlag,
    const int dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount
)
{
    //-----------------------------------------------------------------------------
    // 引数をチェックします。
    if (pDst      == nullptr ||
        pSrc      == nullptr ||
        dstFormat == nullptr ||
        srcFormat == nullptr ||
        quality   == nullptr)
    {
        return false;
    }
    const std::string dstFormatStr = GetAnsiFromUnicode(dstFormat);
    const std::string srcFormatStr = GetAnsiFromUnicode(srcFormat);
    const std::string qualityStr   = GetAnsiFromUnicode(quality);

    //-----------------------------------------------------------------------------
    // 変換します。
    if ((encodeFlag & EncodeFlag_ServerProcess) == 0)
    {
        //-----------------------------------------------------------------------------
        // 現在のプロセスで変換します。
        return ConvertCore(pDst, pSrc, dstFormatStr, srcFormatStr, qualityStr,
            encodeFlag, dimension, imageW, imageH, imageD, mipCount);
    }
    else
    {
        //-----------------------------------------------------------------------------
        // サーバープロセスで変換します。
        const nvtt::Format nvttFormat = GetNvttFormat(dstFormatStr);
        if (nvttFormat == nvtt::Format_Count)
        {
            return false;
        }
        const size_t dstDataSize = GetTextureDataSize(nullptr, nvttFormat,
            dimension, imageW, imageH, imageD, mipCount);
        const size_t srcDataSize = GetTextureDataSize(nullptr, nvtt::Format_RGBA,
            dimension, imageW, imageH, imageD, mipCount) *
            ((IsFloatFormat(srcFormatStr)) ? sizeof(float) : sizeof(uint8_t));
        return ConvertByServer(pDst, pSrc, dstDataSize, srcDataSize,
            dstFormatStr, srcFormatStr, qualityStr,
            encodeFlag, dimension, imageW, imageH, imageD, mipCount);
    }
}

//-----------------------------------------------------------------------------
//! @brief nvtt で GPU によるエンコーディングが可能なら true を返します。
//!
//! @return GPU によるエンコーディングが可能なら true を返します。
//-----------------------------------------------------------------------------
DLLEXPORT bool NvttIsGpuEncodingAvailable()
{
    nvtt::Context context(false);
    return context.isCudaSupported();
}

