﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// CheckRgbDitherIsNeeded

//=============================================================================
// include
//=============================================================================
#include "Common.h"

using namespace std;

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//-----------------------------------------------------------------------------
// float_16 変換用テーブルです。
//-----------------------------------------------------------------------------
const uint16_t Float16ELut[1 << 9] =
{
    // 以下、OpenEXR の eLut.h からコピーしたデータです。

        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,  1024,  2048,  3072,  4096,  5120,  6144,  7168,
     8192,  9216, 10240, 11264, 12288, 13312, 14336, 15360,
    16384, 17408, 18432, 19456, 20480, 21504, 22528, 23552,
    24576, 25600, 26624, 27648, 28672, 29696,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0, 33792, 34816, 35840, 36864, 37888, 38912, 39936,
    40960, 41984, 43008, 44032, 45056, 46080, 47104, 48128,
    49152, 50176, 51200, 52224, 53248, 54272, 55296, 56320,
    57344, 58368, 59392, 60416, 61440, 62464,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
};

//-----------------------------------------------------------------------------
//! @brief float 型を int 型としてリードした値を float_16 の値に変換します。
//-----------------------------------------------------------------------------
uint16_t ConvertToFloat16Value(const int i)
{
    // 以下、OpenEXR の half.cpp の convert() からコピーしたコードです。

    int s =  (i >> 16) & 0x00008000;
    int e = ((i >> 23) & 0x000000ff) - (127 - 15);
    int m =   i        & 0x007fffff;
    if (e <= 0)
    {
        if (e < -10)
        {
            return static_cast<uint16_t>(s);
        }

        m = m | 0x00800000;

        int t = 14 - e;
        int a = (1 << (t - 1)) - 1;
        int b = (m >> t) & 1;

        m = (m + a + b) >> t;
        return static_cast<uint16_t>(s | m);
    }
    else if (e == 0xff - (127 - 15))
    {
        if (m == 0)
        {
            return static_cast<uint16_t>(s | 0x7c00);
        }
        else
        {
            m >>= 13;
            return static_cast<uint16_t>(s | 0x7c00 | m | (m == 0));
        }
    }
    else
    {
        m = m + 0x00000fff + ((m >> 13) & 1);

        if (m & 0x00800000)
        {
            m =  0;
            e += 1;
        }

        if (e > 30)
        {
            // overflow
            return static_cast<uint16_t>(s | 0x7c00);
        }

        return static_cast<uint16_t>(s | (e << 10) | (m >> 13));
    }
}

//-----------------------------------------------------------------------------
//! @brief ゆるやかに RGB 値が変化するピクセルの割合（%）を計算します。
//!
//! @param[out] pSharpPercent 急激に RGB 値が変化するピクセルの割合（%）を格納します。
//!                           nullptr なら格納しません。
//! @param[out] pFlatPercent RGB 値が変化しないピクセルの割合（%）を格納します。
//!                          nullptr なら格納しません。
//! @param[in] pPixels ピクセルデータです。
//!                    1 ピクセルあたり 4 バイトで、RGBA の順に格納します。画像の左上が先頭です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] smoothDiff ゆるやかな変化とみなす RGB 値の差分の最大値です。
//! @param[in] sharpDiff 急激な変化とみなす RGB 値の差分の最小値です。
//!
//! @return ゆるやかに RGB 値が変化するピクセルの割合（%）を返します。
//-----------------------------------------------------------------------------
int CalculateSmoothRgbPercent(
    int* pSharpPercent,
    int* pFlatPercent,
    const void* pPixels,
    const int imageW,
    const int imageH,
    const int smoothDiff,
    const int sharpDiff
)
{
    size_t flatCount = 0;
    size_t smoothCount = 0;
    size_t sharpCount = 0;
    const size_t pixelBytes = EncRgbaBytes;
    const size_t lineBytes = pixelBytes * imageW;
    const uint8_t* pPixC = reinterpret_cast<const uint8_t*>(pPixels);
    for (int y = 0; y < imageH; ++y)
    {
        const uint8_t* pPixU = (y == 0         ) ? pPixC : pPixC - lineBytes;
        const uint8_t* pPixD = (y == imageH - 1) ? pPixC : pPixC + lineBytes;
        for (int x = 0; x < imageW; ++x)
        {
            const int ofsL = (x == 0         ) ? 0 : -EncRgbaCount;
            const int ofsR = (x == imageW - 1) ? 0 :  EncRgbaCount;
            int maxDiff = 0;
            for (int c = 0; c < EncRgbCount; ++c) // RGB 成分のみ計算し、A 成分はスキップします。
            {
                const int value = pPixC[c];
                const int diff0 = std::abs(pPixU[ofsL + c] - value);
                const int diff1 = std::abs(pPixU[       c] - value);
                const int diff2 = std::abs(pPixU[ofsR + c] - value);
                const int diff3 = std::abs(pPixC[ofsL + c] - value);
                const int diff4 = std::abs(pPixC[ofsR + c] - value);
                const int diff5 = std::abs(pPixD[ofsL + c] - value);
                const int diff6 = std::abs(pPixD[       c] - value);
                const int diff7 = std::abs(pPixD[ofsR + c] - value);
                if (diff0 > maxDiff)
                {
                    maxDiff = diff0;
                }
                if (diff1 > maxDiff)
                {
                    maxDiff = diff1;
                }
                if (diff2 > maxDiff)
                {
                    maxDiff = diff2;
                }
                if (diff3 > maxDiff)
                {
                    maxDiff = diff3;
                }
                if (diff4 > maxDiff)
                {
                    maxDiff = diff4;
                }
                if (diff5 > maxDiff)
                {
                    maxDiff = diff5;
                }
                if (diff6 > maxDiff)
                {
                    maxDiff = diff6;
                }
                if (diff7 > maxDiff)
                {
                    maxDiff = diff7;
                }
            }
            if (maxDiff == 0)
            {
                ++flatCount;
            }
            else if (maxDiff <= smoothDiff)
            {
                ++smoothCount;
            }
            else if (maxDiff >= sharpDiff)
            {
                ++sharpCount;
            }
            pPixU += pixelBytes;
            pPixC += pixelBytes;
            pPixD += pixelBytes;
        }
    }
    const size_t pixelCount = static_cast<size_t>(imageW) * imageH;
    if (pSharpPercent != nullptr)
    {
        const double sharpRatio = static_cast<double>(sharpCount) / pixelCount;
        *pSharpPercent = static_cast<int>(sharpRatio * 100.0 + 0.5);
    }
    if (pFlatPercent != nullptr)
    {
        const double flatRatio = static_cast<double>(flatCount) / pixelCount;
        *pFlatPercent = static_cast<int>(flatRatio * 100.0 + 0.5);
    }
    const double smoothRatio = static_cast<double>(smoothCount) / pixelCount;
    return static_cast<int>(smoothRatio * 100.0 + 0.5);
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 開いているメモ帳のウィンドウにメッセージを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void NoteTrace(const char* format, ...)
{
    HWND hWnd = FindWindow("Notepad", nullptr);
    if (hWnd != nullptr)
    {
        static const POINT pt = { 0, 0 };
        hWnd = ChildWindowFromPoint(hWnd, pt);
        if (hWnd != nullptr)
        {
            char strBuf[1024];
            va_list args;
            va_start(args, format);
            const int size = vsprintf_s(strBuf, format, args);
            va_end(args);
            SendMessage(hWnd, EM_REPLACESEL, 0, reinterpret_cast<LPARAM>(strBuf));
            SendMessage(hWnd, EM_REPLACESEL, 0, reinterpret_cast<LPARAM>("\r\n"));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief プロセッサ数を取得します。
//-----------------------------------------------------------------------------
int GetProcessorCount()
{
    SYSTEM_INFO sysinfo;
    GetSystemInfo(&sysinfo);
    return sysinfo.dwNumberOfProcessors;
}

//-----------------------------------------------------------------------------
//! @brief 指定した文字が Shift JIS の 2 バイト文字の 1 バイト目なら true を返します。
//-----------------------------------------------------------------------------
bool IsShiftJisFirstByte(const uint8_t val)
{
    return ((0x81 <= val && val <= 0x9f) ||
            (0xe0 <= val && val <= 0xef));
}

//-----------------------------------------------------------------------------
//! @brief ANSI 文字列からワイド文字列（Unicode）を取得します。
//-----------------------------------------------------------------------------
std::wstring GetUnicodeFromAnsi(const std::string& src)
{
    const int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::wstring();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);
    std::wstring dst(wcharBuf, wcslen(wcharBuf));
    delete[] wcharBuf;
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ワイド文字列（Unicode）から ANSI 文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetAnsiFromUnicode(const std::wstring& src)
{
    const int ansiSize = WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, NULL, 0, NULL, NULL);
    if (ansiSize == 0)
    {
        return std::string();
    }
    char* ansiBuf = new char[ansiSize];
    WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, ansiBuf, ansiSize, NULL, NULL);
    std::string dst(ansiBuf, strlen(ansiBuf));
    delete[] ansiBuf;
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列からフォルダのパス部分を抽出して返します。
//!        返り値の最後に / は付きません。
//-----------------------------------------------------------------------------
std::string GetFolderFromFilePath(const std::string& path)
{
    size_t delimIdx = std::string::npos;
    for (size_t charIdx = 0; charIdx < path.size(); ++charIdx)
    {
        const char c = path[charIdx];
        if (IsShiftJisFirstByte(static_cast<uint8_t>(c)))
        {
            ++charIdx; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            delimIdx = charIdx;
        }
    }
    return (delimIdx != std::string::npos) ? path.substr(0, delimIdx) : ".";
}

//-----------------------------------------------------------------------------
//! @brief ファイルが存在するなら true を返します。
//-----------------------------------------------------------------------------
bool FileExists(const std::string& path)
{
    WIN32_FIND_DATA ffd;
    HANDLE  hFindFile = FindFirstFileA(path.c_str(), &ffd);
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);
    if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
    {
        return true;
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//-----------------------------------------------------------------------------
std::string GetEnvVariable(const char* name)
{
    std::string ret;
    char* pValue;
    size_t length;
    if (_dupenv_s(&pValue, &length, name) == 0)
    {
        // 環境変数が見つからない場合は、pValue が nullptr に
        // length が 0、戻り値が 0（成功）となります。
        if (pValue != nullptr)
        {
            ret = pValue;
            free(pValue);
        }
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief Windows のシステムメッセージ文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetWindowsSystemMessage(const int messageId)
{
    void* pMessage = nullptr;
    FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM |
        FORMAT_MESSAGE_IGNORE_INSERTS, nullptr, messageId,
        MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), reinterpret_cast<LPSTR>(&pMessage), 0, nullptr);
    std::string message;
    if (pMessage != nullptr)
    {
        message = reinterpret_cast<const char*>(pMessage);
        LocalFree(pMessage);
        const size_t spaceIdx = message.find_last_not_of(" \t\r\n"); // 末尾の空白と改行を削除します。
        message = (spaceIdx != std::string::npos) ? message.substr(0, spaceIdx + 1) : "";
    }
    else
    {
        char strBuf[32];
        sprintf_s(strBuf, "0x%08X", messageId);
        message = strBuf;
    }
    return message;
}

//-----------------------------------------------------------------------------
//! @brief Windows の最後のエラーのメッセージ文字列を取得します。
//-----------------------------------------------------------------------------
std::string GetWindowsLastErrorMessage()
{
    return GetWindowsSystemMessage(GetLastError());
}

//-----------------------------------------------------------------------------
//! @brief プレーンな RGBA フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return プレーンな RGBA フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPlainRgbaFormat(const std::string& formatStr)
{
    return (formatStr == "unorm_8_8_8_8"     ||
            formatStr == "snorm_8_8_8_8"     ||
            formatStr == "srgb_8_8_8_8"      ||
            formatStr == "float_32_32_32_32");
}

//-----------------------------------------------------------------------------
//! @brief 浮動小数点数型フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 浮動小数点数型フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsFloatFormat(const std::string& formatStr)
{
    return (formatStr.find("float_" ) == 0 ||
            formatStr.find("ufloat_") == 0);
}

//-----------------------------------------------------------------------------
//! @brief 符号ありフォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 符号ありフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSignedFormat(const std::string& formatStr)
{
    return (formatStr.find("snorm_") == 0 ||
            formatStr.find("sint_" ) == 0 ||
            formatStr.find("float_") == 0);
}

//-----------------------------------------------------------------------------
//! @brief snorm フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return snorm フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSnormFormat(const std::string& formatStr)
{
    return (formatStr.find("snorm_") == 0);
}

//-----------------------------------------------------------------------------
//! @brief snorm または sint フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return snorm または sint フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSnormSintFormat(const std::string& formatStr)
{
    return (formatStr.find("snorm_") == 0 ||
            formatStr.find("sint_" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief sRGB フェッチのフォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return sRGB フェッチのフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsSrgbFetchFormat(const std::string& formatStr)
{
    return (formatStr.find("srgb_") == 0);
}

//-----------------------------------------------------------------------------
//! @brief BC1/BC2/BC3 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsBc123Format(const std::string& formatStr)
{
    return (formatStr.find("unorm_bc1") == 0 ||
            formatStr.find("unorm_bc2") == 0 ||
            formatStr.find("unorm_bc3") == 0 ||
            formatStr.find("srgb_bc1" ) == 0 ||
            formatStr.find("srgb_bc2" ) == 0 ||
            formatStr.find("srgb_bc3" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief BC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsBcFormat(const std::string& formatStr)
{
    return (formatStr.find("unorm_bc" ) == 0 ||
            formatStr.find("snorm_bc" ) == 0 ||
            formatStr.find("srgb_bc"  ) == 0 ||
            formatStr.find("ufloat_bc") == 0 ||
            formatStr.find("float_bc" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief ETC1 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ETC1 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsEtc1Format(const std::string& formatStr)
{
    return (formatStr.find("unorm_etc1") == 0);
}

//-----------------------------------------------------------------------------
//! @brief ETC2 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ETC2 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsEtc2Format(const std::string& formatStr)
{
    return (formatStr.find("unorm_etc2") == 0 ||
            formatStr.find("srgb_etc2" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief ETC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ETC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsEtcFormat(const std::string& formatStr)
{
    return (IsEtc1Format(formatStr) || IsEtc2Format(formatStr));
}

//-----------------------------------------------------------------------------
//! @brief EAC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return EAC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsEacFormat(const std::string& formatStr)
{
    return (formatStr.find("unorm_eac") == 0 ||
            formatStr.find("snorm_eac") == 0);
}

//-----------------------------------------------------------------------------
//! @brief PVRTC1 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return PVRTC1 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPvrtc1Format(const std::string& formatStr)
{
    return (formatStr.find("unorm_pvrtc1") == 0 ||
            formatStr.find("srgb_pvrtc1" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief PVRTC2 フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return PVRTC2 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPvrtc2Format(const std::string& formatStr)
{
    return (formatStr.find("unorm_pvrtc2") == 0 ||
            formatStr.find("srgb_pvrtc2" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief PVRTC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return PVRTC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsPvrtcFormat(const std::string& formatStr)
{
    return (IsPvrtc1Format(formatStr) || IsPvrtc2Format(formatStr));
}

//-----------------------------------------------------------------------------
//! @brief ASTC フォーマットなら true を返します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return ASTC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsAstcFormat(const std::string& formatStr)
{
    return (formatStr.find("unorm_astc") == 0 ||
            formatStr.find("srgb_astc" ) == 0);
}

//-----------------------------------------------------------------------------
//! @brief フォーマット文字列から ASTC ブロックのサイズを取得します。
//!
//! @param[out] pXdim ブロックの幅を格納します。
//! @param[out] pYdim ブロックの高さを格納します。
//! @param[out] pZdim ブロックの奥行きを格納します。
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool GetAstcBlockSize(int* pXdim, int* pYdim, int* pZdim, const std::string& formatStr)
{
    *pXdim = *pYdim = *pZdim = 0;
    size_t iAstc = formatStr.find("astc_");
    if (iAstc != std::string::npos)
    {
        const size_t iXdim = iAstc + strlen("astc_");
        const size_t iX1 = formatStr.find("x", iXdim);
        if (iXdim < iX1 && iX1 + 1 < formatStr.size())
        {
            const size_t iYdim = iX1 + 1;
            const size_t iX2 = formatStr.find("x", iYdim);
            *pXdim = atoi(formatStr.substr(iXdim, iX1 - iXdim).c_str());
            *pYdim = atoi(formatStr.substr(iYdim,
                (iX2 != std::string::npos) ? iX2 - iYdim : std::string::npos).c_str());
            if (iX2 != std::string::npos && iX2 + 1 < formatStr.size())
            {
                *pZdim = atoi(formatStr.substr(iX2 + 1).c_str());
            }
            else
            {
                *pZdim = 1;
            }
        }
    }
    return (*pXdim != 0 && *pYdim != 0 && *pZdim != 0);
}

//-----------------------------------------------------------------------------
//! @brief ASTC フォーマットにエンコードした際のデータサイズを取得します。
//!
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] xdim ブロックの幅です。
//! @param[in] ydim ブロックの高さです。
//! @param[in] zdim ブロックの奥行きです。
//!
//! @return データサイズを返します。
//-----------------------------------------------------------------------------
size_t GetAstcDataSize(
    const int imageW,
    const int imageH,
    const int imageD,
    const int xdim,
    const int ydim,
    const int zdim
)
{
    if (xdim != 0 && ydim != 0 && zdim != 0)
    {
        const int xblocks = (imageW + xdim - 1) / xdim;
        const int yblocks = (imageH + ydim - 1) / ydim;
        const int zblocks = (imageD + zdim - 1) / zdim;
        return static_cast<size_t>(xblocks) * yblocks * zblocks * 16;
    }
    else
    {
        return 0;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャフォーマットから成分数を取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 成分数を返します。
//-----------------------------------------------------------------------------
int GetComponentCount(const std::string& formatStr)
{
    if (formatStr == "unorm_8"      ||
        formatStr == "uint_8"       ||
        formatStr == "snorm_8"      ||
        formatStr == "sint_8"       ||
        formatStr == "unorm_16"     ||
        formatStr == "snorm_16"     ||
        formatStr == "uint_16"      ||
        formatStr == "sint_16"      ||
        formatStr == "uint_32"      ||
        formatStr == "sint_32"      ||
        formatStr == "float_16"     ||
        formatStr == "float_32"     ||
        formatStr == "unorm_bc4"    ||
        formatStr == "snorm_bc4"    ||
        formatStr == "unorm_eac_11" ||
        formatStr == "snorm_eac_11")
    {
        return 1;
    }
    else if (formatStr == "unorm_4_4"       ||
             formatStr == "unorm_8_8"       ||
             formatStr == "uint_8_8"        ||
             formatStr == "snorm_8_8"       ||
             formatStr == "sint_8_8"        ||
             formatStr == "unorm_16_16"     ||
             formatStr == "snorm_16_16"     ||
             formatStr == "uint_16_16"      ||
             formatStr == "sint_16_16"      ||
             formatStr == "uint_32_32"      ||
             formatStr == "sint_32_32"      ||
             formatStr == "float_16_16"     ||
             formatStr == "float_32_32"     ||
             formatStr == "unorm_bc5"       ||
             formatStr == "snorm_bc5"       ||
             formatStr == "unorm_eac_11_11" ||
             formatStr == "snorm_eac_11_11")
    {
        return 2;
    }
    else if (formatStr == "unorm_5_6_5"    ||
             formatStr == "uint_32_32_32"  ||
             formatStr == "sint_32_32_32"  ||
             formatStr == "float_11_11_10" ||
             formatStr == "float_32_32_32" ||
             formatStr == "ufloat_bc6"     ||
             formatStr == "float_bc6"      ||
             formatStr == "unorm_etc1"     ||
             formatStr == "unorm_etc2"     ||
             formatStr == "srgb_etc2")
    {
        return 3;
    }
    else
    {
        return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief 1 ピクセル当たりのビット数を取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return 1 ピクセル当たりのビット数を返します。
//!         ASTC フォーマットの場合は 0 を返します。
//-----------------------------------------------------------------------------
size_t GetBitsPerPixel(const std::string& formatStr)
{
    if (IsBcFormat(formatStr))
    {
        return (
            formatStr.find("_bc1") != std::string::npos ||
            formatStr.find("_bc4") != std::string::npos) ? 4 : 8;
    }
    else if (IsEtc1Format(formatStr))
    {
        return 4;
    }
    else if (IsEtc2Format(formatStr))
    {
        return (formatStr.find("_alpha") != std::string::npos) ? 8 : 4;
    }
    else if (IsEacFormat(formatStr))
    {
        return (formatStr.find("_11_11") != std::string::npos) ? 8 : 4;
    }
    else if (IsPvrtcFormat(formatStr))
    {
        return (formatStr.find("_2bpp") != std::string::npos) ? 2 : 4;
    }
    else if (IsAstcFormat(formatStr))
    {
        return 0;
    }
    else if (
        formatStr == "unorm_8"   ||
        formatStr == "snorm_8"   ||
        formatStr == "uint_8"    ||
        formatStr == "sint_8"    ||
        formatStr == "unorm_4_4")
    {
        return 8;
    }
    else if (
        formatStr == "unorm_16"      ||
        formatStr == "snorm_16"      ||
        formatStr == "uint_16"       ||
        formatStr == "sint_16"       ||
        formatStr == "unorm_8_8"     ||
        formatStr == "snorm_8_8"     ||
        formatStr == "uint_8_8"      ||
        formatStr == "sint_8_8"      ||
        formatStr == "unorm_5_6_5"   ||
        formatStr == "unorm_5_5_5_1" ||
        formatStr == "unorm_4_4_4_4" ||
        formatStr == "float_16")
    {
        return 16;
    }
    else if (
        formatStr == "uint_32"          ||
        formatStr == "sint_32"          ||
        formatStr == "unorm_16_16"      ||
        formatStr == "snorm_16_16"      ||
        formatStr == "uint_16_16"       ||
        formatStr == "sint_16_16"       ||
        formatStr == "unorm_8_8_8_8"    ||
        formatStr == "snorm_8_8_8_8"    ||
        formatStr == "srgb_8_8_8_8"     ||
        formatStr == "uint_8_8_8_8"     ||
        formatStr == "sint_8_8_8_8"     ||
        formatStr == "unorm_10_10_10_2" ||
        formatStr == "uint_10_10_10_2"  ||
        formatStr == "float_32"         ||
        formatStr == "float_16_16"      ||
        formatStr == "float_11_11_10")
    {
        return 32;
    }
    else if (
        formatStr == "uint_32_32"        ||
        formatStr == "sint_32_32"        ||
        formatStr == "unorm_16_16_16_16" ||
        formatStr == "snorm_16_16_16_16" ||
        formatStr == "uint_16_16_16_16"  ||
        formatStr == "sint_16_16_16_16"  ||
        formatStr == "float_32_32"       ||
        formatStr == "float_16_16_16_16")
    {
        return 64;
    }
    else if (
        formatStr == "uint_32_32_32"  ||
        formatStr == "sint_32_32_32"  ||
        formatStr == "float_32_32_32")
    {
        return 96;
    }
    else if (
        formatStr == "uint_32_32_32_32"  ||
        formatStr == "sint_32_32_32_32"  ||
        formatStr == "float_32_32_32_32")
    {
        return 128;
    }
    return 0;
}

//-----------------------------------------------------------------------------
//! @brief フォーマットに対する最小の幅／高さ／奥行きを取得します。
//!
//! @param[out] pMinW 最小の幅を格納します。
//! @param[out] pMinH 最小の高さを格納します。
//! @param[out] pMinD 最小の奥行きを格納します。
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//-----------------------------------------------------------------------------
void GetMinimumWhd(int* pMinW, int* pMinH, int* pMinD, const std::string& formatStr)
{
    *pMinD = 1;

    if (IsAstcFormat(formatStr))
    {
        GetAstcBlockSize(pMinW, pMinH, pMinD, formatStr);
    }
    else if (IsBcFormat(formatStr)  ||
             IsEtcFormat(formatStr) ||
             IsEacFormat(formatStr) ||
             formatStr.find("pvrtc2_alpha_4bpp") != std::string::npos)
    {
        *pMinW = 4;
        *pMinH = 4;
    }
    else if (formatStr.find("pvrtc1_2bpp"      ) != std::string::npos ||
             formatStr.find("pvrtc1_alpha_2bpp") != std::string::npos)
    {
        *pMinW = 16;
        *pMinH = 8;
    }
    else if (formatStr.find("pvrtc1_4bpp"      ) != std::string::npos ||
             formatStr.find("pvrtc1_alpha_4bpp") != std::string::npos)
    {
        *pMinW = 8;
        *pMinH = 8;
    }
    else if (formatStr.find("pvrtc2_alpha_2bpp") != std::string::npos)
    {
        *pMinW = 8;
        *pMinH = 4;
    }
    else
    {
        *pMinW = 1;
        *pMinH = 1;
    }
}

//-----------------------------------------------------------------------------
//! @brief 1 レベルのデータサイズ（バイト数）を取得します。
//!
//! @param[in] levelW 対象レベルの幅です。
//! @param[in] levelH 対象レベルの高さです。
//! @param[in] levelD 対象レベルの奥行きです。
//! @param[in] isAstc ASTC フォーマットなら true です。
//! @param[in] bpp 1 ピクセル当たりのビット数です。
//! @param[in] minW フォーマットに対する最小の幅です。
//! @param[in] minH フォーマットに対する最小の高さです。
//! @param[in] minD フォーマットに対する最小の奥行きです。
//!
//! @return 1 レベルのデータサイズを返します。
//-----------------------------------------------------------------------------
size_t GetLevelDataSize(
    const int levelW,
    const int levelH,
    const int levelD,
    const bool isAstc,
    const size_t bpp,
    const int minW,
    const int minH,
    const int minD
)
{
    if (isAstc)
    {
        return GetAstcDataSize(levelW, levelH, levelD, minW, minH, minD);
    }
    else
    {
        const int alignedW = AlignValue(levelW, minW);
        const int alignedH = AlignValue(levelH, minH);
        return bpp * alignedW * alignedH * levelD / 8;
    }
}

//-----------------------------------------------------------------------------
//! @brief float 型の値から float_16 の値を取得します。
//-----------------------------------------------------------------------------
uint16_t GetFloat16Value(const float valueF32)
{
    const float f = valueF32;

    // 以下、OpenEXR の half.h の half::half(float f) からコピーしたコードです。

    union uif
    {
        uint32_t i;
        float f;
    };

    uif x;
    x.f = f;

    uint16_t hv = 0;
    if (f == 0)
    {
        hv = static_cast<uint16_t>(x.i >> 16);
    }
    else
    {
        int e = (x.i >> 23) & 0x000001ff;
        e = Float16ELut[e];
        if (e)
        {
            int m = x.i & 0x007fffff;
            hv = static_cast<uint16_t>(e + ((m + 0x00000fff + ((m >> 13) & 1)) >> 13));
        }
        else
        {
            hv = ConvertToFloat16Value(static_cast<int>(x.i));
        }
    }
    return hv;
}

//-----------------------------------------------------------------------------
//! @brief float_16 の値から float 型の値を取得します。
//!
//! @param[in] valueF16 float_16 の値です。
//!
//! @return float 型の値を返します。
//-----------------------------------------------------------------------------
float GetFloat32Value(const uint16_t valueF16)
{
    uint32_t ival = valueF16;
    if (ival == 0)
    {
        return 0.0f;
    }
    uint32_t s =   ival & 0x8000;
    int      e = ((ival & 0x7c00) >>10) - 15 + 127;
    uint32_t f =   ival & 0x03ff;
    uint32_t fval = (s << 16) | ((e << 23) & 0x7f800000) | (f << 13);
    return  *reinterpret_cast<float*>(&fval);
}

//-----------------------------------------------------------------------------
//! @brief エンコード品質文字列からエンコードレベルを取得します。
//!
//! @param[in] qualityStr エンコード品質文字列です。
//!
//! @return エンコードレベルを返します。
//-----------------------------------------------------------------------------
int GetQualityLevel(const std::string& qualityStr)
{
    // エンコード品質文字列の先頭が数値なら、それをエンコードレベルとします。
    int digitCount = 0;
    for (size_t charIdx = 0; charIdx < qualityStr.size(); ++charIdx)
    {
        const char c = qualityStr[charIdx];
        if ('0' <= c && c <= '9')
        {
            ++digitCount;
        }
        else
        {
            break;
        }
    }
    return (digitCount > 0) ? atoi(qualityStr.substr(0, digitCount).c_str()) : 1;
}

//-----------------------------------------------------------------------------
//! @brief RGB 成分にディザを使用するべき画像かチェックします。
//!
//! @param[in] pPixels ピクセルデータです。
//!                    1 ピクセルあたり 4 バイトで、RGBA の順に格納します。画像の左上が先頭です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return RGB 成分にディザを使用するべき画像なら true を返します。
//-----------------------------------------------------------------------------
bool CheckRgbDitherIsNeeded(
    const void* pPixels,
    const int imageW,
    const int imageH,
    const int imageD
)
{
    const int MinSize   =   8; // 幅または高さがこのサイズ未満ならディザを使用しません。
    const int ValidSize = 256; // 幅または高さがこのサイズ以上ならピクセルデータに応じてディザを使用します。

    const int SmoothDiff =  1; // ゆるやかな変化とみなす RGB 値の差分の最大値です。
    const int SharpDiff  = 64; // 急激な変化とみなす RGB 値の差分の最小値です。
    const int SmoothThreshold = 50; // ゆるやかな変化の領域がこの値以上ならディザを使用します。
    const int SharpThreshold  = 20; // 急激な変化の領域がこの値以上ならディザを使用しません。

    //-----------------------------------------------------------------------------
    // サイズをチェックします。
    if (imageW < MinSize ||
        imageH < MinSize)
    {
        return false;
    }
    if (imageW < ValidSize &&
        imageH < ValidSize)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // ピクセルデータをチェックします。
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pPixels);
    const size_t pixelBytes = EncRgbaBytes;
    const size_t faceBytes = pixelBytes * imageW * imageH;
    for (int z = 0; z < imageD; ++z)
    {
        int sharpPercent = 0;
        int flatPercent = 0;
        const int smoothPercent = CalculateSmoothRgbPercent(&sharpPercent, &flatPercent,
            pSrc, imageW, imageH, SmoothDiff, SharpDiff);
        const bool isNeeded = (smoothPercent >= SmoothThreshold &&
            sharpPercent < SharpThreshold);
        //cerr << "check dither: " << isNeeded << ": " << smoothPercent << " " << sharpPercent << " " << flatPercent << endl;
        if (isNeeded)
        {
            // いずれかのフェースで、ゆるやかにカラーが変化する領域が広ければ、ディザを使用します。
            return true;
        }
        pSrc += faceBytes;
    }

    return false;
}

