﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include "BcFormat.h"
#include <DirectXTex.h> // 内部で d3d11_1.h をインクルード

using namespace std;
using namespace nn::gfx::tool::texenc;
using namespace DirectX;

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//-----------------------------------------------------------------------------
//! @brief DirectXTex に渡すフォーマットを取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//!
//! @return フォーマットを返します。
//-----------------------------------------------------------------------------
DXGI_FORMAT GetDxgiFormat(const std::string& formatStr)
{
    return
        (formatStr == "unorm_8_8_8_8"    ) ? DXGI_FORMAT_R8G8B8A8_UNORM      :
        (formatStr == "snorm_8_8_8_8"    ) ? DXGI_FORMAT_R8G8B8A8_SNORM      :
        (formatStr == "srgb_8_8_8_8"     ) ? DXGI_FORMAT_R8G8B8A8_UNORM_SRGB :
        (formatStr == "unorm_bc1"        ) ? DXGI_FORMAT_BC1_UNORM           :
        (formatStr == "srgb_bc1"         ) ? DXGI_FORMAT_BC1_UNORM_SRGB      :
        (formatStr == "unorm_bc2"        ) ? DXGI_FORMAT_BC2_UNORM           :
        (formatStr == "srgb_bc2"         ) ? DXGI_FORMAT_BC2_UNORM_SRGB      :
        (formatStr == "unorm_bc3"        ) ? DXGI_FORMAT_BC3_UNORM           :
        (formatStr == "srgb_bc3"         ) ? DXGI_FORMAT_BC3_UNORM_SRGB      :
        (formatStr == "unorm_bc4"        ) ? DXGI_FORMAT_BC4_UNORM           :
        (formatStr == "snorm_bc4"        ) ? DXGI_FORMAT_BC4_SNORM           :
        (formatStr == "unorm_bc5"        ) ? DXGI_FORMAT_BC5_UNORM           :
        (formatStr == "snorm_bc5"        ) ? DXGI_FORMAT_BC5_SNORM           :
        (formatStr == "ufloat_bc6"       ) ? DXGI_FORMAT_BC6H_UF16           :
        (formatStr == "float_bc6"        ) ? DXGI_FORMAT_BC6H_SF16           :
        (formatStr == "unorm_bc7"        ) ? DXGI_FORMAT_BC7_UNORM           :
        (formatStr == "srgb_bc7"         ) ? DXGI_FORMAT_BC7_UNORM_SRGB      :
        (formatStr == "float_16_16_16_16") ? DXGI_FORMAT_R16G16B16A16_FLOAT  :
        (formatStr == "float_32_32_32_32") ? DXGI_FORMAT_R32G32B32A32_FLOAT  :
        DXGI_FORMAT_UNKNOWN;
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief DirectXTex でフォーマットを変換します。
//!
//! @param[in,out] pIsOpenMpUsed NULL でなければ OpenMP を使用した場合に true を格納します。
//!                              使用しなかった場合は値を変更しません。
//!                              NULL でなく値が true なら小さいサイズでも OpenMP を使用します。
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] qualityStr エンコード品質です。
//! @param[in] encodeFlag エンコードフラグです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ConvertByDirectXTex(
    bool* pIsOpenMpUsed,
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const std::string& srcFormatStr,
    const std::string& qualityStr,
    const int encodeFlag,
    const int imageW,
    const int imageH,
    const int imageD
)
{
    //-----------------------------------------------------------------------------
    // 変換前後で unorm / snorm が異なる場合は変換しません。
    // unorm_8_8_8_8 -> snorm_bc5 など
    if (IsFloatFormat(srcFormatStr) && IsSnormFormat(dstFormatStr))
    {
        // float -> snorm は OK
    }
    else if (IsSnormFormat(srcFormatStr) && IsFloatFormat(dstFormatStr))
    {
        // snorm -> float は OK
    }
    else if (IsSnormSintFormat(srcFormatStr) != IsSnormSintFormat(dstFormatStr))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // DirectXTex に渡すフォーマットを取得します。
    const DXGI_FORMAT srcFormat = GetDxgiFormat(srcFormatStr);
    const DXGI_FORMAT dstFormat = GetDxgiFormat(dstFormatStr);
    if (srcFormat == DXGI_FORMAT_UNKNOWN ||
        dstFormat == DXGI_FORMAT_UNKNOWN)
    {
        return false;
    }
    if (!IsCompressed(srcFormat) &&
        !IsCompressed(dstFormat))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // 画像を作成します。
    ScratchImage srcImage;
    const size_t mipLevels = 1;
    srcImage.Initialize2D(srcFormat, imageW, imageH, imageD, mipLevels);
        // ↑内部でピクセルデータのメモリが確保されます。
    memcpy(srcImage.GetPixels(), pSrc, srcImage.GetPixelsSize());

    //-----------------------------------------------------------------------------
    // フォーマットを変換します。
    ScratchImage dstImage;
    HRESULT hr;
    if (IsCompressed(dstFormat))
    {
        DWORD compressFlag = TEX_COMPRESS_DEFAULT;

        const bool isBc67 = (
            dstFormatStr.find("_bc6") != std::string::npos ||
            dstFormatStr.find("_bc7") != std::string::npos);
        size_t mpMinWH = (isBc67) ? 64 : 4096;
            // ↑現在、OpenMP を使用した場合、プロセスデタッチ時に
            // 一定時間スリープするので、スリープ時間を含めても OpenMP 使用のほうが
            // 速いほど大きなサイズであれば使用します。
        if (pIsOpenMpUsed != NULL && *pIsOpenMpUsed)
        {
            // すでに OpenMP が使用されていれば、小さいサイズでも OpenMP を使用します。
            mpMinWH = 16;
        }
        if (static_cast<size_t>(imageW) * imageH >= mpMinWH * mpMinWH)
        {
            compressFlag |= TEX_COMPRESS_PARALLEL; // マルチスレッドでエンコード
            if (pIsOpenMpUsed != NULL)
            {
                *pIsOpenMpUsed = true;
            }
        }

        if (IsSRGB(srcFormat) || IsSRGB(dstFormat))
        {
            // 入力と出力のカラー空間が異なる場合にガンマ変換されるのを防ぐため、
            // 入力または出力が sRGB フェッチなら、
            // 入力も出力も sRGB フェッチとして変換するように指定します。
            compressFlag |= TEX_COMPRESS_SRGB;
        }

        if (qualityStr.find("perceptual") == std::string::npos)
        {
            compressFlag |= TEX_COMPRESS_UNIFORM; // bc1/2/3 で RGB の重み付けなし
        }
        if (GetQualityLevel(qualityStr) >= 1)
        {
            compressFlag |= TEX_COMPRESS_BC7_USE_3SUBSETS; // bc7 で 3 分割パーティションを使用
        }
        if (IsBc123Format(dstFormatStr) && !IsFloatFormat(srcFormatStr))
        {
            if (((encodeFlag & EncodeFlag_Dither    ) != 0) ||
                ((encodeFlag & EncodeFlag_AutoDither) != 0 &&
                 CheckRgbDitherIsNeeded(pSrc, imageW, imageH, imageD)))
            {
                compressFlag |= TEX_COMPRESS_RGB_DITHER; // bc1/2/3 で RGB をディザリング
                //cerr << "bc1/2/3 use dither" << endl;
            }
        }
        hr = Compress(srcImage.GetImages(), srcImage.GetImageCount(), srcImage.GetMetadata(),
            dstFormat, compressFlag, 0.5f, dstImage);
    }
    else
    {
        // 入力と出力のカラー空間が異なる場合にガンマ変換されるのを防ぐため、
        // 出力フォーマットを調整します。
        DXGI_FORMAT decompFormat = dstFormat;
        if (IsSRGB(srcFormat))
        {
            decompFormat = DXGI_FORMAT_R8G8B8A8_UNORM_SRGB;
        }

        hr = Decompress(srcImage.GetImages(), srcImage.GetImageCount(), srcImage.GetMetadata(),
            decompFormat, dstImage);
    }
    srcImage.Release();
    if (FAILED(hr))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // データをコピーします。
    memcpy(pDst, dstImage.GetPixels(), dstImage.GetPixelsSize());
    dstImage.Release();

    return true;
}

