﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iterator>
#include <gfxTool_CompileArg.h>
#include <gfxTool_VariationXml.h>
#include <gfxTool_CommandLineArg.h>
#include <gfxTool_VariationDefinitionArg.h>
#include <gfxTool_VariationValueArg.h>
#include <gfxTool_CompileOption.h>

namespace nn {
namespace gfxTool {

CompileArg::CompileArg()
    : nngfxToolShaderCompilerCompileArg()
{
}

CompileArg::~CompileArg()
{
}

void CompileArg::Initialize( const nngfxToolShaderConverterCreateConvertArgArg* pArg,
        const CommandLineArg* pInput )
{
    static nngfxToolString ( nngfxToolShaderCompilerCompileArg::*pStageSource[] ) =
    {
        &nngfxToolShaderCompilerCompileArg::vertexShaderSource,
        &nngfxToolShaderCompilerCompileArg::hullShaderSource,
        &nngfxToolShaderCompilerCompileArg::domainShaderSource,
        &nngfxToolShaderCompilerCompileArg::geometryShaderSource,
        &nngfxToolShaderCompilerCompileArg::pixelShaderSource,
        &nngfxToolShaderCompilerCompileArg::computeShaderSource
    };

    for( int stage = 0; stage < static_cast< int >( ShaderStage::End ); ++stage )
    {
        if( pInput->shaderFile[ stage ].IsExisting() )
        {
            void* pFileData = nullptr;
            size_t fileSize;
            if( !pArg->pReadInputFileCallback( &pFileData, &fileSize,
                pInput->shaderFile[ stage ].GetValue().data(), pArg->pReadInputFileCallbackParam ) )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_FailedToLoadFile,
                    "%s", pInput->shaderFile[ stage ].GetValue().data() );
            }
            auto pStart = static_cast< const char* >( pFileData );
            if( IsBom( pStart ) )
            {
                pStart += 3;
                fileSize -= 3;
            }
            auto& stageSource = this->*pStageSource[ stage ];
            stageSource.pValue = pStart;
            stageSource.length = NumericCastAuto( fileSize );
        }
    }

    m_ReadIncludeFile.SetCallback(
        pArg->pReadIncludeFileCallback, pArg->pReadIncludeFileCallbackParam );
    for( auto& directory : pInput->includeDirectory.GetValue() )
    {
        m_ReadIncludeFile.AddDirectory( directory.c_str() );
    }
    this->pReadIncludeFileCallback =
        pInput->includeDirectory.IsExisting() ? ReadFileCallback::Callback : nullptr;
    this->pReadIncludeFileCallbackParam = &m_ReadIncludeFile;

    m_WriteDebugInfoFile.SetCallback(
        pArg->pWriteDebugInfoFileCallback, pArg->pWriteDebugInfoFileCallbackParam );
    m_WriteDebugInfoFile.SetDirectory( pInput->debugInfoDirectory.GetValue().c_str() );
    this->pWriteDebugInfoFileCallback =
        pInput->debugInfoDirectory.IsExisting() ? WriteFileCallback::Callback : nullptr;
    this->pWriteDebugInfoFileCallbackParam = &m_WriteDebugInfoFile;

    m_ReadShaderCache.SetCallback(
        pArg->pReadShaderCacheCallback, pArg->pReadShaderCacheCallbackParam );
    m_ReadShaderCache.AddDirectory( pInput->shaderCacheDirectory.GetValue().c_str() );
    this->pReadShaderCacheCallback =
        pInput->shaderCacheDirectory.IsExisting() ? ReadFileCallback::Callback : nullptr;
    this->pReadShaderCacheCallbackParam = &m_ReadShaderCache;
    m_WriteShaderCache.SetCallback(
        pArg->pWriteShaderCacheCallback, pArg->pWriteShaderCacheCallbackParam );
    m_WriteShaderCache.SetDirectory( pInput->shaderCacheDirectory.GetValue().c_str() );
    this->pWriteShaderCacheCallback =
        pInput->shaderCacheDirectory.IsExisting() ? WriteFileCallback::Callback : nullptr;
    this->pWriteShaderCacheCallbackParam = &m_WriteShaderCache;

    this->shaderSourceFormat = StaticCastAuto( pInput->sourceFormat.GetValue() );
    this->targetLowLevelApiType = StaticCastAuto( pInput->apiType.GetValue() );
    // TODO
    if( this->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Gl )
    {
        this->targetLowLevelApiVersion = 4;
    }
    else if( this->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Gx )
    {
        this->targetLowLevelApiVersion = 2;
    }
    else if( this->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_D3d )
    {
        this->targetLowLevelApiVersion = 11;
    }
    else if( this->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Vk )
    {
        this->targetLowLevelApiVersion = 1;
    }

    this->targetCodeType = StaticCastAuto( pInput->codeType.GetValue() );

    m_pCompileOption.reset( new CompileOption() );
    m_pCompileOption->Initialize( pInput );
    this->optionCount = NumericCastAuto( m_pCompileOption->GetCompileOptionArray().size() );
    this->pOptionArray = &m_pCompileOption->GetCompileOptionArray()[ 0 ];

    InitializeVariation( pArg, pInput );
}

void CompileArg::InitializeVariation( const nngfxToolShaderConverterCreateConvertArgArg* pArg,
        const CommandLineArg* pInput )
{
    this->variationCount = 1;

    VariationXml variationXml;
    decltype( variationXml.GetGfxShaderVariation() ) pRoot = nullptr;
    void* pFileData = nullptr;
    if( !pInput->variationFile.IsExisting() )
    {
        return;
    }

    size_t fileSize;
    if( !pArg->pReadInputFileCallback( &pFileData, &fileSize,
        pInput->variationFile.GetValue().data(), pArg->pReadInputFileCallbackParam ) )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_FailedToLoadFile,
            "%s", pInput->variationFile.GetValue().data() );
    }

    // TODO: NULL 終端要求のため無駄なコピーが発生しているので対策する
    Custom< std::vector< char > >::Type fileString;
    fileString.reserve( fileSize + 1 );
    std::copy( static_cast< const char* >( pFileData ),
        static_cast< const char* >( pFileData ) + fileSize, std::back_inserter( fileString ) );
    fileString.push_back( '\0' );
    if( !variationXml.Parse( &fileString[ 0 ] ) )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlParseError, "%s: %s\n",
            variationXml.GetErrorPosition().data(), variationXml.GetErrorString().data() );
    }
    pRoot = variationXml.GetGfxShaderVariation();
    if( pRoot == nullptr )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound,
            "Xml root not found" );
    }
    if( pRoot->GetName().compare( pRoot->GetId() ) )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlParseError,
            "Root is not %s", pRoot->GetId() );
    }
    auto version = pRoot->GetVersion();
    if( version.empty() )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlAttributeNotFound,
            "%s::version", pRoot->GetId() );
    }
    // TODO: バージョン値を見る

    m_pVariationDefinitionArg.reset( new VariationDefinitionArg() );
    auto pShaderVariationDefinitionNode = pRoot->GetShaderVariationDefinition();
    if( pShaderVariationDefinitionNode == nullptr )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s is not found.",
            pShaderVariationDefinitionNode->GetId() );
    }
    m_pVariationDefinitionArg->Initialize( pShaderVariationDefinitionNode );
    this->pVariationDefinitionArg = m_pVariationDefinitionArg.get();

    auto pShaderVariationValueArrayNode = pRoot->GetShaderVariationValueArrayNode();
    if( pShaderVariationValueArrayNode == nullptr )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s is not found.",
            pShaderVariationValueArrayNode->GetId() );
    }
    auto length = pShaderVariationValueArrayNode->GetLengthAttribute();
    if( length.empty() )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlAttributeNotFound, "%s::length is not found.",
            pShaderVariationValueArrayNode->GetId() );
    }
    this->variationCount = NumericCastAuto( LexicalCast< int >( length.data() ) );
    m_VariationValueArrayRaw.resize( this->variationCount );
    m_VariationValueArray.resize( m_VariationValueArrayRaw.size() );
    decltype( pShaderVariationValueArrayNode->GetNextChild( nullptr ) )
        pShaderVariationValueNode = nullptr;
    for( int idxVariation = 0, variationCountMax = NumericCastAuto( m_VariationValueArray.size() );
        idxVariation < variationCountMax; ++idxVariation )
    {
        pShaderVariationValueNode =
            pShaderVariationValueArrayNode->GetNextChild( pShaderVariationValueNode );
        if( pShaderVariationValueNode == nullptr )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_XmlNodeNotFound, "%s is not found.",
                pShaderVariationValueNode->GetId() );
        }
        m_VariationValueArray[ idxVariation ].Initialize( &m_VariationValueArrayRaw[ idxVariation ],
            pShaderVariationValueNode, m_pVariationDefinitionArg.get() );
    }
    this->pVariationValueArray = &m_VariationValueArrayRaw[ 0 ];
}

}
}
